# Copyright (c) 2006-2011, 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import signal
import sys
import CliInputInterface
import LoggingDefs
from _TacUtils import setpdeathsig, threadsigmask
import Tracing

th = Tracing.Handle( "LoggingCli" )

t0 = th.trace0
t1 = th.trace1

# Logging to a terminal is implemented by tailing the output
# of the file /var/log/eos-console to the stdout of the current
# terminal. The tail command runs in a child which
# is created for the console Cli session and by the 'terminal monitor'
# command, and is killed with a SIGTERM whenever the Cli process exits,
# courtesy of setpdeathsig.
loggingSubprocess = None

def monLogSubprocess( logFilename ): # pylint: disable=inconsistent-return-statements
   pid = os.fork()
   if pid == 0:
      # this saves a lot of memory, compared to ManagedSubprocess
      setpdeathsig( signal.SIGTERM )
      # Needed as this is masked in CliShell's CliInputThread
      threadsigmask( signal.SIGTERM, False )
      # avoid being killed by Ctrl-C
      signal.signal( signal.SIGINT, signal.SIG_IGN )
      fd = os.open( '/dev/null', os.O_RDWR )
      os.dup2( fd, 0 )
      os.execvp( 'LogMonUtil', [ 'LogMonUtil', logFilename ] )
   else:
      return pid

def enableLoggingMonitor( ):
   global loggingSubprocess
   if not loggingSubprocess:
      logfile = LoggingDefs.monitorLogFilename
      loggingSubprocess = monLogSubprocess( logfile )

def sendSignalToPid( pid, sig ):
   if not pid:
      return
   try:
      os.kill( pid, sig )
      os.waitpid( pid, 0 )
   except OSError:
      # If pid doesn't exist, then do nothing.
      pass

# Kill the process we started to run tail by sending it a SIGTERM.  We don't
# bother to wait for it, because we don't want to get stuck here if something
# goes wrong.
def disableLoggingMonitor():
   global loggingSubprocess
   sendSignalToPid( loggingSubprocess, signal.SIGTERM )
   loggingSubprocess = None

# keep in sync with LogMon.h
LOGMON_TRUNCATED = 1
LOGMON_REOPENED = 2

class LoggingSynchronousMonitor:
   # When logging synchronous is enabled, this is a callback object from CliInput
   # to synchronous logging vs input.

   def __init__( self, filename ):
      self.maxlines = LoggingDefs.maxSyncMessages
      self.filename = filename
      self.file = None

   def _getLogs( self, fd ):
      # return up to maxlines of lines
      blockSize = 4096
      allLines = []
      prevLine = b''
      while len( allLines ) < self.maxlines:
         data = os.read( fd, blockSize )
         t1( "read: '", data, "'" )
         lines = data.splitlines( True )
         if not lines:
            break

         lines[ 0 ] = prevLine + lines[ 0 ]
         prevLine = b''

         if not lines[ -1 ].endswith( b'\n' ):
            # incomplete line, save for next
            prevLine = lines.pop()

         allLines.extend( lines[ :self.maxlines - len( allLines ) ] )

         if len( data ) < blockSize:
            break

      if prevLine and len( allLines ) < self.maxlines:
         allLines.append( prevLine + b'\n' )

      t0( "read", len( allLines ), "lines" )
      return [ l.decode() for l in allLines ]

   def __call__( self, fd, flags ):
      t0( "__call__", fd, flags )
      note = ''
      lines = []
      if flags & LOGMON_TRUNCATED:
         t0( "file truncated" )
         note = LoggingDefs.consoleLogTruncatedMsg + '\n'
      else:
         lines = self._getLogs( fd )
         if len( lines ) == self.maxlines:
            note = LoggingDefs.consoleSyncLogSkippedMsg + '\n'
            os.lseek( fd, 0, os.SEEK_END )

      if note:
         lines.append( note )
         lines.append( LoggingDefs.consoleSyncLogRunCmdMsg + '\n' )

      if lines:
         # first start with a newline
         sys.stdout.write( '\n\r' )
      for line in lines:
         # '\r' is needed for raw mode
         sys.stdout.write( line + '\r' )
      sys.stdout.flush()
      return len( lines )

loggingSynchronousMonitor = None

def enableLoggingSync( console ):
   global loggingSynchronousMonitor
   if not loggingSynchronousMonitor:
      if console:
         logfile = LoggingDefs.consoleSyncLogFilename
      else:
         logfile = LoggingDefs.monitorSyncLogFilename
      loggingSynchronousMonitor = LoggingSynchronousMonitor( logfile )
      CliInputInterface.loggingSyncEnabledIs( logfile,
                                              loggingSynchronousMonitor )

def disableLoggingSync():
   global loggingSynchronousMonitor
   CliInputInterface.loggingSyncEnabledIs( "", None )
   loggingSynchronousMonitor = None

def Plugin( context ):
   CliInputInterface.registerMethod( 'enableLoggingSync',
                                     enableLoggingSync )
   CliInputInterface.registerMethod( 'disableLoggingSync',
                                     disableLoggingSync )
   CliInputInterface.registerMethod( 'enableLoggingMonitor',
                                     enableLoggingMonitor )
   CliInputInterface.registerMethod( 'disableLoggingMonitor',
                                     disableLoggingMonitor )
