#!/usr/bin/env python3
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# This replaces Logging and Tracing to avoid importing Tac.py

import datetime
import fnmatch
import os
import sys

def _traceFunc( level ):
   # pylint: disable-next=consider-using-f-string,eval-used
   fn = eval( "lambda self, *msg: self._trace( %r, *msg )" % level )
   fn.__name__ = 'trace' + level
   # pylint: disable-next=consider-using-f-string
   fn.__doc__ = 'Traces a message at level %s using this trace handle.' % level
   return fn

class Handle:
   '''A simple version of Tracing.Handle, enabled by exact match in
   in TRACE env'''
   def __init__( self, facilityName ):
      self.facilityName_ = facilityName
      self.levelEnabled_ = set()
      self.file_ = sys.stderr
      self.lineEnd_ = '\r\n'
      self._setEnabled()

   def _setEnabled( self ):
      tracefile = ( os.environ.get( 'CLISHELL_TRACEFILE' ) or
                    os.environ.get( 'TRACEFILE' ) )
      if tracefile:
         try:
            self.file_ = open( tracefile, 'a' ) # pylint: disable=consider-using-with
         except OSError as e:
            print( f"Cannot open file {tracefile}: {e.strerror}",
                   file=sys.stderr, end=self.lineEnd_ )
         else:
            self.lineEnd_ = '\n'
      for s in os.environ.get( 'TRACE', '' ).split( ',' ):
         if s:
            n = s.split( '/', 1 )
            if len( n ) == 1:
               n = [ n[ 0 ], '*' ]
            if fnmatch.fnmatch( self.facilityName_, n[ 0 ] ):
               # do not support range in levels
               if n[ 1 ] == '*':
                  n[ 1 ] = '0123456789'
               for i in n[ 1 ]:
                  self.levelEnabled_.add( i )

   def _trace( self, level, *args ):
      if level in self.levelEnabled_:
         print( datetime.datetime.today(), os.getpid(), self.facilityName_,
                *args, file=self.file_, end=self.lineEnd_ )

   trace0 = _traceFunc( '0' )
   trace1 = _traceFunc( '1' )
   trace2 = _traceFunc( '2' )
   trace3 = _traceFunc( '3' )
   trace4 = _traceFunc( '4' )
   trace5 = _traceFunc( '5' )
   trace6 = _traceFunc( '6' )
   trace7 = _traceFunc( '7' )
   trace8 = _traceFunc( '8' )
   trace9 = _traceFunc( '9' )

def defaultTraceHandle():
   # Just return a dummy handle
   return Handle( 'CliShellTracing' )

def trace( *args ):
   # just don't do anything
   pass

trace0 = trace1 = trace2 = trace3 = trace4 = trace5 = trace6 = trace7 = \
         trace8 = trace9 = trace

class CliShellLogging:
   '''
   Mock some Logging APIs.
   '''
   def nop( self, *args, **kwargs ):
      return None

   LogHandle = nop
   log = nop
   __getattr__ = nop

def install():
   # there is no point in doing this if 'Tac' is already loaded
   assert 'Tac' not in sys.modules
   assert '_Tac' not in sys.modules
   # Now replace Tracing and Logging
   mod = sys.modules[ __name__ ]
   sys.modules[ 'Tracing' ] = mod
   sys.modules[ 'Logging' ] = CliShellLogging()
