#!/usr/bin/env python3
# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from CliCommand import (
   CliExpressionFactory,
   CliExpression,
   Node,
)
from CliPlugin import IpAddrMatcher
from CliPlugin.WanTEShowCli import (
   getVrfNames,
   getAvtNames,
   avtSupportedGuard,
   getAppNames,
)
from CliMatcher import (
   KeywordMatcher,
   DynamicNameMatcher,
   IntegerMatcher,
)

avtKeyword = 'adaptive-virtual-topology'
avtMatcherForShow = KeywordMatcher(
   avtKeyword, helpdesc='Show adaptive virtual topology information' )

avtMatcherForPing = KeywordMatcher(
   avtKeyword,
   helpdesc='Send echo messages over the path(s) belonging to an AVT' )

pathMatcherForShow = KeywordMatcher(
   'path',
   helpdesc='Show adaptive virtual topology path information' )

detailMatcherForShow = KeywordMatcher(
   'detail',
   helpdesc='Show in a detailed view'
)

appProfileMatcherForShow = KeywordMatcher(
    'application-profile',
    helpdesc='Show adaptive virtual topology applicaton profile information' )

avtNameMatcher = DynamicNameMatcher(
   getAvtNames,
   helpdesc='Adaptive virtual topology name',
   helpname='WORD' )

vrfNameMatcher = DynamicNameMatcher(
   getVrfNames,
   helpdesc='VRF name',
   helpname='WORD' )

appNameMatcher = DynamicNameMatcher(
   getAppNames,
   helpdesc="APP Profile name",
   helpname='WORD' )

destinationValueMatcher = IpAddrMatcher.IpAddrMatcher( 'IP address' )
pathIdMatcher = IntegerMatcher( 1, 4096, helpdesc='path-id' )

avtNodeForShow = Node( avtMatcherForShow, guard=avtSupportedGuard )
avtNodeForPing = Node( avtMatcherForPing, guard=avtSupportedGuard )

class AvtPathIdExprFactory( CliExpressionFactory ):
   def __init__( self, helpStr ):
      self.helpStr = helpStr
      CliExpressionFactory.__init__( self )

   def generate( self, name ):
      class AvtPathIdDefinitionExpr( CliExpression ):
         expression = 'path-id PATH-ID'
         data = {
            'path-id': self.helpStr,
            'PATH-ID': pathIdMatcher,
         }
      return AvtPathIdDefinitionExpr

class AvtAppExprFactory( CliExpressionFactory ):
   def __init__( self, helpStr ):
      self.helpStr = helpStr
      CliExpressionFactory.__init__(self)

   def generate( self, name ):
      class AvtAppDefinitionExpr( CliExpression ):
         expression = "application APPLICATION"
         data = {
            'application': Node( matcher=KeywordMatcher( 'application',
                self.helpStr ),  maxMatches=1 ),
            'APPLICATION': appNameMatcher,
                }
      return AvtAppDefinitionExpr

class AvtVrfExprFactory( CliExpressionFactory ):
   def __init__( self, helpStr ):
      self.helpStr = helpStr
      CliExpressionFactory.__init__( self )

   def generate( self, name ):
      class AvtVrfDefinitionExpr( CliExpression ):
         expression = 'vrf VRF'
         data = {
            'vrf': Node( matcher=KeywordMatcher( 'vrf', self.helpStr ),
                         maxMatches=1 ),
            'VRF': vrfNameMatcher,
         }
      return AvtVrfDefinitionExpr

class AvtNameExprFactory( CliExpressionFactory ):
   def __init__( self, helpStr ):
      self.helpStr = helpStr
      CliExpressionFactory.__init__( self )

   def generate( self, name ):
      class AvtNameDefinitionExpr( CliExpression ):
         expression = 'avt AVT'
         data = {
            'avt': Node( matcher=KeywordMatcher( 'avt', self.helpStr ),
                         maxMatches=1 ),
            'AVT': avtNameMatcher,
         }
      return AvtNameDefinitionExpr

class AvtSourceExprFactory( CliExpressionFactory ):
   def __init__( self, helpStr ):
      self.helpStr = helpStr
      CliExpressionFactory.__init__( self )

   def generate( self, name ):
      class AvtSourceDefinitionExpr( CliExpression ):
         expression = 'source SRC'
         data = {
            'source': Node( matcher=KeywordMatcher( 'source', self.helpStr ),
                            maxMatches=1 ),
            'SRC': destinationValueMatcher,
         }
      return AvtSourceDefinitionExpr

class AvtDestinationExprFactory( CliExpressionFactory ):
   def __init__( self, helpStr ):
      self.helpStr = helpStr
      CliExpressionFactory.__init__( self )

   def generate( self, name ):
      class AvtDestinationDefinitionExpr( CliExpression ):
         expression = 'destination DEST'
         data = {
            'destination': Node(
               matcher=KeywordMatcher( 'destination', self.helpStr ),
               maxMatches=1 ),
            'DEST': destinationValueMatcher,
         }
      return AvtDestinationDefinitionExpr
