# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliCommand import (
   CliExpression,
   hiddenKeyword
)
from CliMatcher import (
      EnumMatcher,
      IntegerMatcher,
      KeywordMatcher,
      PatternMatcher,
)
from MultiRangeRule import MultiRangeMatcher
from TypeFuture import TacLazyType

SystemId = TacLazyType( "Dsf::SystemId" )
SystemPortId = TacLazyType( "Dsf::SystemPortId" )

# ----------------------------------------------------------------------------------
#  Keywords
# ----------------------------------------------------------------------------------

# Keyword matchers: please keep these alphabetically sorted
addressKw = KeywordMatcher( "address",
   helpdesc="Device CPU interface address" )
adjacencyKw = KeywordMatcher( "adjacency",
   helpdesc="Adjacency information for hosts" )
attachedHostKw = KeywordMatcher( "attached-host",
   helpdesc="Synchronized tables for attached hosts" )
clusterShowKw = KeywordMatcher( "cluster",
   helpdesc="Show cluster information" )
cpuKw = KeywordMatcher( "cpu",
   helpdesc="Device CPU routing information" )
cpuLocalKw = KeywordMatcher( "cpu",
   helpdesc="Local device CPU routing information" )
deviceVoqKw = KeywordMatcher( "device",
   helpdesc="VOQ devices" )
deviceConfigVoqKw = KeywordMatcher( "device",
   helpdesc="Device configuration" )
disabledKw = KeywordMatcher( "disabled",
   helpdesc="Disable VOQ cluster forwarding" )
disabledDeviceKw = KeywordMatcher( "disabled",
   helpdesc="Disable VOQ cluster forwarding to this device" )
interfaceConfigLeafKw = KeywordMatcher( "interface",
   helpdesc="Leaf device interface config" )
interfaceDeviceKw = KeywordMatcher( "interface",
   helpdesc="Device CPU interface" )
interfaceLocalKw = KeywordMatcher( "interface",
   helpdesc="Local device CPU interface" )
interfacesVoqKw = KeywordMatcher( "interfaces",
   helpdesc="VOQ interfaces" )
ipKw = KeywordMatcher( "ip",
   helpdesc="IP" )
ipv6Kw = KeywordMatcher( "ipv6",
   helpdesc="IPv6" )
leafDevicesKw = KeywordMatcher( "leaf",
   helpdesc="Leaf devices" )
leafConfigKw = KeywordMatcher( "leaf",
   helpdesc="Leaf device config" )
localAdjacencyKw = KeywordMatcher( "local",
   helpdesc="Adjacency information for hosts attached to this device" )
localAttachedHostKw = KeywordMatcher( "local",
   helpdesc="Routing information for hosts attached to this device" )
nameKw = KeywordMatcher( "name",
   helpdesc="Device name" )
portKw = KeywordMatcher( "port",
   helpdesc="System port config" )
profileKw = KeywordMatcher( "profile",
   helpdesc="Leaf device interface profile" )
rangeKw = KeywordMatcher( "range",
   helpdesc="Global system port ID range" )
remoteAdjacencyKw = KeywordMatcher( "remote",
   helpdesc=( "Adjacency information for hosts attached"
              " to remote devices in the cluster" ) )
routeAttachedHostKw = KeywordMatcher( "route",
   helpdesc="Routes for attached hosts" )
spineConfigKw = KeywordMatcher( "spine",
   helpdesc="Spine device config" )
spineDevicesKw = KeywordMatcher( "spine",
   helpdesc="Spine devices" )
summaryKw = KeywordMatcher( "summary",
   helpdesc="Summary of cluster VOQ" )
switchConfigKw = KeywordMatcher( "switch",
   helpdesc="Configure switching behavior" )
systemConfigKw = KeywordMatcher( "system",
   helpdesc="System config" )
systemIdLeafKw = KeywordMatcher( "system-id",
   helpdesc="Unique leaf device identifier within the VOQ cluster" )
systemIdLocalKw = KeywordMatcher( "system-id",
   helpdesc="Unique identifier within the VOQ cluster for the local device" )
systemIdSpineKw = KeywordMatcher( "system-id",
   helpdesc="Unique spine device identifier within the VOQ cluster" )
voqKw = KeywordMatcher( "voq",
   helpdesc="Virtual output queues" )
voqArchKw = KeywordMatcher( "voq",
   helpdesc="Virtual output queue architecture" )

# Hidden Keywords
defaultsLeafHiddenKw = hiddenKeyword( "defaults" )

# ----------------------------------------------------------------------------------
#  Enum Matcher
# ----------------------------------------------------------------------------------
profileEnumMatcher = EnumMatcher(
   {
      "default-max-split-two": "Jericho 3 default mapping profile",
   },
   hiddenHelpDescMapping={
      # Hidden values
      "simulation-lyonsville": "Lyonsville simulation mapping profile",
      "simulation-denali": "Denali simulation mapping profile",
   },
)

# ----------------------------------------------------------------------------------
#  Integer Matcher
# ----------------------------------------------------------------------------------
deviceIdMatcher = IntegerMatcher(
   SystemId.systemIdMin, SystemId.systemIdMax, helpdesc="Device ID" )

spineDeviceIdMatcher = IntegerMatcher(
      SystemId.systemIdSpineMin, SystemId.systemIdSpineMax, helpdesc="Device ID" )

leafDeviceIdMatcher = IntegerMatcher(
      SystemId.systemIdLeafMin, SystemId.systemIdLeafMax, helpdesc="Device ID" )

# ----------------------------------------------------------------------------------
#  Multi Range Matcher
# ----------------------------------------------------------------------------------
leafDeviceIdRangeMatcher = MultiRangeMatcher(
   rangeFn=lambda: ( SystemId.systemIdLeafMin, SystemId.systemIdLeafMax ),
   noSingletons=False, # accept a single System ID
   maxRanges=1,
   helpdesc="Device ID range" )

sppIdRangeMatcher = MultiRangeMatcher(
   rangeFn=lambda: ( SystemPortId.min, SystemPortId.max ),
   noSingletons=False, # accept a single ID
   maxRanges=1,
   helpdesc="System port ID range" )

# ----------------------------------------------------------------------------------
#  Pattern Matcher
# ----------------------------------------------------------------------------------
devNamePattern = r"([A-Za-z0-9_-]+)$"
devNameMatcher = PatternMatcher(
   pattern=devNamePattern, helpname="NAME", helpdesc="Device name" )

# ----------------------------------------------------------------------------------
#  Device filter
# ----------------------------------------------------------------------------------
class DeviceFilter:
   def __init__( self, deviceName=None, systemIdRange=None ):
      self.deviceName = deviceName
      self.systemIdMin, self.systemIdMax = systemIdRange if systemIdRange \
         else ( None, None )

   def match( self, deviceName, systemId ):
      if self.deviceName is not None:
         return self.deviceName == deviceName
      if self.systemIdMin is not None:
         return self.systemIdMin <= systemId <= self.systemIdMax
      # No filter criteria specified
      return True

class LeafDeviceFilterExpression( CliExpression ):
   '''
   Expression for specifying a device filter either by device name or by
   SystemId / range of SystemIds.
   '''
   expression = "device  ( name NAME ) | ( system-id RANGE )"
   data = {
      "device": deviceVoqKw,
      "name": nameKw,
      "NAME": devNameMatcher,
      "system-id": systemIdLeafKw,
      "RANGE": leafDeviceIdRangeMatcher,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ "DEVICE_FILTER" ] = DeviceFilter(
         args.get( "NAME" ),
         args[ "RANGE" ].ranges()[ 0 ] if "RANGE" in args else None )
