# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from Arnet import Ip6AddrWithMask, Ip6Addr
from CliCommand import CliExpression, Node
from CliCommand import hiddenKeyword
from CliMatcher import (
      EnumMatcher,
      KeywordMatcher,
      PatternMatcher,
   )
from CliPlugin.IpAddrMatcher import (
      ipAddrMatcher,
      ipPrefixMatcher,
   )
from CliPlugin.Ip6AddrMatcher import (
      ip6AddrMatcher,
      ip6PrefixMatcher,
   )
from CliPlugin.IpGenAddrMatcher import IpGenAddrMatcher
from CliPlugin.IpGenAddrMatcher import IpGenPrefixMatcher
from CliToken.Community import CommunityConstExpr

# Keyword matchers: please keep these alphabetically sorted
active = KeywordMatcher( "active", helpdesc="Display active configuration" )
advertised = KeywordMatcher( "advertised-routes",
      helpdesc="Display all routes that have been advertised to neighbors" )
aspath = KeywordMatcher( "as-path",
      helpdesc="BGP routes filtered by AS path" )
autoAggr = hiddenKeyword( "auto-aggregation" )
bfd = KeywordMatcher( "bfd", helpdesc="BGP BFD information" )
bgpAfterShow = KeywordMatcher( "bgp", helpdesc="BGP information" )
bgpConfig = KeywordMatcher( "configuration", helpdesc="BGP configuration" )
bgpConvergence = KeywordMatcher( "convergence",
      helpdesc="BGP convergence information" )
bgpInstance = KeywordMatcher( "instance",
      helpdesc="BGP instance related information" )
bgpUpdateGroup = KeywordMatcher( "update-group",
      helpdesc="BGP update group information" )
color = KeywordMatcher( "color", helpdesc="Policy color" )
commRegexp = KeywordMatcher( "regex",
      alternates=[ "regexp" ],
      helpdesc="Display routes matching the communities regular expression" )
communityExact = KeywordMatcher( "exact", helpdesc="Exact match" )
detail = KeywordMatcher( "detail", helpdesc="Detailed view" )
detailConfig = KeywordMatcher( "detail",
      helpdesc="Display diff from current system configuration" )
distinguisher = KeywordMatcher( "distinguisher", helpdesc="Distinguisher" )
endpoint = KeywordMatcher( "endpoint", helpdesc="Policy endpoint" )
errors = KeywordMatcher( "errors", helpdesc="BGP update error information" )
fieldSet = KeywordMatcher( "field-set", helpdesc="IP prefix field set" )
filtered = KeywordMatcher( "filtered",
      helpdesc="Displays all routes that are received and filtered" )
forwarding = KeywordMatcher( "forwarding", helpdesc="BGP LU Forwarding information" )
iar = hiddenKeyword( "iar" )
internal = hiddenKeyword( "internal" )
internalDumpDuplicates = hiddenKeyword( 'dump-duplicates' )
internalRibOut = hiddenKeyword( 'rib-out' )
internalVerbose = hiddenKeyword( 'verbose' )
taskStats = hiddenKeyword( 'task-stats' )
reset = hiddenKeyword( 'reset' )
ipv4AfterShowBgp = KeywordMatcher( "ipv4", helpdesc="Display IPv4 information" )
ipv4Routes = KeywordMatcher( "ipv4", helpdesc="Display only IPv4 routes" )
ipv6AfterShowBgp = KeywordMatcher( "ipv6", helpdesc="Display IPv6 information" )
ipv6Routes = KeywordMatcher( "ipv6", helpdesc="Display only IPv6 routes" )
labUni = KeywordMatcher( "labeled-unicast", helpdesc="labeled-unicast routes only" )
labeledUnicast = KeywordMatcher( "labeled-unicast",
      helpdesc="BGP labeled-unicast information" )
labeledUnicastAfterShowBgp = KeywordMatcher( "labeled-unicast",
      helpdesc="Display labeled-unicast information" )
labeledUnicastRoutes = KeywordMatcher( "labeled-unicast",
      helpdesc="Display labeled-unicast routes" )
lfib = KeywordMatcher( "lfib", helpdesc="BGP LU LFIB information" )
lldpAfterShow = KeywordMatcher( "lldp", helpdesc="LLDP information" )
longerPrefixes = KeywordMatcher( "longer-prefixes",
      helpdesc="Displays the specified route and all more specific routes" )
luNexthop = KeywordMatcher( "next-hop",
      helpdesc="Display labeled-unicast next-hop originated routes",
      value=lambda mode, kw: 'epeOnly' )
luNexthopOriginated = KeywordMatcher( "originated",
      helpdesc="Display labeled-unicast next-hop originated routes",
      value=lambda mode, kw: 'epeOnly' )
mappings = KeywordMatcher( "mappings", helpdesc="Associations of field sets with "
                           "BGP communities" )
match = KeywordMatcher( "match", helpdesc="match the BGP Routes" )
multicast = KeywordMatcher( "multicast", helpdesc="Display multicast routes" )
multicastAfterShowBgp = KeywordMatcher( "multicast",
      helpdesc="Display multicast information" )
neighbors = KeywordMatcher( "neighbors", helpdesc="BGP Neighbor information" )
neighborsDeprecated = KeywordMatcher( "neighbors",
      helpdesc="BGP Neighbor information" )
noLuForwardingFlattening = KeywordMatcher( "raw",
      "Don't flatten BGP LU Forwarding entries" )
peerGroupAfterShow = KeywordMatcher( "peer-group",
      helpdesc="BGP Peer-group information" )
peers = KeywordMatcher( "peers", helpdesc="BGP neighbor information" )
policyInfo = KeywordMatcher( "policy", helpdesc="RFD policy information" )
queuedWithdrawals = KeywordMatcher( "queued-withdrawals",
      helpdesc="Displays advertised routes that have withdrawals pending out-delay" )
received = KeywordMatcher( "routes",
      helpdesc="Display all routes that are received and accepted" )
receivedAll = KeywordMatcher( "received-routes",
      helpdesc="Displays all received routes (both accepted and rejected)" )
regexp = KeywordMatcher( "regex", alternates=[ "regexp" ],
      helpdesc="Display routes matching the AS path regular expression" )
rfdReceived = KeywordMatcher( "received",
      helpdesc="Show all received paths with non zero penalty" )
rfdSuppressed = KeywordMatcher( "suppressed",
      helpdesc="Show route-flap-damping suppressed paths" )
rfdTracked = KeywordMatcher( "tracked",
      helpdesc="Show route-flap-damping tracked paths" )
rfdIgnored = KeywordMatcher( "ignored",
      helpdesc="Show route-flap-damping suppression ignored paths" )
rfdWithdrawn = KeywordMatcher( "withdrawn",
      helpdesc="Show all withdrawn paths with non zero penalty" )
routeFlapDamping = KeywordMatcher( "route-flap-damping",
      helpdesc="Route flap damping information" )
scheduler = Node( matcher=KeywordMatcher( "scheduler",
                                          helpdesc="BGP scheduler statistics" ) )
schedulerVerbose = KeywordMatcher( "verbose",
      helpdesc="Include more information" )
showAccessList = KeywordMatcher( "access-list",
      helpdesc="Display routes matching the AS path access list" )
showCommunity = KeywordMatcher( "community",
      helpdesc="BGP routes filtered by communities" )
showCommunityDeprecated = KeywordMatcher( "community",
      helpdesc="BGP routes filtered by communities" )
showCommunityList = KeywordMatcher( "community-list",
      helpdesc="BGP routes filtered by community list" )
showLargeCommunity = KeywordMatcher( "large-community",
      helpdesc="BGP routes filtered by large community" )
showLargeCommunityList = KeywordMatcher( "large-community-list",
      helpdesc="BGP routes filtered by large community list" )
showAsPathList = KeywordMatcher( "as-path",
      helpdesc="BGP routes filtered by AS path" )
srTeAfterShowBgp = KeywordMatcher( "sr-te",
      helpdesc="Segment Routing Traffic Engineering information" )
statistics = KeywordMatcher( "statistics", helpdesc="BGP statistics" )
summary = KeywordMatcher( "summary", helpdesc="Summarized BGP information" )
summaryGeneric = KeywordMatcher( "summary", helpdesc="Summarized information" )
trafficPolicy = KeywordMatcher( "traffic-policy", helpdesc="Traffic Policy" )
udpTunnelKeyword = KeywordMatcher( "udp", helpdesc="UDP tunnel information" )
# Following Vsp tokens are for show bgp vrf selection policy
vrfVsp = KeywordMatcher( 'vrf', helpdesc='VRF selection policy' )
selectionVsp = KeywordMatcher( 'selection', helpdesc='VRF selection policy' )
policyVsp = KeywordMatcher( 'policy', helpdesc='VRF selection policy' )

unicast = KeywordMatcher( "unicast", helpdesc="Display unicast routes" )
unicastAfterShowBgp = KeywordMatcher( "unicast",
      helpdesc="Display unicast information" )
unsupported = KeywordMatcher( "unsupported",
      helpdesc="Display unsupported configuration" )
update = KeywordMatcher( "update", helpdesc="BGP update information" )
routerId = KeywordMatcher( "router-id", helpdesc="filter by router ID" )
vpn = KeywordMatcher( "vpn", helpdesc="BGP VPN information" )

# Common expressions
ipGenAddrMatcher = IpGenAddrMatcher( 'IP address',
      helpdesc4="IP address", helpdesc6="IPv6 address" )
ipGenPrefixMatcher = IpGenPrefixMatcher( 'IP prefix',
      helpdesc4="IP prefix", helpdesc6="IPv6 prefix" )

# pylint: disable-msg=anomalous-backslash-in-string
aspathRegexMatcher = PatternMatcher(
      pattern=r'[0-9^$_*+.|()?\\\[\]\-{},]+',
      helpname='AS-PATH-REGEX',
      helpdesc='A regular expression to match AS paths.'
      ' Use "Ctrl-v ?" to enter literal "?"' )

afiMatcher = EnumMatcher( {
   'ipv4': ipv4AfterShowBgp.helpdesc_,
   'ipv6': ipv6AfterShowBgp.helpdesc_,
} )
safiMatcher = EnumMatcher( {
   'unicast': unicastAfterShowBgp.helpdesc_,
   'multicast': multicastAfterShowBgp.helpdesc_,
   'labeled-unicast': labeledUnicastAfterShowBgp.helpdesc_,
} )

def afiSafiAdapter( mode, args, argList ):
   afi = args.pop( "AFI", None )
   if afi:
      safi = args.pop( "SAFI" )
      # unicast => unicast
      # multicast => multicast
      # labeled-unicast => mplslabel
      if safi in { "unicast", "multicast" }:
         pass
      elif safi == "labeled-unicast":
         safi = "mplslabel"
      else:
         assert False, "Unknown SAFI " + safi
      args[ "addressFamilyAndType" ] = { "addressFamily": afi, safi: True }
   else:
      args[ "addressFamilyAndType" ] = None

class AddrFamilyAfterShowBgp( CliExpression ):
   expression = "AFI"
   data = {
         "AFI": afiMatcher,
      }

   @staticmethod
   def adapter( mode, args, argList ):
      args[ "addrFamily" ] = args.pop( "AFI", None )

class AddressFamilyAfterShowBgpRule( CliExpression ):
   expression = "AFI SAFI"
   data = {
         "AFI": afiMatcher,
         "SAFI": safiMatcher,
      }
   adapter = afiSafiAdapter

class RouteUnicastAddressFamilyRule( CliExpression ):
   expression = "AFI unicast"
   data = {
         "AFI": afiMatcher,
         "unicast": Node( matcher=unicastAfterShowBgp, alias="SAFI" ),
      }
   adapter = afiSafiAdapter

class CommunityValuesAndExactRule( CliExpression ):
   expression = "community { COMM_VALUE } [ exact ]"
   data = {
         "community": showCommunity,
         "COMM_VALUE": CommunityConstExpr,
         "exact": communityExact,
      }

   @staticmethod
   def adapter( mode, args, argList ):
      if not args.pop( "community", None ):
         return
      result = { "communityValues": args.pop( "communityValue" ) }
      if args.pop( "exact", None ):
         result[ "exact" ] = "exact"
      args[ "communityValuesAndExact" ] = result

routeTypeMatcher = EnumMatcher( {
   'routes': received.helpdesc_,
   'advertised-routes': advertised.helpdesc_,
   'received-routes': receivedAll.helpdesc_,
} )

class RouteTypeMaybeFilteredMatcher( CliExpression ):
   expression = "( routes | advertised-routes | ( received-routes [ filtered ] ) )"
   data = {
         "routes": received,
         "advertised-routes": advertised,
         "received-routes": receivedAll,
         "filtered": filtered
         }

   @staticmethod
   def adapter( mode, args, argList ):
      if args.pop( "routes", None ):
         args[ "route-type" ] = "routes"
      elif args.pop( "advertised-routes", None ):
         args[ "route-type" ] = "advertised-routes"
      elif args.pop( "received-routes", None ):
         args[ "route-type" ] = "received-routes"
         if args.pop( "filtered", None ):
            args[ "filtered" ] = "filtered"
      else:
         assert False

class Ipv4OrIpv6AddrMatcher( CliExpression ):
   expression = "( IPV4_ADDR | IPV6_ADDR )"
   data = {
         "IPV4_ADDR": ipAddrMatcher,
         "IPV6_ADDR": ip6AddrMatcher,
      }

   @staticmethod
   def adapter( mode, args, argList ):
      v4Addr = args.pop( "IPV4_ADDR", None )
      v6Addr = args.pop( "IPV6_ADDR", None )
      if v6Addr is not None:
         # v6Addr has type Arnet::Ip6Addr which evaluates to False
         # when v6Addr is 0::0, so we convert it to a string
         # in that case
         v6Addr = v6Addr if v6Addr else v6Addr.stringValue
      args[ "IPV4_OR_IPV6_ADDR" ] = v4Addr or v6Addr

class Ipv4OrIpv6AddrMaskLongerMatcher( CliExpression ):
   expression = "( IPV4_ADDR_MASK | IPV6_ADDR_MASK ) [ longer-prefixes ]"
   data = {
         "IPV4_ADDR_MASK": ipPrefixMatcher,
         "IPV6_ADDR_MASK": ip6PrefixMatcher,
         "longer-prefixes": longerPrefixes,
      }

   @staticmethod
   def adapter( mode, args, argList ):
      v4AddrMask = args.pop( "IPV4_ADDR_MASK", None )
      v6AddrMask = args.pop( "IPV6_ADDR_MASK", None )
      if v6AddrMask is not None:
         # v6AddrMask has type Arnet::Ip6AddrWithMask which evaluates to False
         # when v6AddrMask is ::/0, so we convert it to a string
         # in that case
         v6AddrMask = v6AddrMask if v6AddrMask else v6AddrMask.stringValue
      args[ "IPV4_OR_IPV6_ADDR_MASK" ] = v4AddrMask or v6AddrMask

class IpOrPrefixMatcher( CliExpression ):
   expression = "( IP | ( IP_PREFIX [longer-prefixes] ) )"
   data = {
         "IP": ipGenAddrMatcher,
         "IP_PREFIX": ipGenPrefixMatcher,
         "longer-prefixes": longerPrefixes,
         }

   @staticmethod
   def adapter( mode, args, argList ):
      ipAddr = args.pop( "IP", None )
      ipPfx = args.pop( "IP_PREFIX", None )
      if ipAddr:
         if ipAddr.af == 'ipv6':
            args[ 'ADDR' ] = Ip6Addr( ipAddr )
         else:
            args[ 'ADDR' ] = ipAddr.stringValue
      elif ipPfx:
         if ipPfx.af == 'ipv6':
            args[ 'ADDR' ] = Ip6AddrWithMask( ipPfx )
         else:
            args[ 'ADDR' ] = ipPfx.stringValue
         if args.pop( "longer-prefixes", None ):
            args[ "longer-prefixes" ] = "longer-prefixes"
      else:
         assert False

class InternalWithVerboseRule( CliExpression ):
   expression = "internal [verbose] [rib-out] [dump-duplicates]"
   data = {
         "internal": internal,
         "verbose": internalVerbose,
         "rib-out": internalRibOut,
         "dump-duplicates": internalDumpDuplicates,
         }

   @staticmethod
   def adapter( mode, args, argList ):
      if args.pop( "internal", None ):
         args[ "INTERNAL" ] = "internal"
         if args.pop( "verbose", None ):
            args[ "INTERNAL" ] += " verbose"
         if args.pop( "rib-out", None ):
            args[ "INTERNAL" ] += " rib-out"
         if args.pop( "dump-duplicates", None ):
            args[ "INTERNAL" ] += " dump-duplicates"

class TaskStatsWithResetRule( CliExpression ):
   expression = "task-stats [ reset ]"
   data = {
         "task-stats": taskStats,
         "reset": reset,
         }

   @staticmethod
   def adapter( mode, args, argList ):
      if args.pop( "task-stats", None ):
         args[ 'TASK_STATS' ] = "task-stats"
         if args.pop( "reset", None ):
            args[ 'TASK_STATS' ] += " reset"
