# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Arnet
from Arnet.MplsLib import boundedMplsLabelStackToLabelList
from PseudowireLib import connStatusMsg as PseudowireConnectorStatusMsg
import collections
import errno
from ForwardingHelper import (
   noMatchNexthopInfo,
   resolveTunnel,
)
from IpLibConsts import DEFAULT_VRF
import os
import re
import struct
import socket
from socket import AF_INET6
import Tac
import Tracing
import BothTrace
import threading
from Toggles.RoutingLibToggleLib import toggleFibGenMountPathEnabled
from TypeFuture import TacLazyType

IPV4, IPV6 = 'ip', 'ipv6'
# string literals used by LspTraceroute/LspPing utilities and some client libs
LspPing = 'lsp-ping'
LspTraceroute = 'lsp-traceroute'

LspPingTypeBgpLu = 'bgpLu'
LspPingTypeGeneric = 'generic'
LspPingTypeLdp = 'ldp'
LspPingTypeMldp = 'mldp'
LspPingTypeNhg = 'nexthop-group'
LspPingTypeNhgTunnel = 'nexthop-group-tunnel'
LspPingTypePwLdp = 'pwLdp'
LspPingTypeRaw = 'raw'
LspPingTypeRsvp = 'rsvp'
LspPingTypeSr = 'segment-routing'
LspPingTypeOspfSr = 'segment-routing-ospf'
LspPingTypeStatic = 'static'
LspPingTypeSrTe = 'SrTePolicy'
LspPingTypeVpn = 'vpn'
LspPingTypes = [ LspPingTypeBgpLu, LspPingTypeGeneric, LspPingTypeVpn,
                 LspPingTypeRaw, LspPingTypeLdp, LspPingTypeRsvp,
                 LspPingTypeSr, LspPingTypeOspfSr, LspPingTypeStatic, LspPingTypeNhg,
                 LspPingTypeSrTe, LspPingTypePwLdp, LspPingTypeNhgTunnel,
                 LspPingTypeMldp ]

LspPingDSMap, LspPingDDMap = 'dsmap', 'ddmap'
LspPingDSTypes = ( LspPingDSMap, LspPingDDMap )

IpGenAddr = Tac.Type( 'Arnet::IpGenAddr' )
IpGenPrefix = Tac.Type( 'Arnet::IpGenPrefix' )
BoundedMplsLabelStack = Tac.Type( 'Arnet::BoundedMplsLabelStack' )
MplsLabelStack = TacLazyType( 'Arnet::MplsLabelOperation' )
ColoredTunnelEndPoint = Tac.Type( "Tunnel::TunnelTable::ColoredTunnelEndpoint" )
ConnectorKey = Tac.Type( 'Pseudowire::ConnectorKey' )
DynTunnelIntfId = Tac.Type( 'Arnet::DynamicTunnelIntfId' )
AddressFamily = Tac.Type( 'Arnet::AddressFamily' )
IPv4, IPv6 = AddressFamily.ipv4, AddressFamily.ipv6
ELI = TacLazyType( 'Arnet::MplsLabel' ).entropyLabelIndicator
implicitNull = TacLazyType( "Arnet::MplsLabel" ).implicitNull
EthAddr = Tac.Type( 'Arnet::EthAddr' )
FecAdjType = TacLazyType( 'Smash::Fib::AdjType' )
FecIdConstants = Tac.Type( 'Smash::Fib::FecIdConstants' )
FecIdIntfId = TacLazyType( 'Arnet::FecIdIntfId' )
FecId = TacLazyType( 'Smash::Fib::FecId' )
IntfId = Tac.Type( 'Arnet::IntfId' )
MplsLabel = Tac.Type( 'Arnet::MplsLabel' )
NexthopGroupIntfIdType = Tac.Type( 'Arnet::NexthopGroupIntfId' )
PolicyKey = TacLazyType( 'SrTePolicy::PolicyKey' )
PseudowireConnectorStatus = Tac.Type( "Pseudowire::PseudowireConnectorStatus" )
PseudowireGenId = TacLazyType( 'Pseudowire::PseudowireGenId' )
PseudowireUserKey = TacLazyType( 'Pseudowire::PseudowireUserKey' )
RoutingOutputType = Tac.Type( 'Routing::RoutingOutputType' )
RvspLerTunnelState = Tac.Type( 'Rsvp::RsvpLerTunnelState' )
RsvpLerSubTunnelStatus = Tac.Type( 'Rsvp::RsvpLerSubTunnelStatus' )
RsvpLerSubTunnelConfigId = Tac.Type( 'Rsvp::RsvpLerSubTunnelConfigId' )
RsvpLerTunnelSource = Tac.Type( 'Rsvp::RsvpLerTunnelSource' )
RsvpNextHopInfo = Tac.Type( 'Rsvp::RsvpNextHopInfo' )
RsvpSessionType = Tac.Type( 'Rsvp::RsvpSessionType' )
TunnelId = Tac.Type( 'Tunnel::TunnelTable::TunnelId' )
TunnelType = TacLazyType( 'Tunnel::TunnelTable::TunnelType' )
LspPingReturnCode = Tac.Type( 'LspPing::LspPingReturnCode' )
LspPingTargetFecSubTlvType = Tac.Type( 'LspPing::LspPingTargetFecSubTlvType' )
LspPingMultipathBitset = Tac.Type( 'LspPing::LspPingMultipathBitset' )
ARP_SMASH_DEFAULT_VRF_ID = Tac.Type( 'Vrf::VrfIdMap::VrfId' ).defaultVrf
RouteType = Tac.Type( 'Routing::Fib::RouteType' )
Route6Type = Tac.Type( 'Routing6::Fib::RouteType' )
igpProtocolType = TacLazyType( 'LspPing::LspPingSrIgpProtocol' )
igpProtocolTypes = [ igpProtocolType.isis, igpProtocolType.ospf, '' ]

tunnelSourceCli = RsvpLerTunnelSource.tunnelSourceCli

NextHopAndLabel = collections.namedtuple( 'NextHopAndLabel', [
                   'nextHopIp',
                   'label',
                   'intfId',
                   ] )

MldpInfo = collections.namedtuple( 'MldpInfo',
              'genOpqVal sourceAddrOpqVal groupAddrOpqVal jitter responderAddr' )

PwLdpInfo = collections.namedtuple( 'PwLdpInfo', [
                   'localRouterId',
                   'remoteRouterId',
                   'pwId',
                   'pwGenId',
                   'pwType',
                   'vcLabel',
                   'cvTypes',
                   'ccTypes',
                   'controlWord',
                   ] )

retCodeStrMap = {
   LspPingReturnCode.noRetCode : 'no return code',
   LspPingReturnCode.malFormEchoRequest : 'malformed echo req',
   LspPingReturnCode.tlvNotUnderstood : 'tlv not understood',
   LspPingReturnCode.repRouterEgress : 'egress ok',
   LspPingReturnCode.noMappingForFec : 'no map for fec',
   LspPingReturnCode.dsMappingMismatch : 'ds map mismatch',
   LspPingReturnCode.usIntfUnknown : 'us intf unknown',
   LspPingReturnCode.reservedRetCode : 'reserved ret code',
   LspPingReturnCode.labelSwitchedAtStackDep : 'label switched',
   LspPingReturnCode.labelSwitchedNoMplsFwd : 'no mpls fwd',
   LspPingReturnCode.labelMappingMismatch : 'label map mismatch',
   LspPingReturnCode.noLabelEntryAtStackDep : 'no label entry',
   LspPingReturnCode.protocolIntfMismatch : 'proto intf mismatch',
   LspPingReturnCode.premTermination : 'prem termination',
   LspPingReturnCode.seeDdmTlv : 'check downstream information',
   LspPingReturnCode.lblSwitchedFecChange : 'label switched with protocol change'
}

# Map for printing the errored Tlv Names.
# If there are any errored Tlv types and
# values added newly for LSP ping in LspPingTlvType
# add here as well

errTlvNameMap = {
   'TargetFecStack' : 'Target Fec Stack',
   'DownstreamMapping' : 'Downstream Mapping',
   'Pad' : 'Pad',
   'NotAssigned4' : 'Unassigned',
   'VendorEnterpriseNum' : 'Vendor Enterprise Number',
   'NotAssigned6' : 'Unassigned',
   'IntfAndLabelStack' : 'Interface and Label Stack',
   'NotAssigned8' : 'Unassigned',
   'ErrorTLV' : 'Errored TLVs',
   'ReplyTosByte' : 'Reply TOS Byte',
   'DownstreamDetailedMapping' : 'Downstream Detailed Mapping',
   'P2mpRespId' : 'P2MP Responder',
   'EchoJitter' : 'Echo Jitter',
   'AristaVendorPrivate' : 'Arista Vendor Private'
}

bgpLuNoTunnelFoundErr = ( 'No BGP labeled unicast tunnel found in tunnel FIB'
                          ' for tunnel ID %d' )
bgpLuNoTunnelViaFoundErr = ( 'No BGP labeled unicast tunnel via found in'
                             ' tunnel ID %d' )

# --------------------------------
# BothTrace Short hand variables
# --------------------------------
__defaultTraceHandle__ = Tracing.Handle( "ClientCommonLib" )
bv = BothTrace.Var
bt8 = BothTrace.trace8

# Map for printing the errored SubTlv Names.
# If there are any errored SubTlv types and
# values added newly for LSP ping in LspPingTlvType
# add here as well

errSubTlvNameMap = {
   'LDPIpv4' : 'LDP IPv4 Prefix',
   'LDPIpv6' : 'LDP IPv6 Prefix',
   'RSVPIpv4Lsp' : 'RSVP IPv4 LSP',
   'RSVPIpv6Lsp' : 'RSVP IPv6 LSP',
   'NotAssigned' : 'Unassigned',
   'VPNIpv4' : 'VPN IPv4 Prefix',
   'VPNIpv6' : 'VPN IPv6 Prefix',
   'L2VPNEndpoint' : 'L2 VPN Endpoint',
   'FEC128Pseudowire' : 'FEC 128 Pseudowire',
   'FEC129Pseudowire' : 'FEC 129 Pseudowire',
   'BGPLabeledIpv4' : 'BGP Labeled IPv4 Prefix',
   'BGPLabeledIpv6' : 'BGP Labeled IPv6 Prefix',
   'GenericIpv4' : 'Generic IPv4 Prefix',
   'GenericIpv6' : 'Generic IPv6 Prefix',
   'NilFec' : 'Nil FEC',
   'P2mpFec' : 'Multicast P2MP LDP FEC Stack',
   'EntropyLabel' : 'Entropy Label FEC'
}

# Map for printing FEC Tlv type
fecTlvRenderMap = {
   'ldpIpv4' : 'LDP IPv4',
   'ldpIpv6' : 'LDP IPv6',
   'rsvpIpv4' : 'RSVP IPv4',
   'rsvpIpv6' : 'RSVP IPv6',
   'vpnIpv4' : 'VPN IPv4',
   'vpnIpv6' : 'VPN IPv6',
   'l2vpnEndpoint' : 'L2VPN Endpoint',
   'fec128Pw' : 'FEC128 PW',
   'fec129Pw' : 'FEC129 PW',
   'bgpLuIpv4' : 'BGP LU IPv4',
   'bgpLuIpv6' : 'BGP LU IPv6',
   'genericIpv4' : 'Generic IPv4',
   'genericIpv6' : 'Generic IPv6',
   'nil' : 'Nil',
   'p2mp' : 'P2MP',
   'srIpv4' : 'SR IPv4',
   'srIpv6' : 'SR IPv6',
   'srIpv4Depr' : 'SR IPv4 Deprecated',
   'srIpv6Depr' : 'SR IPv6 Deprecated',
}

fecTlvTypeMap = {
    LspPingTargetFecSubTlvType.LDPIpv4 : 'ldpIpv4',
    LspPingTargetFecSubTlvType.LDPIpv6 : 'ldpIpv6',
    LspPingTargetFecSubTlvType.RSVPIpv4Lsp : 'rsvpIpv4',
    LspPingTargetFecSubTlvType.RSVPIpv6Lsp : 'rsvpIpv6',
    LspPingTargetFecSubTlvType.VPNIpv4 : 'vpnIpv4',
    LspPingTargetFecSubTlvType.VPNIpv6 : 'vpnIpv6',
    LspPingTargetFecSubTlvType.L2VPNEndpoint : 'l2vpnEndpoint',
    LspPingTargetFecSubTlvType.FEC128Pseudowire : 'fec128Pw',
    LspPingTargetFecSubTlvType.FEC129Pseudowire : 'fec129Pw',
    LspPingTargetFecSubTlvType.BGPLabeledIpv4 : 'bgpLuIpv4',
    LspPingTargetFecSubTlvType.BGPLabeledIpv6 : 'bgpLuIpv6',
    LspPingTargetFecSubTlvType.GenericIpv4 : 'genericIpv4',
    LspPingTargetFecSubTlvType.GenericIpv6 : 'genericIpv6',
    LspPingTargetFecSubTlvType.NilFec : 'nil',
    LspPingTargetFecSubTlvType.P2mpFec : 'p2mp',
    LspPingTargetFecSubTlvType.SRIpv4 : 'srIpv4',
    LspPingTargetFecSubTlvType.SRIpv6 : 'srIpv6',
    LspPingTargetFecSubTlvType.SRIpv4Deprecated : 'srIpv4Depr',
    LspPingTargetFecSubTlvType.SRIpv6Deprecated : 'srIpv6Depr',
}

# Map for printing Well Known MPLS Labels
knownMplsLabelMap = {
      MplsLabel.implicitNull : 'implicit-null',
      MplsLabel.entropyLabelIndicator : 'entropy-label-indicator',
}

# ---------------------------------------------------------
#           ThreadLocal storage getter and setter
# ---------------------------------------------------------
# threadDict maintains variables which are needed per cli thread
# storage like server socket port and tracerouteModel, so two cli sessions
# together running same cmd don't interfere with each other.

def getThreadLocalData( attr ):
   currentThread = threading.current_thread()
   threadDict = currentThread.__dict__
   return threadDict.get( attr, None )

def setThreadLocalData( attr, value ):
   currentThread = threading.current_thread()
   threadDict = currentThread.__dict__
   threadDict[ attr ] = value

# ---------------------------------------------------------
#          Ping/Traceroute model generation helpers
# ---------------------------------------------------------

def sendMessage( sock, msg ):
   # Prefix each message with a 4-byte *sized msg* length
   # ( little-endian marked by <I )
   msg = struct.pack( '<I', len( msg ) ) + msg
   sock.sendall( msg )

def sendViaSocket( sock, msg ):
   dumps = msg.SerializeToString()
   sendMessage( sock, dumps )

def getClientSocket( servSockPort ):
   sock = socket.socket( socket.AF_INET, socket.SOCK_STREAM )
   sock.connect( ( '127.0.0.1', servSockPort ) )
   # SO_LINGER is set with 0 seconds to avoid socket being in Time_Wait
   # state after it has been closed. This avoids gettng any timeout errs
   # if using the same socket tuple assigned before
   lingerEnabled = 1
   lingerTime = 0 # This is in seconds.
   lingerStruct = struct.pack( 'ii', lingerEnabled, lingerTime )
   sock.setsockopt( socket.SOL_SOCKET, socket.SO_LINGER, lingerStruct )
   return sock

def receiveMessage( sock ):
   # Sender prefixes each message with a 4-byte *sized msg* length
   # Read message length first and unpack it into an integer
   msgLenBytes = 4
   rawMsgLen = receiveAll( sock, msgLenBytes )
   if not rawMsgLen:
      return None
   # little-endian marked by <I as protobuf helper functions
   # only supported little-endian writing.
   # struct.unpack always returns a tuple even if it contains exactly one item.
   # Hence [ 0 ] used.
   msgLen = struct.unpack( '<I', rawMsgLen )[ 0 ]
   # Read the message data
   return receiveAll( sock, msgLen )

def receiveAll( sock, n ):
   # Helper function to recv n bytes or return None if EOF is hit
   data = bytearray()
   while len( data ) < n:
      packet = sock.recv( n - len( data ) )
      if not packet:
         return None
      data.extend( packet )
   return data

def createSocket():
   maxConnections = 10
   s = socket.socket( socket.AF_INET, socket.SOCK_STREAM )
   # set socket options SO_REUSEADDR and SO_REUSEPORT by setting to value 1
   s.setsockopt( socket.SOL_SOCKET, socket.SO_REUSEADDR, 1 )
   s.setsockopt( socket.SOL_SOCKET, socket.SO_REUSEPORT, 1 )
   s.bind( ( '127.0.0.1', 0 ) )
   s.listen( maxConnections )
   setThreadLocalData( 's', s )
   # getsockname returns a tuple address and port number
   # portNumber needs to be retuned to send to the subprocess for socket connection
   bt8( "Create server socket ", bv( s.getsockname() ) )
   return s.getsockname()[ 1 ]

# ---------------------------------------------------------

def getMplsLabelStr( label ):
   return knownMplsLabelMap.get( label, str( label ) )

def fecTlvTypeStr( tlvType ):
   return fecTlvTypeMap.get( tlvType, 'unknown' )

def lspPingRetCodeStr( retCode ):
   return retCodeStrMap.get( retCode, 'unknown' )

def errorSubTlvLspPingRetCodeStr( replyPktInfo ):
   retCodeStr = retCodeStrMap.get( replyPktInfo.retCode )
   errSubTlvExists = True
   for errorType in replyPktInfo.errorSubTlvMap:
      retCodeStr += '\n'
      if errSubTlvExists:
         retCodeStr += "Error SubTLVs in Target FEC Stack" + '\n'
         errSubTlvExists = False
      try:
         tlv = Tac.enumName( "LspPing::LspPingTargetFecSubTlvType", errorType )
         errTlvName = str( tlv )
         retCodeStr += "Error SubTLV Name: " + errSubTlvNameMap[ errTlvName ]
      except AttributeError:
         retCodeStr += "Error SubTLV Type: " + str( errorType )
      if replyPktInfo.errorSubTlvMap[ errorType ] > 1:
         retCodeStr += ", occurrence: "
         retCodeStr += str( replyPktInfo.errorSubTlvMap[ errorType ] )
   return retCodeStr

def errorTlvLspPingRetCodeStr( replyPktInfo ):
   retCodeStr = retCodeStrMap.get( replyPktInfo.retCode )
   for errorType in replyPktInfo.errorTlvMap:
      retCodeStr += '\n'
      try:
         errTlvName = str( Tac.enumName( "LspPing::LspPingTlvType", errorType ) )
         retCodeStr += "Error TLV Name: " + errTlvNameMap[ errTlvName ]
      except AttributeError:
         retCodeStr += "Error TLV Type: " + str( errorType )
      if replyPktInfo.errorTlvMap[ errorType ] > 1:
         retCodeStr += ", occurrence: "
         retCodeStr += str( replyPktInfo.errorTlvMap[ errorType ] )
   return retCodeStr

def setProductCodeGlobals():
   '''
   Ensure that we're well behaved when invoked via the CLI as "product" code.

    - Disable tracing to stderr.
    - Use epoll.
    - Disable dropping in to pdb for exceptions.
   '''
   os.environ[ 'NOPDB' ] = '1'
   # We do not want to inherit any tracing or display anything in the CLI output
   # except if tracing is configured to write to TRACEFILE
   if 'TRACEFILE' not in os.environ:
      Tracing.traceSettingIs( '' )

   Tac.activityManager.useEpoll = True

def labelStackToList( labelStack ):
   if isinstance( labelStack, BoundedMplsLabelStack ):
      return boundedMplsLabelStackToLabelList( labelStack )
   labels = []
   for idx in range( labelStack.stackSize ):
      labels.insert( 0, labelStack.labelStack( idx ) )
   return labels

def isIpv6Addr( addr ):
   if hasattr( addr, 'stringValue' ):
      addr = addr.stringValue
   else:
      assert isinstance( addr, str )
   return IpGenAddr.guessAf( addr ) == AddressFamily.ipv6

def isIpv4Addr( addr ):
   return not isIpv6Addr( addr )

def isNexthopGroupVia( via ):
   return NexthopGroupIntfIdType.isNexthopGroupIntfId( via.intfId )

def isNexthopGroupTunnelVia( via ):
   if via.intfId.startswith( 'DynamicTunnel' ):
      tunnelId = DynTunnelIntfId.tunnelId( via.intfId )
      tunnelType = TunnelId( tunnelId ).tunnelType()
      return tunnelType == TunnelType.nexthopGroupTunnel
   return False

def getNexthopGroupId( via ):
   return NexthopGroupIntfIdType.nexthopGroupId( via.intfId )

def getNhgIdToName( nhgId, mount ):
   for key in mount.smashNhgStatus.nexthopGroupEntry:
      entry = mount.smashNhgStatus.nexthopGroupEntry[ key ]
      if entry.nhgId == nhgId:
         return key.nhgName()
   return None

def getNhgId( nhgName, mount ):
   key = Tac.Value( "NexthopGroup::NexthopGroupEntryKey" )
   key.nhgNameIs( nhgName )
   entry = mount.smashNhgStatus.nexthopGroupEntry.get( key )
   if not entry:
      return None
   return entry.nhgId

def getTunnelNhgName( mount, endpoint ):
   coloredTep = ColoredTunnelEndPoint( IpGenPrefix( endpoint ), 0, False )
   ribEntry = mount.nhgTunnelRib.entry.get( coloredTep )
   if not ribEntry or not ribEntry.tunnelId:
      err = 'No nexthop-group tunnel found for prefix %s' % endpoint
      return ( None, None, err )

   tunnelId = ribEntry.tunnelId[ 0 ]
   return resolveNhgTunnelFibEntry( mount, tunnelId )

def genOpaqueValForP2mpGenericOpaqueId( oid ):
   ''' Generates a opaque TLV for Generic opaque LSP identifier '''
   return struct.pack( '!BHI', 1, 4, oid )

def genOpaqueValForP2mpTransitV4SrcOpaqueId( src, grp ):
   ''' Generates a opaque TLV for transit v4 source '''
   unpackSrc = struct.unpack( '!L', socket.inet_aton( src ) )[ 0 ]
   unpackGrp = struct.unpack( '!L', socket.inet_aton( grp ) )[ 0 ]
   return struct.pack( '!BHII', 3, 8, unpackSrc, unpackGrp )

def genOpaqueValForP2mpTransitV6SrcOpaqueId( src, grp ):
   ''' Generates a opaque TLV for transit v6 source '''
   unpackSrc = struct.unpack( '!QQ', socket.inet_pton( AF_INET6, src ) )
   unpackGrp = struct.unpack( '!QQ', socket.inet_pton( AF_INET6, grp ) )
   return struct.pack( '!BHQQQQ', 4, 32,
                       unpackSrc[ 0 ], unpackSrc[ 1 ],
                       unpackGrp[ 0 ], unpackGrp[ 1 ] )

def getIntfPrimaryIpAddrDefaultVrf( mount, intf, ipv=IPV4 ):
   if ipv == IPV4:
      if not mount.vrfIpIntfStatusDefaultVrf:
         return None
      if not mount.vrfIpIntfStatusDefaultVrf.ipIntfStatus:
         return None
      intfStatus = mount.vrfIpIntfStatusDefaultVrf.ipIntfStatus.get( intf )
      if intfStatus:
         return intfStatus.activeAddrWithMask.address
   else:
      if not mount.vrfIp6IntfStatusDefaultVrf:
         return None
      if not mount.vrfIp6IntfStatusDefaultVrf.ip6IntfStatus:
         return None
      intfStatus = mount.vrfIp6IntfStatusDefaultVrf.ip6IntfStatus.get( intf )
      if intfStatus:
         address = intfStatus.selectDefaultSourceAddress()
         if address:
            return address.stringValue
   return None

def getIntfPrimaryIpAddr( mount, intf, ipv=IPV4 ):
   if ipv == IPV4:
      if not mount.vrfIpIntfStatus:
         return None
      if not mount.vrfIpIntfStatus.ipIntfStatus:
         return None
      intfStatus = mount.vrfIpIntfStatus.ipIntfStatus.get( intf )
      if intfStatus:
         return intfStatus.activeAddrWithMask.address
   else:
      if not mount.vrfIp6IntfStatus:
         return None
      if not mount.vrfIp6IntfStatus.ip6IntfStatus:
         return None
      intfStatus = mount.vrfIp6IntfStatus.ip6IntfStatus.get( intf )
      if intfStatus:
         address = intfStatus.selectDefaultSourceAddress()
         if address:
            return address.stringValue
   return None

def generateMplsEntropyLabel( srcIp, dstIp, udpSrcPort ):
   '''
   Generates the entropy label by hashing over the srcIp, dstIp, and UDP src port
   fields in order to preserve flow.

   The entropy label will be generated between [16, 1048575] (inclusive).
   '''
   unreservedMplsLabelBase = 16
   validRange = MplsLabel.max - unreservedMplsLabelBase
   hashVal = hash( ( str( srcIp ), str( dstIp ), udpSrcPort ) )
   entropyLabel = hashVal % validRange + unreservedMplsLabelBase
   return entropyLabel

def getIpv6ArpEntry( mount, nexthop, intf ):
   'Retrieves the Ipv6 ND entry from Arp Smash table given the nexthop and interface'
   arpKey = Tac.Value( 'Arp::Table::ArpKey', ARP_SMASH_DEFAULT_VRF_ID,
                       IpGenAddr( str( nexthop ) ), intf )
   neighborEntry = mount.arpSmash.neighborEntry.get( arpKey )
   if neighborEntry is None:
      return None
   return neighborEntry.ethAddr

def getProtocolIpFec( mount, prefix, protocol, genOpqVal,
                      sourceAddrOpqVal, groupAddrOpqVal, algorithm=None,
                      allowEntropyLabel=False, igpProtocol=None ):
   '''
   Returns a tuple of ( fecVias, errVal ) for the given protocol, where fecVias is
   a list of NextHopAndLabel objects.
   '''
   if protocol == 'ldp':
      nextHopsAndLabels, err, errVal = getLdpFec( mount, prefix, allowEntropyLabel )
      return ( nextHopsAndLabels, err, errVal, None )
   elif protocol == 'mldp':
      nextHopsAndLabels, err, errVal = getMldpFec( mount, prefix, genOpqVal,
                         sourceAddrOpqVal, groupAddrOpqVal )
      return ( nextHopsAndLabels, err, errVal, None )
   elif protocol == 'segment-routing':
      return getSrFec( mount, prefix, algorithm, igpProtocol )
   elif protocol == 'generic':
      nextHopsAndLabels, err, errVal, igpProtocol = getSrFec( mount, prefix,
                                                              algorithm, 
                                                              igpProtocol )
      if not nextHopsAndLabels:
         nextHopsAndLabels, err, errVal = getLdpFec( mount, prefix )
         return ( nextHopsAndLabels, err, errVal, None )
      return ( nextHopsAndLabels, err, errVal, igpProtocol )
   else:
      return ( [], "Not supported protocol", errno.EINVAL, None )

def getRsvpLerFec( mount, tunnelIndex ):

   tunnelIndexMap = mount.rsvpLerStatus.tunnelIndexMap
   rsvpSpecId = tunnelIndexMap.tunnelSpecIdForTunnelIndex.get( tunnelIndex )
   if not rsvpSpecId:
      return None, "No RSVP tunnel for tunnel index: %d" % tunnelIndex
   subTunnelStatusColl = mount.rsvpLerStatus.subTunnelStatusCollTable.\
                           subTunnelStatusColl.get( rsvpSpecId )
   if not subTunnelStatusColl:
      return None, "No RSVP tunnel for tunnel index: %d" % tunnelIndex

   # TODO: BUG543185 should stop hard coding, also check
   # if the warnings can be improved
   subTunnelConfigId = RsvpLerSubTunnelConfigId( rsvpSpecId, 0 )
   subTunnelStatus = subTunnelStatusColl.subTunnelStatus.get( subTunnelConfigId )
   if not subTunnelStatus:
      return None, "No RSVP tunnel for tunnel index: %d" % tunnelIndex
   if subTunnelStatus.tunnelState != RvspLerTunnelState.tunnelStateUp:
      return None, "RSVP tunnel not up for tunnel index: %d" % tunnelIndex
   if subTunnelStatus.nextHop:
      nextHop = subTunnelStatus.nextHop.values()[ 0 ]
   else:
      nextHop = RsvpNextHopInfo()
   nexthopAndLabel = NextHopAndLabel( nextHop.addr,
                                      labelStackToList( nextHop.labelStack ),
                                      nextHop.intf )
   return nexthopAndLabel, None

def getLdpFec( mount, prefix, allowEntropyLabel=False ):
   '''
   Gets the list of L3 resolved IP nexthops with the corresponding label stack. This
   will also resolve over an RSVP tunnel if it is LDP over RSVP.
   '''
   if isIpv6Addr( prefix ):
      err = 'Ipv6 Address not supported'
      return ( None, err, errno.EINVAL )

   coloredTep = ColoredTunnelEndPoint( IpGenPrefix( prefix ), 0, False )
   ribEntry = mount.ldpTunnelRib.entry.get( coloredTep )
   if not ribEntry or not ribEntry.tunnelId:
      err = 'No tunnel found for prefix %s' % prefix
      return ( None, err, errno.EINVAL )

   nextHopsAndLabels = []
   for tId in ribEntry.tunnelId.values():
      # examine all possible tunnel vias in the tunnel table
      tunnelTableEntry = mount.ldpTunnelTable.entry.get( tId )
      if not tunnelTableEntry:
         continue
      for via in tunnelTableEntry.via.values():
         if via.labels.stackSize != 1:
            continue
         intfId = via.intfId
         nexthop = via.nexthop.v4Addr
         if allowEntropyLabel:
            stack = via.labels
            ldpLabelStack = [ stack.calculatedLabelStack( i )
                              for i in range( stack.calculatedStackSize ) ]

            # calculatedLabelStack has index 0 = BOS
            ldpLabelStack.reverse()
         else:
            ldpLabelStack = [ via.labels.labelStack( 0 ) ]

         # resolve subtunnel with forwardingHelper
         if intfId and DynTunnelIntfId.isDynamicTunnelIntfId( intfId ):
            viaTunnelId = TunnelId( DynTunnelIntfId.tunnelId( intfId ) )
            subTunnelInfo = resolveTunnel( mount.fwdingHelper, viaTunnelId )
            if subTunnelInfo.errMsg:
               continue
            nextHopsAndLabels.append( NextHopAndLabel( str( subTunnelInfo.nexthop ),
                                                       subTunnelInfo.labelStack +
                                                         ldpLabelStack,
                                                       subTunnelInfo.intfId ) )
         else:
            nextHopsAndLabels.append( NextHopAndLabel( str( nexthop ), ldpLabelStack,
                                                       intfId ) )

   errVal = errno.EINVAL if not nextHopsAndLabels else 0
   if errVal:
      err = 'No tunnel found or tunnel could not be resolved for tunnelId'
      return( None, err, errVal )
   return ( nextHopsAndLabels, "", errVal )

def getMldpFec( mount, prefix, genOpqVal, sourceAddrOpqVal, groupAddrOpqVal ):
   if isIpv6Addr( prefix ):
      err = 'Ipv6 Address not supported'
      return ( None, err, errno.EINVAL )

   if genOpqVal:
      opqStr = genOpaqueValForP2mpGenericOpaqueId( genOpqVal )
   elif sourceAddrOpqVal and groupAddrOpqVal:
      if isIpv6Addr( sourceAddrOpqVal ) and isIpv6Addr( groupAddrOpqVal ):
         opqStr = genOpaqueValForP2mpTransitV6SrcOpaqueId( sourceAddrOpqVal,
                                                           groupAddrOpqVal )
      elif isIpv4Addr( sourceAddrOpqVal ) and isIpv4Addr( groupAddrOpqVal ):
         opqStr = genOpaqueValForP2mpTransitV4SrcOpaqueId( sourceAddrOpqVal,
                                                           groupAddrOpqVal )
      else:
         err = 'Invalid Source/Group Address Opaque Value'
         return ( None, err, errno.EINVAL )
   else:
      assert False, 'Invalid Opaque value'

   opqIndex = 0 # invalid opaque index
   opqValColl = mount.mldpOpaqueValueTable.opaqueValToIndexColl
   if opqValColl and opqValColl.opaqueVal.get( opqStr ):
      opqIndex = opqValColl.opaqueVal.get( opqStr ).oValIndex

   nextHopsAndLabels = []
   for route in mount.mldpLfib.lfibRoute.values():
      # p2mpfec is formed from rootIp and opqIndex
      if ( route.fec.mldpRootIp.stringValue == prefix and
           route.fec.mldpOpaqueId == opqIndex ):
         vsk = route.viaSetKey
         vs = mount.mldpLfib.viaSet.get( vsk )
         for vk in vs.viaKey.values():
            via = mount.mldpLfib.mplsVia.get( vk )
            if via:
               nexthopAddr = via.nextHop.v4Addr
               label = via.labelStack.topLabel()
               nextHopAndLabel = NextHopAndLabel( nexthopAddr, label, None )
               nextHopsAndLabels.append( nextHopAndLabel )
   errVal = errno.EINVAL if not nextHopsAndLabels else 0
   if errVal:
      err = 'No tunnel found for prefix %s' % prefix
      return ( None, err, errVal )
   return ( nextHopsAndLabels, "", errVal )

def getTunnelTableAndRibEntry( mount, coloredTep, igpProtocol ):
   if igpProtocol == igpProtocolType.isis:
      table = mount.srTunnelTable
      ribEntry = mount.srTunnelRib.entry.get( coloredTep )
   else:
      table = mount.ospfSrTunnelTable
      ribEntry = mount.ospfSrTunnelRib.entry.get( coloredTep )
   return table, ribEntry

def getSrFec( mount, prefix, algorithm=None, igpProtocol=None ):
   if algorithm:
      igpProtocol = igpProtocolType.isis
      table = mount.isisFlexAlgoTunnelTable
      fad = mount.flexAlgoConfig.definition.get( int( algorithm ) )
      if fad is None:
         err = 'Flexible algorithm %s is not defined' % algorithm
         return ( None, err, errno.EINVAL, None )
      color = fad.color if fad.colorDefined else 0
      coloredTep = ColoredTunnelEndPoint( IpGenPrefix( prefix ),
                                          color, fad.colorDefined )
      ribEntry = mount.isisFlexAlgoTunnelRib.entry.get( coloredTep )
   else:
      coloredTep = ColoredTunnelEndPoint( IpGenPrefix( prefix ), 0, False )
      if igpProtocol:
         table, ribEntry = getTunnelTableAndRibEntry( mount, coloredTep,
                                                      igpProtocol )
      else:
         # When igpProtocol is not specified, first look into IS-IS table bindings
         # and then look into OSPF table bindings.
         igpProtocol = igpProtocolType.isis
         table, ribEntry = getTunnelTableAndRibEntry( mount, coloredTep,
                                                      igpProtocolType.isis )
         if not ribEntry or not ribEntry.tunnelId:
            igpProtocol = igpProtocolType.ospf
            table, ribEntry = getTunnelTableAndRibEntry( mount, coloredTep,
                                                         igpProtocolType.ospf )

   if not ribEntry or not ribEntry.tunnelId:
      err = 'No tunnel found for prefix %s' % prefix
      return ( None, err, errno.EINVAL, None )

   nextHopsAndLabels = []
   for tunnelId in ribEntry.tunnelId.values():
      tunnelTableEntry = table.entry.get( tunnelId )
      if not tunnelTableEntry:
         continue

      if algorithm:
         # If algorithm is provided, we will look for flex algo tunnel with
         # matching algorithm ID.
         if TunnelId( tunnelId ).tunnelType() != TunnelType.isisFlexAlgoTunnel:
            continue
         if TunnelId( tunnelId ).extractFlexAlgoId() != algorithm:
            continue

      for via in tunnelTableEntry.via.values():
         nexthopList, intfIdList = [], []
         if via.labels.stackSize != 1:
            continue
         label = getSrNexthopLabel( mount, via )
         if label is None:
            continue

         if DynTunnelIntfId.isDynamicTunnelIntfId( via.intfId ):
            # ISIS SR tunnels might point to a TI-LFA tunnel or RSVP tunnel
            # get the primary nexthop, primary interface from the
            # resolving tunnel
            viaTunnelId = TunnelId( DynTunnelIntfId.tunnelId( via.intfId ) )
            subTunnelInfo = resolveTunnel( mount.fwdingHelper, viaTunnelId )
            if subTunnelInfo.errMsg:
               continue
            nexthop = subTunnelInfo.nexthop
            nexthopAddr = ( nexthop.v4Addr if nexthop.af == IPv4 else
                              nexthop.v6Addr )
            labelStack = subTunnelInfo.labelStack
            if label != 3 or not subTunnelInfo.labelStack:
               labelStack += [ label ]
            nhAndLabel = NextHopAndLabel( nexthopAddr,
                                          labelStack,
                                          subTunnelInfo.intfId )
            nextHopsAndLabels.append( nhAndLabel )

         elif FecIdIntfId.isFecIdIntfId( via.intfId ):
            # ISIS SR tunnels may be optimized as HFECs, so try
            # resolving the NextLevelFecId if possible
            nexthopInfoList = mount.fwdingHelper.resolveHierarchical(
               True, l3IntfId=via.intfId )
            for nexthopInfo in nexthopInfoList:
               nexthopAddr = nexthopInfo.nexthopIp
               labelStack = nexthopInfo.labelStack
               if label != implicitNull or not nexthopInfo.labelStack:
                  labelStack += [ label ]
               nhAndLabel = NextHopAndLabel( nexthopAddr,
                                             labelStack,
                                             nexthopInfo.intf )
               nextHopsAndLabels.append( nhAndLabel )

         else:
            nexthopList, intfIdList = [ via.nexthop ], [ via.intfId ]
            for nexthop, intfId in zip( nexthopList, intfIdList ):
               assert nexthop.af in [ IPv4, IPv6 ]
               nexthopAddr = ( nexthop.v4Addr if nexthop.af == IPv4 else
                               nexthop.v6Addr )
               nextHopAndLabel = NextHopAndLabel( nexthopAddr, label, intfId )
               nextHopsAndLabels.append( nextHopAndLabel )

   errVal = errno.EINVAL if not nextHopsAndLabels else 0
   if errVal:
      err = 'No tunnel found for tunnelId'
      return( None, err, errVal, None )

   return ( nextHopsAndLabels, "", errVal, igpProtocol )

def getPwAttr( mount, connectorKey, instanceName=None, groupName=None,
               remoteRouterId=None ):
   if instanceName:
      # If we've been given a instance name, then we are trying to retrieve
      # a pseudowire created using a profile
      groupKey = Tac.Type( "Pseudowire::VplsGroupKey" )( instanceName, groupName )
      vplsUserKey = PseudowireUserKey.vplsLdpProfilePseudowireUserKey(
         groupKey, remoteRouterId )
      return mount.pwAttr.connectorStatus.get( vplsUserKey )
   patchUserKey = PseudowireUserKey.fromConnectorKey( connectorKey )
   pwAttr = mount.pwAttr.connectorStatus.get( patchUserKey )
   if pwAttr is None:
      vplsUserKey = PseudowireUserKey.vplsLdpPseudowireUserKey( patchUserKey.name() )
      pwAttr = mount.pwAttr.connectorStatus.get( vplsUserKey )
   return pwAttr

def getPwLdpFec( mount, pwLdpName ):
   # Obtain local router id from LDP config from default vrf
   localRouterId = None
   if mount.ldpProtoConfig.protoConfig.get( "default" ):
      localRouterId = mount.ldpProtoConfig.protoConfig[ "default" ].routerId
   if localRouterId is None:
      return None, None, "Local router ID is not configured"

   connectorKey = ConnectorKey()
   remoteRouterId = None
   pwId = 0
   pwGenId = PseudowireGenId.invalid
   instanceName = None
   groupName = None
   m = re.match( r'profile (\S+) (\S+) (\S+)', pwLdpName )
   if m:
      instanceName = m.group( 1 )
      groupName = m.group( 2 )
      remoteRouterId = IpGenAddr( m.group( 3 ) )
   else:
      connectorKey.ldpConnectorKeyIs( pwLdpName )
      pwConnectorName = f"Pseudowire {pwLdpName}"

      remoteConnector = mount.pwConfig.connector.get( connectorKey )
      if remoteConnector is None:
         return None, None, f"{pwConnectorName} is not configured"

   # Obtain remote router id from config
   if not instanceName:
      if not remoteConnector.neighborAddrPresent:
         return ( None, None, ( "Neighbor address for pseudowire "
                                "{} is not configured".format( pwLdpName ) ) )
      remoteRouterId = remoteConnector.neighborAddr
      if not remoteConnector.pwIdPresent:
         return ( None, None, ( "Pseudowire ID for pseudowire "
                                "{} is not configured".format( pwLdpName ) ) )
      pwId = remoteConnector.pwId

   pwLdpInfo = PwLdpInfo( IpGenAddr( localRouterId ), remoteRouterId,
                          pwId, pwGenId, None, None, None, None, None )

   # Obtain PwAttr
   pwAttr = getPwAttr( mount, connectorKey, instanceName, groupName, remoteRouterId )
   # If we parsed an instanceName, then we are trying to ping a pseudowire created
   # using a profile.
   if instanceName:
      if pwAttr is None:
         return ( None, None, (
            "No profile pseudowire for group ({}, {}) with neighbor address "
            "{}".format( instanceName, groupName, remoteRouterId ) ) )
      pwGenId = pwAttr.pwGenId
      pwLdpInfo = PwLdpInfo( IpGenAddr( localRouterId ), remoteRouterId,
                             pwId, pwGenId, None, None, None, None, None )
   else:
      if pwAttr is None or not pwAttr.peerInfo:
         return pwLdpInfo, None, "No tunnel in the tunnel RIB to reach this neighbor"

   status = pwAttr.status
   # pylint: disable-next=consider-using-in
   if ( status != PseudowireConnectorStatus.up and
        status != PseudowireConnectorStatus.standby ):
      statusMsg = PseudowireConnectorStatusMsg.get( status )
      if statusMsg is None:
         statusMsg = "Unknown status %s" % status
      return ( pwLdpInfo, None,
               f"Invalid pseudowire connector status '{statusMsg}'" )

   # Obtain relevant PW info from PwAttr/config
   # BUG497800: add support for multiple active peers
   peerInfo = next( iter( pwAttr.peerInfo.values() ) )
   vcLabel = peerInfo.peerLabel
   pwType = pwAttr.remotePwType
   cvTypes = pwAttr.peerVccvCvTypes
   ccTypes = pwAttr.peerVccvCcTypes
   controlWord = peerInfo.encapControlWord
   pwLdpInfo = PwLdpInfo( IpGenAddr( localRouterId ), remoteRouterId,
                          pwId, pwGenId, pwType, vcLabel, cvTypes, ccTypes,
                          controlWord )

   # Lookup forwarding info.
   # We don't care about transport interface at the moment so just getting the
   # tunnelId is okay.
   tunnelId = peerInfo.firstTunnelId()
   if not tunnelId:
      return pwLdpInfo, None, "No tunnel in the tunnel RIB to reach this neighbor"

   nextHopAndLabels = getTunnelInfo( mount, tunnelId )
   if nextHopAndLabels is None:
      return pwLdpInfo, None, "Tunnel does not use MPLS encapsulation"
   return pwLdpInfo, nextHopAndLabels, None

def getTunnelInfo( mount, tunnelId ):
   if TunnelId( tunnelId ).tunnelType() == TunnelType.srTePolicyTunnel:
      fecId = FecId.tunnelIdToFecId( tunnelId )
      nexthopInfoList = mount.fwdingHelper.resolveHierarchical(
         True, fecId=fecId )
      if not nexthopInfoList:
         return None
      nexthopInfo = nexthopInfoList[ 0 ]
      nhAndLabels = NextHopAndLabel( IpGenAddr( str( nexthopInfo.nexthopIp ) ),
                                     nexthopInfo.labelStack,
                                     nexthopInfo.intf )
   else:
      tunnelInfo = resolveTunnel( mount.fwdingHelper, tunnelId )
      if tunnelInfo.errMsg:
         return None
      nhAndLabels = NextHopAndLabel( nextHopIp=tunnelInfo.nexthop,
                                     label=tunnelInfo.labelStack,
                                     intfId=tunnelInfo.intfId )

   return nhAndLabels

def getFibRoute( mount, prefix ):
   if isIpv6Addr( prefix ):
      tacType = 'Arnet::Ip6Prefix'
      routeStatus = mount.route6Status
      addr, pLen = prefix.split( '/' )
      routeKey = Tac.Value( tacType, Arnet.Ip6Addr( addr ), int( pLen ) )
   else:
      tacType = 'Arnet::Prefix'
      routeStatus = mount.routeStatus
      routeKey = Tac.Value( tacType, stringValue=prefix )
   if routeKey not in routeStatus.route:
      return None

   fibRoute = routeStatus.route.get( routeKey )
   return fibRoute

def getVpnFec( mount, prefix ):
   fibRoute = getFibRoute( mount, prefix )
   useFibGen = toggleFibGenMountPathEnabled()
   if ( not fibRoute or
        # pylint: disable-next=consider-using-in
        ( fibRoute.routeType != RouteType.ibgp and
          fibRoute.routeType != RouteType.ebgp and
          fibRoute.routeType != Route6Type.bgp ) ):
      err = 'Route for %s not configured' % prefix
      return ( None, err, errno.EINVAL )
   if fibRoute.fecId == FecIdConstants.invalidFecId:
      err = 'Invalid nexthop ID'
      return ( None, err, errno.EINVAL )

   if useFibGen:
      fec = ( mount.forwardingGenStatus.fec.get( fibRoute.fecId ) or
              mount.forwardingGenStatusDefaultVrf.fec.get( fibRoute.fecId ) )
   else:
      fec = ( mount.forwardingStatus.fec.get( fibRoute.fecId ) or
              mount.forwarding6Status.fec.get( fibRoute.fecId ) or
              mount.forwardingStatusDefaultVrf.fec.get( fibRoute.fecId ) or
              mount.forwarding6StatusDefaultVrf.fec.get( fibRoute.fecId ) )
   if not fec:
      err = 'No fec found'
      return ( None, err, errno.EINVAL )
   return ( fec, "", 0 )

def getStaticFec( mount, prefix ):
   fibRoute = getFibRoute( mount, prefix )
   if not fibRoute:
      return ( None, errno.EINVAL )
   if fibRoute.fecId == FecIdConstants.invalidFecId:
      print( 'Invalid nexthop ID' )
      return ( None, errno.EINVAL )
   if toggleFibGenMountPathEnabled():
      forwardingStatus = mount.forwardingGenStatus
   elif isIpv6Addr( prefix ):
      forwardingStatus = mount.forwarding6Status
   else:
      forwardingStatus = mount.forwardingStatus
   fec = forwardingStatus.fec.get( fibRoute.fecId )
   if not fec:
      return ( None, errno.EINVAL )
   
   return ( fec, 0 )

def makeSubTunnelConfigId( tunnelSpecId, subTunnelId=0 ):
   subTunnelConfigId = Tac.ValueConst( 'Rsvp::RsvpLerSubTunnelConfigId',
                                       tunnelSpecId, subTunnelId )
   return subTunnelConfigId

def makeTunnelSpecId( tunnelName, tunnelSource=tunnelSourceCli ):
   tunnelSpecId = Tac.ValueConst( 'Rsvp::RsvpLerTunnelSpecId',
                            tunnelName, tunnelSource )
   return tunnelSpecId

def getRsvpTunnelInfo( mount, rsvpTunnel, rsvpSubTunnel ):
   spRxStatusColl = mount.rsvpSharkStatus.spRxStatusColl
   spIdMap = mount.rsvpLerStatus.spIdMap
   subTunnels = []
   rsvpSpIds = []
   nextHopsAndLabels = []
   rsvpSenderAddr = None
   tunnelSpecId = makeTunnelSpecId( rsvpTunnel )
   subTunnelStatusColl = mount.rsvpLerStatus.subTunnelStatusCollTable.\
                            subTunnelStatusColl.get( tunnelSpecId )
   if not subTunnelStatusColl:
      print( 'No RSVP tunnel found' )
      return ( None, errno.EINVAL, None, None, None )

   if not rsvpSubTunnel:
      # Case where only tunnel name is provided
      # get all subtunnels active lsps
      for subTunnelConfigId in subTunnelStatusColl.subTunnelStatus:
         subTunnelStatus = \
            subTunnelStatusColl.subTunnelStatus.get( subTunnelConfigId )
         rsvpLerLspRequestConfigId = subTunnelStatus.activeLsp
         spId = spIdMap.spIdForLspRequestConfigId.get( rsvpLerLspRequestConfigId )
         if not spId:
            continue

         if subTunnelStatus.nextHop:
            nextHop = subTunnelStatus.nextHop.values()[ 0 ]
         else:
            nextHop = RsvpNextHopInfo()
         nexthopAddr = ( nextHop.addr.v4Addr if nextHop.addr.af == IPv4 else
                         nextHop.addr.v6Addr )
         labelStack = labelStackToList( nextHop.labelStack )
         nextHopAndLabel = NextHopAndLabel( nexthopAddr, labelStack, None )
         nextHopsAndLabels.append( nextHopAndLabel )
         rsvpSpIds.append( spId )
         subTunnels.append( subTunnelConfigId.subTunnelId )
   else:
      # get specific subtunnel id info for the provided tunnel
      subTunnelConfigId = makeSubTunnelConfigId( tunnelSpecId, rsvpSubTunnel )
      splitSubTunnelStatus = subTunnelStatusColl.subTunnelStatus.get(
         subTunnelConfigId, RsvpLerSubTunnelStatus( subTunnelConfigId,
                                                    RsvpSessionType.p2pLspTunnel ) )
      if not splitSubTunnelStatus:
         print( 'No RSVP sub-tunnel found' )
         return ( None, errno.EINVAL, None, None, None )
      rsvpLerLspRequestConfigId = splitSubTunnelStatus.activeLsp
      spId = spIdMap.spIdForLspRequestConfigId.get( rsvpLerLspRequestConfigId )

      if splitSubTunnelStatus.nextHop:
         nextHop = splitSubTunnelStatus.nextHop.values()[ 0 ]
      else:
         assert False
         nextHop = RsvpNextHopInfo()
      nexthopAddr = ( nextHop.addr.v4Addr if nextHop.addr.af == IPv4 else
                      nextHop.addr.v6Addr )
      labelStack = labelStackToList( nextHop.labelStack )
      nextHopAndLabel = NextHopAndLabel( nexthopAddr, labelStack, None )
      nextHopsAndLabels.append( nextHopAndLabel )
      if spId:
         rsvpSpIds.append( spId )
      subTunnels.append( rsvpSubTunnel )

   if not rsvpSpIds:
      print( 'No Mapping for Session Path ID found' )
      return ( None, errno.EINVAL, None, None, None )

   spRxStatus = spRxStatusColl.spRxStatus.get( rsvpSpIds[ 0 ] )
   if spRxStatus:
      rsvpSenderAddr = spRxStatus.senderAddr

   if not rsvpSenderAddr:
      print( 'No RSVP sender address found' )
      return ( None, errno.EINVAL, rsvpSpIds, None, None )

   if not nextHopsAndLabels:
      print( 'No Nexthop information found' )
      return ( None, errno.EINVAL, rsvpSpIds, rsvpSenderAddr, None )

   return ( nextHopsAndLabels, 0, rsvpSpIds, rsvpSenderAddr, subTunnels )


def getRsvpFec( mount, session, lsp ):
   sessionIdToCliIdColl = mount.rsvpStatus.sessionIdToCliIdColl
   sessionStateColl = mount.rsvpSharkStatus.sessionStateColl
   spConfigColl = mount.rsvpSharkStatus.spConfigColl
   spStatusColl = mount.rsvpSharkStatus.spStatusColl
   spRxStatusColl = mount.rsvpSharkStatus.spRxStatusColl
   rsvpSpIds = []
   rsvpSenderAddr = None
   spConfigs = []
   spStatuses = []
   spRxStatuses = []
   lspIds = []

   if not sessionStateColl:
      print( 'No RSVP session found' )
      return ( None, errno.EINVAL, None, None, None )

   if not spConfigColl or not spStatusColl or not spRxStatusColl:
      print( 'No RSVP LSP found' )
      return ( None, errno.EINVAL, None, None, None )

   #pylint: disable-msg=too-many-nested-blocks
   # SESSION BY ID
   if isinstance( session, int ):
      for sessionState in sessionStateColl.sessionState.values():
         cliId = sessionIdToCliIdColl.sessionIdToCliId.get( sessionState.sessionId )
         if cliId == session:
            for spId in sessionState.spMember:
               spConfig = spConfigColl.spConfig.get( spId )
               spStatus = spStatusColl.spStatus.get( spId )
               spRxStatus = spRxStatusColl.spRxStatus.get( spId )
               if spConfig is not None and spStatus is not None and \
                  spRxStatus is not None:
                  if not spStatus.operational:
                     # Ignore LSPs that are not operational, ping simply cannot
                     # work on them
                     continue
                  # If the LSP ID was not specified, store all the possible LSPs;
                  # if it was, stop at the first match
                  if lsp is None:
                     rsvpSpIds.append( spConfig.spId )
                     spConfigs.append( spConfig )
                     spStatuses.append( spStatus )
                     spRxStatuses.append( spRxStatus )
                     lspIds.append( spStatus.spCliId )
                     rsvpSenderAddr = spRxStatus.senderAddr
                  elif spStatus.spCliId == lsp:
                     rsvpSpIds = [ spConfig.spId ]
                     spConfigs = [ spConfig ]
                     spStatuses = [ spStatus ]
                     spRxStatuses = [ spRxStatus ]
                     lspIds = [ lsp ]
                     rsvpSenderAddr = spRxStatus.senderAddr
                     break
   # SESSION BY NAME
   else:
      for spId, spConfig in spConfigColl.spConfig.items():
         spStatus = spStatusColl.spStatus.get( spId )
         spRxStatus = spRxStatusColl.spRxStatus.get( spId )
         # Get all the LSP which match sessionName
         if spConfig is not None and spStatus is not None and \
            spRxStatus is not None and \
            spStatus.sessionAttributes.sessionName == session:
            if not spStatus.operational:
               # Ignore LSPs that are not operational, ping simply cannot
               # work on them
               continue
            rsvpSpIds.append( spConfig.spId )
            spConfigs.append( spConfig )
            spStatuses.append( spStatus )
            spRxStatuses.append( spRxStatus )
            lspIds.append( spStatus.spCliId )
            rsvpSenderAddr = spRxStatus.senderAddr
   #pylint: enable-msg=too-many-nested-blocks

   if not rsvpSpIds or not spConfigs or not spStatuses or not spRxStatuses:
      if lsp:
         print( 'No operational RSVP tunnel found for session %s and LSP %s' %
                ( session, lsp ) )
      else:
         print( 'No operational RSVP tunnel found for session %s' % session )
      return ( None, errno.EINVAL, None, None, None )
   
   if spConfigs[ 0 ].sessionRole == 'egressSessionRole':
      print( 'Session role is egress, nothing to do' )
      return ( None, 0, None, None, None )

   # List of all possible adjacencies to reach the destination. In RSVP case, there
   # is only one possible adjacency per SP.
   nextHopsAndLabels = []
   for spConfig, spStatus in zip( spConfigs, spStatuses ):
      # Ping originating from the PLR will need to go through the bypass tunnel in
      # the sceario of FRR, this requires to get the new next hop and the label for
      # this tunnel, in addition to the label of our previously active downstream.
      if spConfig.dsFrrInUse:
         nexthop = spConfig.bypassInfo.bypassNextHop.v4Addr
         bypassLabel = spConfig.bypassInfo.bypassLabel
         # Retrieve the MP label depending on whether link or node protection
         # is used. With link protection, the label of the MP is the same as
         # the previously used downstream label. With node protection, the
         # label of the MP is the next next hop's label.
         # Note: If the MP is at the egress, its label will be implicit null.
         # We add it regardless, it will be removed later and will not actually
         # be encapsulated.
         if spConfig.bypassInfo.bypassNodeProtected:
            mpLabel = spStatus.nnhLabel
         else:
            mpLabel = spStatus.dsLabel
         labelStack = [ bypassLabel, mpLabel ]
      else:
         nexthop = spStatus.primaryDs.neighborIp.v4Addr
         labelStack = [ spStatus.dsLabel ]
      nextHopAndLabel = NextHopAndLabel( nexthop, labelStack, None )
      nextHopsAndLabels.append( nextHopAndLabel )

   # nextHopsAndLabels and rsvpSpIds contain at least one element. They can't
   # be empty. lspIds is used to record the lsp id of each tunnel. It contains
   # at least one element.
   return ( nextHopsAndLabels, 0, rsvpSpIds, rsvpSenderAddr, lspIds )

# Calculate the bitset for the specified multipath type
def getMultipathBitset( multipathType, numMultipathBits ):
   multipathBitset = LspPingMultipathBitset()

   if multipathType != 0:
      # Do a lookup for the specified baseip

      # The size of the multipathBitset should be:
      #  "mask of length 2^(32-prefix length) bits"
      # as per RFC4379, and a minimum of 4 bytes.
      multipathBitset.size = ( 2 ** ( numMultipathBits - 1 ).bit_length() ) // 8

      if multipathBitset.size < 4: # pylint: disable=consider-using-max-builtin
         multipathBitset.size = 4

      for byteIndex in range( 0, multipathBitset.size ):
         if numMultipathBits > 7:
            multipathBitset.bitset[ byteIndex ] = 0xFF
            numMultipathBits = numMultipathBits - 8
         else:
            tempByte = 0
            for _ in range( 0, numMultipathBits ):
               tempByte = tempByte << 1
               tempByte = tempByte + 1

            for _ in range( numMultipathBits, 8 ):
               tempByte = tempByte << 1

            multipathBitset.bitset[ byteIndex ] = tempByte
            numMultipathBits = 0

   return multipathBitset

def getMultipathType( multipath, numMultipathBits ):
   if multipath:
      if numMultipathBits == 0:
         return 0
      else:
         return 8

   return 0

# Note that this may return a different value for numMultipathBits than what
# was passed in
def getMultipathInfo( multipath, numMultipathBits ):
   multipathType = getMultipathType( multipath, numMultipathBits )

   if multipathType == 0:
      numMultipathBits = 0

   multipathBitset = getMultipathBitset( multipathType, numMultipathBits )

   return ( multipathType, numMultipathBits, multipathBitset )

# Generate downstream mapping info for LDP, RSVP and Segment-Routing traceroute
def getDsMappingInfo( nexthop, labelStack, l3IntfMtu, multipath, baseip,
                      numMultipathBits ):
   dsLabelStack = Tac.newInstance( 'LspPing::LspPingDownstreamLabelStack' )
   dsFecStackChangeInfo = Tac.newInstance( "LspPing::LspPingFecStackChangeInfo" )
   # The given via's label stack should always be a tuple or a list
   stackSize = len( labelStack )
   for i, label in enumerate( labelStack ):
      dsLabelEntry = Tac.newInstance( 'LspPing::LspPingDownstreamLabelEntry' )
      dsLabelEntry.label = label
      dsLabelEntry.bos = ( stackSize == i + 1 )
      assert label <= MplsLabel.max, "Cannot assign a non-valid MPLS label"
      dsLabelStack.labelEntry[ i ] = dsLabelEntry
   dsLabelStack.size = stackSize

   ipGenVia = IpGenAddr( nexthop )
   addrType = 1 if ipGenVia.af == AddressFamily.ipv4 else 3

   multipathType, numMultipathBits, multipathBitset = \
      getMultipathInfo( multipath, numMultipathBits )

   dsMappingAddr = IpGenAddr( baseip )
   dsMappingInfo = Tac.newInstance( 'LspPing::LspPingDownstreamMappingInfo',
                                    l3IntfMtu, addrType, 0, ipGenVia, ipGenVia,
                                    multipathType, 0, dsMappingAddr,
                                    multipathBitset, dsLabelStack,
                                    dsFecStackChangeInfo, IntfId(),
                                    0, 0, # retcode, retsubcode
                                    True )
   return dsMappingInfo

def resolveVpnTunnel( mount, fecId ):
   adjToTunnels = {}
   tunnelToAdjs = {}
   nhInfoList = mount.fwdingHelper.resolveHierarchical( allowEncap=True,
                                                        fecId=fecId,
                                                        selectOneVia=False )
   if nhInfoList == [ noMatchNexthopInfo ]:
      err = "None of the segment lists for the policy are valid"
      return ( errno.EINVAL, err )

   # This mapping is useful while we print the ping statistics from
   # the echo reply we get.
   for info in nhInfoList:
      tunEntry = ( info.nexthopIp, tuple( info.labelStack ) )
      tunKey = ( tuple( info.labelStack ), info.dstMac, info.intf )
      adjToTunnels.setdefault( tunKey, [] ).append( tunEntry )

   tunnelToAdjs = { tuple( v ) : k for k, v in adjToTunnels.items() }
   return ( tunnelToAdjs, "" )

def _resolveSrTePolicyTunnels( mount, endpoint, color, trafficAf=None ):
   epAddr = IpGenAddr( str( endpoint ) )
   policyEndpoint = epAddr
   policyKey = PolicyKey( policyEndpoint, color )
   ps = mount.policyStatus.status.get( policyKey, None )
   if not ps:
      print( "There is no active candidate path for the policy" )
      return errno.EINVAL
   policyFecId = ps.labelFecId
   if trafficAf:
      policyFecId = ps.v4FecId if trafficAf == 'v4' else ps.v6FecId
   policyFec = mount.srTeForwardingStatus.fec.get( policyFecId, None )

   if not policyFec:
      print( "The FEC for the policy has not been programmed" )
      return errno.EINVAL

   return _resolveSrTePolicyFec( mount, policyFecId )

def _resolveSrTePolicyFec( mount, policyFecId ):
   nhInfoList = mount.fwdingHelper.resolveHierarchical( allowEncap=True,
                                                        fecId=policyFecId,
                                                        selectOneVia=False )
   if nhInfoList == [ noMatchNexthopInfo ]:
      print( "None of the segment lists for the policy are valid" )
      return errno.EINVAL

   # Map each of the (nhIntf, nhMac, segmentList) tuple to a unique clientId.
   # Also map the (segmentList, nhIp) tuple to a set of (nhIntf,nhMac,labelStack)
   # tuples. So In a way we indirectly map the clientId to a set of
   # ( segmentList, nhIp ) tuples called tunnels. This mapping is useful while we
   # print the ping statistics from the echo reply we get.

   adjToTunnels = {}
   tunnelToAdjacencies = {}
   for info in nhInfoList:
      tunEntry = ( info.nexthopIp, tuple( info.labelStack ) )
      tunKey = ( tuple( info.labelStack ), info.dstMac, info.intf )
      adjToTunnels.setdefault( tunKey, [] ).append( tunEntry )

   tunnelToAdjacencies = { tuple( v ) : k for k, v in adjToTunnels.items() }
   return tunnelToAdjacencies

def resolveNexthopDefaultVrf( mount, state, nexthop, intf=None ):
   isV4 = isIpv4Addr( nexthop )
   if isV4:
      if not state.ipv4RoutingSim:
         ipv4RoutingSim = Tac.newInstance( 'Routing::RoutingSimulator', DEFAULT_VRF,
                                           mount.routingVrfInfoDir,
                                           mount.routeStatusDefaultVrf,
                                           mount.forwardingStatusDefaultVrf,
                                           mount.forwarding6StatusDefaultVrf,
                                           mount.forwardingGenStatusDefaultVrf,
                                           mount.arpSmash,
                                           mount.arpSmashVrfNameToIdMap,
                                           mount.ipStatus,
                                           mount.ip6Status,
                                           None,
                                           None )
         state.ipv4RoutingSim = ipv4RoutingSim
      rsim = state.ipv4RoutingSim
   else:
      if not state.ipv6RoutingSim:
         ipv6RoutingSim = Tac.newInstance( 'Routing6::Routing6Simulator',
                                           DEFAULT_VRF,
                                           mount.routing6VrfInfoDir,
                                           mount.route6StatusDefaultVrf,
                                           mount.forwardingStatusDefaultVrf,
                                           mount.forwarding6StatusDefaultVrf,
                                           mount.forwardingGenStatusDefaultVrf,
                                           mount.arpSmash,
                                           mount.arpSmashVrfNameToIdMap,
                                           mount.ipStatus,
                                           mount.ip6Status,
                                           None,
                                           None )
         state.ipv6RoutingSim = ipv6RoutingSim
      rsim = state.ipv6RoutingSim

   if toggleFibGenMountPathEnabled():
      nexthop = Arnet.IpAddr( nexthop ) if isV4 else Arnet.Ip6Addr( nexthop )

   routingOutput = rsim.route( nexthop )
   if routingOutput.outputType in [ RoutingOutputType.invalid,
                                    RoutingOutputType.cpu ]:
      if isV4 or intf is None:
         return ( None, None )
      else:
         return ( intf, getIpv6ArpEntry( mount, nexthop, intf ) )
   if routingOutput.nextHopEthAddr == EthAddr().stringValue or \
      routingOutput.nextHopIntf == IntfId( '' ):
      return ( None, None )

   # FIXME routingOutput.outputType == multiple
   return ( routingOutput.nextHopIntf, routingOutput.nextHopEthAddr )

def resolveNexthop( mount, state, nexthop, intf=None ):
   isV4 = isIpv4Addr( nexthop )
   if isV4:
      if not state.ipv4RoutingSim:
         ipv4RoutingSim = Tac.newInstance( 'Routing::RoutingSimulator', mount.vrf,
                                           mount.routingVrfInfoDir,
                                           mount.routeStatus, 
                                           mount.forwardingStatus,
                                           mount.forwarding6Status,
                                           mount.forwardingGenStatus,
                                           mount.arpSmash,
                                           mount.arpSmashVrfNameToIdMap,
                                           mount.ipStatus,
                                           mount.ip6Status,
                                           None,
                                           None )
         state.ipv4RoutingSim = ipv4RoutingSim
      rsim = state.ipv4RoutingSim
   else:
      if not state.ipv6RoutingSim:
         ipv6RoutingSim = Tac.newInstance( 'Routing6::Routing6Simulator', mount.vrf,
                                           mount.routing6VrfInfoDir,
                                           mount.route6Status,
                                           mount.forwardingStatus,
                                           mount.forwarding6Status,
                                           mount.forwardingGenStatus,
                                           mount.arpSmash,
                                           mount.arpSmashVrfNameToIdMap,
                                           mount.ipStatus,
                                           mount.ip6Status,
                                           None,
                                           None )
         state.ipv6RoutingSim = ipv6RoutingSim
      rsim = state.ipv6RoutingSim

   if toggleFibGenMountPathEnabled():
      nexthop = Arnet.IpAddr( nexthop ) if isV4 else Arnet.Ip6Addr( nexthop )

   routingOutput = rsim.route( nexthop )

   if routingOutput.outputType in [ RoutingOutputType.invalid,
                                    RoutingOutputType.cpu ]:
      if isV4 or intf is None:
         return ( None, None )
      else:
         return ( intf, getIpv6ArpEntry( mount, nexthop, intf ) )
   if routingOutput.nextHopEthAddr == EthAddr().stringValue or \
      routingOutput.nextHopIntf == IntfId( '' ) :
      return ( None, None )

   # FIXME routingOutput.outputType == multiple
   return ( routingOutput.nextHopIntf, routingOutput.nextHopEthAddr )

def resolveNhgTunnelFibEntry( mount, tunnelId ):
   err = None
   tunnelFibEntry = mount.tunnelFib.entry.get( tunnelId )
   if not tunnelFibEntry:
      err = 'No nexthop-group tunnel found in tunnel fib for tunnelId %s' % tunnelId
      return ( None, None, err )
   nexthopGroupId = getNexthopGroupId( tunnelFibEntry.tunnelVia[ 0 ] )
   nhgName = getNhgIdToName( nexthopGroupId, mount )
   if not nhgName:
      err = 'No nexthop-group with id %d' % nexthopGroupId
      return ( None, None, err )
   tunnelIndex = TunnelId( tunnelId ).tunnelIndex()
   return ( nhgName, tunnelIndex, err )

def resolveHierarchical( mount, intf=None, fecId=None ):
   '''
   Resolves the IntfIdFecId into a set of nexthops and intfIds. Used in the scenario
   when HFECs are enabled.
   If a FIB FecId is given as an argument, this method performs a fib lookup and
   generates a set of nexthops corresponding to the FEC.
   If intf is instead a TI-LFA tunnel, get the nexthop IP and interface from the via
   of the TI-LFA tunnel table entry.
   '''
   assert ( ( intf and not fecId ) or ( not intf and fecId ) ), 'Either intf or '\
      'fecId must be provided and also both must not be provided simultaneously'

   if intf and FecIdIntfId.isFecIdIntfId( intf ):
      fecId = FecIdIntfId.intfIdToFecId( intf )

   if fecId:
      nexthopAndLabelList = []
      if not FecId( fecId ).isValid():
         return None
      fecIdAdjType = FecId( fecId ).adjType()

      # Convert to fibAdj if used by tunnel adj
      if ( fecIdAdjType == FecAdjType.usedByTunnelGenAdj
            and toggleFibGenMountPathEnabled() ):
         fecId = FecId.fecIdToNewAdjType( FecAdjType.fibGenAdj, fecId )
      elif fecIdAdjType == FecAdjType.usedByTunnelV4Adj:
         fecId = FecId.fecIdToNewAdjType( FecAdjType.fibV4Adj, fecId )
      elif fecIdAdjType == FecAdjType.usedByTunnelV6Adj:
         fecId = FecId.fecIdToNewAdjType( FecAdjType.fibV6Adj, fecId )

      # Re-assign in case it changed
      fecIdAdjType = FecId( fecId ).adjType()
      fec = None
      if fecIdAdjType == FecAdjType.fibGenAdj and toggleFibGenMountPathEnabled():
         fec = mount.forwardingGenStatus.fec.get( fecId )
      elif fecIdAdjType == FecAdjType.fibV4Adj:
         fec = mount.forwardingStatus.fec.get( fecId )
      elif fecIdAdjType == FecAdjType.fibV6Adj:
         fec = mount.forwarding6Status.fec.get( fecId )
      if not fec:
         return None

      for i in fec.via:
         genNhAddr = IpGenAddr( str( fec.via[ i ].hop ) )
         intfId = fec.via[ i ].intfId
         if DynTunnelIntfId.isDynamicTunnelIntfId( intfId ):
            tunId = DynTunnelIntfId.tunnelId( intfId )
            if TunnelId( tunId ).tunnelType() == 'rsvpLerTunnel':
               rsvpNhAndLabel, err = getRsvpLerFec( mount,
                                                    TunnelId( tunId ).tunnelIndex() )
               if err:
                  continue
               # Use the RSVP resolved info with the SR label appended
               rsvpNexthopIp = rsvpNhAndLabel.nextHopIp
               nexthopAddr = ( rsvpNexthopIp.v4Addr if rsvpNexthopIp.af == IPv4 else
                               rsvpNexthopIp.v6Addr )
               nexthopAndLabelList.append( NextHopAndLabel(
                  nexthopAddr,
                  rsvpNhAndLabel.label,
                  rsvpNhAndLabel.intfId ) )
         elif not genNhAddr.isAddrZero:
            nexthopAddr = ( genNhAddr.v4Addr if genNhAddr.af == IPv4 else
                            genNhAddr.v6Addr )
            nexthopAndLabelList.append( NextHopAndLabel( nexthopAddr,
                                                         [], intfId ) )
      return nexthopAndLabelList

   if intf and DynTunnelIntfId.isDynamicTunnelIntfId( intf ):
      tunId = DynTunnelIntfId.tunnelId( intf )
      if TunnelId( tunId ).tunnelType() == 'tiLfaTunnel':
         tunEntry = mount.tiLfaTunnelTable.entry.get( tunId )
         if not tunEntry:
            return None
         intfId = tunEntry.via.intfId
         nexthop = IpGenAddr( str( tunEntry.via.nexthop ) )
         nexthopAddr = nexthop.v4Addr if nexthop.af == IPv4 else nexthop.v6Addr
         label = tunEntry.via.labels.labelStack( 0 )
         return [ NextHopAndLabel( nexthopAddr, [ label ], intfId ) ]
      elif TunnelId( tunId ).tunnelType() == 'rsvpLerTunnel':

         rsvpNhAndLabel, err = getRsvpLerFec( mount,
                                              TunnelId( tunId ).tunnelIndex() )
         if err:
            return None
         # Use the RSVP resolved info with the SR label appended
         nexthop = rsvpNhAndLabel.nextHopIp
         nexthopAddr = nexthop.v4Addr if nexthop.af == IPv4 else nexthop.v6Addr
         return [ NextHopAndLabel( nexthopAddr,
                                   rsvpNhAndLabel.label,
                                   rsvpNhAndLabel.intfId ) ]
      else:
         return None

   return None

# Get the nexthop label for an SR FEC. return the top label of via.labels
def getSrNexthopLabel( mount, via ):
   label = via.labels.labelStack( 0 )
   return label

def getBgpLuTunnelFibEntry( mount, prefix ):
   coloredTep = ColoredTunnelEndPoint( IpGenPrefix( prefix ), 0, False )
   ribEntry = mount.bgpLuTunnelRib.entry.get( coloredTep )
   if not ribEntry or not ribEntry.tunnelId:
      err = 'No BGP labeled unicast tunnel found for prefix %s' % prefix
      return None, err
   tunnelId = ribEntry.tunnelId[ 0 ]
   tunnelFibEntry = mount.tunnelFib.entry.get( tunnelId )
   if not tunnelFibEntry:
      err = bgpLuNoTunnelFoundErr % tunnelId
      return None, err
   if not tunnelFibEntry.tunnelVia:
      err = bgpLuNoTunnelViaFoundErr % tunnelId
      return None, err
   return tunnelFibEntry, None

def getLabelsFromTunnelFibVia( mount, tunnelVia ):
   encapId = tunnelVia.encapId
   if encapId.encapType != Tac.Type( "Tunnel::TunnelTable::EncapType" ).mplsEncap:
      err = ( 'Label stack encap information found in tunnel FIB for'
              ' encap ID %d is not MPLS' % encapId.encapIdValue )
      return None, err
   labelStack = None
   if encapId in mount.tunnelFib.labelStackEncap:
      labelStack = mount.tunnelFib.labelStackEncap[ encapId ].labelStack
   else:
      err = ( 'No label stack encap information found in tunnel FIB for'
              ' encap ID %d' % encapId.encapIdValue )
      return None, err
   labels = labelStackToList( labelStack )
   # Remove implicit null label if present, as label stacks will be concatenated
   while MplsLabel.implicitNull in labels:
      labels.remove( MplsLabel.implicitNull )
   return labels, None

def getNhAndLabelFromTunnelFibVia( mount, tunnelVia ):
   labels = []
   # Get label stack from tunnelVia
   labelStack, err = getLabelsFromTunnelFibVia( mount, tunnelVia )
   if err:
      return None, err
   labels = labelStack + labels
   intfId = tunnelVia.intfId
   nexthopIp = tunnelVia.nexthop

   if ( DynTunnelIntfId.isDynamicTunnelIntfId( tunnelVia.intfId ) or
        FecIdIntfId.isFecIdIntfId( tunnelVia.intfId ) ):
      # Using forwardingHelper to resolve hierarchical
      nexthopInfoList = mount.fwdingHelper.resolveHierarchical(
         True, l3IntfId=tunnelVia.intfId )
      if not nexthopInfoList:
         err = 'Can not resolve tunnel via intfId'
         return None, err

      nexthopInfo = nexthopInfoList[ 0 ]
      nexthopIp = nexthopInfo.nexthopIp
      if isinstance( nexthopIp, str ):
         # nexthopIp will be str for resolving fecIdIntfId
         nexthopIp = IpGenAddr( nexthopIp )
      labels = nexthopInfo.labelStack + labels
      intfId = nexthopInfo.intf

   # If label stack is empty, make it implicit null
   if not labels:
      labels = [ MplsLabel.implicitNull ]

   nextHopAndLabel = NextHopAndLabel( nexthopIp, labels, intfId )
   return nextHopAndLabel, None

def validateBgpLuResolvedPushVia( resolvedPushVia, nexthop, label ):
   if not resolvedPushVia:
      err = 'No BGP labeled unicast paths for BGP next hop %s' % nexthop
      if label:
         err += ' label stack %s' % str( label )
      err += ' found'
      return err
   elif len( resolvedPushVia ) > 1 and not label:
      err = ( 'Multiple BGP labeled unicast paths for BGP next hop '
              '%s found, please specify label stack' % nexthop )
      return err

   return None

def fillEntropyLabelPlaceholders( labelStack, proposedEntropyLabel ):
   skipNextLabel = False
   for labelIndex in range( len( labelStack ) - 1 ):

      # the entropy label value might, with very low probability, be the reserved
      # ELI value. We must skip the entropy label explicitly.
      if skipNextLabel:
         skipNextLabel = False
         continue

      if labelStack[ labelIndex ] != ELI:
         continue

      labelStack[ labelIndex + 1 ] = proposedEntropyLabel
      skipNextLabel = True

# Get the labelStack for an entry in the NHG's configuration
def getNhgTunnelLabelStack( nhgConfig, tunEntry, isBackup ):
   nhgConfigEntry = nhgConfig.backupEntry if isBackup else nhgConfig.entry
   if tunEntry in nhgConfigEntry:
      labelStack = nhgConfigEntry[ tunEntry ].mplsLabelStack
   else:
      labelStack = MplsLabelStack()
   return labelStack

# Helper functions for capi support navigation.
def pingUseCapi( protocol ):
   kws = ( [ 'vpn', 'ldp', 'segment-routing', 'segment-routing-ospf',
             'mldp', 'generic', 'bgpLu', 'rsvp', 'pwLdp' ] )
   return protocol in kws

def tracerouteUseCapi( protocol ):
   kws = ( 'ldp', 'segment-routing', 'segment-routing-ospf',
           'generic', 'rsvp', 'mldp' )
   return protocol in kws
