#!/usr/bin/env python3
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-in

import Arnet
import collections
import errno
from functools import wraps
import random
import sys
import Tac
import Tracing
import BothTrace
from TypeFuture import TacLazyType
from PseudowireLib import (
      pwPingCcEnumToVal,
      PwPingCcType,
)
from ClientCommonLib import(
   getMplsLabelStr,
   fecTlvTypeStr,
   getDsMappingInfo,
   getIntfPrimaryIpAddr,
   getIntfPrimaryIpAddrDefaultVrf,
   isIpv6Addr,
   IPV4, IPV6,
   LspPingTypeBgpLu,
   LspPingTypeGeneric,
   LspPingTypeLdp,
   LspPingTypeMldp,
   LspPingTypeRsvp,
   LspPingTypeSr,
   LspPingTypeOspfSr,
   LspPingTypeVpn,
   LspPingDDMap,
   pingUseCapi,
   sendViaSocket,
   tracerouteUseCapi,
)
from MplsPingClientLib import (
   sendOrRenderPingErr,
)
from MplsPingViaInfo import (
   calculateViaInfo,
)
from MplsTracerouteClientLib import (
   sendOrRenderTracerouteErr,
   tracerouteDownstreamInfoRender,
   tracerouteLabelStackRender,
   tracerouteReplyHdrRender,
)
from ClientState import (
   sessionIdIncr,
   cleanupGlobalState,
   getGlobalState
)
from PingModel_pb2 import ( # pylint: disable=no-name-in-module
   MplsPing,
   MplsPingReply,
   MplsPingStatistics,
   MplsPingStatisticsSummary,
   MplsPingVia,
   RsvpModel,
)
from TracerouteModel_pb2 import ( # pylint: disable=no-name-in-module
   DownstreamInfo,
   DownstreamInfos,
   IntfAndLabelStackInfo,
   MplsTraceroute,
   HopPktInfo,
)

#--------------------------------
# BothTrace Short hand variables
#--------------------------------
__defaultTraceHandle__ = Tracing.Handle( "ClientCore" )
bv = BothTrace.Var
bt8 = BothTrace.trace8

# RFC4379
# An MPLS echo request is a UDP packet.  The IP header is set as follows: the source 
# IP address is a routable address of the sender; the destination IP address is a 
# (randomly chosen) IPv4 address from the range 127/8 or IPv6 address from the range 
# 0:0:0:0:0:FFFF:127/104.
# 
# XXX Packets destining to 0:0:0:0:0:FFFF:127.0.0.1 are not trapped on Arad today.
# So use V6 loopback address here. (BUG123983)
DefaultLspPingReqDst = { IPV4 : '127.0.0.1',
                         IPV6 : '::ffff:127.0.0.1' }

AddressFamily = Tac.Type( 'Arnet::AddressFamily' )
IpGenAddr = Tac.Type( 'Arnet::IpGenAddr' )
IpGenPrefix = Tac.Type( 'Arnet::IpGenPrefix' )
Ipv4Unnumbered = Tac.enumValue( 'LspPing::LspPingAddrType', 'ipv4Unnumbered' )
Ipv6Unnumbered = Tac.enumValue( 'LspPing::LspPingAddrType', 'ipv6Unnumbered' )
Ipv4Numbered = Tac.enumValue( 'LspPing::LspPingAddrType', 'ipv4Numbered' )
Ipv6Numbered = Tac.enumValue( 'LspPing::LspPingAddrType', 'ipv6Numbered' )
srOspfIgpProto = Tac.enumValue( 'LspPing::LspPingSrIgpProtocol', 'ospf' )
srIsisIgpProto = Tac.enumValue( 'LspPing::LspPingSrIgpProtocol', 'isis' )
LspPingDownstreamMappingType = TacLazyType(
      'LspPing::LspPingDownstreamMappingType' )
LspPingConstant = Tac.Type( 'LspPing::LspPingConstant' )
LspPingTunnelType = Tac.Type( 'LspPing::LspPingTunnelType' )
LspPingTargetFecSubTlvType = Tac.Type( 'LspPing::LspPingTargetFecSubTlvType' )
LspPingPwInfo = Tac.Type( 'LspPing::LspPingPwInfo' )
LspPingP2mpFecStackSubTlvType = TacLazyType(
      'LspPing::LspPingP2mpFecStackSubTlvType' )
LspPingTxCountConst = Tac.Type( 'LspPing::LspPingTxCountConstants' )
LspPingMultipathBitset = Tac.Type( 'LspPing::LspPingMultipathBitset' )
LspPingReturnCode = Tac.Type( 'LspPing::LspPingReturnCode' )
MplsLabelStack = Tac.Type( 'Arnet::MplsLabelStack' )
MplsLabel = Tac.Type( 'Arnet::MplsLabel' )
NoRetCode = Tac.Type( 'LspPing::LspPingReturnCode' ).noRetCode
TracerouteRetArgs = collections.namedtuple( 'TracerouteRetArgs',
                                            [ 'retVal', 'txPkts', 'replyHostRtts' ] )
TunnelTableIdentifier = TacLazyType( "Tunnel::TunnelTable::TunnelTableIdentifier" )

timerWheel = Tac.newInstance( "Ark::TimerWheel", Tac.activityManager.clock, 
                              100, 10 * 60 * 5, True, 1024 )
IpAddrAllRoutersMulticast = "224.0.0.2"
Ip6AddrAllRoutersMulticast = "ff02::2"

class LspPingResult:
   def __init__( self, time, requests, replyInfo, kbInt ):
      self.duration = time
      self.requests = requests
      self.replyInfo = replyInfo
      self.kbInt = kbInt

# ---------------------------------------------------------
#               Common utility functions
# ---------------------------------------------------------

def createClientRoot( lspPingUtilRoot, mount, vrfIpIntfStatus, vrfIp6IntfStatus,
                      lspPingClientRootUdpPam, tracerouteInfo ):
   lspPingUtilRoot.lspPingClientRoot = ( mount.allIntfStatusDir,
                                         mount.allIntfStatusLocalDir,
                                         mount.kniStatus,
                                         vrfIpIntfStatus,
                                         vrfIp6IntfStatus,
                                         lspPingClientRootUdpPam,
                                         timerWheel, tracerouteInfo )
   return lspPingUtilRoot.lspPingClientRoot

def createLspPingClient( mount, state, interface, src, dst, protocol,
                         tracerouteInfo=None, verbose=False ):
   '''
      Creates, assigns, and returns a global instance of the LspPingClientRoot SM 
   '''
   if not state.clientRoot:
      # Get the udpPam
      lspPingClientRootUdpPam = Tac.newInstance( 'Arnet::UdpPam', 'udppam' )
      vrfIpIntfStatus = vrfIp6IntfStatus = None
      if isIpv6Addr( src ) or isIpv6Addr( dst ):   
         lspPingClientRootUdpPam.ipv6En = True
         lspPingClientRootUdpPam.txDstIp6Addr = Arnet.Ip6Addr( dst )
         vrfIp6IntfStatus = mount.vrfIp6IntfStatus
      else:
         lspPingClientRootUdpPam.txDstIpAddr = dst
         vrfIpIntfStatus = mount.vrfIpIntfStatus
      if state.clientRootUdpPamSrcPort is not None:
         lspPingClientRootUdpPam.rxPortLocal = state.clientRootUdpPamSrcPort

      lspPingClientRootUdpPam.mode = 'server'
      if verbose:
         # pylint: disable-next=consider-using-f-string
         print( "Listening on UDP port %d" % lspPingClientRootUdpPam.rxPort )
      lspPingClientRootUdpPam.maxRxPktData = LspPingConstant.maxRxPktSize

      # Bump up the sock buffer size to 1MB
      if lspPingClientRootUdpPam.incSockBufSize(
            LspPingConstant.sockBufferSize ) == -1:
         return None

      state.lspPingClientRootUdpPam = lspPingClientRootUdpPam
      local = mount.entityManager.root().parent
      if tracerouteInfo:
         # set pam rxPort to be used in multipath code
         tracerouteInfo.srcUdpPort = lspPingClientRootUdpPam.rxPort
      lspPingUtilRoot = local.newEntity( 'LspPing::LspPingUtilRoot', 'LspPingUtil' )
      lspPingClientRoot = createClientRoot( lspPingUtilRoot, mount,
                                            vrfIpIntfStatus, vrfIp6IntfStatus,
                                            lspPingClientRootUdpPam,
                                            tracerouteInfo )
      state.clientRoot = lspPingClientRoot
      state.lspPingUtilRoot = lspPingUtilRoot
   return state.clientRoot

def createLspPingVpnClientConfig( mount, state, clientId, tunnelType, interface,
                                  standard=None, tracerouteInfo=None ):
   lspPingUtilRoot = state.lspPingUtilRoot
   lspPingClientRootUdpPam = state.lspPingClientRootUdpPam
   lspPingClientRoot = createClientRoot( lspPingUtilRoot, mount,
                                         mount.vrfIpIntfStatusDefaultVrf,
                                         mount.vrfIp6IntfStatusDefaultVrf,
                                         lspPingClientRootUdpPam,
                                         tracerouteInfo )
   state.clientRoot = lspPingClientRoot
   lspPingClientConfigColl = lspPingClientRoot.lspPingClientConfigColl

   # Populate the Client config again.
   lspPingClientConfig = lspPingClientConfigColl.newPingClientConfig(
                         clientId )
   lspPingClientConfig.oamStandard = standard or mount.config.oamStandard
   lspPingClientConfig.tunnelType = tunnelType
   lspPingClientConfig.txIntfId = interface

   return lspPingClientConfig

def getTargetFecType( tunnelType, af=None ):
   def tunnelTypeModeRaw( af ):
      if af == AddressFamily.ipv4:
         return LspPingTargetFecSubTlvType.GenericIpv4
      elif af == AddressFamily.ipv6:
         return LspPingTargetFecSubTlvType.GenericIpv6
      else:
         return LspPingTargetFecSubTlvType.NilFec

   tunnelToFecMap = {
      LspPingTunnelType.tunnelStaticConfig : LspPingTargetFecSubTlvType.NilFec,
      LspPingTunnelType.tunnelUnknown : LspPingTargetFecSubTlvType.LDPIpv4,
      LspPingTunnelType.tunnelLdpIpv4 : LspPingTargetFecSubTlvType.LDPIpv4,
      LspPingTunnelType.tunnelMldp : LspPingTargetFecSubTlvType.P2mpFec,
      LspPingTunnelType.tunnelVpnIpv4 : LspPingTargetFecSubTlvType.VPNIpv4,
      LspPingTunnelType.tunnelVpnIpv6 : LspPingTargetFecSubTlvType.VPNIpv6,
      LspPingTunnelType.tunnelRsvpIpv4 : LspPingTargetFecSubTlvType.RSVPIpv4Lsp,
      LspPingTunnelType.tunnelRsvpP2mpIpv4 : LspPingTargetFecSubTlvType.RSVPP2MPIpv4,
      LspPingTunnelType.tunnelSrIpv4 : LspPingTargetFecSubTlvType.SRIpv4,
      LspPingTunnelType.tunnelSrIpv6 : LspPingTargetFecSubTlvType.SRIpv6,
      LspPingTunnelType.tunnelOspfSrIpv4 : LspPingTargetFecSubTlvType.SRIpv4,
      LspPingTunnelType.tunnelPwLdp : LspPingTargetFecSubTlvType.FEC128Pseudowire,
      LspPingTunnelType.tunnelBgpLuIpv4 : LspPingTargetFecSubTlvType.BGPLabeledIpv4,
      LspPingTunnelType.tunnelBgpLuIpv6 : LspPingTargetFecSubTlvType.BGPLabeledIpv6,
      LspPingTunnelType.tunnelGenericIpv4 : LspPingTargetFecSubTlvType.GenericIpv4,
      LspPingTunnelType.tunnelModeRaw : tunnelTypeModeRaw( af ),
   }

   targetFecType = tunnelToFecMap.get( tunnelType )
   return targetFecType

def getTargetFecInfo( tunnelType, mplsLabel, egressValidateAddress=None, src=None,
                      dst=None, dstPrefix=None, algorithm=None, mldpInfo=None,
                      rsvpSpId=None, rsvpSpGroupId=None, rsvpSenderAddr=None,
                      pwInfo=None, rd=None, oamStandard=None ):

   targetFecInfo = Tac.newInstance( "LspPing::LspPingFecInfo" )

   targetFecInfo.mplsLabel = mplsLabel
   targetFecInfo.oamStandard = oamStandard
   targetFecInfo.algorithm = algorithm or 0

   af = None
   if egressValidateAddress is not None:
      egressAddraess = IpGenAddr( egressValidateAddress )
      af = egressAddraess.af
      targetFecInfo.dstPrefix = IpGenPrefix( egressValidateAddress )

   if dstPrefix:
      targetFecInfo.dstPrefix = IpGenPrefix( dstPrefix )

   if tunnelType == LspPingTunnelType.tunnelUnknown:
      targetFecInfo.dstPrefix = IpGenPrefix( dst )
   
   if tunnelType == LspPingTunnelType.tunnelOspfSrIpv4:
      targetFecInfo.igpProtocol = srOspfIgpProto
   else:
      targetFecInfo.igpProtocol = srIsisIgpProto

   if mldpInfo:
      if mldpInfo.genOpqVal:
         targetFecInfo.opqType = LspPingP2mpFecStackSubTlvType.GenericLsp
         targetFecInfo.genericOpqVal = mldpInfo.genOpqVal
      if mldpInfo.sourceAddrOpqVal and mldpInfo.groupAddrOpqVal:
         targetFecInfo.opqType = LspPingP2mpFecStackSubTlvType.SGIpv4
         targetFecInfo.sourceAddrOpqVal = \
               IpGenAddr( mldpInfo.sourceAddrOpqVal )
         targetFecInfo.groupAddrOpqVal = IpGenAddr( mldpInfo.groupAddrOpqVal )

   if rsvpSpId:
      targetFecInfo.rsvpSpId = rsvpSpId
   if rsvpSenderAddr:
      targetFecInfo.rsvpSenderAddr = rsvpSenderAddr
   if rsvpSpGroupId:
      targetFecInfo.rsvpSpGroupId = rsvpSpGroupId

   if pwInfo:
      targetFecInfo.pwInfo = pwInfo

   if rd:
      targetFecInfo.rd = rd

   targetFecInfo.fecTlvType = getTargetFecType( tunnelType, af )
   return targetFecInfo

def updateTargetFecStack( currentTargetFecStack=None,
                          popFecStack=None, pushFecStack=None ):
   '''
      Update the Target FEC stack based on the Fec Stack Change TLV received
   '''
   bt8( "Fec Stack Change event occurred" )
   fecStack = []
   for index, _ in enumerate( currentTargetFecStack.fecInfo ):
      fecStack.append( currentTargetFecStack.fecInfo[ index ] )

   for index, _ in enumerate( popFecStack.fecInfo ):
      if fecStack:
         fecStack.pop()
      else:
         errmsg = 'received more fec stack change pop tlvs than current fec stack'
         return None, errmsg

   for index, _ in enumerate( pushFecStack.fecInfo ):
      fecStack.append( pushFecStack.fecInfo[ index ] )

   if not fecStack:
      # If the FEC stack is empty after processing, we have to drop the echo reply.
      errmsg = 'target fec stack empty after processing fec stack change'
      return None, errmsg

   newTargetFecStack = Tac.newInstance( "LspPing::LspPingFecStack" )
   for index, fecInfo in enumerate( fecStack ):
      newTargetFecStack.fecInfo[ index ] = fecInfo
   return newTargetFecStack, None

def createLspPingClientConfig( mount, state, clientId, protocol, tunnelType,
                               interface, labelStack, src, dst, dmac, smac=None,
                               interval=1, mplsTtl=None, count=None, verbose=False,
                               tc=None, dstPrefix=None, fecStackChangeInfo=None,
                               currentTargetFecStack=None,
                               dsMappingInfo=None, rsvpSpId=None, pwInfo=None,
                               rsvpSenderAddr=None, mldpInfo=None, standard=None,
                               size=None, padReply=False, tos=None,
                               dstype=None, setFecValidateFlag=True,
                               egressValidateAddress=None,
                               lookupLabelCount=1, sock=None, rd=None,
                               algorithm=None, tracerouteInfo=None ):
   '''
      Adds an entry to the global LspPingClientConfig collection, 
      which is used to craft and send packets.
   '''
   lspPingClientRoot = state.clientRoot
   lspPingClientConfigColl = lspPingClientRoot.lspPingClientConfigColl
   lspPingClientRootUdpPam = state.lspPingClientRootUdpPam

   lspPingClientConfig = lspPingClientConfigColl.newPingClientConfig( clientId )
   lspPingClientConfig.oamStandard = standard or mount.config.oamStandard

   lspPingClientConfig.tunnelType = tunnelType
   lspPingClientConfig.txIntfId = interface

   # Before we even proceed, wait for the EthDevPam to be ready.
   pamColl = lspPingClientRoot.lspPingEthDevPamManager.ethDevPamCollection
   try:
      Tac.waitFor( lambda: interface in pamColl.intfPam, timeout=0.5 )
   except Tac.Timeout:
      if protocol == LspPingTypeVpn:
         del lspPingClientRoot
         lspPingClientConfig = createLspPingVpnClientConfig(
                               mount, state, clientId, tunnelType, interface,
                               standard=standard, tracerouteInfo=tracerouteInfo )
         lspPingClientRoot = state.clientRoot
         pamColl = lspPingClientRoot.lspPingEthDevPamManager.ethDevPamCollection
         try:
            Tac.waitFor( lambda: interface in pamColl.intfPam, timeout=0.5 )
         except Tac.Timeout:
            return None
      else:
         return None

   # By default mappingTlv is downstreamMap
   if dstype == LspPingDDMap:
      lspPingClientConfig.mappingTlv = \
         LspPingDownstreamMappingType.downstreamDetailedMap
   if dsMappingInfo and dsMappingInfo.valid:
      lspPingClientConfig.dsMappingInfo = dsMappingInfo
   if fecStackChangeInfo and fecStackChangeInfo.valid:
      lspPingClientConfig.fecStackChangeInfo = fecStackChangeInfo
   if tos:
      lspPingClientConfig.tos = tos
   lspPingClientConfig.mplsTc = tc if tc else 0
   lspPingClientConfig.srcIpAddr = IpGenAddr( src )
   lspPingClientConfig.dstIpAddr = IpGenAddr( dst )
   if dstPrefix:
      lspPingClientConfig.dstPrefix = IpGenPrefix( dstPrefix )

   lspPingClientConfig.srcUdpPort = lspPingClientRootUdpPam.rxPort
   # Use smac if provided, else fall back to interface's routedAddr. Use
   # brigeMacAddr if interface is not found in allIntfStatusDir.
   lspPingClientConfig.srcEthAddr = ( smac or
                                      getattr( mount.allIntfStatusDir.intfStatus.get(
                                                  interface ),
                                               'routedAddr',
                                               mount.bridgingConfig.bridgeMacAddr ) )
   lspPingClientConfig.tunnelNexthopEthAddr = dmac
   lspPingClientConfig.txRate = interval
   lspPingClientConfig.pktDelay = random.randint( 50, 100 ) / 1000.0 # ms
   lspPingClientConfig.txCount = LspPingTxCountConst.infinite if count is None \
                                 else count
  
   # Grab l3 MTU from intfStatus if packet size is specified
   mtu = mount.allIntfStatusDir.intfStatus[ interface ].mtu

   if size:
      lspPingClientConfig.pktSize = size
      lspPingClientConfig.padReply = padReply
      lspPingClientConfig.mtu = mtu

   lspPingClientConfig.lookupLabelCount = lookupLabelCount
   lspPingClientConfig.setFecValidateFlag = setFecValidateFlag

   # mldp config
   if mldpInfo:
      if mldpInfo.jitter:
         lspPingClientConfig.echoJitter = mldpInfo.jitter
      if mldpInfo.responderAddr:
         lspPingClientConfig.nodeResponderAddr = IpGenAddr( mldpInfo.responderAddr )
   if tunnelType == LspPingTunnelType.tunnelMldp:
      # For mldp ,Downstream Detailed Mapping Tlv is added for dsMappingInfo
      lspPingClientConfig.mappingTlv = \
         LspPingDownstreamMappingType.downstreamDetailedMap

   # Add the MPLS labels, possibly including the entropy label
   stackSize = len( labelStack )
   isEntropyLabelNext = False
   for i in range( stackSize ):
      label = int( labelStack[ i ] )
      lspPingClientConfig.mplsLabel[ i ] = MplsLabel( label )
      # If we see the ELI, we need to set it and the following entropy label's
      # TTL to 0 as per RFC6970
      if label == MplsLabel.entropyLabelIndicator:
         isEntropyLabelNext = True
         mplsTtlVal = 0
      elif isEntropyLabelNext:
         isEntropyLabelNext = False
         mplsTtlVal = 0
      else:
         mplsTtlVal = mplsTtl[ i ] if mplsTtl else 255
      lspPingClientConfig.mplsTtl[ i ] = mplsTtlVal

   if pwInfo:
      lspPingClientConfig.pwInfo = pwInfo

   # Get Target FEC Info
   if not currentTargetFecStack:
      targetFecStack = Tac.newInstance( "LspPing::LspPingFecStack" )
      targetFecStack.fecInfo[ 0 ] = getTargetFecInfo( tunnelType,
                                        lspPingClientConfig.mplsLabel[ 0 ],
                                        egressValidateAddress, src, dst,
                                        dstPrefix, algorithm, mldpInfo, rsvpSpId,
                                        None, rsvpSenderAddr, pwInfo, rd,
                                        lspPingClientConfig.oamStandard )
      lspPingClientConfig.targetFecStack = targetFecStack
   else:
      # Fec Stack Change has been received at least once before.
      # In this case, currentTargetFecStack maintains the correct Fec Stack.
      lspPingClientConfig.targetFecStack = currentTargetFecStack

   if not lspPingClientConfig.enabled:
      err = lspPingClientConfig.reason
      sendOrRenderTracerouteErr( err, sock )
      return None

   lspPingClientStatusColl = lspPingClientRoot.lspPingClientStatusColl
   lspPingClientStatus = lspPingClientStatusColl.pingClientStatus.get( clientId )  
   if not lspPingClientStatus:
      return None
   return ( lspPingClientConfig, lspPingClientStatus )

def isValidDestAddr( dst, ipv ):
   '''
   Validate the destination address is in correct range for IPv4/IPv6.
   Returns True if the provided destination address is valid for the ip version,
   False otherwise.
   '''
   if ipv is IPV4:
      if not Arnet.IpAddr( dst ).isLoopback:
         return False
   elif ipv is IPV6:
      ip6DestRange = Arnet.Ip6AddrWithMask( '::ffff:127.0.0.0/104' )
      if not ip6DestRange.contains( Arnet.Ip6Addr( dst ) ):
         return False
   else:
      return False
   return True

def verifyConsistentAfType( addr1Str, addr2Str ):
   addr1 = Arnet.IpGenAddr( addr1Str )
   addr2 = Arnet.IpGenAddr( addr2Str )
   return addr1.af == addr2.af

# ---------------------------------------------------------
#               Core ping helpers
# ---------------------------------------------------------

def sendReplyHdr( replyPktInfo, sock ):
   if replyPktInfo:
      mplsPingReply = MplsPingReply(
                         replyHost=str( replyPktInfo.replyHost ),
                         sequence=replyPktInfo.seqNum,
                         roundTripTime=int( replyPktInfo.roundTrip / 1000 ),
                         roundTripTimeUs=( replyPktInfo.roundTrip % 1000 ),
                         oneWayDelay=replyPktInfo.oneWayDelay,
                         retCode=str( replyPktInfo.retCode ),
                         errorTlvMap=replyPktInfo.errorTlvMap,
                         errorSubTlvMap=replyPktInfo.errorSubTlvMap )
      sendViaSocket( sock, MplsPing( mplsPingReply=mplsPingReply ) )
   else:
      err = "   Request timeout"
      sendOrRenderPingErr( err, sock )

def sendVias( vias, sock, statisticsVia=False, resolved=True, viaHdr=None ):
   for via in vias:
      nextHopIp, labels, interface = via
      if isinstance( labels, int ):
         labels = [ labels ]
      else:
         labels = list( map( int, labels ) )
      lspId = viaHdr.get( 'lspId' )
      subTunnelId = viaHdr.get( 'subTunnelId' )
      if not isinstance( lspId, int ):
         lspId = -1
      mplsPingVia = MplsPingVia( resolved=resolved,
                                 nextHopIp=str( nextHopIp ),
                                 interface=str( interface ),
                                 labelStack=labels,
                                 statisticsVia=statisticsVia,
                                 lspId=lspId,
                                 subTunnelId=subTunnelId )
      sendViaSocket( sock, MplsPing( mplsPingVia=mplsPingVia ) )

def renderPingViaSocket( protocol, clientId, replyPktInfo, renderArgs, sock=None ):
   if not sock:
      return
   vias = renderArgs[ 0 ][ clientId ]
   viaHdr = {}
   if protocol == LspPingTypeRsvp:
      viaHdr[ 'lspId' ] = renderArgs[ 3 ].get( clientId )
      viaHdr[ 'subTunnelId' ] = renderArgs[ 4 ].get( clientId )
   # Some protocols has specific headers for each via entry in some cases.
   # To send these extra attributes through sock, we use a dictionary to contain them
   # which is extendable
   sendVias( vias, sock, viaHdr=viaHdr )
   sendReplyHdr( replyPktInfo, sock )

def renderStatsViaSocket( clientIds, time, txPkts, replyInfo, delayInfo,
                          renderArg, sock=None, protocol=None ):
   if not sock or not txPkts:
      return

   # TODO: update unpacking style to support all variants of renderArgs provided per
   # protocol when capi support is required for other protocols
   if protocol in [ LspPingTypeVpn, LspPingTypeLdp, LspPingTypeMldp, LspPingTypeSr,
                    LspPingTypeOspfSr, LspPingTypeGeneric, LspPingTypeBgpLu ]:
      clientIdToVias, _, unresolvedVias = renderArg
   elif protocol == LspPingTypeRsvp:
      clientIdToVias, _, _, clientIdToLsp, clientIdToSubTunnels = renderArg
      unresolvedVias = None
   else:
      clientIdToVias, _, _ = renderArg
      unresolvedVias = None
   viaHdr = {}
   for clientId in clientIds:
      # sum up all recorded RTTs to figure out the total # of echo replies
      if clientId in replyInfo:
         recvNum = sum( len( v ) for v in replyInfo[ clientId ].values() )
      else:
         recvNum = 0
      lossRate = 100 - recvNum * 100 / txPkts[ clientId ]
      vias = clientIdToVias.get( clientId )
      if vias is not None:
         if protocol == LspPingTypeRsvp:
            viaHdr[ 'lspId' ] = clientIdToLsp.get( clientId )
            viaHdr[ 'subTunnelId' ] = clientIdToSubTunnels.get( clientId )
         sendVias( vias, sock, statisticsVia=True, viaHdr=viaHdr )
      mplsPingStatisticsSummary = MplsPingStatisticsSummary(
                                     packetsTransmitted=txPkts[ clientId ],
                                     packetsReceived=recvNum,
                                     packetLoss=int( lossRate ),
                                     replyTime=time )

      sendViaSocket(
         sock, MplsPing( mplsPingStatisticsSummary=mplsPingStatisticsSummary ) )
      # rtts is a floating value representing round trip time in milliseconds.
      # we have to split this floating value into millisecond and microsecond part
      # to satisfy capi model which have separate fields for millisecond and
      # microsecond.
      # delays is a floaring value representing one way delay in milliseconds.
      # We have to convert this into microseconds as the capi model expectes
      # one way delay in microseconds
      if clientId in replyInfo:
         for host, rtts in replyInfo[ clientId ].items():
            rttMin, rttMax, rttAvg = 0, 0, 0
            if rtts:
               rttMin = min( rtts )
               rttMax = max( rtts )
               rttAvg = sum( rtts ) / len( rtts )

            delayMin, delayMax, delayAvg = 0, 0, 0
            delays = delayInfo[ clientId ][ host ]
            if delays:
               delayMin = int( min( delays ) * 1000 )
               delayMax = int( max( delays ) * 1000 )
               delayAvg = int( sum( delays ) * 1000 / len( delays ) )

            mplsPingStatistics = MplsPingStatistics(
                                    destination=host.stringValue,
                                    roundTripTimeMin=int( rttMin ),
                                    roundTripTimeMax=int( rttMax ),
                                    roundTripTimeAvg=int( rttAvg ),
                                    roundTripTimeMinUs=int( ( rttMin % 1 ) * 1000 ),
                                    roundTripTimeMaxUs=int( ( rttMax % 1 ) * 1000 ),
                                    roundTripTimeAvgUs=int( ( rttAvg % 1 ) * 1000 ),
                                    oneWayDelayMin=delayMin,
                                    oneWayDelayMax=delayMax,
                                    oneWayDelayAvg=delayAvg,
                                    packetsReceived=len( rtts ) )
            sendViaSocket( sock, MplsPing( mplsPingStatistics=mplsPingStatistics ) )

   if unresolvedVias:
      sendVias( unresolvedVias, sock, statisticsVia=True, resolved=False,
                viaHdr=viaHdr )

# RSVP pingModel is sent here instead of in MplsPingHandler as prefix is calculated
# within ping process
def sendRsvpModelViaSocket( prefix, lsp, session, tunnel, subTunnelId, sock ):
   # As protobuf message will pass 0 as default int value while a valid lspId
   # could be 0 in theory. We set lspId to be -1 here to indicate lspId not specified
   # For subTunnelId 0 is invalid so we can just send it through protobuf
   if not lsp:
      lsp = -1
   session = str( session ) if isinstance( session, int ) else session
   rsvpModel = RsvpModel( prefix=prefix, lspId=lsp, session=session,
                          tunnel=tunnel, subTunnelId=subTunnelId )
   sendViaSocket( sock, MplsPing( rsvpModel=rsvpModel ) )

def echoRequestsSent( lspPingClientStatusColl, clientIds, seq ):
   for clientId in clientIds:
      clientStatus = lspPingClientStatusColl.pingClientStatus.get( clientId )
      if not clientStatus or clientStatus.sequenceNum < seq:
         return False
   return True

def echoReplyReceived( clientStatusColl, clientId, seq ):
   clientStatus = clientStatusColl.pingClientStatus.get( clientId )
   if not clientStatus or clientStatus.sequenceNum < seq \
   or not clientStatus.replyPktInfo.get( seq ):
      return False
   return True

def echoRepliesPending( clientStatusColl, clientIds, seq ):
   clientIds[:] = [ clientId for clientId in clientIds \
                    if not echoReplyReceived( clientStatusColl, clientId, seq ) ]
   return clientIds

def sendReceiveAndRenderEchoRequest( state, clientIds, renderArg, seq, startTime,
                                     interval, replyInfo, delayInfo, replyRender,
                                     pktTimeout, txPkts, pollRate, sock, protocol ):
   # check ping client status, XXX timeout hard coded to 3 for now
   '''
      Sends/Receives echo requests/replies and renders ping output and stats
   '''
   clientStatusColl = state.clientRoot.lspPingClientStatusColl
   clientIdBase = state.clientIdBase

   Tac.waitFor( lambda: echoRequestsSent( clientStatusColl, clientIds, seq ),
                maxDelay=0.2, warnAfter=None,
                description="echo request(s) to be sent" )
   bt8( "Echo request sent for sequence no.", bv( seq ) )
   timeout = min( startTime + interval * seq + pktTimeout,
                  Tac.now() + pktTimeout )
   pendingClients = list( clientIds )
   while Tac.now() < timeout:
      pendingClients = echoRepliesPending( clientStatusColl, pendingClients, seq )
      if pendingClients:
         Tac.runActivities( pollRate )
         continue
      break

   for clientId in clientIds:
      clientStatus = clientStatusColl.pingClientStatus.get( clientId )
      replyPktInfo = clientStatus.replyPktInfo.get( seq )
      if replyPktInfo is not None and replyPktInfo.seqNum == seq:
         bt8( "Received reply packet from", bv( replyPktInfo.replyHost ),
               "for sequence no.", bv( replyPktInfo.seqNum ),
               "with retcode", bv( replyPktInfo.retCode ) )
         replyHostRtt = replyInfo.setdefault( clientId - clientIdBase, {} )
         replyHostRtt.setdefault( replyPktInfo.replyHost, [] ).append(
               replyPktInfo.roundTrip / 1000 )
         oneWayDelay = delayInfo.setdefault( clientId - clientIdBase, {} )
         oneWayDelay.setdefault( replyPktInfo.replyHost, [] ).append(
               replyPktInfo.oneWayDelay / 1000 )
      else:
         replyPktInfo = None
      # check useCapi in MplsUtilCli.py for current capi supported protocols
      # In other cases, sock is None and below function does nothing and
      # returns.
      renderPingViaSocket( protocol, clientId - clientIdBase, replyPktInfo,
                           renderArg, sock=sock )
      # replyRender function passing will be removed completely once
      # we support capi ping support for all protocols.
      if replyRender:
         replyRender( clientId - clientIdBase, replyPktInfo, renderArg )
      # update txNum ONLY after we have process the reply
      txPkts[ clientId - clientIdBase ] = seq
   return seq

def awaitPingTermination( clientIds, replyRender, statisticsRender, renderArg,
                          mount, state, pwLdpInfo, ipv, src, dst,
                          baseErrorMsg, smac, tc, mldpInfo, standard, size,
                          padReply, tos, setFecValidateFlag, egressValidateAddress,
                          lookupLabelCount, rd, algorithm, rsvpTunnel, rsvpSubTunnel,
                          session, lsp, pktTimeout,
                          interval=None, count=None, protocol=None, sock=None,
                          pollRate=0.1 ):
   '''
      Wait for either the user to terminate mpls ping or for ping to 
      hit max count of responses received and render output accordingly.
   '''
   txPkts = {}
   seq = 1 # seq # starting from 1
   kbInt, replyInfo, delayInfo = False, {}, {}
   interval = interval or 1
   startTime = Tac.now()
   sentRsvpModel = False
   try:
      while True:
         if protocol == LspPingTypeRsvp:
            # For RSVP nexthop info is calculated on each ping iteration.
            viaInfo, renderArg, rsvpSpIds, rsvpSenderAddr = \
               calculateViaInfo( mount, rsvpTunnel, rsvpSubTunnel,
                                 session, lsp, state )
            if not viaInfo:
               # try pinging continously till count with interval period
               # Maybe we find viaInfo in next iteration.
               if count and seq == count:
                  break
               seq += 1
               Tac.runActivities( pollRate )
               continue
            # RSVP pingModel will be sent out for only once after resolved via
            if not sentRsvpModel:
               prefix = str( renderArg[ 1 ] )
               sendRsvpModelViaSocket( prefix, lsp, session, rsvpTunnel,
                                       rsvpSubTunnel, sock )
               sentRsvpModel = True
            clientIds, renderArg = createClients( viaInfo, mount, state, protocol,
                                                  rsvpSpIds, rsvpSenderAddr,
                                                  pwLdpInfo, ipv, src, dst,
                                                  baseErrorMsg, sock,
                                                  renderArg, smac, interval,
                                                  count, tc, mldpInfo, standard,
                                                  size, padReply, tos,
                                                  setFecValidateFlag,
                                                  egressValidateAddress,
                                                  lookupLabelCount, rd, algorithm )
         seq = sendReceiveAndRenderEchoRequest( state, clientIds, renderArg, seq,
                                                startTime, interval, replyInfo,
                                                delayInfo, replyRender, pktTimeout,
                                                txPkts, pollRate, sock, protocol )
         if count and seq == count:
            break
         seq += 1
   
   except KeyboardInterrupt:
      kbInt = True
   time = int( ( Tac.now() - startTime ) * 1000 ) # microsecond
   relativeClientIds = [ clientId - state.clientIdBase for clientId in clientIds ]
   # capi supported rendering only supported for VPN and BGP LU protocol for now
   # In other cases, sock is None and below function does nothing and returns.
   renderStatsViaSocket( relativeClientIds, time, txPkts, replyInfo, delayInfo,
                         renderArg, protocol=protocol, sock=sock )
   if statisticsRender:
      statisticsRender( relativeClientIds, time, txPkts, replyInfo,
                        delayInfo, renderArg )
   return LspPingResult( time, txPkts, replyInfo, kbInt )

def explicitNullAtBos( labelStack ):
   if labelStack:
      # see if explicitNull present before ( ELI, entropyLabel ) at BOS
      # ( explicit null, ELI, entropyLabel )
      if ( len( labelStack ) > 2 and
           labelStack[ -2 ] is MplsLabel.entropyLabelIndicator ):
         if ( ( labelStack[ -3 ] is MplsLabel.explicitNullIpv4 ) or
              ( labelStack[ -3 ] is MplsLabel.explicitNullIpv6 ) ):
            return True
      # see if explicitNull is present at BOS in labelstack
      else:
         if ( ( labelStack[ -1 ] is MplsLabel.explicitNullIpv4 ) or
              ( labelStack[ -1 ] is MplsLabel.explicitNullIpv6 ) ):
            return True
   return False

def createClients( viaInfo, mount, state, protocol, rsvpSpIds, rsvpSenderAddr,
                   pwLdpInfo, ipv, src, dst, baseErrorMsg, sock, renderArg, smac,
                   interval, count, tc, mldpInfo, standard, size, padReply, tos,
                   setFecValidateFlag, egressValidateAddress, lookupLabelCount,
                   rd, algorithm ):
   # create ping clients
   pwInfo, rsvpSpId = None, None
   prefix = None
   if renderArg:
      # renderArg[1] is prefix
      prefix = renderArg[ 1 ]
   if not ipv:
      ipv = IPV6 if isIpv6Addr( prefix ) else IPV4

   af = AddressFamily.ipv4 if ipv == IPV4 else AddressFamily.ipv6
   if src and IpGenAddr( src ).af != af:
      msg = 'Source ip address must match destination address family'
      err = baseErrorMsg + msg
      sendOrRenderPingErr( err, sock )
      return -1

   clientIds = []
   for idx, ( l3Intf, label, nhMac ) in enumerate( viaInfo ):
      if isinstance( label, int ):
         labelStack = [ label ]
      elif isinstance( label, tuple ):
         labelStack = list( label )
      else:
         labelStack = label
      clientId = state.clientIdBase + idx
      if protocol == LspPingTypeVpn:
         reqSrcIp = ( src or
                      getIntfPrimaryIpAddr( mount, l3Intf, ipv ) or
                      getIntfPrimaryIpAddrDefaultVrf( mount, l3Intf, ipv ) )
         if not reqSrcIp:
            altIpv = IPV6 if ipv is IPV4 else IPV4
            reqSrcIp = ( getIntfPrimaryIpAddr( mount, l3Intf, altIpv ) or
                         getIntfPrimaryIpAddrDefaultVrf( mount, l3Intf, altIpv ) )
      else:
         reqSrcIp = src or getIntfPrimaryIpAddr( mount, l3Intf, ipv )
      if not reqSrcIp:
         msg = 'No usable source ip address for ping'
         err = baseErrorMsg + msg
         sendOrRenderPingErr( err, sock )
         return -1

      # While dst option is not supported via the CLI at this time, some of the tests
      # use it. Also, we may in future start allowing destination address to be
      # specified via the CLI. This check makes sure that if the destination address
      # is provided, it must be in 127.0.0.0/8 range for IPv4 and ::fff:127.0.0.0/104
      # range for IPv6
      if dst:
         # For VPN, the src ip could be IPV4 or IPV6, denoted by reqSrcIp.
         # Hence skip the check as dst is calculated from reqSrcIp.
         if protocol != LspPingTypeVpn and not isValidDestAddr( dst, ipv ):
            msg = 'Destination address must be in ' + ( '127.0.0.0/8' if ipv is IPV4
                  else '::ffff:127.0.0.0/104' ) + ' range.'
            err = baseErrorMsg + msg
            sendOrRenderPingErr( err, sock )
            return -1
      else:
         dst = DefaultLspPingReqDst[
                  IPV4 if IpGenAddr( reqSrcIp ).af == AddressFamily.ipv4 else IPV6 ]

      # ensure both src and dst ip are of the consistent ipv type
      if not verifyConsistentAfType( reqSrcIp, dst ):
         msg = "Inconsistent source (" + str( reqSrcIp ) + ") and destination (" + \
               str( dst ) + ") IP address families.\n"
         err = baseErrorMsg + msg
         sendOrRenderPingErr( err, sock )
         return -1

      rsvpSpId = rsvpSpIds[ idx ] if rsvpSpIds else None
      tunnelType, tunnelPrefix, pwInfo = \
            getTunnelTypeAndPrefix( protocol, prefix, rsvpSpId, ipv, pwLdpInfo )

      if algorithm and ( tunnelType == LspPingTunnelType.tunnelSrIpv4 or
                         tunnelType == LspPingTunnelType.tunnelSrIpv6 ):
         # When printing flex algo tunnels, we append "algorithm <algoName>" to the
         # prefix
         algoDef = mount.flexAlgoConfig.definition.get( algorithm )
         if not algoDef:
            msg = 'failed to find flexible algorithm' + algorithm
            err = baseErrorMsg + msg
            sendOrRenderPingErr( err, sock )
            return -1

         prefixAlgo = str( prefix ) + ', algorithm ' + algoDef.name
         renderArg = ( renderArg[ 0 ], prefixAlgo ) + renderArg[ 2 : ]

      # For some usecases we already push an Exp-NULL label at the BOS. In that case,
      # we do not need to push an extra Exp-NULL label again.
      if ( ( tunnelType == LspPingTunnelType.tunnelModeRaw ) and
           ( not egressValidateAddress ) and
           ( not explicitNullAtBos( labelStack ) ) ):
         nullLabel = ( MplsLabel.explicitNullIpv4 if ipv == IPV4 else
                       MplsLabel.explicitNullIpv6 )
         labelStack.extend( [ nullLabel ] )

      if not createLspPingClient( mount, state, l3Intf, reqSrcIp, dst, protocol ):
         msg = 'failed to create lsp ping client'
         err = baseErrorMsg + msg
         sendOrRenderPingErr( err, sock )
         return -1

      if not createLspPingClientConfig( mount, state, clientId, protocol, tunnelType,
                                        l3Intf, labelStack, reqSrcIp, dst, nhMac,
                                        smac, interval, None, count,
                                        dstPrefix=tunnelPrefix, tc=tc,
                                        rsvpSpId=rsvpSpId, pwInfo=pwInfo,
                                        rsvpSenderAddr=rsvpSenderAddr,
                                        mldpInfo=mldpInfo, standard=standard,
                                        size=size, padReply=padReply, tos=tos,
                                        setFecValidateFlag=setFecValidateFlag,
                                        egressValidateAddress=(
                                           egressValidateAddress ),
                                        lookupLabelCount=lookupLabelCount, rd=rd,
                                        algorithm=algorithm ):
         msg = 'failed to create ping client config'
         err = baseErrorMsg + msg
         sendOrRenderPingErr( err, sock )
         return -1

      clientIds.append( clientId )
   return clientIds, renderArg

# ---------------------------------------------------------
#               Core ping implementation
# ---------------------------------------------------------

def ping( mount, state, src, dst, smac, interval, count, viaInfo, replyRender,
          statisticsRender, renderArg, rsvpTunnel=None, rsvpSubTunnel=None,
          session=None, lsp=None, protocol=None, ipv=None, pwLdpInfo=None,
          rsvpSpIds=None, rsvpSenderAddr=None, tc=None, timeout=5, mldpInfo=None,
          standard=None, size=None, padReply=False, setFecValidateFlag=True,
          egressValidateAddress=None, lookupLabelCount=1, tos=None, rd=None,
          sock=None, algorithm=None, json=False ):
   state = state or getGlobalState()
   if protocol == LspPingTypeRsvp:
      # For RSVP nexthop info is calculated on each ping iteration.
      # Hence not provided by handlers.
      assert not viaInfo
   if json:
      # for json output restrict ping to count 1
      count = 1
   else:
      count = count or None # ping until interrupted
      if not pingUseCapi( protocol ):
         # pylint: disable-next=consider-using-f-string
         print( '   timeout is {}ms, interval is {}ms'.format( timeout * 1000,
                                                               interval * 1000 ) )
   baseErrorMsg = 'Error: Could not proceed with ping\n'
   clientIds = []
   if viaInfo:
      # create ping clients
      clientIds, renderArg = createClients( viaInfo, mount, state, protocol,
                                            rsvpSpIds, rsvpSenderAddr, pwLdpInfo,
                                            ipv, src, dst, baseErrorMsg, sock,
                                            renderArg, smac, interval, count, tc,
                                            mldpInfo, standard, size, padReply, tos,
                                            setFecValidateFlag,
                                            egressValidateAddress,
                                            lookupLabelCount, rd, algorithm )
      if not clientIds:
         return -1

   pingResult = awaitPingTermination( clientIds, replyRender, statisticsRender,
                                      renderArg, mount, state, pwLdpInfo, ipv,
                                      src, dst, baseErrorMsg, smac, tc, mldpInfo,
                                      standard, size, padReply, tos,
                                      setFecValidateFlag, egressValidateAddress,
                                      lookupLabelCount, rd, algorithm, rsvpTunnel,
                                      rsvpSubTunnel, session, lsp,
                                      timeout, interval, count, protocol, sock )
   if pingResult == errno.EINVAL or pingResult == 0:
      return pingResult # error case
   elif pingResult.kbInt:
      return errno.EINTR # keyboard interrupt case
   else:
      return 0

# ---------------------------------------------------------
#                 traceroute helpers 
# ---------------------------------------------------------

# We use this when we do not know what DsMapping info to send in the echo
# request for the next hop. We send the DsMapping Info with the downstream
# address set to 224.0.0.2 in the case of Ipv4 and FF02::2 in the case of
# Ipv6. On doing this, we indicate that the LSR which recieves this echo request
# must ignore DsMapping validation but send a DsMapping TLV in its echo reply.
def getGenericDsMappingInfo( addrFamily=AddressFamily.ipv4 ):
   address = '224.0.0.2' if addrFamily == AddressFamily.ipv4 else 'FF02::2'
   # MTU should not have any significance here as DsMap validation is not performed
   # at the router which receives an echo request with this TLV.
   intfMtu = 1500
   multiPathBaseIp = '0.0.0.0'
   multipath = False
   numMultiPathBits = 0
   labelStack = []
   dsMapInfo = getDsMappingInfo( address, labelStack, intfMtu, multipath,
                                 multiPathBaseIp, numMultiPathBits )
   return dsMapInfo

def tracerouteReplyRender( sock, protocol, ttl, replyPktInfo ):
   if replyPktInfo:
      hopPktInfo = HopPktInfo(
         replyHost=str( replyPktInfo.replyHost ),
         hopMtu=( 0 if not replyPktInfo.downstreamMappingInfo
                    else replyPktInfo.downstreamMappingInfo[ 0 ].mtu ),
         roundTrip=int( replyPktInfo.roundTrip // 1000 ),
         roundTripUs=( replyPktInfo.roundTrip % 1000 ),
         oneWayDelay=replyPktInfo.oneWayDelay,
         retCode=replyPktInfo.retCode,
         ttl=ttl )
   else:
      hopPktInfo = HopPktInfo( replyHost="", hopMtu=0,
                               roundTrip=0, roundTripUs=0,
                               oneWayDelay=0,
                               retCode=NoRetCode,
                               ttl=ttl )
   if tracerouteUseCapi( protocol ) and sock is not None:
      sendViaSocket( sock, MplsTraceroute( hopPktInfo=hopPktInfo ) )
   else:
      tracerouteReplyHdrRender( hopPktInfo )

def downstreamInfoRender( sock, protocol, downstreamInfos ):
   if not downstreamInfos:
      return
   if tracerouteUseCapi( protocol ) and sock is not None:
      dsInfos = DownstreamInfos()
      # pylint: disable=no-member
      dsInfos.downstreamInfo.extend( downstreamInfos )
      # pylint: enable=no-member
      sendViaSocket( sock, MplsTraceroute( downstreamInfos=dsInfos ) )
   else:
      tracerouteDownstreamInfoRender( downstreamInfos )

def getLabelStack( labelStack ):
   labels = [ getMplsLabelStr( labelStack.labelEntry[ entry ].label )
                  for entry in range( labelStack.size ) ]
   return labels

def labelStackRender( sock, protocol, rxLabelStackInfo ):
   if not rxLabelStackInfo.labelStack:
      return

   labels = getLabelStack( rxLabelStackInfo.labelStack )
   intfAndLabelStackInfo = IntfAndLabelStackInfo()
   # pylint: disable=no-member
   intfAndLabelStackInfo.labelStack[ : ] = labels
   addrType = Tac.enumName( 'LspPing::LspPingAddrType', rxLabelStackInfo.addrType )
   intfAndLabelStackInfo.addrType = addrType
   intfAndLabelStackInfo.intfAddr = str( rxLabelStackInfo.intfAddr )
   intfAndLabelStackInfo.addr = str( rxLabelStackInfo.addr )
   intfAndLabelStackInfo.intfIndex = int( rxLabelStackInfo.intfId )
   # pylint: enable=no-member
   if tracerouteUseCapi( protocol ):
      sendViaSocket( sock,
                     MplsTraceroute( intfAndLabelStackInfo=intfAndLabelStackInfo ) )
   else:
      tracerouteLabelStackRender( intfAndLabelStackInfo )

def getTunnelTypeAndPrefix( protocol, prefix,
                            rsvpSpId, ipv, pwLdpInfo=None ):
   tunnelType, tunnelPrefix, pwInfo = None, None, None
   if protocol == LspPingTypeLdp:
      tunnelType = LspPingTunnelType.tunnelLdpIpv4
      tunnelPrefix = prefix
   elif protocol == LspPingTypeGeneric:
      tunnelType = LspPingTunnelType.tunnelGenericIpv4
      tunnelPrefix = prefix
   elif protocol == LspPingTypeMldp:
      tunnelType = LspPingTunnelType.tunnelMldp
      tunnelPrefix = prefix
   elif protocol == LspPingTypeSr:
      assert ipv in [ IPV4, IPV6 ]
      tunnelType = ( LspPingTunnelType.tunnelSrIpv4 if ipv == IPV4
                     else LspPingTunnelType.tunnelSrIpv6 )
      tunnelPrefix = prefix
   elif protocol == LspPingTypeOspfSr:
      tunnelType = LspPingTunnelType.tunnelOspfSrIpv4
      tunnelPrefix = prefix
   elif protocol == LspPingTypeBgpLu:
      assert ipv in [ IPV4, IPV6 ]
      tunnelType = ( LspPingTunnelType.tunnelBgpLuIpv4 if ipv == IPV4
                     else LspPingTunnelType.tunnelBgpLuIpv6 )
      tunnelPrefix = prefix
   elif protocol == LspPingTypeVpn:
      assert ipv in [ IPV4, IPV6 ]
      tunnelType = ( LspPingTunnelType.tunnelVpnIpv4 if ipv == IPV4
                     else LspPingTunnelType.tunnelVpnIpv6 )
      tunnelPrefix = prefix
   elif rsvpSpId:
      tunnelType = LspPingTunnelType.tunnelRsvpIpv4
      tunnelPrefix = rsvpSpId.sessionId.dstIp.stringValue
   elif pwLdpInfo:
      tunnelType = LspPingTunnelType.tunnelPwLdp
      tunnelPrefix = None
      pwInfo = LspPingPwInfo( pwLdpInfo.localRouterId,
                              pwLdpInfo.remoteRouterId,
                              pwLdpInfo.pwId,
                              pwLdpInfo.pwGenId,
                              pwLdpInfo.pwType,
                              pwLdpInfo.controlWord )
      # activeCcType is control word (CW) if CW is being used and is supported by
      # peer. Router alert (RA) is used otherwise. This logic may change when we
      # implement TTL expiry as a ccType.
      if ( pwLdpInfo.controlWord and
           pwLdpInfo.ccTypes & pwPingCcEnumToVal[ PwPingCcType.cw ] ):
         pwInfo.activeCcType = pwPingCcEnumToVal[ PwPingCcType.cw ]
      else:
         pwInfo.activeCcType = pwPingCcEnumToVal[ PwPingCcType.ra ]
   else:
      tunnelType = LspPingTunnelType.tunnelModeRaw
      tunnelPrefix = None

   return ( tunnelType, tunnelPrefix, pwInfo )

def buildDsInfoPbModel( dsType, dsInfo, labels, popFecTypeList, pushFecTypeList ):
   downstreamInfo = DownstreamInfo()
   addrType = Tac.enumName( 'LspPing::LspPingAddrType', dsInfo.addrType )
   downstreamInfo.addrType = addrType
   downstreamInfo.dsIntfAddr = str( dsInfo.dsIntfAddr )
   downstreamInfo.dsIntfIndex = dsInfo.intfIndex()
   downstreamInfo.dsIpAddr = str( dsInfo.dsIpAddr )
   downstreamInfo.retCode = dsInfo.retCode
   downstreamInfo.dsType = dsType
   # pylint: disable=no-member
   downstreamInfo.labelStack[ : ] = labels
   if popFecTypeList:
      downstreamInfo.popFecTypeList.fecTypeList[ : ] = popFecTypeList
   if pushFecTypeList:
      downstreamInfo.pushFecTypeList.fecTypeList[ : ] = pushFecTypeList
   # pylint: enable=no-member
   return downstreamInfo

def getFecStackChangeTlvTypes( dsInfo ):
   popTlvFecs, pushTlvFecs = [], []
   fscInfo = dsInfo.fecStackChangeInfo
   if fscInfo.valid:
      popFecStack = fscInfo.popTlvInfo
      pushFecStack = fscInfo.pushTlvInfo
      for index, _ in enumerate( popFecStack.fecInfo ):
         fecTlvType = popFecStack.fecInfo[ index ].fecTlvType
         popTlvFecs.append( fecTlvTypeStr( fecTlvType ) )
      for index, _ in enumerate( pushFecStack.fecInfo ):
         fecTlvType = pushFecStack.fecInfo[ index ].fecTlvType
         pushTlvFecs.append( fecTlvTypeStr( fecTlvType ) )
   return popTlvFecs, pushTlvFecs

def getAndRenderDownstreamInfoOrIls( sock, protocol, replyPktInfo,
                                     tunnelType, dsInfoExpected=True ):
   mplsLabelStack, downstreamInfos = [], []
   if replyPktInfo.downstreamMappingInfo:
      if tunnelType != LspPingTunnelType.tunnelMldp:
         dsInfos = [ replyPktInfo.downstreamMappingInfo[ 0 ] ]
      else:
         dsInfos = list( replyPktInfo.downstreamMappingInfo.values() )
      for dsInfo in dsInfos:
         if dsInfo.valid and dsInfo.labelStack.size:
            labelStack = dsInfo.labelStack
            labels = getLabelStack( labelStack )
            popFecTypeList, pushFecTypeList = getFecStackChangeTlvTypes( dsInfo )
            downstreamInfo = buildDsInfoPbModel( replyPktInfo.dsType,
                                                 dsInfo, labels,
                                                 popFecTypeList, pushFecTypeList )
            downstreamInfos.append( downstreamInfo )
            mplsLabelStack.append( dsInfo.labelStack )
      downstreamInfoRender( sock, protocol, downstreamInfos )
   elif ( replyPktInfo.intfAndLabelStackInfo.valid and
          replyPktInfo.intfAndLabelStackInfo.labelStack.size ):
      mplsLabelStack.append( replyPktInfo.intfAndLabelStackInfo.labelStack )
      labelStackRender( sock, protocol,
                        replyPktInfo.intfAndLabelStackInfo )
   elif dsInfoExpected:
      # We expected either dsInfo or ILS except for few scenarios where we don't
      # send DSMAP in the first place e.g. NHG traceroute. However we should only do
      # this check for success/labels switched scenario
      if ( replyPktInfo.retCode == LspPingReturnCode.seeDdmTlv or
           replyPktInfo.retCode == LspPingReturnCode.labelSwitchedAtStackDep ):
         err = '     error: downstream information missing in echo response'
         sendOrRenderTracerouteErr( err, sock )
   return mplsLabelStack

def labelStackIsExplicitNull( ipv, mplsLabelStack ):
   for labelStack in mplsLabelStack:
      topLabel = labelStack.labelEntry[ 0 ].label if labelStack.size \
                 else None
      topLabelIsExplicitNull = (
         ( ipv == IPV4 and topLabel == MplsLabel.explicitNullIpv4 ) or
         ( ipv == IPV6 and topLabel == MplsLabel.explicitNullIpv6 )
      )
      isLabelStackSizeOne = ( labelStack.size == 1 )
   # isLabelStackSizeOne and topLabelIsExplicitNull checks for all the
   # downstreaminfo labelstack objects and returns true if it holds true
   # all of them.
   return isLabelStackSizeOne and topLabelIsExplicitNull

def isReturnCodeRouterEgress( retCode ):
   return retCode == LspPingReturnCode.repRouterEgress

def isExpectedReturnCode( prefix, replyPktInfo ):

   retCode = replyPktInfo.retCode

   ret = True
   ret = ( not prefix or
            retCode == 'labelSwitchedAtStackDep' or
            retCode == 'seeDdmTlv' or
            retCode == 'lblSwitchedFecChange' )

   if ( ret and
        replyPktInfo.dsType == LspPingDownstreamMappingType.downstreamDetailedMap ):
      if len( replyPktInfo.downstreamMappingInfo ) == 1:
         # Checks for the retcode in the only DDMAP tlv received in echo reply.
         # Reply header retcode must be labelSwitched/fecStackChange or if
         # reply header retcode is seeDdmTlv, DDMAP retcode must be
         # labelSwitchedAtStackDep.
         # ( As per rfc 8029, we should also confirm that if reply header retCode is
         #   labelSwitched then DDMAP retcode must be noRetCode but here we are not
         #   checking for noRetCode in DDMAPTlv as cisco replies with both reply
         #   header and DDMAP retcode set to labelSwitched )
         ret = ( retCode == LspPingReturnCode.labelSwitchedAtStackDep or
                 ( retCode == LspPingReturnCode.seeDdmTlv and
                   replyPktInfo.downstreamMappingInfo[ 0 ].retCode in
                                      [ LspPingReturnCode.labelSwitchedAtStackDep,
                                        LspPingReturnCode.lblSwitchedFecChange ] ) )

      if len( replyPktInfo.downstreamMappingInfo ) > 1:
         # Checks for the retcode returned in all DDMAP tlvs received in the echo
         # reply. Reply header retcode must be  SeeDdmTlv and all
         # ddmap tlvs must have labelSwitchedAtStackDep in them.
         # Note: We might need to check for fecStackChange with SeeDdmTlv in the
         #       future if required.
         ret = ( ( retCode == LspPingReturnCode.seeDdmTlv ) and
                 all( dsInfo.retCode in [ LspPingReturnCode.labelSwitchedAtStackDep,
                                          LspPingReturnCode.lblSwitchedFecChange ]
                      for dsInfo in replyPktInfo.downstreamMappingInfo.values() ) )
   return ret

def createFirstMultipathSearchNode( packetTracerSm, dsMappingInfo, dstIpStr,
                                    labelStack, l3Intf ):
   # populate search stack with first node to probe
   nexthop = Tac.Value( "LspPing::LspPingGraphNode" )
   nexthop.hopNum = 1
   nexthop.clientId = packetTracerSm.generateClientId()
   nexthop.dsMappingInfo = dsMappingInfo
   for idx, val in enumerate( labelStack ):
      nexthop.mplsLabel[ idx ] = val
   nexthop.intf = l3Intf
   nexthop.dstIpAddr = IpGenAddr( dstIpStr )
   nexthop.isRootNode = True
   return nexthop

def getAndClearPacketTracerSm( lspPingClientRoot ):
   packetTracerSm = lspPingClientRoot.lspPingPacketTracerSm
   # force packetTracerSm state to be empty
   packetTracerSm.searchStack.clear()
   packetTracerSm.tracePath.clear()
   packetTracerSm.searchCompleted = False
   packetTracerSm.sock = 0
   packetTracerSm.clientConfigColl.pingClientConfig.clear()
   return packetTracerSm

def createMultipathTracerouteInfo( tunnelType, srcIpStr, dstIpStr, smac,
                                   mount, dmac, interface, dsMapInfo, tc, count,
                                   interval, topLabel, nextHopIpStr,
                                   dstPrefix=None, dstype=None, tos=None,
                                   standard=None, size=None, padReply=False ):
   # create LspPingMultipathTracerouteInfo object
   tracerouteInfo = Tac.newInstance( "LspPing::LspPingMultipathTracerouteInfo" )
   tracerouteInfo.oamStandard = standard or mount.config.oamStandard
   tracerouteInfo.tunnelType = tunnelType
   tracerouteInfo.nextHopIp = IpGenAddr( nextHopIpStr )
   tracerouteInfo.srcIpAddr = IpGenAddr( srcIpStr )
   tracerouteInfo.dstIpAddr = IpGenAddr( dstIpStr )
   if tos:
      tracerouteInfo.tos = tos
   if dstPrefix:
      tracerouteInfo.dstPrefix = IpGenPrefix( dstPrefix )
   # Use smac if provided, else fall back to interface's routedAddr. Use
   # brigeMacAddr if interface is not found in allIntfStatusDir.
   tracerouteInfo.srcEthAddr = ( smac or
                                 getattr( mount.allIntfStatusDir.intfStatus.get( \
                                          interface ),
                                          'routedAddr',
                                          mount.bridgingConfig.bridgeMacAddr ) )
   tracerouteInfo.tunnelNexthopEthAddr = dmac
   tracerouteInfo.dsMapInfo = dsMapInfo
   # By default tracerouteInfo.dstype is LspPingDownstreamMappingType.downstreamMap
   if dstype == LspPingDDMap:
      tracerouteInfo.dstype = LspPingDownstreamMappingType.downstreamDetailedMap
   if size:
      tracerouteInfo.pktSize = size
      tracerouteInfo.padReply = padReply
   tracerouteInfo.mplsTc = tc
   tracerouteInfo.txCount = count
   tracerouteInfo.txRate = interval
   tracerouteInfo.fd = sys.stdout.fileno()
   # 1 indicates text output for cliPrinter
   tracerouteInfo.format = 1
   tracerouteInfo.topLabel = Tac.newInstance( "Arnet::MplsLabel", topLabel )
   targetFecStack = Tac.newInstance( "LspPing::LspPingFecStack" )
   targetFecStack.fecInfo[ 0 ] = getTargetFecInfo( tunnelType,
                                 tracerouteInfo.topLabel,
                                 src=srcIpStr, dst=dstIpStr,
                                 dstPrefix=dstPrefix,
                                 oamStandard=tracerouteInfo.oamStandard )
   tracerouteInfo.targetFecStack = targetFecStack
   tracerouteInfo.valid = True
   return tracerouteInfo

def doMultipathTraceroute( lspPingClientRoot, dsMappingInfo, dstIpStr,
                           labelStack, l3Intf, sock ):
   # Unlike default traceroute, all logic for multipath traceroute runs in C++
   # and resides within LspPingPacketTracerSm. This utility library waits till
   # packetTracerSm is done, and exits.
   packetTracerSm = getAndClearPacketTracerSm( lspPingClientRoot )
   # set up first node to probe
   nexthop = createFirstMultipathSearchNode( packetTracerSm, dsMappingInfo, dstIpStr,
                                             labelStack, l3Intf )
   packetTracerSm.searchStack.push( nexthop )
   # initiate search
   packetTracerSm.sock = sock.fileno()
   packetTracerSm.probeNextHop()

   # wait till packetTracerSm declares search to be completed
   while not packetTracerSm.searchCompleted:
      Tac.runActivities( 0.050 )

   return 0

# ---------------------------------------------------------
#          Core traceroute implementation
# ---------------------------------------------------------
def bumpSessionId( fn ):
   @wraps( fn )
   def wrapper( *args, **kwargs ):
      ret = fn( *args, **kwargs )
      # cleanup the stuff in existing global state before we actually bump the
      # sessionId.
      cleanupGlobalState()
      # bump session id per fn invocation to force fresh global state
      sessionIdIncr()
      return ret

   return wrapper

# traceroute function returns a tuple of ( success, txPkts and replyHostRtts )
# Returning variable number of args is feasible only in python 3 where
# *retArgs=traceroute(...) can be done to get variable args which will help in
# returning the tuple when required only.
@bumpSessionId
def traceroute( mount, state, l3Intf, labelStack, src, dst, srcMac, dstMac, count,
                interval, protocol=None, prefix=None, verbose=False, hops=None,
                ipv=IPV4, rsvpSpId=None, rsvpSenderAddr=None, dsMappingInfo=None, 
                tc=None, multipath=False, nextHopIp=None,
                mldpInfo=None, standard=None, size=None, padReply=False,
                dstype=None, setFecValidateFlag=True, egressValidateAddress=None,
                lookupLabelCount=1, tos=None, sock=None, algorithm=None,
                pktTimeout=5 ):

   txPkts, replyHostRtts = 0, {}
   retArgs = TracerouteRetArgs( retVal=-1, txPkts=txPkts,
                                replyHostRtts=replyHostRtts )
   baseErrorMsg = 'Error: Could not proceed with traceroute\n'

   af = AddressFamily.ipv4 if ipv == IPV4 else AddressFamily.ipv6
   if src and IpGenAddr( src ).af != af:
      msg = 'Source ip address must match destination address family'
      err = baseErrorMsg + msg
      sendOrRenderTracerouteErr( err, sock )
      return retArgs

   reqSrcIp = src or getIntfPrimaryIpAddr( mount, l3Intf, ipv )
   if not reqSrcIp:
      msg = 'No usable source ip address for traceroute'
      err = baseErrorMsg + msg
      sendOrRenderTracerouteErr( err, sock )
      return retArgs

   # While dst option is not supported via the CLI at this time, some of the tests
   # use it. Also, we may in future start allowing destination address to be
   # specified via the CLI. This check makes sure that if the dst is provided, it's
   # valid.
   if dst:
      # ensure both src and dst ip are of the consistent ipv type
      if not verifyConsistentAfType( reqSrcIp, dst ):
         msg = "Inconsistent source (" + str( reqSrcIp ) + ") and destination (" + \
               str( dst ) + ") IP address families.\n"
         err = baseErrorMsg + msg
         sendOrRenderTracerouteErr( err, sock )
         return retArgs

      # Provided destination address must be in 127.0.0.0/8 range for IPv4 and
      # ::fff:127.0.0.0/104 range for IPv6
      if not isValidDestAddr( dst, ipv ):
         msg = 'Destination address must be in ' + ( '127.0.0.0/8' if ipv is IPV4
               else '::ffff:127.0.0.0/104' ) + ' range.'
         err = baseErrorMsg + msg
         sendOrRenderTracerouteErr( err, sock )
         return retArgs

   # In case of multipath, our destination should be the base address. User specified
   # destination address should match the provided base IP address. If it's not
   # multipath, use the default IP destination address if one is not provided.
   if multipath and dsMappingInfo:
      if dst:
         msg = ( 'Destination address is not supported for multipath traceroute. ' +
                 'Please use multipath base address.' )
         err = baseErrorMsg + msg
         sendOrRenderTracerouteErr( err, sock )
         return retArgs
      dst = dsMappingInfo.dsMultipathBaseAddr.stringValue
   else:
      dst = dst or DefaultLspPingReqDst[ ipv ]

   # pseudowire does not have traceroute implementation yet so
   # pwInfo returned is irrelevant.
   tunnelType, tunnelPrefix, _ = \
               getTunnelTypeAndPrefix( protocol, prefix, rsvpSpId, ipv )

   tracerouteInfo = Tac.newInstance( "LspPing::LspPingMultipathTracerouteInfo" ) \
                    if not multipath else \
                    createMultipathTracerouteInfo( tunnelType, reqSrcIp, dst, srcMac,
                                                   mount, dstMac, l3Intf,
                                                   dsMappingInfo, tc, count,
                                                   interval, labelStack[ 0 ],
                                                   nextHopIp, tunnelPrefix, dstype,
                                                   tos, standard=standard, size=size,
                                                   padReply=padReply )
   # create LspPingClientRootSm
   lspPingClientRoot = createLspPingClient( mount, state, l3Intf, reqSrcIp, dst,
                                            protocol, tracerouteInfo )
   if not lspPingClientRoot:
      err = baseErrorMsg
      sendOrRenderTracerouteErr( err, sock )
      return retArgs

   if multipath:
      ret = doMultipathTraceroute( lspPingClientRoot, dsMappingInfo, dst,
                                   labelStack, l3Intf, sock )
      retArgs = retArgs._replace( retVal=ret )
      return retArgs

   sendDsMappingTlv, nextDsMappingInfo = False, None
   dsFecStackChangeInfo = Tac.newInstance( "LspPing::LspPingDownstreamMappingInfo" )
   currentTargetFecStack = None
   # Send the DsMap TLV in the subsequent echo requests.
   if dsMappingInfo:
      sendDsMappingTlv, nextDsMappingInfo = True, dsMappingInfo

   labelStack = list( labelStack )
   # For some usecases we already push an Exp-NULL label at the BOS. In that case,
   # we do not need to push an extra Exp-NULL label again.
   if ( ( tunnelType == LspPingTunnelType.tunnelModeRaw ) and
        ( not egressValidateAddress ) and
        ( not explicitNullAtBos( labelStack ) ) ):
      nullLabel = ( MplsLabel.explicitNullIpv4 if ipv == IPV4 else
                    MplsLabel.explicitNullIpv6 )
      labelStack.extend( [ nullLabel ] )

   # BUG942452: Get rid of implicit-nulls from the beginning of the labelStack to
   # ensure that we set TTL on the non implicit-null label.
   while len( labelStack ) > 1 and labelStack[ 0 ] == MplsLabel.implicitNull:
      labelStack.pop( 0 )

   labelTtl, numHops = [ 1 ] * len( labelStack ), 0
   maxHops, maxRequests = 64, 1
   clientStatusColl = state.clientRoot.lspPingClientStatusColl
   # pylint: disable-msg=cell-var-from-loop
   # pylint: disable=too-many-nested-blocks
   startTime = Tac.now()
   for hop in range( 1, ( hops or maxHops ) + 1 ):
      numHops += 1
      clientId = state.clientIdBase + hop
      ret = createLspPingClientConfig( mount, state, clientId, protocol, tunnelType,
                                       l3Intf, labelStack, reqSrcIp, dst, dstMac,
                                       srcMac, mplsTtl=labelTtl, count=maxRequests,
                                       dstPrefix=tunnelPrefix, rsvpSpId=rsvpSpId,
                                       rsvpSenderAddr=rsvpSenderAddr,
                                       fecStackChangeInfo=dsFecStackChangeInfo,
                                       currentTargetFecStack=currentTargetFecStack,
                                       dsMappingInfo=nextDsMappingInfo, tc=tc,
                                       mldpInfo=mldpInfo, standard=standard,
                                       size=size, padReply=padReply, tos=tos,
                                       dstype=dstype,
                                       setFecValidateFlag=setFecValidateFlag,
                                       egressValidateAddress=egressValidateAddress,
                                       lookupLabelCount=lookupLabelCount,
                                       sock=sock, algorithm=algorithm,
                                       tracerouteInfo=tracerouteInfo )
      if not ret:
         err = baseErrorMsg
         sendOrRenderTracerouteErr( err, sock )
         state.clientIdBase += numHops
         retArgs = retArgs._replace( retVal=-1 )
         return retArgs

      clientConfig, clientStatus = ret
      unreachable = True
      for seqnum in range( 1, maxRequests + 1 ):
         Tac.waitFor(
            lambda: echoRequestsSent( clientStatusColl, [ clientId ], seqnum ), 
            maxDelay=0.1, warnAfter=None,
            description="echo request(s) to be sent" )
         bt8( "Echo request sent for client id:", bv( clientId ),
              "sequence no::", bv( seqnum ) )
         txPkts += 1
         # wait to receive an echo response
         timeout = min( startTime + ( interval * clientId + seqnum ) + pktTimeout,
                        Tac.now() + pktTimeout )
         while Tac.now() < timeout:
            if echoReplyReceived( clientStatusColl, clientId, seqnum ):
               break
            Tac.runActivities( 0.050 )

         # check if we got an echo response back
         replyPktInfo = clientStatus.replyPktInfo.get( seqnum )
         if replyPktInfo and replyPktInfo.seqNum == seqnum:
            bt8( "Received reply packet from", bv( replyPktInfo.replyHost ),
                  "for client id:", bv( clientId ),
                  "sequence no:", bv( replyPktInfo.seqNum ),
                 "with retcode", bv( replyPktInfo.retCode ) )
            replyHostRtts.setdefault( replyPktInfo.replyHost, [] ).append(
                  replyPktInfo.roundTrip / 1000.0 )
            tracerouteReplyRender( sock, protocol, hop, replyPktInfo )
            dsInfoExpected = dsMappingInfo is not None
            mplsLabelStack = getAndRenderDownstreamInfoOrIls( sock,
                                                              protocol,
                                                              replyPktInfo,
                                                              tunnelType,
                                                              dsInfoExpected )
            # Don't check return code if protocol is not ldp or segment-routing
            # Check for DownstreamInfo and return codes in echo reply header
            # & DDMAP tlvs.
            # Stop probing and return if no downstreamInfo is there or an
            # undesired retcode is returned.
            if ( replyPktInfo.downstreamMappingInfo or
                 replyPktInfo.intfAndLabelStackInfo ):
               if ( not mplsLabelStack or
                    not isExpectedReturnCode( prefix, replyPktInfo ) ):
                  # Skip explicit NULL in echo reply to stop probing further hop(s)
                  # This is necessary when traceroute to single-hop LSP. Otherwise,
                  # will traceroute to more than one hop.
                  state.clientIdBase += numHops
                  retArgs = retArgs._replace( retVal=0, txPkts=txPkts,
                                              replyHostRtts=replyHostRtts )
                  return retArgs
            elif isReturnCodeRouterEgress( replyPktInfo.retCode ):
               state.clientIdBase += numHops
               retArgs = retArgs._replace( retVal=0, txPkts=txPkts,
                                           replyHostRtts=replyHostRtts )
               return retArgs

            unreachable = False
            # increment top label ttl for next ping
            labelTtl[ 0 ] += 1
            # set next dsMapInfo to be used
            if sendDsMappingTlv:
               # Currently, we only populate dsMappingInfo for LDP, RSVP, and SR
               # mpls traceroutes. As this helper will not not be used for multipath
               # traceroutes, we always expect only one dsMapInfo object in the
               # received echo response. If for some reason no dsMapInfo is present
               # in the echo response, don't include the dsMapInfo object in any
               # subsequent echo requests.
               nextDsMappingInfo = (
                        next( iter(
                              replyPktInfo.downstreamMappingInfo.values() ) ) if
                        replyPktInfo.downstreamMappingInfo else None )

               if nextDsMappingInfo and nextDsMappingInfo.fecStackChangeInfo.valid:
                  dsFecStackChangeInfo = nextDsMappingInfo.fecStackChangeInfo
                  currentTargetFecStack, errmsg = updateTargetFecStack(
                        clientConfig.targetFecStack,
                        dsFecStackChangeInfo.popTlvInfo,
                        dsFecStackChangeInfo.pushTlvInfo )
                  if errmsg:
                     err = baseErrorMsg + errmsg
                     sendOrRenderTracerouteErr( err, sock )
                     return retArgs

               # As per rfc 6425, section 4.3.4, If the echo request is destined
               # for more than one node, then the Downstream IP Address field of
               # the Downstream Detailed Mapping TLV MUST be set to the
               # ALLROUTERS multicast address and the Address Type field
               # MUST be set to either IPv4 Unnumbered or IPv6 Unnumbered
               # depending on the Target FEC Stack TLV.
               # As per rfc 4379, section 4.6, the interface index MUST be set to 0.
               # Arnet::IntfId does not support taking value 0, so 0 value is set in
               # LspPingClientTxSm depending on dsMappingInfo.dsIpAddr being
               # AllRoutersMulticast address.
               if ( protocol == 'mldp'
                    and ( len( replyPktInfo.downstreamMappingInfo.values() ) > 1 ) ):
                  if ipv == IPV4:
                     dsIpAddr = IpGenAddr( IpAddrAllRoutersMulticast )
                     addrType = Ipv4Unnumbered

                  else:
                     dsIpAddr = IpGenAddr( Ip6AddrAllRoutersMulticast )
                     addrType = Ipv6Unnumbered

                  nextDsMappingInfo = Tac.newInstance(
                        'LspPing::LspPingDownstreamMappingInfo',
                        nextDsMappingInfo.mtu, addrType, nextDsMappingInfo.dsFlags,
                        dsIpAddr, IpGenAddr(), 0, 0, IpGenAddr(),
                        LspPingMultipathBitset( 0 ), nextDsMappingInfo.labelStack,
                        nextDsMappingInfo.fecStackChangeInfo, Arnet.IntfId( '' ),
                        0, 0, True )
         else:
            if seqnum < maxRequests:
               clientConfig.txCount = 1

      if unreachable:
         tracerouteReplyRender( sock, protocol, hop, None )
         labelTtl[ 0 ] += 1
         if sendDsMappingTlv:
            # Traceroute to this particular hop timed out. Hence we do not have
            # the DsMapping Info TLV to send in the next echo request to the next hop
            # Set the nextDsMappingInfo to None here. The subsequent lines of code
            # would handle this, and sets an appropriate DsMapping TLV to be sent in
            # the echo request to subsequent hops.
            nextDsMappingInfo = None

      if sendDsMappingTlv and ( not nextDsMappingInfo ):
         addrFamily = AddressFamily.ipv4 if ipv == IPV4 else AddressFamily.ipv6
         nextDsMappingInfo = getGenericDsMappingInfo( addrFamily=addrFamily )

   #pylint: enable-msg=cell-var-from-loop
   state.clientIdBase += numHops

   retArgs = retArgs._replace( retVal=-1 )
   return retArgs
