#!/usr/bin/env python3
# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import functools
from collections import defaultdict

import CliParser
import Tac
from CliCommand import SetEnumMatcher, guardedKeyword
from CliPlugin.ShowConfigConsistencyCliModels import ConfigConsistencyMultiTableModel


class UndefinedReferenceChecker:
   referenceGatherers = defaultdict( list )
   definitionGatherers = {}

   @classmethod
   def addReferenceGatherer( cls, features, gatherer ):
      for feature in features:
         cls.referenceGatherers[ feature ].append( gatherer )

   @classmethod
   def addDefinitionGatherer( cls, features, gatherer ):
      for feature in features:
         cls.definitionGatherers[ feature ] = gatherer

   @classmethod
   def check( cls, features ):
      undefinedReferences = {}
      for feature in features:
         if feature in cls.referenceGatherers and feature in cls.definitionGatherers:
            references = set()
            for gatherer in cls.referenceGatherers[ feature ]:
               references |= gatherer.gather( feature )
            definitions = cls.definitionGatherers[ feature ].gather( feature )
            undefinedReferences[ feature ] = references - definitions
      return undefinedReferences

class ConfigConsistencyChecker:
   """
   Config consistency checker.
   Given cluster and categories, run registered check functions
   """

   @classmethod
   def getClusterChecker( cls, cluster ):
      if not hasattr( cls, cluster ):
         # Add empty matcher to satisfy ShowCommandClass
         setattr( cls,
                  cluster,
                  type( cluster,
                        (),
                        { 'clusterMatcher': lambda: SetEnumMatcher( {} ),
                          'categoryMatcher': lambda: SetEnumMatcher( {} ) } ) )
      return getattr( cls, cluster )

   @classmethod
   def getCategoryChecker( cls, cluster, category ):
      ClusterCategory = cls.getClusterChecker( cluster )
      if not hasattr( ClusterCategory, category ):
         setattr( ClusterCategory, category,
                  type( category, (), { 'checks': [] } ) )
         setattr( cls, cluster, ClusterCategory, )
      return getattr( ClusterCategory, category )

def configConsistencyCheck( cluster, category ):
   """
   Decorator to register the check function to
   `show config consistency ( cluster ) [ { category } ]`

   By default the CLI model class is ConfigConsistencyMultiTableModel.
   ConfigConsistencyMultiTableModelBuilder is the preferred way to build the model.
   """
   def decorator( func ):
      ClusterChecker = ConfigConsistencyChecker.getClusterChecker( cluster )
      CategoryChecker = ConfigConsistencyChecker.getCategoryChecker(
                           cluster, category )
      CategoryChecker.checks.append( func )
      setattr( ClusterChecker, category, CategoryChecker )
      setattr( ConfigConsistencyChecker, cluster, ClusterChecker )

      @functools.wraps( func )
      def wrapper( *args, **kw ):
         return func( *args, **kw )
      return wrapper
   return decorator

def configConsistencyCluster(
      cluster, desc, cliModelClass=ConfigConsistencyMultiTableModel ):
   """
   Create a cluster checker class

   class ConfigConsistencyChecker:
      class < cluster >:  # <--- Create this
         def getCategories() -> Dict[str, str]:...
         def clusterMatcher() -> Node: ...
         def categoryMatcher() -> Node: ...
   """
   def decorator( func ):
      # each cluster can only declare once
      assert not hasattr( ConfigConsistencyChecker, cluster )
      ClusterChecker = ConfigConsistencyChecker.getClusterChecker( cluster )

      def getCategories( mode, context=None ):
         res = {}
         for name, CategoryChecker in ClusterChecker.__dict__.items():
            if ( not name.startswith( '_' ) and
                 isinstance( CategoryChecker, type ) and
                 CategoryChecker.valid( mode, context ) ):
               res[ name ] = CategoryChecker.desc
         return res

      def check( categories, mode, args, context=None ):
         model = cliModelClass()
         if not categories:
            categories = getCategories( mode, context )

         for ctg in categories:
            CategoryChecker = getattr( ClusterChecker, ctg )
            for func in CategoryChecker.checks:
               func( model, mode, args )

         return model

      def categoryMatcher():
         return SetEnumMatcher( getCategories )

      def clusterMatcher():
         def guard( mode, token ):
            if func( mode ):
               return None
            else:
               return CliParser.guardNotThisPlatform
         return guardedKeyword( cluster, desc, guard )

      setattr( ClusterChecker, 'getCategories', getCategories )
      setattr( ClusterChecker, 'check', check )
      setattr( ClusterChecker, 'categoryMatcher', categoryMatcher )
      setattr( ClusterChecker, 'clusterMatcher', clusterMatcher )
      setattr( ConfigConsistencyChecker, cluster, ClusterChecker )

      @functools.wraps( func )
      def wrapper( *args, **kw ):
         return func( *args, **kw )
      return wrapper
   return decorator

def configConsistencyCategory( cluster, category, desc ):
   """
   Create a category checker class

   class ConfigConsistencyChecker:
      class < cluster >: ...
         class < category >:  # <--- Create this
            checks = []
            desc = < desc >
            def valid( mode ) -> Bool: ...
   """
   def decorator( func ):
      ClusterChecker = ConfigConsistencyChecker.getClusterChecker( cluster )
      # each category can only declare once
      assert not hasattr( ClusterChecker, category )
      CategoryChecker = ConfigConsistencyChecker.getCategoryChecker(
                           cluster, category )

      setattr( CategoryChecker, 'valid', func )
      setattr( CategoryChecker, 'desc', desc )
      setattr( ClusterChecker, category, CategoryChecker )
      setattr( ConfigConsistencyChecker, cluster, ClusterChecker )

      @functools.wraps( func )
      def wrapper( *args, **kw ):
         return func( *args, **kw )
      return wrapper
   return decorator
