# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AirStreamLib
import CliSession
import GnmiSetCliSession
import Plugins
import Tac
import Tracing
import LazyMount

t0 = Tracing.Handle( "AleCapacityConfigSession" ).trace0

ThresholdEntry = Tac.Type( "AleCapacity::ThresholdEntry" )
ThresholdConfig = Tac.Type( "AleCapacity::OpenConfig::ThresholdConfig" )
ThresholdConfigElem = Tac.Type( "AleCapacity::OpenConfig::ThresholdConfigElem" )

@Plugins.plugin( requires=( 'session', ) )
def Plugin( entMan ):
   # Precommnit handler for hw platform utilization thresholds

   class ToNativePlatformUtilThresholdSyncher( GnmiSetCliSession.PreCommitHandler ):
      externalPathList = [ 'hardware/capacity/openconfig/config' ]
      nativePathList = [ 'hardware/capacity/config' ]

      @classmethod
      def run( cls, sessionName ):
         externalConfig = AirStreamLib.getSessionEntity(
               entMan, sessionName, 'hardware/capacity/openconfig/config' )
         nativeConfig = AirStreamLib.getSessionEntity(
               entMan, sessionName, 'hardware/capacity/config' )
         nativeStatus = LazyMount.mount( entMan, 'hardware/capacity/status', 
                                         'Tac::Dir', 'r' )
         
         t0( 'Construct lookup cache for resource names' )
         rnameSet = set()
         for agentDir in nativeStatus.values():
            if not agentDir.tableList:
               continue
            featureMapping = agentDir.tableList.featureMapping
            for tname in featureMapping:
               for fname in featureMapping[ tname ].feature:
                  # fname can be ''
                  if fname != '':
                     rnameSet.add( tname + '-' + fname )
                  else:
                     rnameSet.add( tname )
 
         for rname in list( externalConfig.threshold ):
            t0( f"Resource name validity checking for {rname}" )
            rnameNative = ''
            names = rname.split( '/' )
            numNames = len( names )
            if numNames <= 2:
               rnameNative = names[ 0 ] if numNames == 1 \
                        else names[ 0 ] + '-' + names[ 1 ]
               if rnameNative not in rnameSet:
                  raise AirStreamLib.ToNativeSyncherError( sessionName, cls.__name__,
                     f"non-existing hw platform utilization resource name {rname}" )
            else:
               raise AirStreamLib.ToNativeSyncherError( sessionName, cls.__name__,
                     f"invalid table/feature name format: {rname}" )
           
            threshold = externalConfig.threshold[ rname ]

            t0( 'Make resource name consistent' )
            # Resource name is a leafref to the config name; if config name is
            # not set, then copy the resource name to config name; otherwise, 
            # if the resource name and config name are different, then error out
            if not threshold.config.name:
               thresholdConfig = ThresholdConfig( rname )
               thresholdConfig.usedThresholdUpper = \
                     threshold.config.usedThresholdUpper
               thresholdConfig.usedThresholdUpperClear = \
                     threshold.config.usedThresholdUpperClear
               thresholdConfigElem = ThresholdConfigElem( rname )
               thresholdConfigElem.config = thresholdConfig
               externalConfig.threshold.addMember( thresholdConfigElem )
            elif threshold.config.name != rname:
               raise AirStreamLib.ToNativeSyncherError( sessionName, cls.__name__,
                     f"Resource name is different from the inner config name; "
                     f"{rname} != {threshold.config.name}" )

            t0( 'Validity check for thresholds' )
            if not threshold.config.usedThresholdUpper:
               if threshold.config.usedThresholdUpperClear:
                  # Setting the clear-threshold value without having a threshold
                  # value is meaningless
                  raise AirStreamLib.ToNativeSyncherError( sessionName, cls.__name__,
                        f"used-threshold-upper must be set for "
                        f"used-threshold-upper-clear to be set for {rname}" )
            
               raise AirStreamLib.ToNativeSyncherError( sessionName, cls.__name__,
                     f"both used-threshold-upper and used-threshold-upper-clear "
                     f"are unconfigured for {rname}; set the thresholds or "
                     f"remove {rname} from /system/utilization/resources/resource" )

            if threshold.config.usedThresholdUpper and \
                  threshold.config.usedThresholdUpperClear and \
                  ( threshold.config.usedThresholdUpper <
                    threshold.config.usedThresholdUpperClear ):
               raise AirStreamLib.ToNativeSyncherError( sessionName, cls.__name__,
                     f"used-threshold-upper must be greater than or equal to " 
                     f"used-threshold-upper-clear for {rname}" ) 

            t0( 'Copying thresholds from external to native for {rnameNative}' )
            # At this point, we know that threshold.config.usedThresholdUpper
            # is configured
            clear = threshold.config.usedThresholdUpperClear
            implicit = False
            if not clear:
               # In the case when usedThresholdUpperClear is None, implicitly
               # set usedThresholdUpperClear as usedThresholdUpper
               clear = threshold.config.usedThresholdUpper
               implicit = True
            nativeConfig.threshold[ rnameNative ] = \
                  ThresholdEntry( threshold.config.usedThresholdUpper, clear,
                                  implicit )

         # Clean up thresholds that are not in the external config
         for rnameNative in list( nativeConfig.threshold ):
            rname = rnameNative.replace( '-', '/' )
            if rname not in externalConfig.threshold:
               del nativeConfig.threshold[ rnameNative ]

   GnmiSetCliSession.registerPreCommitHandler( ToNativePlatformUtilThresholdSyncher )

   AirStreamLib.registerCopyHandler( entMan, "PlatformUtilThreshold",
         typeName="AleCapacity::OpenConfig::PlatformUtilizationConfig" )

   CliSession.registerConfigGroup( entMan, "airstream-cmv",
         'hardware/capacity/openconfig/config' )
