# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import enum
import AirStreamLib
import CliSession
import GnmiSetCliSession
import Tac
import Tracing
from BgpLib import createKey
from CliPlugin.RoutingBgpCli import checkForConflictingPolicy
from IpLibConsts import DEFAULT_VRF
from Toggles.BgpCommonToggleLib import (
   toggleOCBgpPolicyDefUniPOAEnabled,
   toggleOCBgpPolicyDefLuPOAEnabled,
   toggleOCBgpDefaultPolicyPOAEnabled,
   toggleOCBgpUcmpConfigEnabled,
)
from Toggles.RcfLibToggleLib import toggleOCBgpPeerRcfPOAEnabled
from TypeFuture import TacLazyType

th = Tracing.Handle( "ConfigSessionBgp" )
t0 = th.trace0

AfiSafi = Tac.Type( "Routing::Bgp::AfiSafi" )
PolicyActionType = Tac.Type( "Routing::Bgp::PolicyActionType" )

class PolicyDirection( enum.Enum ):
   IN = "In"
   OUT = "Out"

def registerBgpPolicy( entMan ):
   defaultPolicyYANGToEOSMapping = {
      "ACCEPT_ROUTE": PolicyActionType.policyActionPermit,
      "REJECT_ROUTE": PolicyActionType.policyActionDeny,
      # This key corresponds with the default-*-policy leaf not being present
      None: PolicyActionType.policyActionUnspecified,
   }

   afiSafiYANGToEosMapping = {
      "IPV4_UNICAST": AfiSafi.ipv4Uni(),
      "IPV6_UNICAST": AfiSafi.ipv6Uni(),
      "IPV4_LABELED_UNICAST": AfiSafi.ipv4UniLab(),
      "IPV6_LABELED_UNICAST": AfiSafi.ipv6UniLab(),
   }

   def toNativeBgpPolicySyncher( cls, sessionName ):
      extNeighborConfigDir = AirStreamLib.getSessionEntity( entMan,
         sessionName, cls.externalPathList[ 0 ] )
      extNeighborApplyPolicyColl = extNeighborConfigDir.applyPolicy
      extPeerGroupConfigDir = AirStreamLib.getSessionEntity( entMan,
         sessionName, cls.externalPathList[ 1 ] )
      extPeerGroupApplyPolicyColl = extPeerGroupConfigDir.applyPolicy
      nativeBgpConfig = AirStreamLib.getSessionEntity( entMan, sessionName,
         cls.nativePathList[ 0 ] )
      nativeBgpVrfConfigDir = AirStreamLib.getSessionEntity( entMan, sessionName,
         cls.nativePathList[ 1 ] )

      def getBgpConfig( vrfName ):
         if vrfName == DEFAULT_VRF:
            return nativeBgpConfig
         else:
            bgpVrfConfig = nativeBgpVrfConfigDir.vrfConfig.get( vrfName )
            if not bgpVrfConfig:
               raise AirStreamLib.ToNativeSyncherError( sessionName,
                  cls.__name__ + "::toNativeBgpPolicySyncher",
                  f"Vrf {vrfName} not found" )
            return bgpVrfConfig

      def getNeighborConfig( bgpConfig, neighborKeyValue ):
         peerConfigKey = createKey( neighborKeyValue )
         neighborConfig = bgpConfig.neighborConfig.get( peerConfigKey )
         if not neighborConfig:
            neighborConfig = bgpConfig.neighborConfig.newMember( peerConfigKey )
         return neighborConfig

      def checkForSupportedAfiSafiNames( afiSafiName ):
         # Once we add support for other afisafis, it will probably be worthwhile
         # to use something like BgpLib.peerConfigAttrsAfMap to get the correct
         # route map attributes in this function
         afiSafiName = afiSafiName.lstrip( "openconfig-bgp-types:" )
         supportedAfiSafis = [ "IPV4_UNICAST", "IPV6_UNICAST" ]
         if toggleOCBgpPolicyDefLuPOAEnabled():
            supportedAfiSafis += [ "IPV4_LABELED_UNICAST", "IPV6_LABELED_UNICAST" ]
         if afiSafiName not in supportedAfiSafis:
            supportedAfiSafisStr = "IPV4_UNICAST and IPV6_UNICAST"
            if toggleOCBgpPolicyDefLuPOAEnabled():
               supportedAfiSafisStr = ( "IPV4_UNICAST, IPV6_UNICAST, "
                                        "IPV4_LABELED_UNICAST, and "
                                        "IPV6_LABELED_UNICAST" )
            raise AirStreamLib.ToNativeSyncherError( sessionName,
               cls.__name__ + "::toNativeBgpPolicySyncher",
               f"Policy can only be applied on {supportedAfiSafisStr} "
               "AFI-SAFI types" )


      def setPolicy( key, neighborKeyValue, policyList, policyType, bgpConfig,
                     inOut ):
         assert inOut in PolicyDirection
         assert policyType in [ "ROUTE_MAP", "RCF" ]
         if policyType == "RCF" and not toggleOCBgpPeerRcfPOAEnabled():
            err = ( "Policy type RCF is not supported" )
            raise AirStreamLib.ToNativeSyncherError( sessionName,
               cls.__name__ + "::toNativeBgpPolicySyncher", err )
         # import-policy and export-policy are leaf-lists in the public model,
         # but in EOS only one route map can be specified at a point of application.
         # Our YANG deviations ensure that this is the case. The list can also be
         # empty, which will set no route map.
         if not policyList:
            return
         if len( policyList ) > 1:
            direction = "inbound" if inOut == PolicyDirection.IN else "outbound"
            err = ( f"Only one {direction} policy can be applied per neighbor "
                    "or peer group." )
            raise AirStreamLib.ToNativeSyncherError( sessionName,
               cls.__name__ + "::toNativeBgpPolicySyncher", err )
         afiSafiName = key.afiSafiName
         checkForSupportedAfiSafiNames( afiSafiName )

         neighborConfig = getNeighborConfig( bgpConfig, neighborKeyValue )

         peerConfigKey = createKey( neighborKeyValue )
         neighborConfig = bgpConfig.neighborConfig.get( peerConfigKey )
         if not neighborConfig:
            neighborConfig = bgpConfig.neighborConfig.newMember( peerConfigKey )

         addrFamily = "ipv4"
         if afiSafiName in [ "IPV6_UNICAST", "IPV6_LABELED_UNICAST" ]:
            addrFamily = "ipv6"
         safi = "Uni"
         if afiSafiName in [ "IPV4_LABELED_UNICAST", "IPV6_LABELED_UNICAST" ]:
            safi = "UniLab"
         attrPrefix = "rcf" if policyType == "RCF" else "routeMap"
         afVersion = 4 if addrFamily == 'ipv4' else 6
         policyAttr = f"{attrPrefix}V{afVersion}{safi}{inOut.value}"

         policyVal = policyList[ 0 ]
         err = checkForConflictingPolicy( neighborConfig,
                                          f"{attrPrefix}{inOut.value}", policyVal,
                                          addrFamily,
                                          "route-map" if policyType == "ROUTE_MAP"
                                          else "rcf" )
         if err:
            raise AirStreamLib.ToNativeSyncherError( sessionName,
               cls.__name__ + "::toNativeBgpPolicySyncher", err )

         setattr( neighborConfig, policyAttr, policyVal )
         setattr( neighborConfig, f"{policyAttr}Present", True )

      def setDefaultPolicy( key, neighborKeyValue, defaultPolicy, bgpConfig, inOut ):
         assert inOut in PolicyDirection
         afiSafiName = key.afiSafiName
         checkForSupportedAfiSafiNames( key.afiSafiName )
         neighborConfig = getNeighborConfig( bgpConfig, neighborKeyValue )

         eosAfiSafi = afiSafiYANGToEosMapping[ afiSafiName ]
         missingPolicyColl = (
            neighborConfig.missingPolicyActionAfIn if inOut == PolicyDirection.IN
            else neighborConfig.missingPolicyActionAfOut )
         missingPolicyAction = defaultPolicyYANGToEOSMapping[ defaultPolicy ]

         if defaultPolicy is not None and not toggleOCBgpDefaultPolicyPOAEnabled():
            raise AirStreamLib.ToNativeSyncherError( sessionName,
               cls.__name__ + "::toNativeBgpPolicySyncher",
               "default-import-policy and default-export-policy are not supported" )

         missingPolicyColl[ eosAfiSafi ] = missingPolicyAction

      t0( "Removing stale native policy config" )
      bgpConfigs = [ nativeBgpConfig ]
      bgpConfigs += list( nativeBgpVrfConfigDir.vrfConfig.values() )
      routeMapAttrs = [ "routeMapV4UniIn", "routeMapV4UniOut", "routeMapV4UniLabIn",
                        "routeMapV4UniLabOut", "routeMapV6UniIn", "routeMapV6UniOut",
                        "routeMapV6UniLabIn", "routeMapV6UniLabOut" ]
      rcfAttrs = [ "rcfV4UniIn", "rcfV4UniOut", "rcfV4UniLabIn", "rcfV4UniLabOut",
                   "rcfV6UniIn", "rcfV6UniOut", "rcfV6UniLabIn", "rcfV6UniLabOut" ]
      policyAttrs = routeMapAttrs + rcfAttrs
      for bgpConfig in bgpConfigs:
         for neighborConfig in bgpConfig.neighborConfig.values():
            if toggleOCBgpDefaultPolicyPOAEnabled():
               neighborConfig.missingPolicyActionAfIn.clear()
               neighborConfig.missingPolicyActionAfOut.clear()
            for attr in policyAttrs:
               setattr( neighborConfig, f"{attr}Present", False )
               setattr( neighborConfig, attr, getattr( neighborConfig,
                                                       f"{attr}Default" ) )

      t0( f"Synching from {extNeighborApplyPolicyColl} to native BGP config" )
      for key, applyPolicy in extNeighborApplyPolicyColl.items():
         bgpConfig = getBgpConfig( key.vrfName )
         cfg = applyPolicy.config
         setPolicy( key, key.neighborAddress, cfg.importPolicy, cfg.importPolicyType,
                    bgpConfig, PolicyDirection.IN )
         setPolicy( key, key.neighborAddress, cfg.exportPolicy, cfg.exportPolicyType,
                    bgpConfig, PolicyDirection.OUT )
         setDefaultPolicy( key, key.neighborAddress, cfg.defaultImportPolicy,
                           bgpConfig, PolicyDirection.IN )
         setDefaultPolicy( key, key.neighborAddress, cfg.defaultExportPolicy,
                           bgpConfig, PolicyDirection.OUT )

      t0( f"Synching from {extPeerGroupApplyPolicyColl} to native BGP config" )
      for key, applyPolicy in extPeerGroupApplyPolicyColl.items():
         bgpConfig = getBgpConfig( key.vrfName )
         cfg = applyPolicy.config
         setPolicy( key, key.peerGroupName, cfg.importPolicy, cfg.importPolicyType,
                    bgpConfig, PolicyDirection.IN )
         setPolicy( key, key.peerGroupName, cfg.exportPolicy, cfg.exportPolicyType,
                    bgpConfig, PolicyDirection.OUT )
         setDefaultPolicy( key, key.peerGroupName, cfg.defaultImportPolicy,
                           bgpConfig, PolicyDirection.IN )
         setDefaultPolicy( key, key.peerGroupName, cfg.defaultExportPolicy,
                           bgpConfig, PolicyDirection.OUT )

   class ToNativeBgpSyncher( GnmiSetCliSession.PreCommitHandler ):
      externalPathList = [ "routing/bgp/openconfig/neighborAfiSafiPolicyConfigDir",
                           "routing/bgp/openconfig/peerGroupAfiSafiPolicyConfigDir" ]
      nativePathList = [ "routing/bgp/config", "routing/bgp/vrf/config" ]

      @classmethod
      def run( cls, sessionName ):
         toNativeBgpPolicySyncher( cls, sessionName )

   CliSession.registerConfigGroup( entMan, "airstream-cmv",
      "routing/bgp/openconfig/neighborAfiSafiPolicyConfigDir" )
   CliSession.registerConfigGroup( entMan, "airstream-cmv",
      "routing/bgp/openconfig/peerGroupAfiSafiPolicyConfigDir" )

   if toggleOCBgpPolicyDefUniPOAEnabled():
      GnmiSetCliSession.registerPreCommitHandler( ToNativeBgpSyncher )
      AirStreamLib.registerCopyHandler( entMan, "OCBgpPolicyConfig",
         typeName="Routing::Bgp::OCNeighborAfiSafiPolicyConfigDir" )

def registerBgpUcmpConfig( entMan ):
   OcUcmpPreCommitHandler = \
      TacLazyType( "Routing::Bgp::OpenConfig::OcUcmpPreCommitHandler" )
   OcUcmpEnabledError = \
         TacLazyType( "Routing::Bgp::OpenConfig::OcUcmpEnabledError::ErrorType" )

   class ToNativeBgpUcmpSyncher( GnmiSetCliSession.PreCommitHandler ):
      nativePathList = [ "routing/ucmp/bgp/config", "routing/ucmp/bgp/vrf/config" ]
      externalPathList = [ "routing/bgp/openconfig/ucmp" ]

      @classmethod
      def run( cls, sessionName ):
         ucmpConfig = AirStreamLib.getSessionEntity( entMan, sessionName,
            "routing/ucmp/bgp/config" )
         ucmpVrfConfig = AirStreamLib.getSessionEntity( entMan, sessionName,
            "routing/ucmp/bgp/vrf/config" )
         ocUcmpRoot = AirStreamLib.getSessionEntity( entMan, sessionName,
            "routing/bgp/openconfig/ucmp" )

         preCommitHandler = OcUcmpPreCommitHandler( ucmpConfig,
                                                    ucmpVrfConfig,
                                                    ocUcmpRoot )

         result = preCommitHandler.validateEnabledConfig()
         if result.error != OcUcmpEnabledError.ucmpSyncNoError:
            raise AirStreamLib.ToNativeSyncherError( sessionName, cls.__name__,
                                                     result.message() )

         preCommitHandler.syncEnabledConfig()

   CliSession.registerConfigGroup( entMan, "airstream-cmv",
      "routing/bgp/openconfig/ucmp" )
   GnmiSetCliSession.registerPreCommitHandler( ToNativeBgpUcmpSyncher )
   AirStreamLib.registerCopyHandler( entMan, "OcBgpUcmpConfig",
         path="routing/bgp/openconfig/ucmp",
         typeName="Routing::Bgp::OpenConfig::OcUcmpRoot" )

def Plugin( entMan ):
   registerBgpPolicy( entMan )

   if toggleOCBgpUcmpConfigEnabled():
      registerBgpUcmpConfig( entMan )
