# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AirStreamLib
import CliSession
import GnmiSetCliSession
import Tac
import Tracing
from abc import ABC, abstractmethod
from FieldSetOpenConfigLib import portNumRangeToPortRange

th = Tracing.Handle( "ConfigSessionFieldSet" )
t0 = th.trace0
t9 = th.trace9

ipv4 = Tac.Type( "Arnet::AddressFamily" ).ipv4
ipv6 = Tac.Type( "Arnet::AddressFamily" ).ipv6
FieldSetType = Tac.Type( "Classification::FieldSetType" )
cli = Tac.enumValue( 'Classification::ContentsSource', 'cli' )
controller = Tac.enumValue( 'Classification::ContentsSource', 'controller' )
ContentSource = Tac.Type( 'Classification::ContentsSource' )
syncAction = Tac.Type( "Classification::SyncActionType" )
# Entity Paths
extControllerIpv4EntityPath = \
      "trafficPolicies/openconfig/fieldset/controller/config/ipv4"
extControllerIpv6EntityPath = \
      "trafficPolicies/openconfig/fieldset/controller/config/ipv6"
extCliIpv4EntityPath = "trafficPolicies/openconfig/fieldset/cli/config/ipv4"
extCliIpv6EntityPath = "trafficPolicies/openconfig/fieldset/cli/config/ipv6"
extCliPortEntityPath = "trafficPolicies/openconfig/fieldset/cli/config/port"
nativeControllerEntityPath = "trafficPolicies/fieldset/input/controller"
nativeCliEntityPath = "trafficPolicies/fieldset/input/cli"

def Plugin( entMan ):
   CliSession.registerConfigGroup( entMan, "airstream-cmv",
                                   extControllerIpv4EntityPath )
   CliSession.registerConfigGroup( entMan, "airstream-cmv",
                                   extControllerIpv6EntityPath )
   CliSession.registerConfigGroup( entMan, "airstream-cmv", extCliIpv4EntityPath )
   CliSession.registerConfigGroup( entMan, "airstream-cmv", extCliIpv6EntityPath )
   CliSession.registerConfigGroup( entMan, "airstream-cmv", extCliPortEntityPath )

   class ToNativeFieldSetSyncherBase( GnmiSetCliSession.PreCommitHandler, ABC ):

      nativeFsConf = {} # Native Entities
      extFsConf = {} # External Entities

      @classmethod
      @abstractmethod
      def copyFsElementsToSubConf( cls, fsSubConf, fsType, fsName ):
         raise NotImplementedError( "Child class must implement copyPrefixes() " )

      @classmethod
      def getConfig( cls, fsType, sourceDir ):
         config = cls.nativeFsConf[ sourceDir ]
         if fsType == FieldSetType.IPv4Prefix:
            return config.fieldSetIpPrefix
         if fsType == FieldSetType.IPv6Prefix:
            return config.fieldSetIpv6Prefix
         if fsType == FieldSetType.L4Port:
            return config.fieldSetL4Port
         return None

      @classmethod
      def getFsConfig( cls, fsType, fsName, sourceDir ):
         return cls.getConfig( fsType, sourceDir )[ fsName ]

      @classmethod
      def fieldSetExists( cls, fsType, fsName, sourceDir ):
         return fsName in cls.getConfig( fsType, sourceDir )

      @classmethod
      def createFieldSet( cls, fsType, fsName, sourceDir ):
         config = cls.getConfig( fsType, sourceDir )
         if fsType in [ FieldSetType.IPv4Prefix, FieldSetType.IPv6Prefix ]:
            af = ipv4 if fsType == FieldSetType.IPv4Prefix else ipv6
            config.newMember( fsName, af )
         if fsType == FieldSetType.L4Port:
            config.newMember( fsName )

      @classmethod
      def updateFieldSet( cls, fsType, fsName, sourceDir ):
         '''
         By now field set entry is created. Populate all the required fields.
         '''
         fsConf = cls.getFsConfig( fsType, fsName, sourceDir )
         currVersion = fsConf.currCfg.version if fsConf.currCfg else None
         newVersion = Tac.Value( "Ark::UniqueId" )
         fsConf.subConfig.newMember( fsName, newVersion )
         fsSubConf = fsConf.subConfig[ newVersion ]
         fsSubConf.source = sourceDir
         # Sync prefixes from external entity to fsSubConf
         cls.copyFsElementsToSubConf( fsSubConf, fsType, fsName )
         # Update currCfg only when the underlying data has changed. Otherwise, save
         # the platform from unnecessary churn
         if fsConf.currCfg and fsConf.currCfg.isSame( fsSubConf ):
            t0( f"{fsName} field set data remained the same, skipping update" )
            del fsConf.subConfig[ newVersion ]
         else:
            # currCfg pointer must point to the latest subConf.
            fsConf.currCfg = fsSubConf
            # Remove old subConf if present.
            if currVersion and fsConf.subConfig[ currVersion ]:
               del fsConf.subConfig[ currVersion ]

      @classmethod
      def syncFieldSet( cls, fsType, fsName, action, sourceDir ):
         '''
         Initiate sync between external and native field set.
         Check if the field set already exists. If not, create a new field set.
         '''
         t9( "sync field set: " + fsName )
         if action == syncAction.Update:
            if not cls.fieldSetExists( fsType, fsName, sourceDir ):
               cls.createFieldSet( fsType, fsName, sourceDir )
            cls.updateFieldSet( fsType, fsName, sourceDir )
         elif action == syncAction.Delete:
            del cls.getConfig( fsType, sourceDir )[ fsName ]
         else:
            t0( "Unsupported action:", action )
         # Done with the update. Remove fieldSet from the fieldSetToSync set.
         cls.extFsConf[ fsType ].fieldSetSyncComplete( fsName )

   class ToNativeControllerFieldSetSyncher( ToNativeFieldSetSyncherBase ):
      externalPathList = [ extControllerIpv4EntityPath, extControllerIpv6EntityPath ]
      nativePathList = [ nativeControllerEntityPath, nativeCliEntityPath ]

      @classmethod
      def copyFsElementsToSubConf( cls, fsSubConf, fsType, fsName ):
         extFsConf = cls.extFsConf[ fsType ].config[ fsName ]
         if fsType in [ FieldSetType.IPv4Prefix, FieldSetType.IPv6Prefix ]:
            for prefix in extFsConf.prefixes:
               fsSubConf.prefixes.add( prefix )

      @classmethod
      def run( cls, sessionName ):
         def checkConfigValidity( fsType, fsName, action ):
            if action == syncAction.Delete:
               return
            # STATIC type is not supported, raise error in that case.
            configType = extFsConf[ fsType ].config[ fsName ].configType
            if configType != "CONTROLLER":
               raise AirStreamLib.ToNativeSyncherError( sessionName, cls.__name__,
                     f"configType {configType} is not supported; fieldSet:{fsName}" )
            # If this field set exists in cli, verify if source is 'controller'. If
            # not, 'fsName' is an already existing field set.
            if cls.fieldSetExists( fsType, fsName, cli ):
               ipConf = cls.getFsConfig( fsType, fsName, cli )
               if ipConf.currCfg.source != ContentSource.controller:
                  af = ipv4 if fsType == FieldSetType.IPv4Prefix else ipv6
                  raise AirStreamLib.ToNativeSyncherError( sessionName, cls.__name__,
                        f"{af} field set {fsName} already exists in source "
                        f"{ipConf.currCfg.source}" )

         # Fetch External and Native entities to work on
         extFsConf = {
               FieldSetType.IPv4Prefix: AirStreamLib.getSessionEntity( entMan,
                                       sessionName, extControllerIpv4EntityPath ),
               FieldSetType.IPv6Prefix: AirStreamLib.getSessionEntity( entMan,
                                       sessionName, extControllerIpv6EntityPath ) }
         nativeFsConf = {
               controller: AirStreamLib.getSessionEntity( entMan,
                                          sessionName, nativeControllerEntityPath ),
               cli: AirStreamLib.getSessionEntity( entMan, sessionName,
                                                    nativeCliEntityPath ) }
         ToNativeFieldSetSyncherBase.nativeFsConf = nativeFsConf
         ToNativeFieldSetSyncherBase.extFsConf = extFsConf
         for fsType, extConfDir in extFsConf.items():
            # Iterate over the field set names in the fieldSetToSync. These are the
            # field sets that need external to native syncing.
            fieldSetToSync = set( extConfDir.fieldSetToSync.items() )
            for fsName, action in fieldSetToSync:
               checkConfigValidity( fsType, fsName, action )
               cls.syncFieldSet( fsType, fsName, action, controller )

   class ToNativeCliFieldSetSyncher( ToNativeFieldSetSyncherBase ):
      externalPathList = [ extCliIpv4EntityPath,
                           extCliIpv6EntityPath,
                           extCliPortEntityPath ]
      nativePathList = [ nativeCliEntityPath ]

      @classmethod
      def copyFsElementsToSubConf( cls, fsSubConf, fsType, fsName ):
         extFsConf = cls.extFsConf[ fsType ].config[ fsName ]
         if fsType in [ FieldSetType.IPv4Prefix, FieldSetType.IPv6Prefix ]:
            for p in extFsConf.prefix:
               fsSubConf.prefixes.add( p )
         if fsType == FieldSetType.L4Port:
            portRangeList = portNumRangeToPortRange( extFsConf.port )
            for portRange in portRangeList:
               fsSubConf.ports.add( portRange )

      @classmethod
      def run( cls, sessionName ):
         # Fetch External and Native entities to work on
         extFsConf = {
               FieldSetType.IPv4Prefix: AirStreamLib.getSessionEntity( entMan,
                                                sessionName, extCliIpv4EntityPath ),
               FieldSetType.IPv6Prefix: AirStreamLib.getSessionEntity( entMan,
                                                sessionName, extCliIpv6EntityPath ),
               FieldSetType.L4Port: AirStreamLib.getSessionEntity( entMan,
                                                sessionName, extCliPortEntityPath ) }
         nativeFsConf = {
               cli: AirStreamLib.getSessionEntity( entMan, sessionName,
                                                    nativeCliEntityPath ) }
         ToNativeFieldSetSyncherBase.nativeFsConf = nativeFsConf
         ToNativeFieldSetSyncherBase.extFsConf = extFsConf
         for fsType, extConfDir in extFsConf.items():
            # Iterate over the field set names in the fieldSetToSync. These are the
            # field sets that need external to native syncing.
            fieldSetToSync = set( extConfDir.fieldSetToSync.items() )
            for fsName, action in fieldSetToSync:
               cls.syncFieldSet( fsType, fsName, action, cli )

   # MonitorExtFieldSetSm is used to determine the field sets that need
   # to be synced in the ToNativeFieldSetSyncher for session commit.
   class MonitorExtControllerFieldSet( GnmiSetCliSession.GnmiSetCliSessionSm ):
      entityPaths = [ extControllerIpv4EntityPath, extControllerIpv6EntityPath ]

      def __init__( self ):
         self.MonitorExtFieldSetSm = None

      def run( self ):
         extIpv4FsConf = self.sessionEntities[ extControllerIpv4EntityPath ]
         extIpv6FsConf = self.sessionEntities[ extControllerIpv6EntityPath ]
         self.MonitorExtFieldSetSm = Tac.newInstance(
                              "Classification::OcController::MonitorExtFieldSetSm",
                              extIpv4FsConf, extIpv6FsConf )

   # MonitorExtFieldSetSm is used to determine the field sets that need
   # to be synced in the ToNativeFieldSetSyncher for session commit.
   class MonitorExtCliFieldSet( GnmiSetCliSession.GnmiSetCliSessionSm ):
      entityPaths = [ extCliIpv4EntityPath,
                      extCliIpv6EntityPath,
                      extCliPortEntityPath ]

      def __init__( self ):
         self.MonitorExtFieldSetSm = None

      def run( self ):
         extIpv4FsConf = self.sessionEntities[ extCliIpv4EntityPath ]
         extIpv6FsConf = self.sessionEntities[ extCliIpv6EntityPath ]
         extPortFsConf = self.sessionEntities[ extCliPortEntityPath ]
         self.MonitorExtFieldSetSm = Tac.newInstance(
                              "Classification::OcFieldSet::MonitorExtFieldSetSm",
                              extIpv4FsConf, extIpv6FsConf, extPortFsConf )

   GnmiSetCliSession.registerPreCommitSm( MonitorExtControllerFieldSet )
   GnmiSetCliSession.registerPreCommitSm( MonitorExtCliFieldSet )
   GnmiSetCliSession.registerPreCommitHandler( ToNativeControllerFieldSetSyncher )
   GnmiSetCliSession.registerPreCommitHandler( ToNativeCliFieldSetSyncher )

   # Register a special copy handler for native & external entities.
   # Because the 'input/controller' path is populated by an external controller,
   # it does not expect this config to be impacted by operations such as a
   # 'rollback clean-config' or 'config-replace'.  Use the airstream custom copy
   # handler to exclude these entities from non-gnmi config session operations.
   # The 'Classification::FieldSetConfig' entity is used by other cli config
   # as well, so we will register the handler for the specific path.
   AirStreamLib.registerCopyHandler(
      entMan, "OCDynamicExtIpv4FieldSetConfig",
      typeName="Classification::OcController::OcIpv4FieldSetConfigDir" )
   AirStreamLib.registerCopyHandler(
      entMan, "OCDynamicExtIpv6FieldSetConfig",
      typeName="Classification::OcController::OcIpv6FieldSetConfigDir" )
   AirStreamLib.registerCopyHandler(
      entMan, "OCExtIpv4FieldSetConfig",
      typeName="Classification::OcFieldSet::OcIpv4FieldSetConfigDir" )
   AirStreamLib.registerCopyHandler(
      entMan, "OCExtIpv6FieldSetConfig",
      typeName="Classification::OcFieldSet::OcIpv6FieldSetConfigDir" )
   AirStreamLib.registerCopyHandler(
      entMan, "OCExtPortFieldSetConfig",
      typeName="Classification::OcFieldSet::OcPortFieldSetConfigDir" )
   # Although this is a native entity type, it is treated as an external entity for
   # config session commit purpose. So, override the commit flag.
   AirStreamLib.registerCopyHandler(
      entMan, "OCDynamicNativeFieldSetConfig",
      path="trafficPolicies/fieldset/input/controller",
      typeName="Classification::FieldSetConfig",
      commitExternal=True )
