# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import AirStreamLib
import CliSession
import GnmiSetCliSession
import LazyMount
import Tac
import Tracing
from CliPlugin.XcvrConfigCli import getXcvrConfigCliDirHelper

t5 = Tracing.Handle( 'L1OpenConfigAgent' ).trace5
t6 = Tracing.Handle( 'L1OpenConfigAgent' ).trace6

def Plugin( entMan ):
   '''
   Plugin to handle the synchronization between OpenConfig transceiver configuration
   attributes which were modified in a config session and the native attributes which
   implement the configuration.
   '''
   optChanPath = 'hardware/l1/openConfig/xcvrConfig/opticalChannel'
   physChanPath = 'hardware/l1/openConfig/xcvrConfig/physicalChannel'
   ocXcvrPath = 'hardware/l1/openConfig/xcvrConfig/transceiver'
   ocLogicalPath = 'hardware/l1/openConfig/xcvrConfig/logicalChannel'
   ocLogicalIngressPath = 'hardware/l1/openConfig/xcvrConfig/logicalChannelIngress'
   ocLogicalAssignmentPath = \
      'hardware/l1/openConfig/xcvrConfig/logicalChannelAssignment'
   nativeXgcPath = 'hardware/xcvr/xgc'
   nativeXcvrCliPath = 'hardware/xcvr/cli/config/slice'
   allConfigDirPath = 'hardware/xcvr/config/all'
   ocOpticalStatusPath = 'hardware/l1/openConfig/xcvr/opticalChannel'
   ocLogicalStatusPath = 'hardware/l1/openConfig/xcvr/logicalChannel'
   lcIngressStatePath = 'hardware/l1/openConfig/xcvr/logicalChannelIngress'
   lcAssignmentPath = 'hardware/l1/openConfig/xcvr/logicalChannelAssignment'
   lcMapPath = 'hardware/l1/openConfig/xcvr/logicalChannelMap'

   CliSession.registerConfigGroup( entMan, 'airstream-cmv', optChanPath )
   CliSession.registerConfigGroup( entMan, 'airstream-cmv', physChanPath )
   CliSession.registerConfigGroup( entMan, 'airstream-cmv', ocXcvrPath )
   CliSession.registerConfigGroup( entMan, 'airstream-cmv', ocLogicalPath )
   CliSession.registerConfigGroup( entMan, 'airstream-cmv', ocLogicalIngressPath )
   CliSession.registerConfigGroup( entMan, 'airstream-cmv', ocLogicalAssignmentPath )
   t5( 'L1OpenConfig session plugin' )

   # Get the entity which maps interface name to transceiver names
   intfToXcvrName = entMan.root().entity[ allConfigDirPath ].intfToXcvrName

   def getIntfName( xcvrName, intfToXcvrMap ):
      '''
      Multiple xcvrNames may be associated with an interface; eg, Ethernet1/1
      Ethernet1/2, Ethernet1/3 and Ethernet1/4 may all be associated with
      the transceiver Ethernet1.  We really want the association between the
      transceiver name and the lead Ethernet interface.  To do this, we
      create a list of all the interface names associated with a given
      transceiver, sort this list, and return the first element.

      Parameters:
      xcvrName: The transceiver name to map to an interface name
      intfToXcvrMap: The mapping of interface names to transceiver names

      Returns: The first interface of the given transceiver
      '''
      intfNames = [ intfName for intfName, currXcvrName in intfToXcvrMap.items()
                    if currXcvrName == xcvrName ]
      return min( intfNames, default=None )

   def getIntfNameFromEthIndex( lcIndex, lcIngressState ):
      if lcIndex not in lcIngressState.state:
         return None
      else:
         return lcIngressState.state[ lcIndex ].interface

   def getIntfNameFromOtnIndex( lcIndex, lcAssignment, lcIngressState ):
      for lcai, lca in lcAssignment.assignment.items():
         if lcai.lcIndex != lcIndex:
            continue
         if lca.state is None or lca.state.logicalChannel is None:
            continue
         subordinateIndex = lca.state.logicalChannel
         return getIntfNameFromEthIndex( subordinateIndex, lcIngressState )

      return None


   def toNativeTransceiverSyncher( cls, sessionName ):
      '''
      Copy transceiver attributes from the OpenConfig tree to the native attributes
      which implement the desired configuration.

      Parameters:
      cls: The name of the syncher class (child of CliSession.PreCommitHandler )
      sessionName: The name of the config session
      '''
      # The AirStreamLib provides a map of entities indexed by path.  Get both
      # the open config entities and the native entities.
      xcvrConfig = AirStreamLib.getSessionEntity( entMan, sessionName, ocXcvrPath )
      nativeXcvrCliConfig = AirStreamLib.getSessionEntity(
         entMan, sessionName, nativeXcvrCliPath )
      t5( 'Synching from {} to {}'.format( str( xcvrConfig ),
                                           str( nativeXcvrCliConfig ) ) )
      for xcvrName, xcvrValues in xcvrConfig.config.items():
         # Go through each transceiver in the OpenConfig configuration tree
         try:
            intfName = getIntfName( xcvrName, intfToXcvrName )
            if intfName is None:
               continue
            # The xcvr cli config attributes are stored in slice directories. On
            # modular systems, we get the slice name from the transceiver name.
            slicePath = getXcvrConfigCliDirHelper( intfName, nativeXcvrCliConfig )
            if slicePath is None:
               continue
            if intfName not in slicePath.xcvrConfigCli:
               if xcvrValues.enabled is False:
                  t6( f'xcvrConfigCli[{intfName}] does not exist. Create it' )
                  slicePath.newXcvrConfigCli( intfName )
               else:
                  continue
            xcvrConfig = slicePath.xcvrConfigCli[ intfName ]
            if xcvrValues.enabled is None:
               externValue = True
            else:
               externValue = xcvrValues.enabled
            if xcvrConfig.forceModuleRemoved == externValue:
               t6( f'Setting enable for {intfName} to {externValue}' )
               xcvrConfig.forceModuleRemoved = not externValue
               xcvrConfig.forceModuleRemovedChanged += 1
         except Exception as e: # pylint: disable-msg=W0703
            t5( 'Transceiver syncher raised exception' )
            raise AirStreamLib.ToNativeSyncherError( sessionName,
                  cls.__name__ + '::toNativeTransceiverSyncher', str( e ) )

   # Register one-time-sync callback to be called before commit
   class ToNativeTransceiverSyncher( GnmiSetCliSession.PreCommitHandler ):
      externalPathList = [ ocXcvrPath ]
      nativePathList = [ nativeXcvrCliPath ]

      @classmethod
      def run( cls, sessionName ):
         toNativeTransceiverSyncher( cls, sessionName )

   GnmiSetCliSession.registerPreCommitHandler( ToNativeTransceiverSyncher )

   AirStreamLib.registerCopyHandler(
      entMan, 'OcTransceiver',
      typeName='L1OpenConfig::Xcvr::TransceiverConfigBase' )

   def toNativeOpticalChannelSyncher( cls, sessionName ):
      '''
      Copy optical channel attributes from the OpenConfig tree to the native
      attributes which implement the desired configuration.

      Note: This only supports transceivers with a single optical lane.  It's
      unclear to me how setting parameters for multiple optical lanes would
      work with Xgc.

      Parameters:
      cls: The name of the syncher class (child of CliSession.PreCommitHandler )
      sessionName: The name of the config session
      '''
      ocStatus = LazyMount.mount( entMan, ocOpticalStatusPath,
                                  "L1OpenConfig::Xcvr::OpticalChannel", "ri" )

      # The AirStreamLib provides a map of entities indexed by path.  Get both
      # the open config entities and the native entities.
      nativeXcvrCliConfig = AirStreamLib.getSessionEntity(
         entMan, sessionName, nativeXcvrCliPath )
      ocConfig = AirStreamLib.getSessionEntity( entMan, sessionName, optChanPath )
      nativeXgcConfig = AirStreamLib.getSessionEntity(
         entMan, sessionName, nativeXgcPath )

      t5( 'Synching from {} to {}'.format( str( ocConfig ),
                                           str( nativeXgcConfig ) ) )
      for ocName, ocValues in ocConfig.config.items():
         try:
            if ocName not in ocStatus.state:
               t5( f'skipping {ocName}' )
               continue
            xcvrName = ocName.split( '-' )[ 0 ]
            intfName = getIntfName( xcvrName, intfToXcvrName )
            if intfName is None:
               continue

            slicePath = getXcvrConfigCliDirHelper( intfName, nativeXcvrCliConfig )
            if slicePath is not None:
               if intfName not in slicePath.xcvrConfigCli:
                  if ocValues.operationalMode:
                     t6( f'xcvrConfigCli[{intfName}] does not exist. Create it' )
                     slicePath.newXcvrConfigCli( intfName )
               if intfName in slicePath.xcvrConfigCli:
                  xcvrConfig = slicePath.xcvrConfigCli[ intfName ]
                  if ocValues.operationalMode is None:
                     del xcvrConfig.cmisOverrideApplication[ 0 ]
                  else:
                     coa = Tac.Value( 'Xcvr::CmisOverrideApplication' )
                     coa.apSel = ocValues.operationalMode
                     xcvrConfig.cmisOverrideApplication[ 0 ] = coa

            if ocValues.frequency is None:
               del nativeXgcConfig.tuningConfig[ intfName ]
            else:
               if intfName not in nativeXgcConfig.tuningConfig:
                  t6( 'tuningConfig[{}] does not exist. Create it'.format(
                     xcvrName ) )
                  nativeXgcConfig.newTuningConfig( intfName )
               config = nativeXgcConfig.tuningConfig[ intfName ]
               if config.frequency != ocValues.frequency:
                  t6( f'Copying frequency {ocValues.frequency} to {intfName}' )
                  config.frequency = ocValues.frequency
            if ocValues.targetOutputPower is None:
               del nativeXgcConfig.txPowerConfig[ intfName ]
            else:
               if intfName not in nativeXgcConfig.txPowerConfig:
                  t6( f'txPowerConfig[{intfName}] does not exist. Create it' )
                  nativeXgcConfig.newTxPowerConfig( intfName )
               config = nativeXgcConfig.txPowerConfig[ intfName ]
               if config.signalPower != ocValues.targetOutputPower:
                  t6( f'Copying txPower {ocValues.targetOutputPower} to {intfName}' )
                  config.signalPower = ocValues.targetOutputPower
         except Exception as e: # pylint: disable-msg=W0703
            t5( 'Optical channel syncher raised exception' )
            raise AirStreamLib.ToNativeSyncherError( sessionName,
                  cls.__name__ + '::toNativeOpticalChannelSyncher', str( e ) )

      # delete frequency configs which no longer exist in external model
      for intfName in nativeXgcConfig.tuningConfig:
         if intfName not in nativeXgcConfig.tuningConfig:
            continue
         ocName = intfToXcvrName[ intfName ] + '-Optical0'
         if( ocName not in ocConfig.config or
             ocConfig.config[ ocName ].frequency is None ):
            t6( 'Deleting frequency {}'.format( intfName ) )
            del nativeXgcConfig.tuningConfig[ intfName ]

      # delete power configs which no longer exist in external model
      for intfName in nativeXgcConfig.txPowerConfig:
         if intfName not in nativeXgcConfig.tuningConfig:
            continue
         ocName = intfToXcvrName[ intfName ] + '-Optical0'
         if( ocName not in ocConfig.config or
             ocConfig.config[ ocName ].targetOutputPower is None ):
            t6( 'Deleting txPowerConfig {}'.format( intfName ) )
            del nativeXgcConfig.txPowerConfig[ intfName ]

   # Register one-time-sync callback to be called before commit
   class ToNativeOpticalChannelSyncher( GnmiSetCliSession.PreCommitHandler ):
      externalPathList = [ optChanPath ]
      nativePathList = [ nativeXgcPath, nativeXcvrCliPath ]

      @classmethod
      def run( cls, sessionName ):
         toNativeOpticalChannelSyncher( cls, sessionName )

   GnmiSetCliSession.registerPreCommitHandler( ToNativeOpticalChannelSyncher )

   AirStreamLib.registerCopyHandler(
      entMan, 'OpticalChannel',
      typeName='L1OpenConfig::Xcvr::OpticalChannelConfigBase' )

   def toNativePhysicalChannelSyncher( cls, sessionName ):
      '''
      Copy physical-channel attributes from the OpenConfig tree to the native
      attributes which implement the desired configuration.

      Parameters:
      cls: The name of the syncher class (child of CliSession.PreCommitHandler )
      sessionName: The name of the config session
      '''

      # The AirStreamLib provides a map of entities indexed by path.  Get both
      # the OpenConfig entity and the native entity.
      nativeXgcConfig = AirStreamLib.getSessionEntity(
         entMan, sessionName, nativeXgcPath )
      physChanConfigBase = AirStreamLib.getSessionEntity(
         entMan, sessionName, physChanPath )

      t5( 'Synching from {} to {}'.format( str( physChanConfigBase ),
                                           str( nativeXgcConfig ) ) )
      for physChanKey, physChanConfig in physChanConfigBase.config.items():
         if physChanKey.channelIndex == 0:
            try:
               intfName = getIntfName( physChanKey.xcvrName, intfToXcvrName )
               if intfName is None:
                  t5( f'skipping {physChanKey}' )
                  continue

               if physChanConfig.txLaser is False:
                  nativeXgcConfig.txDisabled.add( intfName )
               else:
                  # None is equivalent to True.
                  del nativeXgcConfig.txDisabled[ intfName ]
            except Exception as e: # pylint: disable-msg=W0703
               t5( 'Physical channel syncher raised exception' )
               raise AirStreamLib.ToNativeSyncherError( sessionName,
                     cls.__name__ + '::toNativePhysicalChannelSyncher', str( e ) )
         elif physChanConfig.txLaser is not None:
            raise AirStreamLib.ToNativeSyncherError(
               sessionName,
               cls.__name__ + '::toNativePhysicalChannelSyncher',
               f'Physical-channel index not supported: {physChanKey.channelIndex}' )
         # else the session involves a channel > 0 but not tx-laser, so this handler
         # is not relevant ==> ignore.

   # Register one-time-sync callback to be called before commit
   class ToNativePhysicalChannelSyncher( GnmiSetCliSession.PreCommitHandler ):
      externalPathList = [ physChanPath ]
      nativePathList = [ nativeXgcPath ]

      @classmethod
      def run( cls, sessionName ):
         toNativePhysicalChannelSyncher( cls, sessionName )

   GnmiSetCliSession.registerPreCommitHandler( ToNativePhysicalChannelSyncher )

   AirStreamLib.registerCopyHandler(
      entMan, 'PhysicalChannel',
      typeName='L1OpenConfig::Xcvr::PhysicalChannelConfigBase' )

   def toNativeLogicalChannelSyncher( cls, sessionName ):
      '''
      Copy logical channel attributes from the OpenConfig tree to the native
      attributes which implement the desired configuration.

      Note: This only supports transceivers with a single optical lane.  It's
      unclear to me how setting parameters for multiple optical lanes would
      work with Xgc.

      Parameters:
      cls: The name of the syncher class (child of CliSession.PreCommitHandler )
      sessionName: The name of the config session
      '''
      lcStatus = LazyMount.mount( entMan, ocLogicalStatusPath,
                                  "L1OpenConfig::Xcvr::LogicalChannel", "ri" )
      lcIngressState = LazyMount.mount( entMan, lcIngressStatePath,
                        "L1OpenConfig::Xcvr::LogicalChannelIngressState", "ri" )
      lcAssignment = LazyMount.mount( entMan, lcAssignmentPath,
                        "L1OpenConfig::Xcvr::LogicalChannelAssignment", "ri" )
      lcMap = LazyMount.mount( entMan, lcMapPath,
                               "L1OpenConfig::Xcvr::LogicalChannelMap", "ri" )

      # The AirStreamLib provides a map of entities indexed by path.  Get both
      # the open config entities and the native entities.
      nativeXcvrCliConfig = AirStreamLib.getSessionEntity(
         entMan, sessionName, nativeXcvrCliPath )
      lcConfig = AirStreamLib.getSessionEntity( entMan, sessionName, ocLogicalPath )

      t5( 'Synching from {} to {}'.format( str( lcConfig ),
                                           str( nativeXcvrCliConfig ) ) )

      for lcIndex, lcValues in lcConfig.config.items():
         try:
            if lcIndex not in lcStatus.channel:
               t5( f'skipping Index {lcIndex}' )
               continue

            lcType = Tac.Type( 'L1OpenConfig::Xcvr::LogicalChannelType' )
            otnChannel = lcValues.logicalChannelType == lcType.PROT_OTN
            if otnChannel:
               intfName = getIntfNameFromOtnIndex( lcIndex, lcAssignment,
                                                   lcIngressState )
            else:
               intfName = getIntfNameFromEthIndex( lcIndex, lcIngressState )

            if intfName is None:
               continue
            xcvrName = intfToXcvrName[ intfName ]
            leadIntfName = getIntfName( xcvrName, intfToXcvrName )

            lcMapType = None
            for info in lcMap.lcInfo.values():
               if info.intfName == intfName:
                  lcMapType = info
            if lcMapType is None:
               continue

            slicePath = getXcvrConfigCliDirHelper( leadIntfName,
                                                   nativeXcvrCliConfig )
            if slicePath is None:
               continue
            if leadIntfName not in slicePath.xcvrConfigCli:
               if lcValues.loopbackMode:
                  t6( f'xcvrConfigCli[{leadIntfName}] does not exist. Create it' )
                  slicePath.newXcvrConfigCli( leadIntfName )
            if leadIntfName in slicePath.xcvrConfigCli:
               xcvrConfig = slicePath.xcvrConfigCli[ leadIntfName ]
            else:
               continue

            lpConfig = xcvrConfig.cmisLoopbackConfig
            if not lpConfig:
               xcvrConfig.cmisLoopbackConfig = ( "cmisLoopbackConfig", )
               lpConfig = xcvrConfig.cmisLoopbackConfig

            if otnChannel:
               if lcValues.loopbackMode is None:
                  lpConfig.mediaLoopback.clear()
               else:
                  cmisLbMode = Tac.Type( 'Xcvr::CmisLoopback::Mode' )
                  OcLbMode = Tac.Type( 'L1OpenConfig::Xcvr::OcLoopbackMode' )
                  otnModeConversion = { OcLbMode.TERMINAL: cmisLbMode.Output,
                                          OcLbMode.FACILITY: cmisLbMode.Input }
                  for lane in range( lcMapType.numOpticalLanes ):
                     lpConfig.mediaLoopback[ lane ] = \
                        otnModeConversion[ lcValues.loopbackMode ]
               lpConfig.generationId += 1
            else:
               if lcValues.loopbackMode is None:
                  for lane in range( lcMapType.numHostLanes ):
                     del lpConfig.hostLoopback[ lcMapType.xcvrAndLane.lane + lane ]
               else:
                  cmisLbMode = Tac.Type( 'Xcvr::CmisLoopback::Mode' )
                  OcLbMode = Tac.Type( 'L1OpenConfig::Xcvr::OcLoopbackMode' )
                  lbModeConversion = { OcLbMode.TERMINAL: cmisLbMode.Input,
                                          OcLbMode.FACILITY: cmisLbMode.Output }
                  for lane in range( lcMapType.numHostLanes ):
                     lpConfig.hostLoopback[ lcMapType.xcvrAndLane.lane + lane ] = \
                        lbModeConversion[ lcValues.loopbackMode ]
               lpConfig.generationId += 1

         except Exception as e: # pylint: disable-msg=W0703
            t5( 'Logical channel syncher raised exception' )
            raise AirStreamLib.ToNativeSyncherError( sessionName,
                  cls.__name__ + '::toNativeLogicalChannelSyncher', str( e ) )

   # Register one-time-sync callback to be called before commit
   class ToNativeLogicalChannelSyncher( GnmiSetCliSession.PreCommitHandler ):
      externalPathList = [ ocLogicalPath ]
      nativePathList = [ nativeXcvrCliPath ]

      @classmethod
      def run( cls, sessionName ):
         toNativeLogicalChannelSyncher( cls, sessionName )

   GnmiSetCliSession.registerPreCommitHandler( ToNativeLogicalChannelSyncher )

   AirStreamLib.registerCopyHandler(
      entMan, 'LogicalChannel',
      typeName='L1OpenConfig::Xcvr::LogicalChannelConfig' )
