# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AirStreamLib
import CliSession
import GnmiSetCliSession
import Tac
import Tracing
from Toggles.AirStreamGnmiSetToggleLib import \
       toggleAirstreamExternalEntityCopyBypassEnabled

t0 = Tracing.Handle( "ConfigSessionMacsec" ).trace0
t9 = Tracing.Handle( "ConfigSessionMacsec" ).trace9

macsecCipher = Tac.Value( 'Macsec::OCEOSMacsecCipher' )

def Plugin( entMan ):
   cliConfig = entMan.lookup( "macsec/input/cli" )
   profilePath = cliConfig.fullName + ".profile"

   # we need to add the attribute edit log bypass due to the randomness of
   # "SecretCliLib::Secret" attributes salt and initial vector.
   for attr in [ "key", "defaultKey" ]:
      CliSession.registerAttributeEditLogBypass( entMan, profilePath, attr,
                                                 "Macsec::Cak", "set" )
      CliSession.registerAttributeEditLogBypass( entMan, profilePath, attr,
                                                 "Macsec::Cak", "del" )
   for attr in [ "txStaticSak", "rxStaticSak" ]:
      CliSession.registerAttributeEditLogBypass( entMan, profilePath, attr,
                                                 "Macsec::CliSak", "set" )
      CliSession.registerAttributeEditLogBypass( entMan, profilePath, attr,
                                                 "Macsec::CliSak", "del" )

   CliSession.registerConfigGroup( entMan, "airstream-cmv",
         "macsec/openconfig/policies" )
   CliSession.registerConfigGroup( entMan, "airstream-cmv",
         "macsec/openconfig/interfaces" )

   def toNativeMacsecPolicySyncher( cls, sessionName ):
      externalPolicies = AirStreamLib.getSessionEntity( entMan, sessionName, 
                           "macsec/openconfig/policies" )
      nativeConfig = AirStreamLib.getSessionEntity( entMan, sessionName,
                           "macsec/input/cli" )

      t0( "Synching from " + str( externalPolicies ) + " to " + str( nativeConfig ) )
      for name, value in externalPolicies.policy.items():
         try:
            if name not in nativeConfig.profile:
               t9( "profile " + name + " does not exist. Create it" )
               nativeConfig.newProfile( name )
            t9( "Copying profile " + name )
            nativeProfile = nativeConfig.profile[ name ]
            nativeProfile.keyServerPriority = value.keyServerPriority
            if not value.macsecCipherSuite:
               raise AirStreamLib.ToNativeSyncherError( sessionName,
                     cls.__name__ + '::toNativeMacsecPolicySyncher',
                     "delete on macsec-cipher-suite is not supported " ) 
            if len( value.macsecCipherSuite ) == 1:
               nativeProfile.cipherSuite = \
                  macsecCipher.yangToEOSCipher( value.macsecCipherSuite[0] )
            else:
               raise AirStreamLib.ToNativeSyncherError( sessionName,
                     cls.__name__ + '::toNativeMacsecPolicySyncher',
                     "more than 1 macsec-cipher-suite is not supported " + 
                     str( value.macsecCipherSuite.values() ) )
            nativeProfile.sessionReKeyPeriod = value.sakRekeyInterval
         except Exception as e: # pylint: disable-msg=W0703
            raise AirStreamLib.ToNativeSyncherError( sessionName,
                  cls.__name__ + '::toNativeMacsecPolicySyncher', str( e ) )

      # delete profile which no longer exist in external model
      for profile in nativeConfig.profile:
         if profile not in externalPolicies.policy:
            t9( "Deleting profile " + profile )
            del nativeConfig.profile[ profile ]

   def toNativeMacsecInterfaceSyncher( cls, sessionName ):
      externalInterfaces = AirStreamLib.getSessionEntity( entMan, sessionName,
                             "macsec/openconfig/interfaces" )
      nativeConfig = AirStreamLib.getSessionEntity( entMan, sessionName,
                             "macsec/input/cli" )

      t0( "Synching from " + str( externalInterfaces ) + " to " + 
          str( nativeConfig ) )
      for name, value in externalInterfaces.interface.items():
         try:
            if name not in nativeConfig.intfConfig:
               t9( "Config for interface " + name +
                   " doesn't exist. Create it" )
               nativeConfig.newIntfConfig( name )

            if value.config.enable and value.mka and value.mka.config.mkaPolicy:
               t9( "Copying interface config " + name )
               nativeIntfConfig = nativeConfig.intfConfig[ name ]
               oldProfileName = nativeIntfConfig.profileName
            
               if oldProfileName and oldProfileName in nativeConfig.profile:
                  oldProfile = nativeConfig.profile[ oldProfileName ]
                  # reset old profile parameters. It shouldn't affect any other
                  # interface as currently there is a limitation to apply a unique
                  # profile per interface.
                  oldProfile.secretProfileName = ''
                  oldProfile.replayProtectionWindow = 0
                  del oldProfile.intf[ name ]
               nativeIntfConfig.profileName = value.mka.config.mkaPolicy
            
               if value.mka.config.mkaPolicy in nativeConfig.profile:
                  nativeProfile = nativeConfig.profile[ value.mka.config.mkaPolicy ]
                  nativeProfile.intf[ name ] = True
                  nativeProfile.secretProfileName = value.mka.config.keyChain
                  nativeProfile.replayProtectionWindow = \
                                  value.config.replayProtection

            elif not ( value.config.enable or
                       ( value.mka and value.mka.config.mkaPolicy ) ):
               for profile in nativeConfig.profile.values():
                  if name in profile.intf:
                     del profile.intf[ name ]

               nativeConfig.intfConfig[ name ].profileName = ''
               
               if not toggleAirstreamExternalEntityCopyBypassEnabled():
                  # If 'AirstreamExternalEntityCopyBypass' toggle is not enabled,
                  # we have to explicitly reset the stale interface attributes in
                  # session.externalConf to avoid direct copy of such attributes
                  # from session.externalConf to global.externalConf. This is done to
                  # avoid any config differance between nativeConf and externalConf.
                  externalInterfaces.interface[ name ].config.replayProtection = 0

            elif value.config.enable:
               raise AirStreamLib.ToNativeSyncherError( sessionName,
                     cls.__name__ + '::toNativeMacsecPolicySyncher',
                     "Enabling macsec without profile is not supported" )

            elif value.mka and value.mka.config.mkaPolicy:
               raise AirStreamLib.ToNativeSyncherError( sessionName,
                     cls.__name__ + '::toNativeMacsecPolicySyncher',
                     "Configuring profile without enabling macsec is not supported" )

         except Exception as e: # pylint: disable-msg=W0703
            raise AirStreamLib.ToNativeSyncherError( sessionName,
                  cls.__name__ + '::toNativeMacsecInterfaceSyncher', str( e ) )

      for intf in nativeConfig.intfConfig:
         if intf not in externalInterfaces.interface:
            t9( "Deleting intf " + intf )
            del nativeConfig.intfConfig[ intf ]

   # Register one-time-sync callback to be called before commit
   class ToNativeMacsecSyncher( GnmiSetCliSession.PreCommitHandler ):
      externalPathList = [ 'macsec/openconfig/policies',
                           'macsec/openconfig/interfaces' ]
      nativePathList = [ 'macsec/input/cli' ]
      @classmethod
      def run( cls, sessionName ):
         # Both the syncher needs to be called in this specific order
         # to handle any change on policy path or interface path.
         toNativeMacsecPolicySyncher( cls, sessionName )
         toNativeMacsecInterfaceSyncher( cls, sessionName )

   GnmiSetCliSession.registerPreCommitHandler( ToNativeMacsecSyncher )

   AirStreamLib.registerCopyHandler( entMan, "MacsecPolicy",
                                     typeName="Macsec::OCMacsecPolicyDir" )

   AirStreamLib.registerCopyHandler( entMan, "MacsecInterfaces",
                                     typeName="Macsec::OCMacsecInterfaceDir" )
