# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AirStreamLib
from Arnet.MplsLib import labelStackToMplsLabelOperation as LabelOp
import CliSession
import GnmiSetCliSession
import Tac
import Tracing
from Toggles.MplsToggleLib import toggleStaticMplsSwapOCEnabled

import QuickTrace

qv = QuickTrace.Var
qt0 = QuickTrace.trace0
t0 = Tracing.Handle( "ConfigSessionMplsStatic" ).trace0

DEFAULT_VRF = 'default'

extConfigPath = "routing/mpls/openconfig/staticLsps"

LabelRangeInfo = Tac.Type( 'Mpls::LabelRangeInfo' )
MplsLabel = Tac.Type( 'Arnet::MplsLabel' )
BoundedMplsLabelStack = Tac.Type( 'Arnet::BoundedMplsLabelStack' )
RouteKeyAndMetric = Tac.Type( 'Mpls::RouteKeyAndMetric' )
RouteMetric = Tac.Type( 'Mpls::RouteMetric' )
nativePayloadTypeEnum = Tac.Type( 'Mpls::PayloadType' )
MplsLabelActionEnum = Tac.Type( 'Arnet::MplsLabelAction' )
OpenConfigMplsLabelHelper = Tac.Type(
   'Arnet::OpenConfig::OpenConfigMplsLabelHelper' )
implicitNullLabel = MplsLabel.implicitNull

syncAction = Tac.Type( "Mpls::SyncActionType" )

# const
mplsIntfIdType = Tac.Type( "Mpls::OCMplsStaticLspInterface" )
ViaIndex = Tac.Type( "Mpls::ViaIndex" )

def Plugin( entMan ):
   CliSession.registerConfigGroup( entMan, "airstream-cmv",
         extConfigPath )

   # /network-instances/network-instance/mpls/lsps/static-lsps/static-lsp
   def toNativeMplsStaticLspSyncher( cls, sessionName ):
      externalConfig = AirStreamLib.getSessionEntity( entMan, sessionName,
                           extConfigPath )
      nativeConfig = AirStreamLib.getSessionEntity( entMan, sessionName,
                        "routing/mpls/route/input/cli" )
      nativeMplsConfig = AirStreamLib.getSessionEntity( entMan, sessionName,
                           "routing/mpls/config" )
      cliHelper = Tac.newInstance( "Mpls::CliHelper",
                                   "toNativeMplsStaticLspSyncher", nativeConfig )

      def raiseToNativeSyncherError( msg ):
         raise AirStreamLib.ToNativeSyncherError( sessionName,
               cls.__name__ + '::toNativeMplsStaticLspSyncher', msg )

      def getNativePayloadType( extPayloadType ):
         nativePayloadType = nativePayloadTypeEnum.undefinedPayload
         if extPayloadType == 'PAYLOAD_MPLS':
            nativePayloadType = nativePayloadTypeEnum.mpls
         elif extPayloadType == 'PAYLOAD_IPV4':
            nativePayloadType = nativePayloadTypeEnum.ipv4
         elif extPayloadType == 'PAYLOAD_IPV6':
            nativePayloadType = nativePayloadTypeEnum.ipv6
         return nativePayloadType

      def shouldSkipNativeRoute( nativeRoute ):
         # some Mpls routes are not supported via OpenConfig and therefore not
         # streamed as well. For example: routes with via nh-group or bgp-peer.
         # Return True if such routes are present otherwise False
         
         # Create via info
         vias = nativeRoute.via
         for via in vias:
            nexthop = via.nextHop
            intf = via.intf
            payload = via.payloadType
            action = via.labelAction
            outLabelStk = via.outLabelStack
            # skipEgressAcl is set to True by default for swap label routes.
            skipEgressAcl = via.skipEgressAcl
            validVia = Tac.Value( "Mpls::Via",
                                  nextHop=nexthop,
                                  labelAction=action,
                                  outLabelStack=outLabelStk,
                                  skipEgressAcl=skipEgressAcl,
                                  intf=intf,
                                  payloadType=payload )
            if via != validVia:
               # If both are same means that it only has attributes supported by both
               # CLI as well as OC, otherwise native has some attributes which are
               # not supported by OC.
               return True
         return False

      def getRouteKeyAndMetric( extInLabel, extMetric ):
         labelStack = BoundedMplsLabelStack()
         labelStack.prepend( extInLabel )
         routeKeyAndMetric = RouteKeyAndMetric.fromLabelStack( labelStack,
               extMetric )
         return routeKeyAndMetric

      def checkUnsupportedAttrsInNativeRoute( route ):
         # Check if nativeConfig has unsupported attributes
         shouldSkipRoute = shouldSkipNativeRoute( route )
         if shouldSkipRoute:
            raiseToNativeSyncherError( "Route CLI configurable only,"
               " unsupported attributes present!" )

      def validateInLabelRange( extInLabel, extMetric ):
         t0( 'validateLabelRange' )
         qt0( 'validateLabelRange' )
         staticLabelRange = nativeMplsConfig.labelRange.get( 'static',
               nativeMplsConfig.labelRangeDefault[ 'static' ] )
         base = staticLabelRange.base
         size = staticLabelRange.size
         minLabel = base
         maxLabel = base + size - 1
         t0( 'extInLabel', extInLabel )
         qt0( 'extInLabel', qv( extInLabel ) )
         t0( 'minLabel', minLabel )
         t0( 'maxLabel', maxLabel )
         # inLabel need to be within staticLabelRange. But under certain
         # circumstances CLI allows to configure outside the range label
         # such as during startup config, config replace. OC needs to ignore
         # for these routes and shouldn't throw error.
         routeKeyAndMetric = getRouteKeyAndMetric( extInLabel, extMetric )
         if extInLabel == MplsLabel.null or \
               ( ( extInLabel < minLabel or extInLabel > maxLabel ) and
                     routeKeyAndMetric not in nativeConfig.route ):
            raiseToNativeSyncherError( "Invalid incoming label value!" )

      def validateMetric( extMetric ):
         t0( 'validateMetric' )
         qt0( 'validateMetric' )
         t0( 'metric', extMetric )
         qt0( 'metric', qv( extMetric ) )
         if extMetric < RouteMetric.min or extMetric > RouteMetric.max:
            raiseToNativeSyncherError( "Invalid metric value!" )

      # Validate that incoming-label and metric are available
      # iff next-hops is populated for both egress and transit
      def validateExtLabelAndMetric( extStaticLsp ):
         t0( 'validateExtLabelAndMetric' )
         egressInLabel = None
         egressMetric = None
         transitInLabel = None
         transitMetric = None
         if extStaticLsp.egress:
            egressConfig = extStaticLsp.egress.config

            egressMetric = egressConfig.metric
            validateMetric( egressMetric )

            egressInLabel = OpenConfigMplsLabelHelper.toMplsLabel(
                  egressConfig.incomingLabel )
            validateInLabelRange( egressInLabel, egressMetric )

            if not extStaticLsp.egress.lspNextHops or \
                  not extStaticLsp.egress.lspNextHops.lspNextHop:
               raiseToNativeSyncherError( "Egress next-hops cannot be empty!" )

         if extStaticLsp.transit:
            transitConfig = extStaticLsp.transit.config

            transitMetric = transitConfig.metric
            validateMetric( transitMetric )

            transitInLabel = OpenConfigMplsLabelHelper.toMplsLabel(
                  transitConfig.incomingLabel )
            validateInLabelRange( transitInLabel, transitMetric )

            if not extStaticLsp.transit.lspNextHops or \
                  not extStaticLsp.transit.lspNextHops.lspNextHop:
               raiseToNativeSyncherError( "Transit next-hops cannot be empty!" )

         if egressInLabel and transitInLabel:
            if egressInLabel != transitInLabel:
               raiseToNativeSyncherError( "Incoming labels for egress and "
                     "transit must be same!" )

         if egressMetric and transitMetric:
            if egressMetric != transitMetric:
               raiseToNativeSyncherError( "Metric values for egress and "
                     "transit must be same!" )

      def processExtDeletedNextHops( extStaticLsp ):
         t0( 'processExtDeletedNextHops' )
         rkm = nativeConfig.lspNameToRouteKeyAndMetric[
               extStaticLsp.lspKey.identifier ]
         if extStaticLsp.egress and extStaticLsp.egress.lspNextHops:
            for index, value in nativeConfig.route[ rkm ].indexToPopVia.items():
               if index not in extStaticLsp.egress.lspNextHops.lspNextHop:
                  t0( 'Deleting index', index, 'in nativeConfig indexToPopVia' )
                  qt0( 'Deleting index', qv( index ),
                        'in nativeConfig indexToPopVia' )
                  del nativeConfig.route[ rkm ].via[ value ]
                  del nativeConfig.route[ rkm ].indexToPopVia[ index ]
                  del nativeConfig.route[ rkm ].viaToIndex[ value ]
         if extStaticLsp.transit and extStaticLsp.transit.lspNextHops:
            for index, value in nativeConfig.route[ rkm ].indexToSwapVia.items():
               if index not in extStaticLsp.transit.lspNextHops.lspNextHop:
                  t0( 'Deleting index', index, 'in nativeConfig indexToSwapVia' )
                  qt0( 'Deleting index', qv( index ),
                        'in nativeConfig indexToSwapVia' )
                  del nativeConfig.route[ rkm ].via[ value ]
                  del nativeConfig.route[ rkm ].indexToSwapVia[ index ]
                  del nativeConfig.route[ rkm ].viaToIndex[ value ]

      def checkAndAddNativeRoute( extStaticLsp ):
         t0( 'checkAndAddNativeRoute' )
         qt0( 'checkAndAddNativeRoute' )
         extInLabel = None
         extMetric = None
         if extStaticLsp.egress:
            extInLabel = OpenConfigMplsLabelHelper.toMplsLabel(
                  extStaticLsp.egress.config.incomingLabel )
            extMetric = extStaticLsp.egress.config.metric
         elif extStaticLsp.transit:
            extInLabel = OpenConfigMplsLabelHelper.toMplsLabel(
                  extStaticLsp.transit.config.incomingLabel )
            extMetric = extStaticLsp.transit.config.metric

         qt0( 'extInLabel', qv( extInLabel ) )
         qt0( 'extMetric', qv( extMetric ) )
         routeKeyAndMetric = getRouteKeyAndMetric( extInLabel, extMetric )
         route = None

         # Reject if different lsp-name is provided for an existing
         # RouteKeyAndMetric
         lspName = extStaticLsp.lspKey.identifier
         if routeKeyAndMetric in nativeConfig.route:
            route = nativeConfig.route[ routeKeyAndMetric ]
            if route.lspName != lspName:
               raiseToNativeSyncherError(
                  "RouteKeyAndMetric already present with a different lspName!" )
            else:
               # Check if nativeConfig has unsupported attributes
               checkUnsupportedAttrsInNativeRoute( route )
         else:
            # Before adding this route check if we have same lspName configured
            # Delete old RouteKeyAndMetric if we received a different
            # RouteKeyAndMetric for an existing lsp-name
            if lspName in nativeConfig.lspNameToRouteKeyAndMetric:
               oldRouteKeyAndMetric = \
                     nativeConfig.lspNameToRouteKeyAndMetric[ lspName ]
               # Check if nativeConfig has unsupported attributes
               checkUnsupportedAttrsInNativeRoute(
                     nativeConfig.route[ oldRouteKeyAndMetric ] )
               # if not delete old route.
               if oldRouteKeyAndMetric != routeKeyAndMetric:
                  t0( 'deleting old RouteKeyAndMetric',
                        lspName, oldRouteKeyAndMetric )
                  qt0( 'deleting old RouteKeyAndMetric', qv( lspName ),
                        qv( oldRouteKeyAndMetric ) )
                  cliHelper.delMplsRouteByRouteKey( oldRouteKeyAndMetric )
            # now add new route
            labelStack = BoundedMplsLabelStack()
            labelStack.prepend( extInLabel )
            route = cliHelper.addMplsRoute( labelStack, extMetric, lspName )
            route.lspNameConfigured = True

         return route

      # update with lsp-next-hops/lsp-next-hop[index]/config/access-list-bypass
      def updateSkipEgressAclInNextHopConfig( extNextHop, value ):
         t0( 'updateSkipEgressAclInNextHopConfig' )
         extNhConfig = Tac.Value( "Mpls::OCMplsStaticLspEgressNextHopConfig" )
         extNhConfig.index = extNextHop.config.index
         extNhConfig.ipAddress = extNextHop.config.ipAddress
         extNhConfig.interface = extNextHop.config.interface
         extNhConfig.payloadType = extNextHop.config.payloadType
         extNhConfig.accessListBypass = value
         extNextHop.config = extNhConfig

      def validatePopViaPayload( extStaticLsp, route, newVia, extIndex ):
         t0( 'validatePopViaPayload' )
         qt0( 'validatePopViaPayload' )
         # pylint: disable-msg=pointless-string-statement
         '''
         This function checks all the vias for a the label to see if there are any 
         pop vias configured. If there are any then validates that the payload type 
         is the same for all the pop vias. If not then it raises an error.
         In addition if there is a pop via configured with the same
         nexthop, payload-type as the new incoming via, but a different option for 
         the access-list bypass parameter then that via is deleted. The new via is 
         going to replace the deleted via. 
         eg. 'mpls static top-label x 10.1.1.1 pop payload-type ipv4' then if the 
         user configures 'mpls static top-label x 10.1.1.1 pop payload-type ipv4 
         access-list bypass' then the old via gets replaced with the new via 
         configured.
         '''
         vias = route.via
         if vias is None:
            return
         for via in vias:
            if via.labelAction == MplsLabelActionEnum.pop:
               if newVia.payloadType != via.payloadType:
                  raiseToNativeSyncherError(
                        "Pop vias for the same route cannot have different "
                        "payload type!" )

         viaToCheck = Tac.Value( "Mpls::Via",
                                  nextHop=newVia.nextHop,
                                  labelAction=newVia.labelAction,
                                  outLabelStack=newVia.outLabelStack,
                                  skipEgressAcl=not newVia.skipEgressAcl )
         viaToCheck.payloadType = newVia.payloadType
         viaToCheck.intf = newVia.intf
         nativeIndex = None
         if viaToCheck in route.viaToIndex:
            t0( 'Deleting via from maps' )
            nativeIndex = route.viaToIndex[ viaToCheck ].index
            del route.viaToIndex[ viaToCheck ]
            del route.indexToPopVia[ nativeIndex ]
         if viaToCheck in vias:
            t0( 'Deleting via with index', nativeIndex )
            del vias[ viaToCheck ]
            # Need to delete this via from ext model as well.
            # Otherwise this will be processed as a new
            # via (if index is not the same) in the next iteration
            # and will recursively add back again the deleted via.
            if nativeIndex is None:
               return
            if nativeIndex != extIndex:
               del extStaticLsp.egress.lspNextHops.lspNextHop[ nativeIndex ]
            else:
               # update only the skipEgressAcl flag. This is because if we
               # simply delete it from ext config, then there can be a race b/w
               # nativeToExternal SM and OcToEOS syncher in copying the
               # local copy of ext entity to the global config and we might
               # end up in wrong state for ext entity.
               extNextHop = extStaticLsp.egress.lspNextHops.lspNextHop[
                     nativeIndex ]
               updateSkipEgressAclInNextHopConfig( extNextHop, newVia.skipEgressAcl )

      # Updates the 'lsp-next-hops/lsp-next-hop[index]/config/index' attribute
      # with lsp-next-hops/lsp-next-hop[index]/index ( which is a leaf-ref )
      def updateIndexInNextHopConfig( extNextHop, action, index ):
         t0( 'updateIndexInNextHopConfig' )
         extNhConfig = None
         if action == 'pop':
            extNhConfig = Tac.Value( "Mpls::OCMplsStaticLspEgressNextHopConfig" )
            extNhConfig.payloadType = extNextHop.config.payloadType
            extNhConfig.accessListBypass = extNextHop.config.accessListBypass
         else:
            extNhConfig = Tac.Value( "Mpls::OCMplsStaticLspTransitNextHopConfig" )
            extNhConfig.pushLabel = extNextHop.config.pushLabel
         extNhConfig.index = index
         extNhConfig.ipAddress = extNextHop.config.ipAddress
         extNhConfig.interface = extNextHop.config.interface
         extNextHop.config = extNhConfig

      def processNextHops( extStaticLsp, action, route ):
         t0( 'processNextHops' )
         qt0( 'processNextHops' )
         # pylint: disable=too-many-nested-blocks
         if action == 'pop':
            extNode = extStaticLsp.egress
         else:
            extNode = extStaticLsp.transit

         if extNode:
            t0( 'action', action )
            qt0( 'action', qv( action ) )

            # next-hops
            extNextHop = extNode.lspNextHops.lspNextHop
            t0( 'len', len( extNode.lspNextHops.lspNextHop ) )
            for index in list( extNextHop.keys() ):
               # index can get removed in validatePopViaPayload,
               # dynamically during the loop run. So validate each
               # time in every iteration if the index is present
               if index not in extNextHop:
                  continue

               extConfig = extNextHop[ index ].config

               t0( 'index', index )
               qt0( 'index', qv( index ) )

               # Update the index leafref in the config manually
               updateIndexInNextHopConfig( extNextHop[ index ], action, index )

               # nextHop
               extNhAddr = extConfig.ipAddress
               t0( 'extNhAddr', extNhAddr )
               qt0( 'extNhAddr', qv( extNhAddr ) )
               if extNhAddr.isAddrZero:
                  raiseToNativeSyncherError( "Invalid next hop address!" )

               # payload-type, skipEgressAcl, push-label
               if action == 'swap':
                  extPayloadType = 'PAYLOAD_MPLS'
                  extPushLabel = OpenConfigMplsLabelHelper.toMplsLabel(
                     extConfig.pushLabel )
                  t0( 'extPushLabel', extPushLabel )
                  qt0( 'extPushLabel', qv( extPushLabel ) )
                  minPushLabel = MplsLabel.unassignedMin
                  maxPushLabel = MplsLabel.max
                  t0( 'minPushLabel', minPushLabel, 'maxPushLabel', maxPushLabel )
                  if extPushLabel == MplsLabel.null or \
                     ( extPushLabel < minPushLabel or extPushLabel > maxPushLabel ):
                     raiseToNativeSyncherError( "Invalid push label value!" )
                  skipEgressAcl = True
               else:
                  extPayloadType = extConfig.payloadType
                  extPushLabel = implicitNullLabel
                  if extPayloadType == 'PAYLOAD_MPLS' and extConfig.accessListBypass:
                     raiseToNativeSyncherError(
                           "access-list-bypass cannot be used with mpls payload" )
                  skipEgressAcl = extConfig.accessListBypass

                  t0( 'skipEgressAcl', skipEgressAcl )
                  qt0( 'skipEgressAcl', qv( skipEgressAcl ) )

                  t0( 'extPayloadType', extPayloadType )
                  qt0( 'extPayloadType', qv( extPayloadType ) )
                  if extPayloadType == 'PAYLOAD_NONE':
                     raiseToNativeSyncherError( "Invalid payload type!" )

               # Create via info
               via = Tac.Value( "Mpls::Via",
                                nextHop=extNhAddr,
                                labelAction=action,
                                outLabelStack=LabelOp(
                                   [ extPushLabel ], operation='unknown',
                                   unboundedMplsLabelOperation=True ),
                                skipEgressAcl=skipEgressAcl,
                                payloadType=getNativePayloadType( extPayloadType ) )

               # interface
               if extConfig.interface != mplsIntfIdType.interfaceDefault:
                  t0( 'extIntfName', extConfig.interface )
                  qt0( 'extIntfName', qv( extConfig.interface ) )
                  via.intf = Tac.Value( "Arnet::IntfId", extConfig.interface )

               # weight
               via.weight = 1

               # Validate if pop via
               if action == 'pop' and route:
                  validatePopViaPayload( extStaticLsp, route, via, index )

               # Handle if Via index conflicts with existing Vias
               # Case 1: Same Via different index - Reject
               if via in route.via:
                  t0( 'Via already present in native route!' )
                  qt0( 'Via already present in native route!' )
                  nativeViaIndex = route.viaToIndex.get( via )
                  if nativeViaIndex is not None and nativeViaIndex.index != index:
                     raiseToNativeSyncherError( "Index cannot be "
                           "changed for a next-hop!" )

               # Case 2: Different Via same index - Overwrite
               else:
                  nativeVia = None
                  if action == 'swap' and index in route.indexToSwapVia:
                     nativeVia = route.indexToSwapVia[ index ]
                  elif action == 'pop' and index in route.indexToPopVia:
                     nativeVia = route.indexToPopVia[ index ]
                  if nativeVia is not None:
                     t0( 'Index', index, 'already found in native route!' )
                     qt0( 'Index', qv( index ), 'already found in native route!' )
                     if nativeVia != via:
                        del route.via[ nativeVia ]
                        del route.viaToIndex[ nativeVia ]
                        if action == 'swap':
                           del route.indexToSwapVia[ index ]
                        else:
                           del route.indexToPopVia[ index ]

               # Check if max ECMP size is reached already
               maxEcmpSize = Tac.Type( "Mpls::Constants" ).maxEcmpSize
               if len( route.via ) >= maxEcmpSize:
                  raiseToNativeSyncherError( f"Maximum number of next hops "
                     f"supported across both egress and transit is {maxEcmpSize}" )
               
               # Add via now
               route.viaToIndex[ via ] = ViaIndex( index, True )
               # Update indexToViaMap
               if action == 'swap':
                  t0( 'Updating indexToSwapVia' )
                  route.indexToSwapVia[ index ] = via
               else:
                  t0( 'Updating indexToPopVia' )
                  route.indexToPopVia[ index ] = via
               route.via[ via ] = True

      def handleStaticLsp( extStaticLsp ):
         t0( 'handleStaticLsp' )
         validateExtLabelAndMetric( extStaticLsp )
         route = checkAndAddNativeRoute( extStaticLsp )
         processExtDeletedNextHops( extStaticLsp )
         processNextHops( extStaticLsp, 'pop', route )
         processNextHops( extStaticLsp, 'swap', route )

      def createOrUpdateLspKey( lspKey ):
         t0( 'createOrUpdateLspKey' )
         qt0( 'createOrUpdateLspKey' )
         vrfName = lspKey.vrfName
         if vrfName != DEFAULT_VRF:
            raiseToNativeSyncherError( "Only default vrf is supported!" )

         # Begin handling for staticLsp
         value = externalConfig.staticLsp[ lspKey ]
         if value.transit:
            if not toggleStaticMplsSwapOCEnabled():
               raiseToNativeSyncherError( "Swap label not supported!" )

         handleStaticLsp( value )

      def deleteLspKey( lspKey ):
         t0( 'deleteLspKey' )
         lspName = lspKey.identifier
         if lspName in nativeConfig.lspNameToRouteKeyAndMetric:
            routeKeyAndMetric = \
                  nativeConfig.lspNameToRouteKeyAndMetric[ lspName ]
            # Check for OC unsupported attributes in nativeConfig
            checkUnsupportedAttrsInNativeRoute( 
                  nativeConfig.route[ routeKeyAndMetric ] )
            t0( 'Deleting native lspName, routeKeyAndMetric',
                lspName, routeKeyAndMetric )
            qt0( 'Deleting native lspName, routeKeyAndMetric',
                  qv( lspName ), qv( routeKeyAndMetric ) )

            cliHelper.delMplsRouteByRouteKey( routeKeyAndMetric )
         # delete staticLsp[ lspKey ] in case only egress/transit is deleted
         del externalConfig.staticLsp[ lspKey ]

      # Initiate sync between external and native field set.
      def syncLspKey( lspKey, action ):
         t0( 'syncLspKey' )
         qt0( 'syncLspKey' )
         if action == syncAction.Update:
            # Make the required changes in field set.
            createOrUpdateLspKey( lspKey )
         elif action == syncAction.Delete:
            deleteLspKey( lspKey )
         else:
            t0( 'Unsupported action:', action )
         # Done with the update. Remove lspKey from the fieldSetToSync set.
         externalConfig.fieldSetSyncComplete( lspKey )

      # BEGIN: Synching
      t0( 'Synching from ', str( externalConfig ), ' to ', str( nativeConfig ) )
      qt0( 'Synching from', qv( str( externalConfig ) ), ' to ',
            qv( str( nativeConfig ) ) )
      try:
         fieldSetToSync = set( externalConfig.fieldSetToSync.items() )
         # pylint: disable-msg=pointless-string-statement
         '''
         Process all the affected LSP keys. These LSP keys are updated in
         MplsOpenConfigStaticLspOcToEosSm with their corresponding actions
         either delete or update. deleted keys need to be handled first 
         before processing updated keys.
         This is because consider a test case as below:
         - Add a Mpls static route with LSP name L1 and routeKeyMetric R1
         - Add a Mpls static route with LSP name L2 and routeKeyMetric R2
         - Replace both the  LSP routes with only one route as below:
            - Mpls static route with LSP name L1 and routeKeyMetric R2
         
         This would result in error saying L1 is already associated with R1.
         This is because if L1 is processed first as an update before L2 as delete, 
         we would see that L1 is still associated with R1 in native config ( which
         would be deleted if L2 was processed first ) and rejects the config
         as same LSP name can't be associated to a different RouteKeyMetric.
         To handle this, deletion of all mpls routes should be processed first
         before updates. 
         '''
         for lspKey, action in fieldSetToSync:
            if action == syncAction.Update: # process only deleted keys first
               continue
            t0( 'lspKey', lspKey, 'action', action )
            qt0( 'lspKey', qv( lspKey ), 'action', qv( action ) )
            syncLspKey( lspKey, action ) # Removes the key from the coll as well
         
         fieldSetToSync = set( externalConfig.fieldSetToSync.items() )
         for lspKey, action in fieldSetToSync:
            t0( 'lspKey', lspKey, 'action', action )
            qt0( 'lspKey', qv( lspKey ), 'action', qv( action ) )
            syncLspKey( lspKey, action )

      except Exception as e: # pylint: disable-msg=W0703
         raiseToNativeSyncherError( str( e ) )
      # END: synching

   # Register one-time-sync callback to be called before commit
   class ToNativeMplsStaticLspSyncher( GnmiSetCliSession.PreCommitHandler ):
      externalPathList = [ extConfigPath ]
      nativePathList = [ 'routing/mpls/route/input/cli',
                         'routing/mpls/config' ]

      @classmethod
      def run( cls, sessionName ):
         toNativeMplsStaticLspSyncher( cls, sessionName )

   # MplsOpenConfigStaticLspOcToEosSm is used to determine the field sets that
   # need to be synced in the ToNativeFieldSetSyncher for session commit.
   class MplsOpenConfigStaticLspOcToEos( GnmiSetCliSession.GnmiSetCliSessionSm ):
      entityPaths = [ extConfigPath ]

      def __init__( self ):
         self.MplsOpenConfigStaticLspOcToEosSm = None

      def run( self ):
         extMplsStaticConf = self.sessionEntities[ extConfigPath ]
         self.MplsOpenConfigStaticLspOcToEosSm = Tac.newInstance(
            "Mpls::MplsOpenConfigStaticLspOcToEosSm", extMplsStaticConf )

   GnmiSetCliSession.registerPreCommitSm( MplsOpenConfigStaticLspOcToEos )
   GnmiSetCliSession.registerPreCommitHandler( ToNativeMplsStaticLspSyncher )

   AirStreamLib.registerCopyHandler( entMan, "MplsOpenConfigStaticLsp",
         typeName="Mpls::OCMplsStaticLspDir" )
