# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AirStreamLib
import CliSession
import GnmiSetCliSession
import Tac
import Tracing
from CliPlugin import IraVrfCli
import Arnet
import ReversibleSecretCli
import QuickTrace

qv = QuickTrace.Var
qt0 = QuickTrace.trace0
t0 = Tracing.Handle( "ConfigSessionOspf" ).trace0

intfTypeEnum = Tac.Type( "Routing::Ospf::IntfType" )
OCOspfAuthTypeTypeEnum = Tac.Type( "Routing::Ospf::OspfAuthType" )
OCOspfAuthEncryptionType = Tac.Type( "Routing::Ospf::OCOspfAuthEncryption" )
OCOspfAuthPasswordType = Tac.Type( "Routing::Ospf::OCOspfAuthPassword" )
nativeMdAuthEncryptionEnum = Tac.Type( "Routing::Ospf::mdAlgorithm" )
nativeOspfIntfBfdStateEnum = Tac.Type( "Routing::Ospf::OspfIntfBfdState" )
adjLoggingTypeEnum = Tac.Type( "Routing::Ospf::AdjacencyLoggingType" )

DEFAULT_VRF = 'default'

def Plugin( entMan ):
   CliSession.registerConfigGroup( entMan, "airstream-cmv",
         "routing/ospf/openconfig/instances" )

   # /network-instances/network-instance/protocols/protocol/ospfv2/global
   def toNativeOspfSyncher( cls, sessionName ):
      externalConfig = AirStreamLib.getSessionEntity( entMan, sessionName,
                           "routing/ospf/openconfig/instances" )
      nativeConfig = AirStreamLib.getSessionEntity( entMan, sessionName,
                           "routing/ospf/config" )
      vrfConfigDir = AirStreamLib.getSessionEntity( entMan, sessionName,
                           "ip/vrf/config" )
      l3IntfConfigDir = AirStreamLib.getSessionEntity( entMan, sessionName,
                           "l3/intf/config" )

      def getIntfVrf( intfName ):
         intfConfig = l3IntfConfigDir.intfConfig.get( intfName, None )
         if intfConfig:
            t0( "IntfConfig", intfName, "exists in l3IntfConfigDir" )
            return intfConfig.vrf
         else:
            qt0( "IntfConfig", qv( intfName ), "doesn't exist in l3IntfConfigDir" )
         return DEFAULT_VRF

      def checkIfVrfExists( vrfName ):
         if vrfName == DEFAULT_VRF:
            return True
         if vrfName in vrfConfigDir.vrf:
            return True
         t0( "Vrf", vrfName, "doesn't exist in vrfConfigDir" )
         qt0( "Vrf", qv( vrfName ), "doesn't exist in vrfConfigDir" )
         return False

      def vrfNameToInstance( instanceConfig, vrf=DEFAULT_VRF ):
         _x = [ i for i in instanceConfig.values() if i.vrfName == vrf ]
         return ( [ i.instance for i in _x ] if _x else [] )

      def nInstancesInVrf( instanceConfig, vrf ):
         return len( [ i for i in instanceConfig.values() if i.vrfName == vrf ] )

      def getNativeMdEncryption( extEncryption ):
         nativeMdEncryption = nativeMdAuthEncryptionEnum.mdnone
         if extEncryption == 'MD5':
            nativeMdEncryption = nativeMdAuthEncryptionEnum.md5
         elif extEncryption == 'SHA1':
            nativeMdEncryption = nativeMdAuthEncryptionEnum.sha1
         elif extEncryption == 'SHA256':
            nativeMdEncryption = nativeMdAuthEncryptionEnum.sha256
         elif extEncryption == 'SHA384':
            nativeMdEncryption = nativeMdAuthEncryptionEnum.sha384
         elif extEncryption == 'SHA512':
            nativeMdEncryption = nativeMdAuthEncryptionEnum.sha512
         return nativeMdEncryption

      def updateExtIntfConfigId( extIntf, intfName ):
         extIntfConfig = extIntf[ intfName ].config
         intfConfig = Tac.Value( "Routing::Ospf::OCOspfIntfConfig" )
         intfConfig.id = intfName
         intfConfig.networkType = extIntfConfig.networkType
         intfConfig.priority = extIntfConfig.priority
         intfConfig.metric = extIntfConfig.metric
         intfConfig.passive = extIntfConfig.passive
         extIntf[ intfName ].config = intfConfig

      def raiseToNativeSyncherError( msg ):
         raise AirStreamLib.ToNativeSyncherError( sessionName,
               cls.__name__ + '::toNativeOspfSyncher', msg )

      def handleIntfConfig( extIntfConfig, nativeIntfConfig, instanceId ):
         # /network-instances/network-instance/protocols/protocol/ospfv2/
         # areas/area/interfaces/interface/config
         t0( "networkType", extIntfConfig.networkType )
         if extIntfConfig.networkType == 'POINT_TO_POINT_NETWORK':
            nativeIntfConfig.intfType = intfTypeEnum.intfTypePointToPoint
         elif extIntfConfig.networkType == 'BROADCAST_NETWORK':
            nativeIntfConfig.intfType = intfTypeEnum.intfTypeBroadcast
         elif extIntfConfig.networkType == 'NON_BROADCAST_NETWORK':
            nativeIntfConfig.intfType = intfTypeEnum.intfTypeNonBroadcast

         t0( "priority", extIntfConfig.priority )
         nativeIntfConfig.priority = extIntfConfig.priority

         t0( "metric", extIntfConfig.metric )
         nativeIntfConfig.cost = extIntfConfig.metric

         t0( "passive", extIntfConfig.passive )
         passiveIntfCfg = nativeConfig.passiveIntfConfig.\
               newMember( nativeIntfConfig.name )
         if extIntfConfig.passive:
            passiveIntfCfg.passiveInstances[ instanceId ] = 'ospfIntfPassive'
         else:
            passiveIntfCfg.passiveInstances[ instanceId ] = 'ospfIntfActive'

      def handleIntfAuth( extIntfAuth, nativeIntfConfig ):
         extIntfAuthConfig = extIntfAuth.config
         # /network-instances/network-instance/protocols/protocol/ospfv2/
         # areas/area/interfaces/interface/authentication
         t0( "authMode", extIntfAuthConfig.authMode )
         if extIntfAuthConfig.authMode == "AUTH_NONE":
            nativeIntfConfig.authType = OCOspfAuthTypeTypeEnum.ospfAuthTypeNone
         elif extIntfAuthConfig.authMode == "AUTH_SIMPLE":
            nativeIntfConfig.authType = OCOspfAuthTypeTypeEnum.ospfAuthTypeSimple
         elif extIntfAuthConfig.authMode == "AUTH_MESSAGE_DIGEST":
            nativeIntfConfig.authType = OCOspfAuthTypeTypeEnum.ospfAuthTypeMd
         # External simple auth is stored encrypted in native model
         simplePassword = ReversibleSecretCli.generateSecretEntity(
               extIntfAuthConfig.simplePassword )
         nativeIntfConfig.simpleAuth = simplePassword
         t0( "handleIntfAuth nativeIntfConfig", nativeIntfConfig.name )
         t0( "handleIntfAuth simpleAuth", nativeIntfConfig.simpleAuth
               .getClearText() )
         # Del all md auth first
         nativeIntfConfig.mdAuth.clear()
         for mdKeyId in extIntfAuthConfig.messageDigest:
            t0( "handleIntfAuth mdKeyId", mdKeyId )
            extMd = extIntfAuthConfig.messageDigest[ mdKeyId ]
            # Ensure that both mdEncryption and mdPassword are populated
            if extMd.mdEncryption == OCOspfAuthEncryptionType.MDNONE or\
               extMd.mdPassword == OCOspfAuthPasswordType.authPasswordDefault:
               raiseToNativeSyncherError( "Both md-encryption and md-password "
                     "need to set for message-digest key" )
            # External md auth is stored encrypted in native model
            mdPassword = ReversibleSecretCli.generateSecretEntity(
                  extMd.mdPassword )
            nativeIntfConfig.mdAuth[ mdKeyId ] = Tac.Value( "Routing::Ospf::"
                  "OspfMdAuth", mdPassword,
                  getNativeMdEncryption( extMd.mdEncryption ) )

      def handleIntfTimers( extIntfTimers, nativeIntfConfig ):
         extIntfTimersConfig = extIntfTimers.config
         # /network-instances/network-instance/protocols/protocol/ospfv2/
         # areas/area/interfaces/interface/timers
         nativeIntfConfig.routerDeadInterval = extIntfTimersConfig.deadInterval
         nativeIntfConfig.helloInterval = extIntfTimersConfig.helloInterval
         nativeIntfConfig.rxInt = extIntfTimersConfig.retransmissionInterval

      def handleIntfBfd( extIntfBfd, nativeIntfConfig ):
         extIntfBfdConfig = extIntfBfd.config
         # /network-instances/network-instance/protocols/protocol/ospfv2/
         # areas/area/interfaces/interface/enable-bfd
         t0( "Ext Bfd enabled", extIntfBfdConfig.enabled )
         if extIntfBfdConfig.enabled is True:
            nativeIntfConfig.bfdIntfState = \
                  nativeOspfIntfBfdStateEnum.ospfIntfBfdEnabled
         else:
            # In EOS, Bfd states are enums with an additional state ospfIntfBfdNone
            # This state is applicable when "bfd default" command is run from
            # router ospf mode. This is not applicable from Openconfig though
            nativeIntfConfig.bfdIntfState = \
                  nativeOspfIntfBfdStateEnum.ospfIntfBfdDisabled

      def handleExtAreas( extAreas, nativeInstConfig ):
         if extAreas:
            # areas/area
            extArea = extAreas.area
            for areaIndex, areaValue in extArea.items():
               # area config identifier is expected to be updated by
               # pre-commit handler
               areaConfig = Tac.Value( "Routing::Ospf::OCOspfAreaConfig" )
               areaConfig.identifier = areaIndex
               extArea[ areaIndex ].config = areaConfig
               areaId = None
               try:
                  areaId = areaIndex.areaIdInt
                  t0( "areaId received as int", areaId )
               except NotImplementedError:
                  areaId = areaIndex.areaIdIpAddr
                  t0( "areaId received as str", areaId )
               areaId = Arnet.IpAddress( areaId )
               areaId = areaId.stringValue
               if areaId not in nativeInstConfig.areaConfig:
                  t0( "area", areaId, "does not exist. Create it" )
                  nativeInstConfig.newAreaConfig( areaId )

               # /network-instances/network-instance/protocols/protocol/ospfv2/
               # areas/area/interfaces/interface
               if areaValue.interfaces:
                  extIntf = areaValue.interfaces.interface
                  for intfName, intfValue in extIntf.items():
                     # Management intf not supported
                     if intfName.startswith( "Management" ):
                        raiseToNativeSyncherError( "Management interface is "
                              "not supported!" )

                     # interface id is expected to be updated by pre-commit handler
                     updateExtIntfConfigId( extIntf, intfName )

                     if intfName not in nativeConfig.intfConfig:
                        t0( "interface", intfName, "does not exist. Create it" )
                        nativeConfig.newIntfConfig( intfName )

                     # Verify that interface really belongs to the VRF corresponding
                     # to the received request
                     if getIntfVrf( intfName ) != vrfName:
                        raiseToNativeSyncherError( f"intf {intfName} "
                              f"does not belong to VRF {vrfName}" )

                     # Verify interface is associated with only one instance
                     if intfName in intfToInstMap and \
                           intfToInstMap[ intfName ] != instanceId:
                        raiseToNativeSyncherError( f"intf {intfName} "
                              "is already associated with another instance" )

                     t0( "Adding", intfName, "to intfToInstMap" )
                     qt0( "Adding", qv( intfName ), "to intfToInstMap" )
                     intfToInstMap[ intfName ] = instanceId

                     # Verify interface is associated with only one area within an
                     # instance
                     if intfName in intfToAreaMap and \
                           intfToAreaMap[ intfName ] != areaId:
                        raiseToNativeSyncherError( f"intf {intfName} "
                              "is already associated with another area" )

                     t0( "Adding", intfName, "to intfToAreaMap" )
                     qt0( "Adding", qv( intfName ), "to intfToAreaMap" )
                     intfToAreaMap[ intfName ] = areaId
                     nativeIntfConfig = nativeConfig.intfConfig[ intfName ]
                     nativeIntfConfig.areaId = areaId
                     nativeIntfConfig.areaIdPresent = True

                     # Initialise if not already set, so that default values
                     # for the interface
                     if not intfValue.timers:
                        intfValue.timers = ()
                     if not intfValue.authentication:
                        intfValue.authentication = ()
                     if not intfValue.authentication.config:
                        intfValue.authentication.config = ()
                     if not intfValue.enableBfd:
                        intfValue.enableBfd = ()

                     # interface/config
                     extIntfConfig = intfValue.config
                     handleIntfConfig( extIntfConfig, nativeIntfConfig, instanceId )

                     # interface/timers
                     extIntfTimers = intfValue.timers
                     handleIntfTimers( extIntfTimers, nativeIntfConfig )

                     # interface/authentication
                     extIntfAuth = intfValue.authentication
                     handleIntfAuth( extIntfAuth, nativeIntfConfig )

                     # interface/enable-bfd
                     extIntfBfd = intfValue.enableBfd
                     handleIntfBfd( extIntfBfd, nativeIntfConfig )

         # Delete all areas in native config which no longer exist in external
         # model
         for area in nativeInstConfig.areaConfig:
            areaIp = Arnet.IpAddress( area )
            areaExtStr = Tac.Value( 'Routing::Ospf::OCOspfAreaId',
                  areaIdIpAddr=areaIp.stringValue )
            areaExtInt = Tac.Value( 'Routing::Ospf::OCOspfAreaId',
                  areaIdInt=areaIp.value )
            if extAreas:
               if areaExtStr not in extArea and areaExtInt not in extArea:
                  t0( "Deleting area", area, "in native config" )
                  qt0( "Deleting area", qv( area ), "in native config" )
                  del nativeInstConfig.areaConfig[ area ]
            else:
               t0( "Deleting area", area, "in native config" )
               qt0( "Deleting area", qv( area ), "in native config" )
               del nativeInstConfig.areaConfig[ area ]

      def handleGlobalConfig( extGlobalConfig, nativeInstConfig ):
         t0( "routerId", extGlobalConfig.routerId )
         t0( "logAdjacencyChanges", extGlobalConfig.logAdjacencyChanges )
         nativeInstConfig.routerId = extGlobalConfig.routerId
         if extGlobalConfig.logAdjacencyChanges:
            nativeInstConfig.adjacencyLogging = \
                  adjLoggingTypeEnum.adjacencyLoggingTypeNormal
         else:
            nativeInstConfig.adjacencyLogging = \
                  adjLoggingTypeEnum.adjacencyLoggingTypeNone

      def handleGlobalSpfTimers( extGlobalSpfTimers, nativeInstConfig ):
         # global/timers/spf/config
         extGlobalSpfTimersConfig = extGlobalSpfTimers.config
         spfStartInt = extGlobalSpfTimersConfig.initialDelay
         spfHoldInt = nativeInstConfig.spfHoldIntDefault
         spfMaxWaitInt = extGlobalSpfTimersConfig.maximumDelay
         # spfMaxWaitInt must be >= spfStartInt
         # adjust configured values if they are invalid
         # Also, openconfig user cannot configure spfHoldInt.
         # So it is being set to spfMaxWaitInt
         spfMaxWaitInt = max( spfMaxWaitInt, spfStartInt )
         spfHoldInt = spfMaxWaitInt
         t0( "Throttle timers setting to: ", spfStartInt, " ", spfHoldInt,
               " ", spfMaxWaitInt )
         nativeInstConfig.spfThrottleTimerConfig = \
               Tac.Value( "Routing::Ospf::SpfThrottleTimer",
                                         spfStartInt, spfHoldInt, spfMaxWaitInt )

      def handleGlobalMaxMetricTimers( extGlobalMaxMetricTimers, nativeInstConfig ):
         # global/timers/max-metric/config
         extGlobalMaxMetricTimersConfig = extGlobalMaxMetricTimers.config
         t0( "Setting maxMetric set attribute" )
         maxMetricFlag = extGlobalMaxMetricTimersConfig.maxMetricSet
         includeStubFlag = nativeInstConfig.maxMetricStubFlagDefault
         onStartTime = extGlobalMaxMetricTimersConfig.timeout
         onStartupDefaultTime = Tac.Value( "Routing::Ospf::"
               "OCOspfGlobalMaxMetricOnStartupTimeout" ).onStartupDefault
         onStartupTrigger = False

         for extIncludeType in extGlobalMaxMetricTimersConfig.include.values():
            t0( "extIncludeType ", extIncludeType )
            if extIncludeType == 'MAX_METRIC_INCLUDE_STUB':
               includeStubFlag = True
            else:
               raiseToNativeSyncherError(
                     f"Include type: {extIncludeType} not supported" )

         for extTriggerType in extGlobalMaxMetricTimersConfig.trigger.values():
            t0( "extTriggerType ", extTriggerType )
            if extTriggerType == 'MAX_METRIC_ON_SYSTEM_BOOT':
               onStartupTrigger = True
            else:
               raiseToNativeSyncherError(
                     f"Trigger type: {extTriggerType} not supported" )

         # Ensure that max metric flag is prerequisite for setting any
         # other attribute. Raise error otherwise.
         if not maxMetricFlag:
            if includeStubFlag or onStartupTrigger or \
                  extGlobalMaxMetricTimersConfig.timeout != onStartupDefaultTime:
               raiseToNativeSyncherError( "Max metric not set!" )
         else:
            # Verify for timeout and trigger that either both are set
            # or none are set. Because, both are required if max metric
            # on startup needs to be triggered.
            if onStartupTrigger ^ ( extGlobalMaxMetricTimersConfig.timeout !=
                  onStartupDefaultTime ):
               raiseToNativeSyncherError( "Trigger and timeout both "
                     "need to be set to trigger max metric on startup!" )

         # Below are not really configurable through openconfig,
         # setting them to eos default
         extLsaMetric = nativeInstConfig.maxMetricExtMetricDefault
         onStartWaitBgpFlag = nativeInstConfig.maxMetricWaitBgpFlagDefault
         sumLsaMetric = nativeInstConfig.maxMetricSumMetricDefault

         t0( "Max metric external", extLsaMetric, "includeStubFlag",
                     includeStubFlag, "onStartTime", onStartTime )
         qt0( "Max metric external", qv( extLsaMetric ), "includeStubFlag",
               qv( includeStubFlag ), "onStartTime", qv( onStartTime ) )
         nativeInstConfig.maxMetricConfig = \
               Tac.Value( 'Routing::Ospf::MaxMetric', maxMetricFlag,
                           extLsaMetric, includeStubFlag,
                           onStartTime, onStartWaitBgpFlag, sumLsaMetric )

      def handleGlobalLsaGenTimers( extGlobalLsaGenTimers, nativeInstConfig ):
         # global/timers/lsa-generation/config
         extGlobalLsaGenTimersConfig = extGlobalLsaGenTimers.config
         lsaStartInt = extGlobalLsaGenTimersConfig.initialDelay
         lsaHoldInt = nativeInstConfig.lsaHoldIntDefault
         lsaMaxWaitInt = extGlobalLsaGenTimersConfig.maximumDelay
         # lsaMaxWaitInt must be >= lsaHoldInt
         # adjust configured values if they are invalid
         # Also, openconfig user cannot configure lsaHoldInt.
         # So it is being set to lsaMaxWaitInt.
         lsaMaxWaitInt = max( lsaMaxWaitInt, lsaStartInt )
         lsaHoldInt = lsaMaxWaitInt
         t0( "LSA generation timers setting to: ", lsaStartInt, " ",
               lsaHoldInt, " ", lsaMaxWaitInt )
         nativeInstConfig.lsaThrottleTimerConfig = \
               Tac.Value( "Routing::Ospf::LsaThrottleTimer",
                           lsaStartInt, lsaHoldInt, lsaMaxWaitInt )

      def handleGlobalTimers( extGlobalTimers, nativeInstConfig ):
         handleGlobalSpfTimers( extGlobalTimers.spf, nativeInstConfig )
         handleGlobalMaxMetricTimers( extGlobalTimers.maxMetric,
               nativeInstConfig )
         handleGlobalLsaGenTimers( extGlobalTimers.lsaGeneration,
               nativeInstConfig )

      def handleGlobalGR( extGlobalGR, nativeInstConfig ):
         # global/graceful-restart/config
         extGlobalGRConfig = extGlobalGR.config
         grEnabled = extGlobalGRConfig.enabled
         grHelper = extGlobalGRConfig.helperOnly
         t0( 'grEnabled', grEnabled )
         t0( 'grHelper', grHelper )
         nativeInstConfig.gracefulRestart = grEnabled
         nativeInstConfig.grHelper = grHelper

      def handleExtGlobal( extGlobal, nativeInstConfig ):
         handleGlobalConfig( extGlobal.config, nativeInstConfig )
         handleGlobalTimers( extGlobal.timers, nativeInstConfig )
         handleGlobalGR( extGlobal.gracefulRestart, nativeInstConfig )

      def handleOspfv2( extOspfv2, nativeInstConfig ):
         handleExtGlobal( extOspfv2.instGlobal, nativeInstConfig )
         handleExtAreas( extOspfv2.areas, nativeInstConfig )

      t0( "Synching from ", str( externalConfig ), " to ", str( nativeConfig ) )

      intfToInstMap = {}
      intfToAreaMap = {}
      for key, value in externalConfig.instance.items():
         try:
            instanceId = int( key.instanceId ) # instance is U32 in native tac model
            vrfName = key.vrfName
            if checkIfVrfExists( vrfName ) is False:
               raiseToNativeSyncherError( f"Vrf {vrfName} not found" )

            t0( "instanceId ", instanceId, ", vrfName ", vrfName )
            if instanceId not in nativeConfig.instanceConfig:
               t0( "instance ", str( instanceId ), " does not exist. Create it" )
               # Ensure that multi-instance is not allowed. This is because
               # in openconfig, we can configure only area interfaces
               if nInstancesInVrf( nativeConfig.instanceConfig, vrfName ) == 1:
                  raiseToNativeSyncherError(
                        "Multi-instance is not supported from openconfig" )

               nativeConfig.newInstanceConfig( instanceId, vrfName )
               IraVrfCli.addAgentVrfEntry( vrfName, "Ospf" )

            nativeInstConfig = nativeConfig.instanceConfig[ instanceId ]

            if nativeInstConfig.vrfName != vrfName:
               raiseToNativeSyncherError(
                     f"OSPF instance {instanceId} already exists in VRF "
                     f"{nativeInstConfig.vrfName}" )

            # Update external instance attributes
            extConfig = Tac.Value( "Routing::Ospf::OCOspfInstanceConfig" )
            extConfig.identifier = 'OSPF'
            extConfig.name = key.instanceId
            extConfig.enabled = value.config.enabled
            value.config = extConfig
            # Enable/disable native instance
            nativeInstConfig.enable = value.config.enabled

            # Iniialise entities if not already done, so that default values
            # are set for the instance.
            if not value.ospfv2:
               value.ospfv2 = ()
            if not value.ospfv2.instGlobal:
               value.ospfv2.instGlobal = ()
            if not value.ospfv2.instGlobal.config:
               value.ospfv2.instGlobal.config = ()
            if not value.ospfv2.instGlobal.timers:
               value.ospfv2.instGlobal.timers = ()
            if not value.ospfv2.instGlobal.timers.spf:
               value.ospfv2.instGlobal.timers.spf = ()
            if not value.ospfv2.instGlobal.timers.maxMetric:
               value.ospfv2.instGlobal.timers.maxMetric = ()
            if not value.ospfv2.instGlobal.timers.lsaGeneration:
               value.ospfv2.instGlobal.timers.lsaGeneration = ()
            if not value.ospfv2.instGlobal.gracefulRestart:
               value.ospfv2.instGlobal.gracefulRestart = ()

            # Begin handling ospfv2 node from here
            extOspfv2 = value.ospfv2
            handleOspfv2( extOspfv2, nativeInstConfig )

         except Exception as e: # pylint: disable-msg=W0703
            raiseToNativeSyncherError( str( e ) )

      # Delete intfConfig which no longer exist in external model
      for nativeIntfName in nativeConfig.intfConfig:
         if nativeConfig.intfConfig[ nativeIntfName ].areaIdPresent is False:
            continue
         vrfName = getIntfVrf( nativeIntfName )
         t0( "Intf vrf", vrfName )
         instanceIdList = vrfNameToInstance( nativeConfig.instanceConfig,
               vrfName )
         if len( instanceIdList ) == 0:
            continue
         instanceId = instanceIdList[ 0 ]
         instKey = Tac.Value( "Routing::Ospf::OCOspfInstanceKey",
                              vrfName, "OSPF", str( instanceId ) )
         if instKey in externalConfig.instance:
            t0( "Intf instance", instanceId )
            extAreaId = None
            areaIp = nativeConfig.intfConfig[ nativeIntfName ].areaId
            areaIp = Arnet.IpAddress( areaIp )
            areaExtStr = Tac.Value( 'Routing::Ospf::OCOspfAreaId',
                  areaIdIpAddr=areaIp.stringValue )
            areaExtInt = Tac.Value( 'Routing::Ospf::OCOspfAreaId',
                  areaIdInt=areaIp.value )
            extOspfv2 = externalConfig.instance[ instKey ].ospfv2
            if extOspfv2.areas:
               extAreas = extOspfv2.areas
               if areaExtStr in extAreas.area:
                  extAreaId = areaExtStr
               if areaExtInt in extAreas.area:
                  extAreaId = areaExtInt
               t0( "Intf extAreaId", extAreaId )
               if extAreaId and extAreas.area[ extAreaId ].interfaces:
                  extIntfs = extAreas.area[ extAreaId ].interfaces
                  t0( "Check if Intf exists in external", nativeIntfName )
                  if nativeIntfName in extIntfs.interface:
                     t0( "Intf exists in external", nativeIntfName )
                     continue
         t0( "Deleting interface", nativeIntfName, "in native config" )
         qt0( "Deleting interface", qv( nativeIntfName ), "in native config" )
         del nativeConfig.intfConfig[ nativeIntfName ]

      # delete instances which no longer exist in external model
      for instanceId, value in nativeConfig.instanceConfig.items():
         vrfName = value.vrfName
         instKey = Tac.Value( "Routing::Ospf::OCOspfInstanceKey",
                              vrfName, "OSPF", str( instanceId ) )
         if instKey not in externalConfig.instance:
            t0( "Deleting instance", str( instanceId ) )
            qt0( "Deleting instance", qv( str( instanceId ) ) )
            nativeConfig.instanceConfig[ instanceId ].summaryAddrConfig.clear()
            del nativeConfig.instanceConfig[ instanceId ]
            if not nInstancesInVrf( nativeConfig.instanceConfig, vrfName ):
               IraVrfCli.removeAgentVrfEntry( vrfName, "Ospf" )

   # Register one-time-sync callback to be called before commit
   class ToNativeOspfSyncher( GnmiSetCliSession.PreCommitHandler ):
      externalPathList = [ 'routing/ospf/openconfig/instances' ]
      nativePathList = [ 'routing/ospf/config',
                         'ip/vrf/config',
                         'l3/intf/config' ]

      @classmethod
      def run( cls, sessionName ):
         toNativeOspfSyncher( cls, sessionName )

   GnmiSetCliSession.registerPreCommitHandler( ToNativeOspfSyncher )
   AirStreamLib.registerCopyHandler( entMan, "OspfOpenConfig",
            typeName="Routing::Ospf::OCOspfInstanceDir" )
