# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import threading

import Ark
import Logging
import Syscall
import Tac
import UtmpDump

SYS_CONFIG_LOCK_CLEAR = Logging.LogHandle( 'SYS_CONFIG_LOCK_CLEAR',
   severity=Logging.logNotice,
   fmt='The configuration lock was released manually by \'%s\'.',
   explanation=( 'The configuration lock was released by a user not holding '
                 'the lock.' ),
   recommendedAction=Logging.NO_ACTION_REQUIRED )

MAX_HISTORY_SIZE = 100
LOCK = threading.RLock()

class ConfigLockException( Exception ):
   pass

class UnableToAcquireLockException( ConfigLockException ):
   pass

class UnableToReleaseLockException( ConfigLockException ):
   pass

class ConfigureLockHistory:
   def __init__( self, user, tty, location, tid, transactionName, acquireReason ):
      self.user = user
      self.tty = tty
      self.location = location if location != 'UnknownIpAddr' else None
      self.tid = tid
      self.transactionName = transactionName
      self.grabTime = Tac.now()
      self.acquireReason = acquireReason
      self.releaseTime = None
      self.releaseReason = None

class LockBase:
   @property
   def transactionName( self ):
      raise NotImplementedError

   def isLockOwner( self ):
      raise NotImplementedError

   def continueLock( self ):
      raise NotImplementedError

   def maybeCleanupLock( self ):
      raise NotImplementedError

   def autoReleaseLock( self ):
      raise NotImplementedError

   def _logForceMessage( self ):
      userInfo = UtmpDump.getUserInfo()
      SYS_CONFIG_LOCK_CLEAR( userInfo[ 'user' ] )

class ThreadLock( LockBase ):
   '''
      Lock that can only be held by a single thread/cli session. When the lock
      is created it's automatically aquired by thread. The lock is released
      either by a CLI explictly calling the lock release or when a  cli session
      goes away. Also, the release doesn't work with transactions at all.
   '''
   def __init__( self ):
      self.ownerTid_ = Syscall.gettid()

   @property
   def transactionName( self ):
      return None

   def isLockOwner( self ):
      return self.ownerTid_ == Syscall.gettid()

   def checkReleaseLock( self, transactionName, force ):
      '''Before the lock is released check to see if it can be released.'''
      # a thread lock doesn't take a transaction name. This is almost
      # tantamount to a syntax error
      if transactionName is not None:
         raise UnableToReleaseLockException(
               'Unable to release Configure Lock. Not a valid transaction' )

      # if this thread owns the lock then we can release the lock
      if self.isLockOwner():
         return

      # this means that this user/thread is not the owner of the lock.
      # if we are being forced then log a message, otherwise throw an error
      if force:
         self._logForceMessage()
         return

      raise UnableToReleaseLockException(
            'Unable to release Configure Lock. Not held by thread/user.' )

   def continueLock( self ):
      '''A thread lock cannot continue a lock. This is used in the TransactionLock
      where a lock transaction lock is continued by a cli session'''
      raise UnableToAcquireLockException(
            'Unable to continue Configure Lock. Held by another thread/user.' )

   def maybeCleanupLock( self ):
      '''This lock has nothing to cleanup when a thread goes away '''
      pass # pylint: disable=unnecessary-pass

   def autoReleaseLock( self ):
      return True

class TransactionLock( LockBase ):
   '''
      Lock that can held by several threads/cli sessions at the same time. When
      the lock is created the creating thread will also implictly grab the lock.
      If additional threads also want to participate in the lock they would
      continue the lock which means that that thread will also be marked as holding
      the lock. When the thread/cli session goes away that thread should also be
      marked as not participating in the lock anymore since the thread ids can be
      reused.
   '''
   def __init__( self, transactionName ):
      self.tids_ = set()
      self.continueLock()
      self.transactionName_ = transactionName

   @property
   def transactionName( self ):
      return self.transactionName_

   def isLockOwner( self ):
      '''Returns true if this thread co-owns this lock'''
      tid = Syscall.gettid()
      return tid in self.tids_

   def checkReleaseLock( self, transactionName, force ):
      '''Before the lock is released check to see if it can be released.'''

      # if the transaction names match then good to go. Please note that
      # the thread doesn't have to own the lock in order to release the lock
      if self.transactionName == transactionName:
         return

      # if it's being forced, don't do anymore checks
      if force:
         self._logForceMessage()
         return

      # if we don't have a transaction name, say that we should have one.
      # otherwise the user requested a transaction name different than
      # what is actually created so also throw an error
      if transactionName is None: # pylint: disable=no-else-raise
         raise UnableToReleaseLockException(
            'Unable to release Configure Lock. Requires a transaction' )
      else:
         raise UnableToReleaseLockException(
            'Unable to release Configure Lock. Not a valid transaction' )

   def continueLock( self ):
      '''Add the current thread to threads that co-own this lock'''
      self.tids_.add( Syscall.gettid() )

   def maybeCleanupLock( self ):
      '''Remove this thread from threads that co-own this lock'''
      assert self.isLockOwner()
      self.tids_.remove( Syscall.gettid() )

   def autoReleaseLock( self ):
      return False

class ConfigureLock:
   '''Singleton object that that will lock the configuration of the switch'''

   def __init__( self ):
      self.lock_ = None # when active stores a type of lock
      self.history_ = []
      self.currHistory_ = None

   def getCurrUserInfo( self ):
      return self.currHistory_

   def getHistory( self ):
      return self.history_

   @Ark.synchronized( LOCK )
   def acquireLock( self, reason=None, transactionName=None ):
      if self.isLockedOwned():
         raise UnableToAcquireLockException(
               'Unable to acquire Configure Lock. Held by another thread/user.' )

      # setup the information for the lock
      if transactionName is None:
         self.lock_ = ThreadLock()
      else:
         self.lock_ = TransactionLock( transactionName )

      info = UtmpDump.getUserInfo()
      tid = Syscall.gettid()
      self.currHistory_ = ConfigureLockHistory( info[ 'user' ],
            info[ 'tty' ], info[ 'ipAddr' ], tid, transactionName, reason )
      self.history_.append( self.currHistory_ )
      if len( self.history_ ) > MAX_HISTORY_SIZE:
         self.history_.pop( 0 )

   @Ark.synchronized( LOCK )
   def continueLock( self, reason=None, transactionName=None ):
      # this should be called for a transaction lock to know that a new cli session
      # has requested the lock.
      assert transactionName is not None
      if not self.isLockedOwned():
         raise UnableToAcquireLockException(
               'Unable to continue Configure Lock. Not currently held by a user.' )

      self.lock_.continueLock()

   @Ark.synchronized( LOCK )
   def releaseLock( self, force=False, transactionName=None, reason=None ):
      if not self.isLockedOwned():
         # if there is no owner we shouldn't do the rest of this
         return
      self.lock_.checkReleaseLock( transactionName, force )
      self.lock_ = None
      self.currHistory_.releaseTime = Tac.now()
      self.currHistory_.releaseReason = reason
      self.currHistory_ = None

   @Ark.synchronized( LOCK )
   def maybeCleanupLock( self, reason=None ):
      # When a CLI session goes away we should maybe release the lock. There are 2
      # cases here. If it's a thread lock, and we own it, we should fully release
      # the lock. If it's a transaction lock then we should just remove this thread
      # from the list of threads using the lock.
      if not self.isLockOwner():
         return

      self.lock_.maybeCleanupLock()
      if self.lock_.autoReleaseLock():
         self.releaseLock( reason=reason )

   @Ark.synchronized( LOCK )
   def canRunConfigureCmds( self ):
      # if the lock isn't owned then it's clear to run. If it's owned
      # then only if it's the lock owner can it run.
      return not self.isLockedOwned() or self.isLockOwner()

   @Ark.synchronized( LOCK )
   def isLockOwner( self ):
      # if this thread has the lock
      return self.isLockedOwned() and self.lock_.isLockOwner()

   @Ark.synchronized( LOCK )
   def isLockedOwned( self ):
      # if any thread has the lock
      return self.lock_ is not None
