#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ArnetModel import IpGenericPrefix
from CliModel import Model, Dict
import TableOutput

class LeakedConnectedRoutes( Model ):
   '''Connected routes that have been leaked into a specific VRF'''

   routes = Dict( help="A mapping of route prefix to the VRF that leaked it",
                        keyType=IpGenericPrefix, valueType=str )

class LeakedConnectedRoutesByVrf( Model ):
   '''Connected routes that have been leaked from one VRF to another VRF'''

   destVrfs = Dict( help="A mapping of destination VRF to leaked connected "
                    "routes", keyType=str, valueType=LeakedConnectedRoutes )

   def skipValue( self, value, valueFilter ):
      '''Utility method to apply a filter'''

      if value is None:
         return True

      if valueFilter is None:
         return False

      return value != valueFilter

   # pylint: disable-msg=unsupported-assignment-operation
   # pylint: disable-msg=unsubscriptable-object
   def initFromTacModel( self, leakedRouteStatus, vrfIdStatus, destVrfFilter=None,
                         prefixFilter=None, sourceVrfFilter=None ):

      for destVrfPrefix, leakedRoute in \
          leakedRouteStatus.leakedRoute.items():

         destVrfEntry = vrfIdStatus.vrfIdToName.get( destVrfPrefix.destVrfId )
         if destVrfEntry is None:
            continue

         destVrfName = destVrfEntry.vrfName
         if self.skipValue( destVrfName, destVrfFilter ):
            continue

         sourceVrfEntry = vrfIdStatus.vrfIdToName.get( leakedRoute.sourceVrfId )
         if sourceVrfEntry is None:
            continue

         sourceVrfName = sourceVrfEntry.vrfName
         if self.skipValue( sourceVrfName, sourceVrfFilter ):
            continue

         prefix = str( destVrfPrefix.prefix )
         if self.skipValue( prefix, prefixFilter ):
            continue

         vrfModel = self.destVrfs.get( destVrfName )
         if vrfModel is None:
            vrfModel = LeakedConnectedRoutes()
            self.destVrfs[ destVrfName ] = vrfModel

         vrfModel.routes[ prefix ] = sourceVrfName

   def render( self ):
      '''Output Format:

      Destination VRF    Leaked Prefix           Source VRF
      ------------------ ----------------------- ----------
      Green              2.2.2.0/24              Blue
      Red                1.1.1.0/24              Blue
      Red                3.3.3.0/24              Default
      Yellow             1.1.1.0/24              Green'''

      headers = ( "Destination VRF", "Leaked Prefix", "Source VRF" )

      formatLeft = TableOutput.Format( justify="left" )
      formatLeft.noPadLeftIs( True )

      table = TableOutput.createTable( headers )
      table.formatColumns( formatLeft, formatLeft, formatLeft )

      for destVrf in sorted( self.destVrfs ):
         vrfLeakedRoutes = self.destVrfs[ destVrf ]
         for leakedPrefix in sorted( vrfLeakedRoutes.routes ):
            sourceVrf = vrfLeakedRoutes.routes[ leakedPrefix ]
            table.newRow( destVrf, str( leakedPrefix ), sourceVrf )

      print( table.output() )

   def parseRenderOutput( self, text, validateSorting=True ):
      '''Given the rendered output:

      Destination VRF    Leaked Prefix           Source VRF
      ------------------ ----------------------- ----------
      Green              2.2.2.0/24              Blue
      Red                1.1.1.0/24              Blue
      Red                3.3.3.0/24              Default
      Yellow             1.1.1.0/24              Green

      returns the following JSON Model:

      {'destVrfs': {'Green': {'routes': {'2.2.2.0/24': 'Blue'}}.
                    'Red': {'routes': {'1.1.1.0/24': 'Blue',
                                       '3.3.3.0/24': 'Default'}},
                    'Yellow': {'routes': {'1.1.1.0/24': 'Green'}}}}

      This is useful in tests'''

      model = {}
      model[ 'destVrfs' ] = {}
      destVrfs = model[ 'destVrfs' ]

      # Items 0 through 8 are part of the header
      start = 9
      textList = text.split()
      end = len( textList )
      numColumns = 3

      prevPrefix = None
      prevDestVrf = None

      for i in range( start, end, numColumns ):

         destVrf = textList[ i ]
         prefix = textList[ i + 1 ]
         sourceVrf = textList[ i + 2 ]

         if validateSorting:

            if prevDestVrf is not None:
               assert destVrf >= prevDestVrf

            if destVrf == prevDestVrf:
               if prevPrefix is not None:
                  assert prefix >= prevPrefix

            prevDestVrf = destVrf
            prevPrefix = prefix

         vrfModel = destVrfs.get( destVrf )
         if vrfModel is None:
            destVrfs[ destVrf ] = {}
            vrfModel = destVrfs.get( destVrf )
            vrfModel[ 'routes' ] = {}

         vrfLeakedRoutes = vrfModel.get( 'routes' )
         vrfLeakedRoutes[ prefix ] = sourceVrf

      return model
