# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from ContainerMgrCommon import serviceInstalled
import Logging
import PyWrappers.Docker as docker
import QuickTrace
import SuperServer
import Tac
import Tracing
import os
import sys
from IpLibConsts import DEFAULT_VRF


traceHandle = Tracing.Handle( "ContainerMgrConfigSm" )
KernelRoutingIntfState = Tac.Type( "Routing::KernelRoutingIntfState" )
IpTristateBool = Tac.Type( "Ip::TristateBool" )
IntfId = Tac.Type( "Arnet::IntfId" )
DockerIntfId = Tac.Type( "Arnet::DockerIntfId" )

t0 = traceHandle.trace0 # function calls
t1 = traceHandle.trace1 # error/exception
t2 = traceHandle.trace2 # login/logout from a registry
t3 = traceHandle.trace3 # container info traces

qv = QuickTrace.Var
qt0 = QuickTrace.trace0 # Important function calls
qt1 = QuickTrace.trace1 # error/exception
qt2 = QuickTrace.trace2 # Other important info

CONTAINERMGR_DOCKER_NOT_INSTALLED = None
CONTAINERMGR_DOCKER_NOT_INSTALLED = Logging.LogHandle(
      "CONTAINERMGR_DOCKER_NOT_INSTALLED",
      severity=Logging.logError,
      fmt="Docker is not installed",
      explanation="ContainerMgr depends on docker to work properly. "
                  "Docker RPM is not packaged with EOS.swi. "
                  "docker.x86_64.swix needs to be installed before "
                  "ContainerMgr is configured.",
      recommendedAction="Install docker.x86_64.swix, "
                                " and then reconfigure ContainerMgr." )

class ContainerMgrConfigReactor( SuperServer.LinuxService ):
   notifierTypeName = "ContainerMgr::ContainerMgrSuperServerConfig"

   def __init__( self,
                 containerMgrConfig,
                 containerMgrSuperServerConfig,
                 containerMgrSuperServerStatus,
                 kernelRoutingOverrideStatus,
                 registryConfigDir,
                 master ):
      SuperServer.LinuxService.__init__( self, "ContainerMgr", docker.name(),
                                         containerMgrSuperServerConfig,
                                         "/etc/docker/daemon.json",
                                         configFileHeaderEnabled=False )
      self.master_ = master
      self.daemonStarted = False
      self.config_ = containerMgrConfig
      self.superServerConfig_ = containerMgrSuperServerConfig
      self.superServerStatus_ = containerMgrSuperServerStatus
      self.routingOverrideStatus = kernelRoutingOverrideStatus
      self.registryConfigDir = registryConfigDir

      self.pollDaemonActivity = Tac.ClockNotifiee()
      self.pollDaemonActivity.timeMin = Tac.endOfTime
      self.pollDaemonActivity.handler = self.pollDaemon
      self.pollDaemonInterval = 0.5 # Poll every 0.5s

      self.handleDaemonEnable()

   def serviceEnabled( self ):
      return serviceInstalled() and self.superServerConfig_.daemonEnable

   def serviceProcessWarm( self ):
      t0( "serviceProcessWarm called." )
      return True

   def startService( self ):
      if serviceInstalled():
         t0( "startService" )
         SuperServer.LinuxService.startService( self )
      else:
         t0( "Skipping startService as docker not installed" )

   def stopService( self ):
      if serviceInstalled():
         t0( "stopService" )
         SuperServer.LinuxService.stopService( self )
         Tac.run( [ "systemctl", "stop", "docker.socket" ],
                  stdout=sys.stdout, stderr=sys.stderr, asRoot=True,
                  ignoreReturnCode=True )
         # remove the docker interface from routing disable override
         vrf = DEFAULT_VRF
         krosVrf = self.routingOverrideStatus.kernelRoutingIntfByVrf
         if not krosVrf:
            return
         routingIntfByVrf = krosVrf[ vrf ]
         routingIntf = routingIntfByVrf.kernelRoutingIntf
         docker0IntfName = DockerIntfId.docker0IntfName
         if docker0IntfName in routingIntf:
            del routingIntf[ docker0IntfName ]
            if not routingIntf:
               del krosVrf[ vrf ]
         try:
            if os.path.exists( "/sys/class/net/docker0" ):
               cmd = [ 'ip', 'link', 'delete', 'docker0' ]
               Tac.run( cmd, stdout=sys.stdout, stderr=sys.stderr, asRoot=True )
         except Tac.SystemCommandError:
            t1( "ip link delete cmd failed. cmd is %s" % cmd )
            qt1( "ip link delete cmd failed. cmd is ", qv( cmd ) )
      else:
         t0( "Skipping stopService as docker not installed" )

   def restartService( self ):
      if serviceInstalled():
         t0( "restartService" )
         SuperServer.LinuxService.restartService( self )
      else:
         t0( "Skipping restartService as docker not installed" )

   def conf( self ):
      t0( "conf called." )
      qt0( "conf called." )
      cfTemplate = '{%s\n  "storage-driver" : "overlay2"\n}\n'
      insecureRegistries = self.master_.getInsecureServers()
      if insecureRegistries:
         insecureRegistriesLine = ( '\n  "insecure-registries" : [%s],' %
                                    insecureRegistries )
      else:
         insecureRegistriesLine = ''
      cf = cfTemplate % insecureRegistriesLine
      return cf

   def loadImages( self ):
      t0( "loadImages called." )
      qt0( "loadImages called." )
      files = []
      loadCmd = [ 'docker', 'load' ]
      persistPaths = [ self.config_.defaultPersistentPath ]
      if self.config_.persistentPath != self.config_.defaultPersistentPath:
         persistPaths += [ self.config_.persistentPath ]

      for path in persistPaths:
         persistPath = path + '.containermgr/'
         if os.path.exists( persistPath ):
            files += os.listdir( persistPath )
         print( "Loading backed up containers from %s" % persistPath,
                file=sys.stderr )
         for fileName in files:
            absolutePath = persistPath + fileName
            t3( "Loading from absolute path %s" % absolutePath )
            cmd = loadCmd + [ '-i', absolutePath ]
            try:
               Tac.run( cmd, stdout=sys.stdout, stderr=sys.stderr, asRoot=True )
               qt2( "Image loaded using ", qv( cmd ) )
               t3( "Image loaded using %s", cmd )
            except Tac.SystemCommandError as e:
               t1( 'ContainerMgr image backed up at %s cannot be loaded'
               ' due to %s' % ( absolutePath, e.output ) )
               qt1( "ContainerMgr image backed up at ", qv( absolutePath ),
                    " cannot be loaded due to ", qv( e.output ) )

   def onDaemonStarted( self ):
      t0( "onDaemonStarted" )
      qt0( "onDaemonStarted" )
      docker0IntfName = DockerIntfId.docker0IntfName
      dockerIntfState = KernelRoutingIntfState( docker0IntfName )
      dockerIntfState.kernelOnlyDeviceIntfId = IntfId( docker0IntfName )
      dockerIntfState.disableKernelRouting = IpTristateBool.isFalse
      intfByVrf = self.routingOverrideStatus.kernelRoutingIntfByVrf. \
         newMember( DEFAULT_VRF )
      intfByVrf.kernelRoutingIntf.addMember( dockerIntfState )
      self.loadImages()
      self.master_.applyInitialConfig()
      self.superServerStatus_.enabled = True

   def isDaemonRunning( self ):
      # Run docker info to see if dockerd is up
      cmd = [ 'docker', 'info' ]
      try:
         Tac.run( cmd,
                  stdout=Tac.DISCARD, stderr=Tac.CAPTURE )
      except Tac.SystemCommandError as e:
         traceMsg = ( f"isDaemonRunning: returing False, "
                      f"status command failed: {e.output}" )
         t1( traceMsg )
         qt1( qv( traceMsg ) )
         return False
      return True

   def pollDaemon( self ):
      t1( "pollDaemon" )
      qt1( "pollDaemon" )
      if not self.serviceEnabled():
         t0( "pollDaemon: service disabled, stop polling" )
         qt0( "pollDaemon: service disabled, stop polling" )
         self.pollDaemonActivity.timeMin = Tac.endOfTime
         return

      if not self.isDaemonRunning():
         t1( "pollDaemon: scheduling next poll" )
         qt1( "pollDaemon: scheduling next poll" )
         # dockerd still not up, schedule another poll
         self.pollDaemonActivity.timeMin = Tac.now() + self.pollDaemonInterval
         return
      else:
         # dockerd is up, stop polling, also do any pending initialization
         t0( "pollDaemon: successful" )
         qt0( "pollDaemon: successful" )
         self.pollDaemonActivity.timeMin = Tac.endOfTime
         self.daemonStarted = True
         self.onDaemonStarted()

   def handleServiceEnabled( self ):
      t0( "handleServiceEnabled called." )
      qt0( "handleServiceEnabled called." )

      if self.serviceEnabled():
         self.sync() # schedule dockerd start
         self.pollDaemonActivity.timeMin = Tac.now() + self.pollDaemonInterval
      elif self.daemonStarted:
         self.master_.cleanup()
         self.stopService()
         self.daemonStarted = False
         # TODO: Remove after ContainerMgr is decoupled from SuperServer
         # But for now, make separate ContainerMgr agent shutdown.
         self.superServerStatus_.enabled = False

   @Tac.handler( 'daemonEnable' )
   def handleDaemonEnable( self ):
      t0( "handleDaemonEnable called." )
      qt0( "handleDaemonEnable called." )
      if self.superServerConfig_.daemonEnable:
         if not serviceInstalled():
            Logging.log( CONTAINERMGR_DOCKER_NOT_INSTALLED )
            return
      self.handleServiceEnabled()

   def close( self ):
      t0( "close for ContainerMgrConfigReactor called." )
      Tac.Notifiee.close( self )
