# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ContainerMgrCommon import runCmd
import QuickTrace
import Tac
import Tracing
from TypeFuture import TacLazyType

traceHandle = Tracing.Handle( "ContainerMgrRegistrySm" )

ServerInfo = TacLazyType( 'ContainerMgr::ServerInfo' )
RegistryState = TacLazyType( 'ContainerMgr::RegistryState' )
RegistryParams = TacLazyType( 'ContainerMgr::RegistryParams' )

t0 = traceHandle.trace0 # important function calls
t1 = traceHandle.trace1 # error/exception
t2 = traceHandle.trace2 # other function calls

qv = QuickTrace.Var
qt0 = QuickTrace.trace0 # Important function calls
qt1 = QuickTrace.trace1 # error/exception
qt2 = QuickTrace.trace2 # other function calls

class RegistryReactor( Tac.Notifiee ):
   notifierTypeName = "ContainerMgr::RegistryConfig"

   def __init__( self, notifier, master ):
      Tac.Notifiee.__init__( self, notifier )
      self.master_ = master
      self.registryConfig = notifier
      self.prevServerName = ""
      self.activity_ = Tac.ClockNotifiee()
      self.activity_.handler = self.handleRegistryConfig
      self.activity_.timeMin = Tac.endOfTime
      self.activityInterval = 30
      self.retryCounter = 0
      self.maxRetry = 30
      self.registryStatusDir = master.registryStatusDir
      self.registryStatus = \
            self.registryStatusDir.newRegistry( self.registryConfig.name )
      self.handleRegistryConfig()

   def login( self ):
      registryName = self.registryConfig.name
      t2( f"login called for registry {registryName}" )
      qt2( "login called for registry", qv( registryName ) )

      params = self.registryConfig.params
      assert params.serverName
      assert params.userName
      assert params.password
      cmd = [ "docker", "login",
              "-u", params.userName,
              "-p", params.password,
              params.serverName ]
      cmdDesc = f"logging into {params.serverName} with username {params.userName}"
      hasFailed, errText = runCmd( cmd, cmdDesc )
      if not hasFailed:
         serverInfo = ServerInfo( params, self.registryConfig.name )
         self.master_.registryStatusDir.secureServer.addMember( serverInfo )

      return ( hasFailed, errText )

   @staticmethod
   def logout( registryStatusDir, serverName ):
      t2( f"logout called for {serverName}" )
      qt2( "logout called for", qv( serverName ) )

      cmd = [ "docker", "logout", serverName ]
      runCmd( cmd,
              f"Logging out from {serverName}" )
      del registryStatusDir.secureServer[ serverName ]

   def attemptToScheduleRetry( self ):
      ''' Returns if retry was scheduled'''
      registryName = self.registryConfig.name
      if self.retryCounter < self.maxRetry:
         self.retryCounter += 1

         t1( f"Scheduling retry {self.retryCounter} for registry {registryName}" )
         qt1( "Scheduling retry", qv( self.retryCounter ),
              "for registry", registryName )
         self.activity_.timeMin = Tac.now() + self.activityInterval
         return True
      else:
         t1( f"Retries exhausted for registry {registryName}, marking failed" )
         qt1( "Retries exhausted for registry", registryName,
              ", marking failed" )
         self.activity_.timeMin = Tac.endOfTime
         return False

   def clearRetryState( self ):
      self.retryCounter = 0
      self.activity_.timeMin = Tac.endOfTime

   def handleRegistryConfig( self ):
      registryName = self.registryConfig.name
      t2( f"handleRegistryConfig called. Registry is {registryName}" )
      qt2( "handleRegistryConfig called. Registry is",
           qv( registryName ) )

      if not self.master_.daemonStarted():
         t2( "handleRegistryConfig: skip" )
         qt2( "handleRegistryConfig: skip" )
         return

      configParams = self.registryConfig.params
      statusParams = self.registryStatus.params

      isSecureRegistryConfig = ( not configParams.insecure and
                                 configParams.serverName and
                                 configParams.userName and
                                 configParams.password )

      prevState = self.registryStatus.state
      errText = ""
      if isSecureRegistryConfig:
         configChanged = ( configParams != statusParams )
         if configChanged:
            self.clearRetryState()

         isRetryAttempt = ( prevState == RegistryState.regStateLoginRetry )
         if configChanged or isRetryAttempt:
            hasFailed, errText = self.login()
            if not hasFailed:
               self.clearRetryState()
               nextState = RegistryState.regStateLoggedIn
            else:
               retryScheduled = self.attemptToScheduleRetry()
               if retryScheduled:
                  nextState = RegistryState.regStateLoginRetry
               else:
                  nextState = RegistryState.regStateLoginFailed
         else:
            # not config change and not retry, do nothing, retain state
            nextState = prevState
      else:
         # insecure registry, or incomplete secure registry config,
         # marked logged out, clear retry
         self.clearRetryState()
         nextState = RegistryState.regStateLoggedOut

      self.registryStatus.params = configParams
      self.registryStatus.state = nextState
      self.registryStatus.errText = errText

   @Tac.handler( 'params' )
   def handleParams( self ):
      registryName = self.registryConfig.name
      t2( f"handleParams called for {registryName}" )
      qt2( "handleParams called for", qv( registryName ) )
      self.handleRegistryConfig()
      self.master_.handleRegistryUpdate()

   def cleanup( self ):
      registryName = self.registryConfig.name
      t2( f"close for RegistryReactorReactor called for {registryName}" )
      qt2( "close for RegistryReactorReactor called for", qv( registryName ) )
      self.activity_.timeMin = Tac.endOfTime
      del self.registryStatusDir.registry[ registryName ]
      self.master_.handleRegistryUpdate()

   def close( self ):
      registryName = self.registryConfig.name
      t2( f"close for RegistryReactorReactor called {registryName}" )
      qt2( "close for RegistryReactorReactor called", qv( registryName ) )
      self.cleanup()
      Tac.Notifiee.close( self )
