# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing
import Agent
import BothTrace
import Cell
import MlagMountHelper
from ControllerClientConfigMonitor import ConfigMonitor
from GenericReactor import GenericReactor

from Toggles.ControllerCommonToggleLib import toggleControllerMountMonitorEnabled

# following import is to aid in dependency generation
# pkgdeps: import SysMgrLib

__defaultTraceHandle__ = Tracing.Handle( 'ControllerClient' )
bv = BothTrace.Var
bt5 = BothTrace.tracef5

Constants = Tac.Type( "Controller::Constants" )
MgmtSecuritySslConstants = Tac.Type( "Mgmt::Security::Ssl::Constants" )
SslProfileDisableReason = Tac.Type( "Controller::SslProfileStatus::DisableReason" )
ProfileState = Tac.Type( "Mgmt::Security::Ssl::ProfileState" )
mySysname = None

def agentName():
   return 'ControllerClient'

# This class's job is to mount client-side controller state and
# do setup/tear-down of the ConfigMonitor.
class ControllerClient( Agent.Agent ):
   def __init__( self, entityMgr, plugins=None, pluginPath=None ):
      self.plugins = plugins
      self.pluginPath = pluginPath
      self.root = None
      self.servicePluginCtx = None
      self.configMonitor = None
      self.configReactor = None
      self.config = None
      self.status = None
      self.serviceConfig = None
      self.overrideServiceConfig = None
      self.bridgingConfig = None
      self.netConfig = None
      self.ipStatus = None
      self.ipConfig = None
      self.mlagConfig = None
      self.controllerSysdbConfig = None
      self.controllerSysdbStatus = None
      self.mgmtSecSslStatus = None
      self.vrfStatusLocal = None
      self.cleanupReactor = None
      self.hostnameReactor = None
      self.local = None

      Agent.Agent.__init__( self, entityMgr )
      global mySysname
      mySysname = entityMgr.sysname()

   def doInfraMounts( self, mg ):
      self.config = mg.mountPath( "mgmt/controller/config" )
      self.status = mg.mountPath( "mgmt/controller/status" )
      self.controllerSysdbConfig = \
            mg.mountPath( "mgmt/controller/publish/config" )
      self.controllerSysdbStatus = \
            mg.mountPath( "mgmt/controller/publish/status" )
      self.serviceConfig = mg.mountPath( "mgmt/controller/service/config" )
      self.overrideServiceConfig = \
            mg.mountPath( "mgmt/controller/debug/overrideService" )

   def doInit( self, entityManager ):
      bt5()
      self.local = entityManager.root().parent
      mg = entityManager.mountGroup()
      self.doInfraMounts( mg )
      self.bridgingConfig = mg.mountPath( "bridging/config" )
      self.netConfig = mg.mountPath( "sys/net/config" )
      self.ipConfig = mg.mountPath( "ip/config" )
      mg.mountPath( "l3/intf/config" )
      Tac.Type( "Ira::IraIpStatusMounter" ).doMountEntities( mg.cMg_, True, False )
      self.ipStatus = mg.mountPath( "ip/status" )
      self.mgmtSecSslStatus = mg.mountPath ( "mgmt/security/ssl/status" )
      # Mount mlag/config, Mlag::Config and its dependent paths
      self.mlagConfig = MlagMountHelper.mountMlagConfig( mg )
      self.vrfStatusLocal = mg.mountPath( Cell.path( "ip/vrf/status/local" ) )

      mg.close( self.doMountsComplete )

   def doMountsComplete( self ):
      bt5( "Creating root and its state-machines" )
      self.root = self.local.newEntity( "ControllerClient::Root", "root" )
      self.root.sourceIntfDir = ( "sourceIntfDir", )
      self.root.mySysname = mySysname
      self.status.controllerStatus.clear()
      self.status.sslProfileStatus = None
      self.status.sslProfileStatus = ( "sslProfileStatus", )
      if not self.status.established:
         self.status.established = ( "established", )
      if not self.status.leaderServiceStatusDir:
         self.status.leaderServiceStatusDir = ( "leaderServiceStatusDir", )
      for serviceName in self.serviceConfig.service:
         # No need of reactor because ServiceConfig is published by
         # ConfigAgent/Sysdb before ControllerClient is spawned by Launcher
         self.status.leaderServiceStatusDir.newService( serviceName )
      self.status.initialized = True
      self.configReactor = GenericReactor( self.config,
                                            [ 'enabled' ],
                                            self.handleEnabled,
                                            callBackNow=True )
      self.hostnameReactor = GenericReactor( self.netConfig,
                                             [ 'hostname' ],
                                             self.handleHostname )

   def handleEnabled( self, notifiee=None ):
      enabled = self.config.enabled
      bt5( "CVX enabled: ", bv( enabled ) )
      if enabled:
         if self.cleanupReactor:
            self.cleanupReactor.close()
            del self.cleanupReactor
            self.cleanupReactor = None

         if self.configMonitor:
            self.configMonitor.doCleanup( force=True )
            del self.configMonitor
         self.configMonitor = ConfigMonitor( self.config, self.status,
                                             self.overrideServiceConfig,
                                             self.serviceConfig,
                                             self.mgmtSecSslStatus,
                                             self.bridgingConfig,
                                             self.netConfig,
                                             self.ipStatus,
                                             self.ipConfig,
                                             self.mlagConfig,
                                             self.vrfStatusLocal,
                                             self.controllerSysdbConfig,
                                             self.root )

         if not self.root.leadershipSm:
            self.root.leadershipSm = ( self.status, self.controllerSysdbConfig )

         if not self.root.leaderServiceStatusSm:
            self.root.leaderServiceStatusSm = ( self.status,
                                                self.controllerSysdbConfig )
         
         if toggleControllerMountMonitorEnabled() and \
            not self.root.publishMountMonitorSm:
            self.root.publishMountMounitorSm =\
               ( self.controllerSysdbConfig, self.controllerSysdbStatus )

         self.status.enabled = enabled
      else:
         self.doCleanup()

         if not self.cleanupReactor:
            self.cleanupReactor = GenericReactor( self.status,
                                                  [ 'controllerStatus' ],
                                                  self.handleControllerStatus,
                                                  callBackNow=True )

   def handleControllerStatus( self, notifiee=None, key=None ):
      if not self.status.controllerStatus:
         del self.configMonitor
         self.configMonitor = None
         self.status.enabled = self.config.enabled

   def handleHostname( self, notifiee=None ):
      bt5( "setting connection hostname to", self.netConfig.hostname )
      for controllerStatus in self.status.controllerStatus.values():
         controllerStatus.connectionStatus.switchHost.hostname = \
               self.netConfig.hostname

   def doCleanup( self ):
      bt5()
      if self.configMonitor:
         self.configMonitor.doCleanup()
      if self.root.leadershipSm:
         self.root.leadershipSm.doCleanup()
         self.root.leadershipSm = None
      if self.root.leaderServiceStatusSm:
         self.root.leaderServiceStatusSm.doCleanup()
         self.root.leaderServiceStatusSm = None

