#!/usr/bin/env python3
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

# This file implements a simple protobuf-to-tacc compiler. It
# assumes that:
# - The protobuf model includes only messages and enums;
# - Message fields are either scalar value types or enums, except
#   in the top-level Message, which may contain other messages;
# - Submessages in the top-level Message are named according to
#   their type, so a field of type CvxFooMessage is named "foo".

import re
import Tac

PROTOFILE = "/src/ControllerCommon/Controller.proto"

DEBUG = False

_cog = None # cog isn't a proper module we can import
_parseTree = None

_tacc64Type = { "string" : "Tac::String" }

_taccType = { "double" : "F64",
              "float" : "F32",
              "int16" : "S16",
              "int32" : "S32",
              "int64" : "S64",
              "uint32": "U32",
              "uint64" : "U64",
              "sint32" : "S32",
              "sint64" : "S64",
              "bool" : "bool",
              "string" : "Tac::String" }
_pbNamespace = "ControllerProtobuf"

# Output a line of text, including a specified number of indents.
def _out( line, indent=0 ):
   idt = "   " * indent
   # lines longer than 85*2 characters aren't supported
   op = ( idt + line )[ :84 ]
   op2 = None
   if len( op ) < ( len( line ) + len( idt ) ):
      op += "\\"
      op2 = ( line[ ( 85 - len( idt ) - 1 ): ] )
   if DEBUG:
      print( op )
      if op2:
         print( op2 )
   else:
      _cog.outl( op )
      if op2:
         _cog.outl( op2 )

# Return all instances of the specified pattern in a text block.
def _parse( regex, block ):
   return re.findall( regex, block, re.DOTALL )

def _tType( pbType ):
   return _taccType[ pbType ] if pbType in _taccType else pbType
def _tMessageEnum( pbMsg ):
   return pbMsg[ 0 ].lower() + pbMsg[ 1: ]
def _tMessageField( pbType ):
   return pbType.replace( "Cvx", "" ).replace( "Message", "" ).lower()

# Every node in the parse tree is a PbNode. Each derived node
# type is instantiated with some text that matched its regular
# expression, so, e.g., the PbMessage type gets a message name
# and a chunk of unparsed message fields that it then parses to
# create PbMessageField nodes. We need to be able to generate
# three things: tacc type definitions for each protobuf message
# and enum; C++ implementations of those types; and an enum of
# message types. Each node type knows how to emit text for each
# of those functions.
class PbNode:
   re = ""
   children_ = []
   def generateType( self ):
      for c in self.children_:
         c.generateType()
   def generateImpl( self ):
      for c in self.children_:
         c.generateImpl()
   def generateMessageEnum( self ):
      for c in self.children_:
         c.generateMessageEnum()

# PbNode has no __init__ for its derived classes to call.
# pylint: disable-msg=W0231

class PbMessageField( PbNode ):
   re = r"(required|optional)\s+(\w+)\s+(\w+)\s+=\s+(\d+);"
   def __init__( self, block, parent ):
      self.parent_ = parent
      ( self.type_, self.name_, self.val_ ) = block[ 1: ]

   def generateType( self ):
      _out( f"{self.name_} : {_tType( self.type_ )} {{", 1 )
      _out( "`hasExternalMutator;", 2 )
      _out( "`hasExternalAccessor;", 2 )
      _out( "`hasDataMember=false;", 2 )
      _out( "`hasInlinedAccessor=false;", 2 )
      _out( "}", 1 )

   def generateImpl( self ):
      self._generateAccessor()
      self._generateMutator()

   def generateMessageEnum( self ):
      # ignore basic types
      if self.type_ not in _taccType and \
         self.type_ not in self.parent_.blacklist:
         _out( f"{_tMessageEnum( self.type_ )} : {self.val_};", 1 )

   def _generateAccessor( self ):
      #Tac.pdb()
      _out( _tType( self.type_ ) )
      _out( f"{self.parent_.name_}::{self.name_}() const {{" )
      pba = "pbMessage_->{}().{}()".format( _tMessageField( self.parent_.name_ ),
                                            self.name_.lower() )
      if self.type_ == "string":
         _out( "return Tac::String( %s );" % pba, 1 )
      elif self.type_ not in _taccType:
         # Cast for enums
         _out( f"return ({_tType( self.type_ )})({pba});", 1 )
      else:
         _out( "return %s;" % pba, 1 )
      _out( "}" )

   def _generateMutator( self ):
      _out( "void" )
      
      # if type is 64 bit then you pass in by reference
      formalParam = ( "%s const & x" if self.type_ in _tacc64Type 
            else "%s x" ) % _tType( self.type_ )
      _out( "{}::{}Is( {} ) {{".format( self.parent_.name_, self.name_, 
         formalParam ) )

      # Cast for enums
      cast = ( f"({_pbNamespace}::{self.type_})" ) \
            if self.type_ not in _taccType else ""
      _out( "pbMessage_->mutable_%s()->set_%s( %sx%s );" %
            ( _tMessageField( self.parent_.name_ ), self.name_.lower(), cast,
              ".charPtr()" if self.type_ == "string" else "" ), 1 )
      _out( "}\n" )

class PbMessage( PbNode ):
   re = r"message\s+(\w+)\s+?\{(.*?)\}"
   blacklist = [ "Message", "CvxVersion", "CvxVersionResponse", "Heartbeat" ]
   rxMsgMethodName = "rxMessage"
   txMsgMethodName = "newTxMessage"
   cmeSocketType = "ControllerMessageSocket::Ptr"
   def __init__( self, block, parent ):
      self.parent_ = parent
      self.name_ = block[ 0 ]
      self.children_ = [ PbMessageField( f, self ) for f in \
                         _parse( PbMessageField.re, block[ 1 ] ) ]


   def generateType( self ):
      if self.name_ in self.blacklist:
         return
      _out( "%s : Tac::Type(socket) : ControllerMessage {" % self.name_ )
      _out( "`hasFactoryFunction;", 1 )
      _out( "messageType = initially %s;" % _tMessageEnum( self.name_ ), 1 )
      for c in self.children_:
         c.generateType()
      for methodName in self.rxMsgMethodName, self.txMsgMethodName:
         _out( f"{methodName} : static extern {self.name_}::Ptr(", 1 )
         _out( "socket : %s);" % self.cmeSocketType, 2 )
      _out( "}\n" )

   def _generateRxTxMsgMethodImpl( self, methodName ):
      factoryMethod = self.name_[ 0 ].lower() + self.name_[ 1: ] + "Factory"
      _out( "%s::Ptr" % self.name_ )
      _out( f"{self.name_}::{methodName}(" )
      _out( "%s const &socket ) {" % self.cmeSocketType, 1 )
      _out( "auto msg = %s( socket );" % factoryMethod, 1 )
      if methodName == self.txMsgMethodName:
         _out( "msg->emptyIs( true );", 1 )
      _out( "return msg;", 1 )
      _out( "}\n" )

   def generateImpl( self ):
      if self.name_ in self.blacklist:
         return
      for c in self.children_:
         c.generateImpl()
      for methodName in self.rxMsgMethodName, self.txMsgMethodName:
         self._generateRxTxMsgMethodImpl( methodName )

   def generateMessageEnum( self ):
      if self.name_ != "Message":
         return
      _out( "ControllerMessageId : invasive Tac::Enum : U32 {" )
      _out( "CvxDefaultMessage : 0;", 1 )
      for c in self.children_:
         c.generateMessageEnum()
      _out( "}" )

class PbEnumField( PbNode ):
   re = r"(\w+)\s+=\s+(\d+);"
   def __init__( self, block, parent ):
      ( self.name_, self.val_ ) = block
      self.parent_ = parent

   def generateType( self ):
      _out( f"{self.name_} : {self.val_};", 1 )

class PbEnum( PbNode ):
   re = r"enum\s+(\w+)\s+?\{(.*?)\}"
   def __init__( self, block, parent ):
      self.name_ = block[ 0 ]
      self.parent_ = parent
      self.children_ = [ PbEnumField( f, self ) for f in \
                         _parse( PbEnumField.re, block[ 1 ] ) ]

   def generateType( self ):
      _out( "%s : invasive Tac::Enum : U32 {" % _tType( self.name_ ) )
      for c in self.children_:
         c.generateType()
      _out( "}\n" )

class PbTree( PbNode ):
   def __init__( self, block, parent ):
      self.children_ = \
            [ PbEnum( b, self ) for b in _parse( PbEnum.re, block ) ] + \
            [ PbMessage( b, self ) for b in _parse( PbMessage.re, block ) ]

def _getParseTree():
   f = open( PROTOFILE ) # pylint: disable=consider-using-with
   text = f.read()
   f.close()
   return PbTree( text, None )

#################################################################
###################### Generator functions ######################
#################################################################

# Create the parse tree just once and output a warning about this
# file being autogenerated.
def init( cog ):
   global _parseTree, _cog
   _cog = cog
   _parseTree = _getParseTree()
   _out( "//\n//" )
   _out( "// NOTE: THIS FILE WAS AUTOGENERATED. DO NOT EDIT IT DIRECTLY." )
   _out( "//       EDIT THE PRETAC/PRETIN AND THEN REGENERATE THIS FILE." )
   _out( "//\n//" )

# tacc enum for the message types, e.g.,
#    ControllerMessageId : invasive Tac::Enum {
#       ...
#       versionStatusMessage : 7;
#       ...
#    }
def generateMessageIdEnum():
   _parseTree.generateMessageEnum()

# tacc types for the message types in Controller.proto, e.g.,
#    CvxVersionStatusMessage : Tac::Type(pbMessage) : ControllerMessage {
#       ...
#    }
def generateTypes():
   _parseTree.generateType()

# Accessors and mutators for the protobuf messages, e.g.,
#    void VersionStatus::versionIs( U32 v ) { ... }
def generateImpls():
   _parseTree.generateImpl()
