#!/usr/bin/env python3
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, EntityManager, os
import Plugins
import Tracing
from collections import namedtuple

__defaultTraceHandle__ = Tracing.Handle( 'ControllerdbEntityManager' )
t0 = Tracing.trace0

Constants = Tac.Value( "Controller::Constants" )
controllerdbPortEnvVar = 'CONTROLLERDBPORT'
controllerdbSocknameEnvVar = 'CONTROLLERDBSOCKNAME'

def controllerdbPort():
   """Returns the port on which Controllerdb is (or
   should be) listening.  Raises a MountError if ControllerdbPort is in
   an illegal format"""
   ( host, port ) = controllerdbHostPort()
   if host != "localhost":
      # pylint: disable-next=consider-using-f-string
      raise EntityManager.MountError( "Error: CONTROLLERDBPORT is '%s' but it must "
                                      "be set to localhost"
                                      % os.getenv( controllerdbPortEnvVar ) )
   return port

def controllerdbPortIs( port ):
   """Makes the environment variable consistent with the actual port
   that Controllerdb is listening on."""
   os.environ[ controllerdbPortEnvVar ] = str( port )

def controllerdbHostPort():
   """Returns a tuple of the host and port on which Controller is (or
   should be) listening.  host is a string, port is an int.

   Raises MountError if the CONTROLLERPORT environment variable does not
   have a legal format.
   """
   return EntityManager.parseHostPort(
      os.getenv( controllerdbPortEnvVar, str( Constants.controllerdbDefaultPort ) ) )

def controllerdbSockname():
   """Returns the socket name which Controllerdb is (or
   should be) listening."""
   sockname = os.getenv( controllerdbSocknameEnvVar,
                         Constants.controllerdbDefaultSockname )
   return sockname

def controllerdbSocknameIs( sockname ):
   """Makes the environment variable consistent with the actual socket name
   that Controllerdb is listening on."""
   os.environ[ controllerdbSocknameEnvVar ] = sockname

class Controllerdb( EntityManager.Remote ):
   """This entity manager obtains all requested entities by mounting them from
   Controllerdb. It is intended for use by all agents on the Controller besides
   Controllerdb"""

   def __init__( self, sysname,
                 controllerdbSockname_=None,
                 controllerdbHostport_=None,
                 dieOnDisconnect=True, mountRoot=True, persistent=False ):
      sysnameDir = Tac.root.newEntity( 'Tac::Dir', sysname )
      root = sysnameDir.newEntity( 'Tac::Dir', 'Controllerdb' )
      # Unless overridden by the user, we try to use the host port
      # to connect to Controllerdb
      serveraddr = controllerdbSockname_ or controllerdbHostport_ or \
                   controllerdbHostPort() or controllerdbSockname()
      EntityManager.Remote.__init__( self, sysname, serveraddr,
                                     root=root, remoteRootPath=root.fullName,
                                     dieOnDisconnect=dieOnDisconnect,
                                     mountRoot=mountRoot, scheduled=False,
                                     persistent=persistent )

# pylint: disable-msg=W0212
class Local( EntityManager._NotSimple ):
   """ This EntityManager manages the entities within the Local Controllerdb """
   ServiceConfig = namedtuple( "ServiceConfig", [ "name" ] ) 

   def __init__( self, sysname, plugins=None, pluginPath=None ):
      t0( 'ControllerdbLocal: __init__' )
      rootname = "Controllerdb"
      sysnameDir = Tac.root.newEntity( 'Tac::Dir', sysname )
      root = sysnameDir.newEntity( 'Tac::Dir', rootname )
      EntityManager._NotSimple.__init__( self, sysname, root )

      emRoot = emRemotePath = f"{sysname}/{rootname}"
      cEm_config = Tac.Value( "Sysdb::EntityManager::Config",
                     sysname=sysname, root=emRoot, remotePath=emRemotePath,
                     isLocalEm=True, dieOnDisconnect=False,
                     isServer=True )
      self.cEm_.config = cEm_config
      self.cEm_.doInit()

      self.serviceConfigList = []
      Plugins.loadPlugins( 'ControllerdbPlugin', self, plugins, pluginPath )

   def register( self, path, typenameOrType, force=True ):
      """Used by Sysdb plugins to register an entity to be added to Sysdb.
      The typenameOrType parameter can either be a C++ type name, or a
      subclass of DynEntity.Type."""
      # ignore force, irrelevant on Local but kept for consistency.
      return self.cEm_.doRegister( path, typenameOrType, "wi" )

   def mount( self, path, typeName, mode='r', mountGroup=None ):
      return self.register( path, typeName )
   entity = mount

   def registerService( self, serviceName ):
      # pylint: disable-next=unidiomatic-typecheck
      assert type( serviceName ) is str, repr( serviceName )
      assert not serviceName.endswith( Constants.proxyServiceSuffix )
      self.serviceConfigList.append( Local.ServiceConfig( serviceName ) )

   def unregisterService( self, serviceName ):
      pass
