#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ArnetModel import MacAddress
from CliModel import Bool, Dict, Int, Model, Str, Submodel
from IntfModels import Interface
from TableOutput import createTable

class TrafficClasses( Model ):
   __public__ = False
   rx = Dict( keyType=int, valueType=int,
              help="Successful Rx frames per traffic class" )
   tx = Dict( keyType=int, valueType=int,
              help="Successful Tx frames per traffic class" )

class CountersField( Model ):
   __public__ = False
   cpu_code = Dict( valueType=int, help="Summary of Rx per cpu codes counters" )
   fdma = Dict( valueType=int, help="Summary of FDMA aggregagte counters" )
   tx = Dict( valueType=int, help="Summary of Tx counters" )
   tx_errors = Dict( valueType=int, help="Summary of Tx errors counters" )
   rx = Dict( valueType=int, help="Summary of Rx counters" )
   rx_errors = Dict( valueType=int, help="Summary of Rx errors counters" )
   traffic_classes = Submodel( valueType=TrafficClasses,
                               help="Summary of Traffic classes counters" )
   pcimapfail = Int( help="SKB pci dma mapping failure counter" )
   skballocfail = Int( help="SKB allocataion failure counter" )

   def render( self ):
      t = createTable( [ "Group", "Counter", "# of packets" ] )
      for code, count in self.cpu_code.items():
         t.newRow( "CPU Code", code, count )
      for counter, count in self.fdma.items():
         t.newRow( "FAP Summary", counter, count )
      for tx_counter, count in self.tx.items():
         t.newRow( "TX Counter", tx_counter, count )
      for tx_fail_counter, count in self.tx_errors.items():
         t.newRow( "TX Error", tx_fail_counter, count )
      for rx_counter, count in self.rx.items():
         t.newRow( "RX Counter", rx_counter, count )
      for rx_fail_counter, count in self.rx_errors.items():
         t.newRow( "RX Error", rx_fail_counter, count )
      for tc in self.traffic_classes.rx:
         t.newRow( "Traffic Class", f"TC{tc}_RX", self.traffic_classes.rx[ tc ] )
         t.newRow( "Traffic Class", f"TC{tc}_TX", self.traffic_classes.tx[ tc ] )
      t.newRow( "Misc", "pcimapfail", self.pcimapfail )
      t.newRow( "Misc", "skballocfail", self.skballocfail )

      print( t.output() )

class FabParamsField ( Model ):
   __public__ = False
   bfd_napi_time_limit_ns = Int( help="Bfd napi poll timeout value" )
   debug = Int( help="Fab debug tracing value" )
   fab_usr_hdr_sz = Int( help="Fab user header size" )
   fdevtx_locktime = Int( help="Fab lock max time(us) for fdev_tx" )
   ifdel_nice = Int( help="Device deletion nice value" )
   islocked = Bool( help="True if fab locked" )
   lock = Int( help="Fab lock reference" )
   macsec_eapol_etype = Int( help="Macsec configurable EAPOL ethertype" )
   mlag_peer_link_system_port = Int( help="Mlag peer link system port" )
   napi_poll_active = Int( help="Active napi poll count on all cpus" )
   napi_poll_inactive_cnt = Int( help="Inactive napi poll count" )
   napi_time_limit_ns = Int( help="Napi-poll timout value" )
   nfdev = Int( help="Network interfaces for tx flow-control" )
   oldest_faptype = Str( help="Oldest fap type name" )
   pciremove_locktime = Int( help="Fab lock max time(us) for pci-removal" )
   rxtype = Int( help="Rx type for Rx processing path" )
   sflow_drop_threshold_jericho = Int(
      help="Number of sflow packets drop threshold" )
   vcable_proto = Int( help="Virtual cable outer tag id" )
   vxlan_mcast_evpn_state = Int( help="VXLAN multi cast state in ethernet vpn" )
   vxlan_vtep_learn_mode = Int( help="VXLAN virtual tunnel endpoint learn mode" )

   def render( self ):
      t = createTable( [ "Config", "Value" ] )
      for field, val in self.toDict().items():
         t.newRow( field, val )
      print( t.output() )

class Fdev( Model ):
   __public__ = False
   bypass_lldp = Int( optional=True,
                     help="True if LLDP process skkipped to send packet in clear" )
   bypass_ptp = Int( optional=True,
                     help="True if PTP process skipped to send packet in clear" )
   ctrl = Bool( help="Set pph pkt_is_ctrl bit" )
   dot1q_tunnel = Bool( help="True if switch port mode is dot1q tunnel" )
   encapvlan = Int( optional=True, help="Vlan encap for packtes on routed-port" )
   fap_port = Int( help="Sand fap port id" )
   fapid = Int( help="Fap id" )
   fdma = Str( help="Device logical name" )
   ilag = Bool( help="True if internal lag master" )
   ilag_fdev = Str( optional=True, help="Interface name for internal lag fdev" )
   ivlan = Int( optional=True, help="Internal vlan id if fdev is routed-port" )
   lagname = Str( optional=True, help="LAG interface name" )
   lagid = Int( help="Lag id if fdev is LAG or LAG member interface" )
   macsec = Bool( help="True if port is macsec capable" )
   macsec_client_name = Str( optional=True,
                             help="Interface name is client of macsec proxy" )
   macsec_client_tag = Int( optional=True,
                            help="Macsec proxy tag identifying the client port" )
   macsec_proxy = Bool( help="True if interface is macsec proxy port" )
   mark4 = Int( help="Mark ipv4 skbs on receive side" )
   mark6 = Int( help="Mark ipv6 skbs on receive side" )
   native_vlan = Int( optional=True, help="Port's native vlan id" )
   netif_queue_stopped = Bool( 
      help="True if transmit queue on device is currently unable to send" )
   selected_lag_member = Str( optional=True,
      help="Interface name for selected lag member fdev for tx to lag destination" )
   sflow_cookie = Int( help="Sflow opaque data" )
   mirror_on_drop_cookie = Int( help="Mirror-on-drop opaque data" )
   static = Bool( help="True if fdma is static else dynamic" )
   svtag = Int( optional=True, help="32 bit security vlan tag added to packet" )
   sub = Bool( help="True if sub-interfaces linked to this fdev" )
   sys_port = Int( help="Sand system port id" )
   tc_vpri_map = Dict( keyType=int, valueType=int, optional=True,
                       help="Mapping from Per-intf traffic class to cos" )
   tc_vpri_map_id = Int( optional=True, help="Per-intf tc->cos map id")
   traffic_class = Int( help="Transmit ITMH traffic-class" )
   vcable_muxid = Int( optional=True, help="Virtual cable muxid" )

class Channel( Model ):
   __public__ = False
   bytes = Int( help="Channel Tx/Rx bytes" )
   direction = Str( help="Channel Tx/Rx direction" )
   overflow = Int( optional=True, help="Tx ring/chain out of descriptors" )
   packets = Int( help="Successful Tx/Rx frames" )
   poll = Int( help="napi-poll tx-chain-done/rx-desc-done" )

class Fdma( Model ):
   __public__ = False
   arad = Int( help="Cpu injection port" )
   aradbr = Int( help="Cpu port for bridging" )
   bfd_complete = Int( help="BFD napi complete" )
   bfd_incomplete = Int( help="BFD napi incomplete" )
   bfd_napi_time_squeeze = Int( help="Exceeded time budget count in bfd_napi_poll" )
   channel = Dict( keyType=int, valueType=Channel,
                   help="Summary of DMA Channel 0-3" )
   complete = Int( help="Napi complete count" )
   incomplete = Int( help="Napi incomplete count" )
   intr = Int( help="Interrupts received" )
   napi_state = Int( help="Napi state" )
   napi_time_squeeze = Int( help="Exceeded time budget count in napi_poll" )
   chain_recover = Int( help="DMA chain recovery" )
   pcireadfail = Int( help="Pci register read failures" )
   pdev = Str( help="Device name" )
   poll = Int( help="Napi poll calls count" )
   shutdown = Int( help="fdma shutdown call count" )
   state = Bool( help="Actual operational state" )
   sysport = Int( help="Cpu system port" )
   tm = Int( help="Tx on cpu interface" )
   type = Int( help="Fap chip type" )

class VSI( Model ):
   __public__ = False
   fdev = Str( help="VLAN-VSI Interface name" )
   outer_vlanid = Int( help="Outer VLAN ID" )
   outer_tpid = Int( help="Outer VLAN TPID" )
   inner_vlanid = Int( help="Inner VLAN ID" )
   inner_tpid = Int( help="Inner VLAN TPID" )
   vsi = Int( help="VSI to be applied for the VLAN IDs" )

class LabelMappedVlan( Model ):
   __public__ = False
   vlanid_ipv4 = Int( help="Label-VLAN id for ipv4" )
   vlanid_ipv6 = Int( help="Label-VLAN id for ipv6" )
   controlword = Int( help="Indicates whether controlWord follows BOS label" )
   dot1q_xlate = Int(
      help="Indicates whether inner L2 payload needs dot1q translation" )
   l2_payload = Int( help="Indicates whether the VPN label payload is L2 packet" )
   strip_label = Int( help="Strip the mpls label after processing" )

class LabeltagMappedVlan( Model ):
   __public__ = False
   label = Int( help="L2 EVPN label" )
   dot1qTag = Int( help="Normalized dot1q Tag" )
   vlanid = Int( help="Destination VLAN ID" )

class Mappings( Model ):
   __public__ = False
   dscp_to_tc = Dict( keyType=int, valueType=int, 
                     help="Mapping from DSCP to traffic class" )
   tc_to_vpri = Dict( keyType=int, valueType=int,
                      help="Mapping from traffic class to vlan priority" )
   tc_vpri_map_id_to_value = Dict( keyType=int, valueType=int,
                                   help="Mapping from egress TC vpri to COS map" )
   vpri_to_tc = Dict( keyType=int, valueType=int,
                      help="Mapping from vlan priority to traffic class" )
   ivlan_to_encap_vlan = Dict( keyType=int, valueType=int, optional=True,
                               help="Mapping from internal vlan to encap vlan" )
   pvlanmap = Dict( keyType=int, valueType=int, optional=True,
         help="Mapping from secondary vlan to primary vlan for pvlan" )
   routedports_ivlan_map = Dict( keyType=int, valueType=str, optional=True, 
                  help="Mapping from internal vlan id to fdev interface name" )
   vlan_to_vsi = Dict( valueType=VSI, optional=True,
                       help="Mapping from VLAN tag to VSI information" )
   vni_to_vlan = Dict( keyType=int, valueType=int, optional=True,
                       help="Mapping from VNI to VLAN id" )
   fapid_to_oamp = Dict( keyType=int, valueType=int, optional=True,
                         help="Mapping from Fap ID to SPPID of OAMP" )
   labeltag_to_vlanId = Dict( valueType=LabeltagMappedVlan, optional=True,
                             help="Mapping from Label dot1qtag to VLAN information" )
   label_to_vlanId = Dict( keyType=int, valueType=LabelMappedVlan, optional=True,
                           help="Mapping from Label to VLAN information" )

   def render( self ):
      def renderBasicMapping( mapping, col1, col2 ):
         t = createTable( [ col1, col2 ] )
         for elem1, elem2 in mapping.items():
            t.newRow( elem1, elem2 )
         print( t.output() )

      renderBasicMapping( self.dscp_to_tc, "DSCP", "TC" )
      renderBasicMapping( self.tc_to_vpri, "TC", "VLAN Priority" )
      renderBasicMapping( self.vpri_to_tc, "VLAN Priority", "TC" )
      renderBasicMapping( self.tc_vpri_map_id_to_value, "TC VLAN Priority Map ID",
                          "COS Maps" )
      renderBasicMapping( self.ivlan_to_encap_vlan, "VSI", "VLAN" )
      renderBasicMapping( self.pvlanmap, "Secondary VLAN", "Primary VLAN" )
      renderBasicMapping( self.routedports_ivlan_map, "VSI", "Fdev" )
      renderBasicMapping( self.vni_to_vlan, "VNI", "VLAN" )
      renderBasicMapping( self.fapid_to_oamp, "FAP ID", "Source Sys Port ID" )

      t = createTable( [ "Interface", "Outer VLAN ID", "Outer TPID",
                       "Inner VLAN ID", "Inner TPID", "VSI" ] )
      for _, vsi_entry in self.vlan_to_vsi.items():
         t.newRow( vsi_entry.fdev, vsi_entry.outer_vlanid, vsi_entry.outer_tpid,
                  vsi_entry.inner_vlanid, vsi_entry.inner_tpid, vsi_entry.vsi )
      print( t.output() )

      t = createTable( [ "Label Tag", "dot1q Tag", "VLAN ID" ] )
      for _, labeltag_entry in self.labeltag_to_vlanId.items():
         t.newRow( labeltag_entry.label, labeltag_entry.dot1qTag,
                  labeltag_entry.vlanid )
      print( t.output() )

      t = createTable( [ "VPN Label", "VLAN ID for IPv4", "VLAN ID for IPv6",
                       "Control Word", "Has VLAN Translation", "Is L2 packet",
                       "Strip Label" ] )
      for vpn_label, vlan in self.label_to_vlanId.items():
         t.newRow( vpn_label, vlan.vlanid_ipv4, vlan.vlanid_ipv6, vlan.controlword,
                  vlan.dot1q_xlate, vlan.l2_payload, vlan.strip_label )
      print( t.output() )

class Lag( Model ):
   __public__ = False
   lag_name = Str( help="Lag fdev interface name" )
   members = Dict( optional=True, keyType=int, valueType=str,
                   help="Mapping from System port to lag interface" )

class LagField( Model ):
   __public__ = False
   nlags = Int( help="Number of lags supported" )
   nlagmems = Int( help="Number of lag members per lag supported" )
   nlagshift = Int( help="Number of bit position in sys_port_id" )
   npartitionshift = Int( help="Number of bit position in sys_port_id" )
   lagidshift = Int( help="Number of bit position for lag id in sys_port_id" )
   lags = Dict( keyType=int, valueType=Lag, optional=True,
                help="Mapping from Lag id to Lag information" )

   def render( self ):
      t = createTable( [ "LAG Property", "Value" ] )
      for key, val in self.toDict().items():
         if key == "lags":
            continue
         t.newRow( key, val )
      print( t.output() )

      print( "LAGs:" )
      for lag in self.lags.values():
         print( f"\t{lag.lag_name}:" )
         for member in lag.members.values():
            print( f"\t\t{member}" )

class Vlan( Model ):
   __public__ = False
   routing_enabled = Bool( optional=True, default=False,
                          help="VLAN routing enabled" )
   vxlan_enabled = Bool( optional=True, default=False, help="VLAN vxlan enabled" )
   has_virtual_IPv4 = Bool( optional=True, default=False,
                           help="True if VLAN with virtual IPv4 address" )
   has_virtual_IPv6 = Bool( optional=True, default=False,
                           help="True if VLAN with virtual IPv6 address" )
   arp_snoop_enabled = Bool( optional=True, default=False,
                            help="VLAN with ARP snooping enabled" )
   mvpn_pmsi = Bool( optional=True, default=False,
                    help="VLAN with mvpn pmsi enabled" )

class VlanField( Model ):
   __public__ = False
   number_of_vxlan_enabled_vlan = Int( help="Number of VXLAN enabled VLANs" )
   number_of_arpsnoop_enabled_vlan = Int( 
      help="Number of ARP snooping enabled VLANs" )
   vlans = Dict( keyType=int, optional=True, valueType=Vlan,
                 help="Mapping from VLAN id to VLAN information" )

   def render( self ):
      t = createTable( [ "VLAN ID", "Routing", "VXLAN", "Virtual IPv4",
                       "Virtual IPv6", "ARP Snoop", "MVPN" ] )
      for vid, vlan in self.vlans.items():
         t.newRow( vid, vlan.routing_enabled, vlan.vxlan_enabled,
                  vlan.has_virtual_IPv4, vlan.has_virtual_IPv6,
                  vlan.arp_snoop_enabled, vlan.mvpn_pmsi )
      print( t.output() )

class Sand( Model ):
   __public__ = False
   counters = Submodel( valueType=CountersField,
                        help="Sand fab counters information" )
   debug_intfs = Dict( valueType=Interface, help="Debug Interfaces" )
   fab_params = Submodel( valueType=FabParamsField,
                          help="Sand fab parameters information" )
   fdevs = Dict( valueType=Fdev, help="Sand fdev information" )
   fdma = Dict( valueType=Fdma, help="Fab independent information" )
   lag_configs = Submodel( valueType=LagField, help="Details of Lag configurations" )
   mac_addrs = Dict( valueType=MacAddress, help="Mac Address information" )
   mappings = Submodel( valueType=Mappings, help="Sand Fab mappings inforamtion" )
   toggles = Dict( valueType=bool, help="List of toggles fields in Sand Fab" )
   vlan_configs = Submodel( valueType=VlanField,
                            help="Details of VLAN configurations" )

   def render( self ):
      self.counters.render()
      t = createTable( [ "Debug Interface Index", "Interface Name" ] )
      for key, intf in self.debug_intfs.items():
         t.newRow( key, intf )
      print( t.output() )
      self.fab_params.render()

      fdevHeaders = Fdev.__attributes__.keys()
      t = createTable( [ "Fdev Name", *fdevHeaders ] )
      for name, fdev in self.fdevs.items():
         vals = [ getattr( fdev, col ) for col in fdevHeaders ]
         t.newRow( name, *vals )
      print( t.output() )

      # printing general fdma attributes
      fdmaHeaders = Fdma.__attributes__.keys() - { "channel" }
      t = createTable( [ "FDMA Name", *fdmaHeaders ] )
      for name, fdma in self.fdma.items():
         vals = [ getattr( fdma, col ) for col in fdmaHeaders ]
         t.newRow( name, *vals )
      print( t.output() )

      # printing DMA channel specific values
      channelAttrs = Channel.__attributes__.keys()
      t = createTable( [ "FDMA Name", "Channel #", *channelAttrs ] )
      for name, fdma in self.fdma.items():
         for channelNumber, channel in fdma.channel.items():
            vals = [ getattr( channel, col ) for col in channelAttrs ]
            t.newRow( name, channelNumber, *vals )
      print( t.output() )

      self.lag_configs.render()

      print( "MAC Addresses:" )
      t = createTable( [ "Name", "Address" ] )
      for name, addr in self.mac_addrs.items():
         t.newRow( name, addr )
      print( t.output() )

      self.mappings.render()

      t = createTable( [ "Toggle", "State" ] )
      for toggle, val in self.toggles.items():
         t.newRow( toggle, val )
      print( t.output() )

      self.vlan_configs.render()
