#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ArnetModel import MacAddress
from CliModel import Bool, Dict, Int, Model, Str, Submodel
from TableOutput import createTable

class SkbMemStats( Model ):
   __public__ = False
   skb_last_ptm = Dict( keyType=int, valueType=int,
                        help="Last PTM Processed" )
   skb_processed = Dict( keyType=int, valueType=int,
                         help="Number of SKB processed" )
   skb_duplicated = Dict( keyType=int, valueType=int,
                          help="Number of SKB duplicated" )
   skb_forwarded = Dict( keyType=int, valueType=int,
                         help="Forwarded to kernel/cimic" )
   skb_freed = Dict( keyType=int, valueType=int,
                     help="Number of SKB released" )

class CountersField( Model ):
   __public__ = False
   skbmem = Submodel( valueType=SkbMemStats,
                      help="Summary of SKB Memory counters" )
   tx = Dict( valueType=int, help="Summary of Tx counters" )
   tx_raw = Dict( valueType=int, help="Summary of Tx raw counters" )
   rx = Dict( valueType=int, help="Summary of Rx counters" )
   rx_cpu_code = Dict( valueType=int, help="Summary of Rx CPU Codes" )
   misc = Dict( valueType=int, help="Summary of misc counters" )

   def render( self ):
      t = createTable( [ "Group", "Counter", "# of packets" ] )
      for tx_counter, count in self.tx.items():
         t.newRow( "TX Counter", tx_counter, count )
      for tx_raw_counter, count in self.tx_raw.items():
         t.newRow( "TX Raw Counter", tx_raw_counter, count )
      for rx_counter, count in self.rx.items():
         t.newRow( "RX Counter", rx_counter, count )
      for rx_cpu_code, count in self.rx_cpu_code.items():
         t.newRow( "RX CPU Code", rx_cpu_code, count )
      for misc_counter, count in self.misc.items():
         t.newRow( "Misc Counter", misc_counter, count )
      print( t.output() )
      skbHeaders = SkbMemStats.__attributes__.keys()
      t = createTable( [ "SKB Memory ID", *skbHeaders ] )
      for idx in self.skbmem.skb_last_ptm:
         vals = [ getattr(self.skbmem, col)[idx] for col in skbHeaders]
         t.newRow(idx, *vals)         
      print( t.output() )
      
class FlexFieldStats( Model ):
   __public__ = False
   symbols = Dict( keyType=int, valueType=str,
                   help="Mapping of Flow and EP2 CPU Field symbols" )
   pos = Dict( keyType=int, valueType=int,
               help="Mapping of Flow and Flex Field Position" )
   len = Dict( keyType=int, valueType=int,
               help="Mapping of Flow and Flex Field Length" )

class FlexField( Model ):
   __public__ = False
   unit = Dict( keyType=int, valueType=FlexFieldStats, help="Flex Field unit" )
   rxflexreasonmap = Dict( keyType=int, valueType=int,
                           help="Flex reason codes and cpu drop code" )
   def render( self ):
      t = createTable( [ "Flow", "Flex Symbol", "Flex Position", "Flex Length" ] )
      for flow, flex in self.unit.items():
         for c in flex.symbols.keys():
            t.newRow( flow, flex.symbols[c], flex.pos[c], flex.len[c] )
      print( t.output() )
      t = createTable( [ "Flex RX Reason", "Value" ] )
      for c in self.rxflexreasonmap:
         t.newRow( f"rxflexreason{c}", self.rxflexreasonmap[c] )
      print( t.output() )


class TrafficField( Model ):
   __public__ = False
   loopback = Dict( valueType=int, help="Mapping of Loopback traffic" )
   def render( self ):
      t = createTable( [ "Loopback Traffic Field", "Value" ] )
      for f, v in self.loopback.items():
         t.newRow( f, v )
      print( t.output() )

class MacField( Model ):
   __public__ = False
   virtualmac = MacAddress( help="virtual router mac" )
   mlagpeermac = MacAddress( help="MLAG peer router mac" )
   shared_router_encap_mac = MacAddress( help="Shared Router Encap mac" )
   vxpeermac = MacAddress( help="MLAG peer mac used for vxlan virtual ipv6 addrs" )
   virtual_linklocal_addr6 = Int( help="Link local ipv6 addr for the virtual mac" )
   def render( self ):
      t = createTable( [ "Name", "Value" ] )
      for field, val in self.toDict().items():
         t.newRow( field, val )
      print( t.output() )

class FabParamsField ( Model ):
   __public__ = False
   debug = Int( help="Fab debug tracing value" )
   jiffies = Int( help="Kernel Jiffies" )
   ifdel_nice = Int( help="Device deletion nice value" )
   islocked = Bool( help="True if fab locked" )
   lock = Int( help="Fab lock reference" )
   napi_polls_active = Int( help="Active napi poll count on all cpus" )
   napi_polls_inactive_cnt = Int( help="Inactive napi poll count" )
   napi_time_limit_ns = Int( help="Napi-poll timout value" )
   mirror_ethertype = Int( help="Etype for t3 egress mirror to cpu" )
   mpc_mode = Bool( help="MPC mode flag" )
   mirror_egress_internal_loopback = Int( help="Egress mirror internal loopback" )
   internal_lpbk0 = Int( help="Portid of internal loopback0" )
   internal_lpbk1 = Int( help="Portid of internal loopback1" )
   internal_lpbk_feature = Int( help="Feature using internal loopback" )
   rxipv4 = Int( help="Rx ipv4 packets" )
   rxipv6 = Int( help="Rx ipv6 packets" )
   rxvmac_nd_modify = Int( help="Rx ND packets to virtual mac" )
   rxvmac_nd_notsenttokernel = Int( help="Rx ND packets not sent to kernel" )
   allow_tx_runts = Bool( help="Configurable via fab ioctl" )
   sai_mode = Bool( help="True if dma customizations needed for Sonic" )
   vxlan_nd_snooping = Bool( help="True if send ipv6 ND pkts to VXLAN agent" )
   global_mld_snooping_enabled = Bool(
      help="True if MLD snooping is globally enabled" )
   use_defer_free = Bool( help="True if defer free needed" )
   partial_multicast_fwd_enabled = Bool(
      help="True if partial multicast fwd plane enabled" )
   def render( self ):
      t = createTable( [ "Config", "Value" ] )
      for field, val in self.toDict().items():
         t.newRow( field, val )
      print( t.output() )

class FdevField( Model ):
   __public__ = False
   unit = Int( help="Chip unit" )
   port = Int( help="Logical port" )
   lagid = Int( help="Lag ID" )
   mark4 = Int( optional=True, help="Rx side in skb->mark for ipv4 packets" )
   mark6 = Int( optional=True, help="Rx side in skb->mark for ipv6 packets" )
   uc_qbase = Int( optional=True,
                   help="Base queue number for unicast in SOB header" )
   mc_qbase = Int( optional=True,
                   help="Base queue number use for multicast SOB header" )
   routedport = Bool( optional=True, help="Set if routed port" )
   sflow_cookie = Int( optional=True, help="Sflow ifindex (set via ioctl)" )
   macsec_subintf = Bool( optional=True, help="Macsec subintf on this port" )
   mirror_on_drop_cookie = Int( optional=True, help="SNMP ifindex (set via ioctl)" )
   netif_q_stopped = Bool( optional=True, help="True if Tx queue is stopped" )
   

class Fdev( Model ):
   __public__ = False
   intf = Dict( optional=True, valueType=FdevField,
                help="Mapping of Fabric interface" )  
   reap = Dict( optional=True, valueType=FdevField,
                help="Mapping of Fdev to cleanup" )
   def render( self ):
      fdevHeaders = FdevField.__attributes__.keys()
      t = createTable( [ "FDEV Name", *fdevHeaders ] )      
      for name, intf in self.intf.items():
         vals = [ getattr( intf, col ) for col in fdevHeaders ]
         t.newRow( name, *vals )
      for name, reap in self.reap.items():
         vals = [ getattr( reap, col ) for col in fdevHeaders ]
         t.newRow( name, *vals )
      print( t.output() )

class DeferedCqField( Model ):
   __public__ = False
   channel = Str( help="DMA channel" )
   size = Int( help="Number of DCBs allocated in descriptor ring" )
   active = Int( help="Index to start DCB set when chain is active" )
   active_count = Int( help="Actual size of active DCBs in descriptor ring" )
   active_rx = Int( help="Index to current DCB to be processed (RX only)" )

class BitField( Model ):
   __public__ = False
   bit = Dict( keyType=int, valueType=str, help="Mapping of bit to RX reason" )

class CdmaRxReasonUnit( Model ):
   __public__ = False
   unit = Dict( optional=True, keyType=int, valueType=BitField,
                help="Mapping of CMIC DMA unit to bit field" )
   def render( self ):
      t = createTable( [ "Unit", "Bit", "Value" ] )
      for u, f in self.unit.items():
         for b, val in f.bit.items():
            t.newRow( u, b, val )
      print( t.output() )


class CdmaRxTcUnit( Model ):
   __public__ = False
   unit = Dict( optional=True, keyType=int, valueType=int,
                help="Mapping of CMIC DMA unit to Rx traffic class" )
   def render( self ):
      t = createTable( [ "Unit", "Value" ] )
      for u, v in self.unit.items():
         t.newRow( u, v )
      print( t.output() )

class CdmaDebugField( Model ):
   __public__ = False
   rx_reason = Submodel( valueType=CdmaRxReasonUnit, optional=True,
                         help="CMIC DMA Rx reason" )
   rxtc = Submodel( valueType=CdmaRxTcUnit, optional=True,
                    help="CMIC DMA Rx traffic class" )
   def render( self ):
      if self.rx_reason:
         print( "RX Reason" )
         self.rx_reason.render()
      if self.rxtc:
         print( "RX Traffic Class" )
         self.rxtc.render()

class Cdma( Model ):
   __public__ = False
   defered_cq_blocks = Dict( valueType=DeferedCqField, optional=True,
                             help="Mapping of deferred channel queue" )
   cdma_desired_state_up = Dict( keyType=int, valueType=bool, optional=True,
                                 help="Mapping of device state" )
   cdma_desired_role_unit = Dict( keyType=int, valueType=int, optional=True,
                                  help="Mapping of device role" )
   cdma_desired_backplane_unit = Dict( keyType=int, valueType=int, optional=True,
                                       help="Mapping of backplane unit" )
   cdma_designated = Int( help="CDMA designated unit" )
   desired_designated_unit = Int( help="Save designated unit" )
   debug = Submodel( valueType=CdmaDebugField, optional=True,
                     help="Summary of CDMA debug information" )
   def render( self ):
      print( "Defered Channel Queue" )
      t = createTable( [ "Interface", "Field", "Value" ] )
      for intf, field in self.defered_cq_blocks.items():
         for f, val in field.toDict().items():
            t.newRow( intf, f, val )
      print( t.output() )
      print( "Device State" )
      t = createTable( [ "Unit", "State UP" ] )
      for u, state in self.cdma_desired_state_up.items():
         t.newRow( u, state )
      print( t.output() )
      print( "Device Role" )
      t = createTable( [ "Unit", "Role" ] )
      for u, role in self.cdma_desired_role_unit.items():
         t.newRow( u, role )
      print( t.output() )
      print( "Backplane Unit" )
      t = createTable( [ "Unit", "Value" ] )
      for u, val in self.cdma_desired_backplane_unit.items():
         t.newRow( u, val )
      print( t.output() )
      t = createTable( [ "Field", "Value" ] )
      t.newRow( "Cdma Designated",  self.cdma_designated )
      t.newRow( "Cdma Designated Unit",  self.desired_designated_unit )
      print( t.output() )
      if self.debug:
         print( "Debug" )
         self.debug.render()

class VfiVlanMap( Model ):
   __public__ = False
   vlanid = Int( help="Vlan id" )

class VlanVfiMap( Model ):
   __public__ = False
   vfi = Int( help="VFI id" )

class MacMap( Model ):
   __public__ = False
   mac = Int( help="MAC Address" )
   vlanid = Int( help="Vlan id" )
   is_mba = Int( help="Is Mba" )

class VxlanMap( Model ):
   __public__ = False
   vni = Int( help="Virtual Network id" )
   vlanid = Int( help="Vlan id" )

class LagField( Model ):
   __public__ = False
   lagid = Int( help="Lag id" )

class PortLagField( Model ):
   __public__ = False
   port = Dict( keyType=int, valueType=LagField, help="Lag port" )

class UnitLagField( Model ):
   __public__ = False
   unit = Dict( keyType=int, valueType=PortLagField, help="Lag unit" )

class Port( Model ):
   __public__ = False
   port = Dict( keyType=int, valueType=int, help="Port id" )

class Unit( Model ):
   __public__ = False
   unit = Dict( keyType=int, valueType=Port, help="Unit id" )

class Mappings( Model ):
   __public__ = False
   fdevroutedporttab = Dict( keyType=int, valueType=str, optional=True,
                     help="Mapping from vlan id to fdev" )
   subintfencapvlanmap = Dict( keyType=int, valueType=int,
                      help="Mapping from internalvlan to subinterface encap vlan " )
   vprimap = Dict( keyType=int, valueType=int,
                      help="Mapping from vlan priority to traffic class" )
   dscpmap = Dict( keyType=int, valueType=int, optional=True,
                               help="Mapping from dscp to traffic class" )
   uctxqmap = Dict( keyType=int, valueType=int, optional=True,
         help="Mapping from traffic class to uc-tx queue" )
   mctxqmap = Dict( keyType=int, valueType=int, optional=True, 
                  help="Mapping from traffic class to mc-tx queue" )
   pvlanmap = Dict( keyType=int, valueType=int, optional=True,
                       help="Mapping from secondary to primary vlan map" )
   vfi = Dict( keyType=int, valueType=VfiVlanMap, optional=True,
               help="Mapping from vfi-vlan map for all vfi/vxlan asics" )
   vlanid = Dict( keyType=int, valueType=VlanVfiMap, optional=True,
               help="Mapping from vlan-vfi map for all dynamic vfi asics" )
   mac = Dict( keyType=int, valueType=MacMap, optional=True,
               help="Mapping from SRC MAC to VLAN information " )   
   vxlan = Dict( keyType=int, valueType=VxlanMap, optional=True,
                 help="Mapping from VNI to VLANId  information" )
   lag  = Dict( keyType=int, valueType=LagField, optional=True,
                help="Mapping form LAG ID to FDEV information" )
   lagmap = Submodel( valueType=UnitLagField, optional=True,
                     help="Mapping from unit,port to lag" )
   lagconfigmap = Submodel( valueType=UnitLagField, optional=True,
                            help="Mapping from unit,port to lag id" )
   encap_vlan = Submodel( valueType=Unit, optional=True,
                          help="Mapping from unit,port to encapsulation vlan" )
   default_vlan = Submodel( valueType=Unit, optional=True,
                            help="Mapping from unit, port to default vlan id" )
   mlagpeerlink = Submodel( valueType=Unit, optional=True,
                            help="Mapping from unit, port to mlag peer link" )
   portflags = Submodel( valueType=Unit, optional=True,
                         help="Mapping from unit, port to port flags" )
   def render( self ):
      def renderBasicMapping( mapping, col1, col2 ):
         if mapping is None:
            return
         t = createTable( [ col1, col2 ] )
         for elem1, elem2 in mapping.items():
            t.newRow( elem1, elem2 )
         print( t.output() )
      def renderUnitMapping( mapping, col1, col2, col3 ):
         if mapping is None:
            return
         t = createTable( [ col1, col2, col3 ] )
         for elem1, m in mapping.unit.items():
            for elem2, elem3 in m.port.items():
               t.newRow( elem1, elem2, elem3 )
         print( t.output() )
      renderBasicMapping( self.fdevroutedporttab, "VLAN ID", "NAME" )
      renderBasicMapping( self.subintfencapvlanmap,
                          "Internal VLAN", "Sub Intf VLAN" )
      renderBasicMapping( self.vprimap, "VLAN Priority", "TC" )
      renderBasicMapping( self.dscpmap, "DSCP", "TC" )
      renderBasicMapping( self.pvlanmap, "Secondary VLAN", "Primary VLAN" )
      renderBasicMapping( self.uctxqmap, "TC", "UC-TX" )
      renderBasicMapping( self.mctxqmap, "TC", "MC-TX" )
      renderUnitMapping( self.encap_vlan, "ENCAP VLAN UNIT", "PORT", " Value" )
      renderUnitMapping( self.default_vlan, "Default VLAN UNIT", "PORT", " Value" )
      renderUnitMapping( self.mlagpeerlink, "MLAG Peer Link UNIT", "PORT", " Value" )
      renderUnitMapping( self.portflags, "PORT Flags UNIT", "PORT", " Value" )
      t = createTable( [ "MAC HASH", "MAC", "VLAN ID", "Is Mba" ] )
      for hashcode, macfield in self.mac.items():
         t.newRow( hashcode, macfield.mac, macfield.vlanid, macfield.is_mba )
      print( t.output() )

      t = createTable( [ "VNI HASH", "VNI", "VLAN ID" ] )
      for hashcode, vxlanfield in self.vxlan.items():
         t.newRow( hashcode, vxlanfield.vni, vxlanfield.vlanid )
      print( t.output() )

      t = createTable( [ "LAG HASH", "LAG ID" ] )
      for hashcode, lag in self.lag.items():
         t.newRow( hashcode, lag.lagid )
      print( t.output() )

      t = createTable( [ "LAG UNIT", "PORT", "LAG ID" ] )
      if self.lagmap:
         for unit, portmap in self.lagmap.unit.items():
            for port, portf in portmap.port.items():
               t.newRow( unit, port, portf.lagid )
      print( t.output() )
      
      t = createTable( [ "LAG Config UNIT", "PORT", "LAG ID" ] )
      if self.lagconfigmap:
         for unit, portmap in self.lagconfigmap.unit.items():
            for port, portf in portmap.port.items():
               t.newRow( unit, port, portf.lagid )
      print( t.output() )

      t = createTable( [ "VFI ID", "VLAN ID" ] )
      for vfi, vlan in self.vfi.items():
         t.newRow( vfi, vlan.vlanid )
      print( t.output() )

      t = createTable( [ "VLAN ID", "VFI ID" ] )
      for vlan, vfi in self.vlanid.items():
         t.newRow( vlan, vfi.vfi )
      print( t.output() )

class VlanField( Model ):
   __public__ = False
   vxlan_enabled = Bool( optional=True, help="True if Vxlan enabled" )
   routing_enabled = Bool( optional=True,
                           help="True if vlan routing status enabled" )
   has_mac_vlans_refcnt = Bool( optional=True,
                                help="mac based vlan reference count" )
   vlantxdisable = Bool( optional=True, help="VXLAN tx disable" )
   has_vlan_ipv4_virt = Bool( optional=True,
                              help="True if VLAN with virtual IPv4 address" )
   has_vlan_ipv6_virt = Bool( optional=True,
                              help="True if VLAN with virtual IPv6 address" )

class Strata( Model ):
   __public__ = False
   counters = Submodel( valueType=CountersField,
                        help="Summary of counters information" )
   flexfields = Submodel( valueType=FlexField,
                          help="Summary of Flex field information" )
   traffic = Submodel( valueType=TrafficField,
                       help="Summary of loopback traffic information" )
   mac_addrs = Submodel( valueType=MacField,
                         help="Summary Mac Address information" )
   cdma = Submodel( valueType=Cdma, help="Summary CMIC DMA information" )
   fab_params = Submodel( valueType=FabParamsField,
                          help="Summary of Strata fab parameters information" )
   fdev = Submodel( valueType=Fdev, help="Summary of Strata fdev information" ) 
   mappings = Submodel( valueType=Mappings,
                        help="Summary of Fab mappings inforamtion" )
   vlan_config = Dict( keyType=int, valueType=VlanField,
                       help="Details of VLAN configurations" )

   def render( self ):
      if self.counters:
         self.counters.render()
      if self.fab_params:
         self.fab_params.render()
      if self.traffic:
         self.traffic.render()
      if self.mac_addrs:
         self.mac_addrs.render()
      if self.flexfields:
         self.flexfields.render()
      if self.cdma:
         self.cdma.render()
      if self.fdev:
         self.fdev.render()
      if self.mappings:
         self.mappings.render()

      t = createTable( [ "VLAN ID", "VLAN Config", "Value" ] )
      for idx, vlan in self.vlan_config.items():
         for config, val in vlan.toDict().items():
            t.newRow( idx, config, val )
      print( t.output() )
