# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Tac
from DcbxTypes import cliSel
import TableOutput

class IeeeEtsTlv:
   def __init__( self, rawData=None ):
      self.willing = False
      self.cbs = False
      self.numTcSupported = 0
      self.cosTc = { x : 0 for x in range( 0, 8 ) }
      self.tcBw = { x : 0 for x in range( 0, 8 ) }
      self.tcAlgorithm = { x : 255 for x in range( 0, 8 ) }
      if rawData:
         self.parseRawData( rawData )

   def parseRawData( self, rawData ):
      # Moving this to the top level causes multithreading issues:
      # bitstring accesses global variables that are not allowed in the
      # Cli Agent
      # pylint: disable-next=import-outside-toplevel
      from bitstring import ConstBitStream
      dataStream = ConstBitStream( rawData )
      self.willing = dataStream.read( 'uint:1' ) == 1
      self.cbs = dataStream.read( 'uint:1' )  == 1
      dataStream.read( 'bin:3' ) # skip 3 reserved bits
      self.numTcSupported = dataStream.read( 'uint:3' )

      for x in range( 0, 8 ):
         self.cosTc[ x ] = dataStream.read( 'uint:4' )

      for x in range( 0, 8 ):
         self.tcBw[ x ] = dataStream.read( 'uint:8' )

      for x in range( 0, 8 ):
         self.tcAlgorithm[ x ] = dataStream.read( 'uint:8' )

   def dumpData( self ):
      """
      Create unknownOrgDef buffer as bytearray
      """
      data = [0] * 21

      if self.willing:
         data[0] |= 1 << 7
      if self.cbs:
         data[0] |= 1 << 6
      data[0] |= self.numTcSupported

      for i in range(4):
         data[1 + i] |= self.cosTc[2 * i] << 4
         data[1 + i] |= self.cosTc[2 * i + 1]

      for i in range(8):
         data[5 + i] |= self.tcBw[i]

      for i in range(8):
         data[13 + i] |= self.tcAlgorithm[i]

      return bytearray( data )

   def __repr__( self ):
      return repr( vars( self ) )

   def __eq__( self, other ):
      if not isinstance( other, IeeeEtsTlv ):
         # don't attempt to compare against unrelated types
         return NotImplemented
      return hash( self ) == hash( other )

   def __hash__( self ):
      return hash( ( self.willing,
                     self.cbs,
                     self.numTcSupported,
                     frozenset( self.cosTc.items() ),
                     frozenset( self.tcBw.items() ),
                     frozenset( self.tcAlgorithm.items() ) ) )

def trafficClassesSupportedInfoStr( tcsi ):
   return [ "Traffic classes supported: %d" % tcsi.trafficClassesSupported ]

def priorityFlowControlConfigInfoStr( pfcci ):
   lines = []
   if pfcci.willing:
      lines.append( "PFC configuration: willing" )
   else:
      lines.append( "PFC configuration: not willing" )
   if pfcci.macSecBypassCapability:
      lines.append( "not capable of bypassing MACsec" )
   else:
      lines.append( "capable of bypassing MACsec" )
   lines.append( "supports PFC on up to %d traffic classes" %
         pfcci.pfcCapability )
   if pfcci.pfcEnable:
      lines.append( "PFC enabled on priorities: %s" %
               " ".join(  str( x ) for x in range( 8 )
                     if ( 1 << x ) & pfcci.pfcEnable  ) )
   else:
      lines.append( "PFC not enabled on any priorities" )
   return lines

def applicationPriorityConfigInfoStr( apci ):
   lines = [ "Application priority configuration:" ]
   entryCount = len( apci.applicationPriorityEntry )
   lines.append( "%d application priorities configured%s" % (
         entryCount, ":" if entryCount else "" ) )
   for entry in apci.applicationPriorityEntry.values():
      lines.append( "  %s %d priority %d" % (
               cliSel( entry.sel ), entry.protocolId, entry.priority ) )
   return lines

def etsConfigInfoStr( priorities, bandwidth, algorithm, recommendation=True ):
   columns = [
      'Traffic Class', 'Priority', 'Bandwidth (%)', 'Algorithm'
   ]
   table = TableOutput.createTable( columns )

   justifyRight = TableOutput.Format( justify="right" )
   table.formatColumns( *[ justifyRight ] * len( columns ) )

   columnData = [
      priorities, bandwidth, algorithm
   ]

   for i in range( 8 ):
      table.newRow( *([i] + [ col[ i ] for col in columnData ]) )

   tlvName = "Recommendation" if recommendation else "Configuration"
   output = [ "ETS %s:" % tlvName ] + table.output().split('\n')

   return output

_config = Tac.newInstance( "Dcbx::Config", "tmp" )
ieeeOui = 0x80c2 # 802.1 OUI is 00-80-C2
etsConfigSubtype = _config.etsConfigSubtype
etsRecommendationSubtype = _config.etsRecommendationSubtype
pfcConfigSubtype = _config.pfcConfigSubtype
applicationPriorityConfigSubtype = _config.applicationPriorityConfigSubtype
ceeOui = 0x1b21
ceeBaseLineSubType = _config.ceeBaseLineSubType
ceeControlType = _config.ceeControlType
typeAndLenFmt = "uint:7=featureType,uint:9=dataLength"
ceeControlFmt = "uint:8=operVersion,uint:8=maxVersion,uint:32=seqNum,uint:32=ackNum"
