# Copyright (c) 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import Tac
from collections import namedtuple

DEBUG_MESSAGE_TYPE = 'Debug::MessageType'
def newDebugMessage( name, helpDesc ):
   return Tac.Value( DEBUG_MESSAGE_TYPE, name, helpDesc )

#------------------------------------------------------------------------------------
# Functions for walking the debug tree and calling a user-supplied function on each
# category.
#------------------------------------------------------------------------------------

def _walkDebugCategory( category, categoryStack, categoryFunction,
                        categoryStackPusher ):
   # Push the category onto the stack, call the user's handler function, 
   # recurse, and then pop the current category from the stack.
   categoryStackPusher( category, categoryStack )
   categoryFunction( category, categoryStack )

   for categoryName in category.subcategory:
      _walkDebugCategory( category.subcategory[ categoryName ], categoryStack,
                          categoryFunction, categoryStackPusher )

   categoryStack.pop()

def categoryNameStackPusher( category, categoryStack ):
   """Default category stack pusher. Pushes the category name onto the stack."""
   categoryStack.append( category.name )

def categoryObjectStackPusher( category, categoryStack ):
   """An alternative pusher, which pushes the category object."""
   categoryStack.append( category )

def walkDebugCliTree( categoryRoot, categoryFunction,
                      categoryStackPusher=categoryNameStackPusher,
                      processChildrenOnly=False ):
   """Walk through the tree, calling a function for each category encountered.

The function is called on the current category if it is not the config root and the
processChildrenOnly argument is False.

The category function has the prototype function( category, categoryStack ) in which
categoryStack is a list of the names of the categories up to and including the
current category.

If a custom stack is needed, to contain extra information or for other reasons,
the categoryStackPusher argument may be set to a function with the prototype
function( category, categoryStack ). It is expected that this function will just
modify the category stack, and will push one and only one item to the end of the 
list.
"""

   categoryStack = []

   # If this is the Debug::Config object, then it doesn't have a messageType 
   # attribute. If it has messageType, then it is a category and should be
   # processed with everything else if the caller hasn't specified otherwise.
   if hasattr( categoryRoot, 'messageType' ) and not processChildrenOnly:
      categoryStackPusher( categoryRoot, categoryStack )
      categoryFunction( categoryRoot, categoryStack )

   for categoryName in categoryRoot.subcategory:
      _walkDebugCategory( categoryRoot.subcategory[ categoryName ], categoryStack,
                          categoryFunction, categoryStackPusher )

#------------------------------------------------------------------------------------
# Functions for translating a message/category path string to the corresponding 
# messages or categories
#------------------------------------------------------------------------------------

def walkPathList( root, fullPathList ):
   category = root
   for categoryName in fullPathList:
      category = category.subcategory[ categoryName ]
   return category


CategoryMessagePair = namedtuple( 'CategoryMessagePair', 'category messageName' )

def walkMessagePath( root, fullPath ):
   messageName = fullPath.pop()
   category = walkPathList( root, fullPath )
   return CategoryMessagePair( category, messageName )

def walkCategoryPath( root, fullPath ):
   return walkPathList( root, fullPath )
