# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import SysdbUtil
import Tac
import json
import os

# Read dpi default config json file and load default config
def registerDpiDefaultConfig( entMan, configFile=None ):
   appRecConfig = entMan.lookup( "classification/app-recognition/config" )

   constants = Tac.Type( "Classification::ClassificationConstants" )
   defService = constants.defaultServiceName
   defServiceId = constants.defaultServiceId
   defCategory = constants.defaultCategoryName
   defCategoryId = constants.defaultCategoryId

   # Always create service "default" and category 'general' even if
   # json doesn't file has it because ipv4 apps use service 'default'
   # and category 'general'.
   serviceConfig = appRecConfig.service.newMember( defService )
   serviceConfig.serviceId = defServiceId
   catConfig = appRecConfig.category.newMember( defCategory )
   catConfig.defaultCategory = True
   catConfig.categoryId = defCategoryId

   testFile = os.getenv( "DPI_DEFAULT_CONFIG_FILE" )
   configFile = configFile if not testFile else testFile
   try:
      with open( configFile ) as f:
         config = json.load( f )
   except Exception as error: # pylint: disable=broad-except
      print( f"Error loading dpi default config file: {error}" )
      return

   # Populate default services
   for service, serviceId in config[ 'services' ].items():
      serviceConfig = appRecConfig.service.newMember( service )
      serviceConfig.serviceId = serviceId

   # Populate default categories
   for cat, catId in config[ 'categories' ].items():
      catConfig = appRecConfig.category.newMember( cat )
      catConfig.defaultCategory = True
      catConfig.categoryId = catId
   assert appRecConfig.service[ defService ].serviceId == defServiceId
   assert appRecConfig.category[ defCategory ].categoryId == defCategoryId

   # Populate default category assignments of built-in dpi apps
   for app, appData in config[ 'applications' ].items():
      appConfig = appRecConfig.app.newMember( app )
      appConfig.defaultApp = True
      appConfig.appId = appData[ 'id' ]
      if not 'service-category' in appData:
         appConfig.defaultServiceCategory[ defService ] = defCategory
         continue
      for service, category in appData[ 'service-category' ].items():
         appConfig.defaultServiceCategory[ service ] = category
      appConfig.version = 1

def Plugin( entMan ):
   SysdbUtil.addLogFacility( entMan, "CLASSIFICATION" )

   if os.getenv( "SIMULATION_CLASSIFICATION" ):
      registerDpiDefaultConfig( entMan,
         configFile="/usr/share/veos/DpiDefaultConfig.json" )
