# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re

import ArPyUtils
from EventLib import builtinEventHandlerIs

def totalSystemMemoryGb():
   # Default of 0 would force the caller to use a safe value for any system, but as
   # /proc/meminfo format is fixed, should never be used in practice.
   memoryKb = 0
   memRe = re.compile( r"MemTotal: {0,20} (\d{1,12}) kB" )
   with open( "/proc/meminfo" ) as f:
      for line in f:
         m = memRe.match( line )
         if m:
            memoryKb = int( m.groups()[ 0 ] )
   return memoryKb // 1024 // 1024

def Plugin( entMan ):
   config = entMan.lookup( 'sys/event/config' )

   # Low memory 64-bit systems can misbehave if too many concurrent actions are
   # started, so we reduce the limit from its default on these systems.
   if ArPyUtils.arch() == 64 and totalSystemMemoryGb() <= 4:
      config.concurrentAsyncActionLimit = 20

   builtinEventHandlerIs( config,
      "DropCountersHandler", {
         'triggerType': 'on-counters',
         'operstatus': False,
         'ip': False,
         'ip6': False,
         'intfName': '',
         'actionKind': 'bash',
         'command': 'DropCounterLog.py -l',
         'delay': 0,
         'repeatInterval': 0,
         'maxRepeatActionCount': 1,
         'asynchronous': False,
         'timeout': 20,
         'thresholdCount': 1,
         'threshold': 0,
         'pollInterval': 60,
         'countersCondition': 'bashCmd."DropCounterMonitor.py" > 0',
         'logRegex': '',
         'maintenanceUnitName': '',
         'maintenanceOper': '',
         'maintenanceStage': '',
         'maintenanceBgpPeer': '',
         'vrfName': '',
         'metricName': 'hardware-drops',
         'hasSubhandlers': False,
         'runUponConfig': True,
      } )

   builtinEventHandlerIs( config,
      "VoqDeleteEventHandler", {
         'triggerType': 'on-counters',
         'operstatus': False,
         'ip': False,
         'ip6': False,
         'intfName': '',
         'actionKind': 'log',
         'command': '',
         'delay': 0,
         'repeatInterval': 0,
         'maxRepeatActionCount': 1,
         'asynchronous': False,
         'timeout': 20,
         'thresholdCount': 3,
         'threshold': 900,
         'pollInterval': 30,
         'countersCondition': '*.voqDeleteEvent.delta > 0',
         'logRegex': '',
         'maintenanceUnitName': '',
         'maintenanceOper': '',
         'maintenanceStage': '',
         'maintenanceBgpPeer': '',
         'vrfName': '',
         'metricName': '',
         'hasSubhandlers': True,
         'runUponConfig': True,
      } )
