# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import errno

import Tac
import Tracing
import SslCertKey

t0 = Tracing.trace0

Constants = Tac.Type( "Mgmt::Security::Ssl::Constants" )

TlsVersionsToStrMap = { Constants.tlsv1: '1.0',
                        Constants.tlsv1_1: '1.1',
                        Constants.tlsv1_2: '1.2',
                        Constants.tlsv1_3: '1.3' }
TlsStrToVersionsMap = { version: mask
                        for mask, version in TlsVersionsToStrMap.items() }

CipherSuitesTlsV1_3 = [ "TLS_AES_256_GCM_SHA384",
                        "TLS_CHACHA20_POLY1305_SHA256",
                        "TLS_AES_128_GCM_SHA256",
                        "TLS_AES_128_CCM_8_SHA256",
                        "TLS_AES_128_CCM_SHA256" ]

def writeFile( filename, contents ):
   try:
      with open( filename, 'w' ) as fd:
         fd.write( contents )
   except IOError as e:
      t0( "Failed to write default SSL profile certs:", e.strerror )
      raise

def createDirs():
   try:
      SslCertKey.dirCreate( Constants.certsDirPath() )
      SslCertKey.dirCreate( Constants.autoCertsDirPath() )
      SslCertKey.dirCreate( Constants.keysDirPath() )
      SslCertKey.dirCreate( Constants.autoKeysDirPath() )
   except OSError as e:
      if e.errno == errno.EACCES:
         # This could happen in workspace or testing environment
         t0( "Permission denied when creating SSL certs/keys path" )
      else:
         t0( "Unknown error when creating SSL certs/keys path:", e.strerror )
      return False
   else:
      return True

def tlsVersionMaskToStrList( tlsVersion ):
   tlsVersionsEnabled = []
   for mask, version in TlsVersionsToStrMap.items():
      if tlsVersion & mask:
         tlsVersionsEnabled.append( version )
   return tlsVersionsEnabled
