# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
"""
Drop-in replacement for Artest.desiredTracing and ArosTest.desiredTracing

Before:
   from Artest import applyDesiredTracing, desiredTracingIs
   desiredTracingIs( "one" )
   applyDesiredTracing()

After:
   from DesiredTracing import applyDesiredTracing, desiredTracingIs
   desiredTracingIs( "one" )
   applyDesiredTracing()

Also, tracing is shared between all three modules, allowing gradual replacement of
ArosTest/Artest desiredTracing-related calls with those from DesiredTracing:
   import ArosTest
   import Artest
   import DesiredTracing
   ArosTest.desiredTracingIs( "foo" )
   Artest.desiredTracingIs( "bar" )

   DesiredTracing.desiredTracing => [ "foo", "bar" ]
"""

import os
import traceback
from typing import Any

import Tracing

class SafeTracing:
   '''This context manager helps Plugin systems discern if Tracing.traceSettingIs
   calls are safe or not. The direct usage of Tracing.traceSettingIs in Plugin
   systems is generally considered unsafe as maintainers working on their plugin
   need not to be aware of other existing settings imposed by other users of the same
   system, so having this in place enforces best practices.

   During the lifetime of this context manager, calls to Tracing.traceSettingIs will
   be flagged and an exception will be raised.
   Guidance will be provided with the raised Exception.

   Example (please read as there are some caveats):
   import DesiredTracing
   import Plugins
   from Tracing import traceSettingIs
   with DesiredTracing.SafeTracing():
      Plugins.loadPlugins( ... )
      # This traceSettingIs call will NOT be intercepted because the symbol
      # was brought into scope BEFORE SafeTracing was created
      traceSettingIs( ... )
      from Tracing import traceSettingIs
      # This traceSettingIs call will be intercepted because the symbol was brought
      # into scope during the lifespan of SafeTracing
      traceSettingIs( ... )
   '''

   def __init__( self ) -> None:
      self.trueTraceSettingIs = Tracing.traceSettingIs

   def guardedTraceSettingIs( self, handle: str ) -> None:
      callStack = traceback.extract_stack()
      caller = callStack[ -2 ]

      # We consider applyDesiredTracing as the only safe caller of traceSettingIs due
      # to how it is implemented (later down in this file)
      # Any other callers require further auditing
      if caller[ 2 ] == 'applyDesiredTracing':
         self.trueTraceSettingIs( handle )
         return

      raise Exception( "Unsafe usage of traceSettingIs detected.\n"
         "Please call DesiredTracing.desiredTracingIs() instead.\n"
         f"{traceback.format_list( ( caller, ) )[ 0 ]}" )

   def __enter__( self ) -> None:
      Tracing.traceSettingIs = self.guardedTraceSettingIs

   def __exit__( self, exc_type: Any, exc_value: Any, err_traceback: Any ) -> None:
      Tracing.traceSettingIs = self.trueTraceSettingIs

class DesiredTracingContext:
   """
   DesiredTracingContext can be used to separate tracing from other sources.

   The other sources are:
      ArosTest.desiredTracingIs
      Artest.desiredTracingIs
      DesiredTracing.desiredTracingIs

   The default context is shared by all three sources, so each of the above call
   contributes to the same desiredTracing list.

   To have separate desiredTracing:
     context = DesiredTracingContext()
     context.desiredTracingIs( "foo" )
     context.applyDesiredTracing()
   """

   def __init__( self ) -> None:
      self.desiredTracing: list[ str ] = []

   def desiredTracingIs( self, tracespec: str ) -> None:
      self.desiredTracing.append( tracespec )

   def desiredTracingReset( self ) -> None:
      del self.desiredTracing[ : ]

   def applyDesiredTracing( self ) -> None:
      """Apply desired tracing as configured using desiredTracingIs"""
      if self.desiredTracing:
         # De-dup, because we get called multiple times.
         traceValues = set( self.desiredTracing )
         if "TRACE" in os.environ:
            traceValues |= set( os.environ[ "TRACE" ].split( "," ) )
         Tracing.traceSettingIs( ",".join( sorted( traceValues ) ) )

_context = DesiredTracingContext()
desiredTracing = _context.desiredTracing

def desiredTracingIs( tracespec: str ) -> None:
   _context.desiredTracingIs( tracespec )

def desiredTracingReset() -> None:
   _context.desiredTracingReset()

def applyDesiredTracing() -> None:
   _context.applyDesiredTracing()
