#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re
import Tac

def kernelIntfToEosIntf( intf ):
   rules = [
      ( '_', '/' ),
      ( 'et', 'Ethernet' ),
      ( 'sw', 'Switch' ),
      ( 'vlan', 'Vlan' ),
      ( 'po', 'Port-Channel' ),
      ( 'lo', 'Loopback' ),
      ( 'tu', 'Tunnel' ),
      ( 'ma', 'Management' ),
      ( 'cpu', 'Cpu' ),
      ( 'mirror', 'Mirror' ),
      ( 'dp', 'Dps' ),
   ]
   for rule in rules:
      intf = intf.replace( *rule )
   return intf

def eosIntfToKernelIntf( intf ):
   rules = [
      ( '/', '_' ),
      ( 'Ethernet', 'et' ),
      ( 'Switch', 'sw' ),
      ( 'Vlan', 'vlan' ),
      ( 'Port-Channel', 'po' ),
      ( 'Loopback', 'lo' ),
      ( 'Tunnel', 'tu' ),
      ( 'Management', 'ma' ),
      ( 'Cpu', 'cpu' ),
      ( 'Mirror', 'mirror' ),
      ( 'Dps', 'dp' ),
   ]
   for rule in rules:
      intf = intf.replace( *rule )
   return intf

def fixKernelIntf( m ):
   # replace kernel interface name by EOS interface name
   s = m.group()
   # We're passed trailing whitespace so that we can try to adjust the
   # whitespace along with the string.
   origLen = len( s )
   ret = s.rstrip()
   hadWhitespace = ( s != ret )
   ret = kernelIntfToEosIntf( ret )
   if hadWhitespace:
      if len( ret ) < origLen:
         ret += " " * ( origLen - len( ret ) )
      else:
         # We are returning a longer string, but at least give it one space.
         ret += " "
   return ret

def kernelIntfFilter( out ):
   # replace kernel intfs in output with EOS style interfaces
   out = re.sub( r'\b((?:vlan|et|sw|po|lo|tu|ma|cpu|mirror)[0-9_]+) *\b',
                 fixKernelIntf, out )
   return out

intfId = Tac.Type( 'Arnet::IntfId' )

def intfLongName( intf ):
   return intfId.fromShortName( intf )  # implicit conversion to string
                                        # due to "defaultAttribute"
