#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from EosDhcpServerLib import hasLinkLocalAddr
from EosDhcpServerLib import IFA_F_TENTATIVE
import QuickTrace
import Smash
import Tac

qv = QuickTrace.Var
qt0 = QuickTrace.trace0

addrFamilyEnum = Tac.Type( "Arnet::AddressFamily" )

class GenericDhcpServiceReactor( Tac.Notifiee ):
   notifierTypeName = '*'

   def __init__( self, config, service ):
      self.config_ = config
      self.service_ = service
      Tac.Notifiee.__init__( self, self.config_ )

   def onAttribute( self, attr, key ):
      Tac.Notifiee.onAttribute( self, attr, key )
      self.service_.sync()

# This class will react to any changes in the keyToTypeMap
# collection and calls sync() to update the kea config.
class OptionReactorBase( GenericDhcpServiceReactor ):
   notifierTypeName = '*'
   def __init__( self, config, service ):
      GenericDhcpServiceReactor.__init__( self, config, service )

   def close( self ):
      self.service_.sync()
      GenericDhcpServiceReactor.close( self )

   @Tac.handler( 'keyToTypeMap' )
   def handleKeyToTypeMap( self, *args ):
      self.service_.sync()

class PrivateOptionV4Reactor( OptionReactorBase ):
   notifierTypeName = 'DhcpServer::PrivateOptionsConfig'

class ArbitraryOptionV4Reactor( OptionReactorBase ):
   notifierTypeName = 'DhcpServer::ArbitraryOptionsConfig'

class VendorOptionV4Reactor( GenericDhcpServiceReactor ):
   notifierTypeName = 'DhcpServer::VendorOptionIpv4'

   def __init__( self, config, service ):
      GenericDhcpServiceReactor.__init__( self, config, service )
      self.service_.handleVendorOptionAdded( self.config_.vendorId )

   def close( self ):
      self.service_.handleVendorOptionRemoved( self.config_.vendorId )
      GenericDhcpServiceReactor.close( self )

   @Tac.handler( 'subOptionConfig' )
   def handleChange( self, *args ):
      pass

class ReservationMacAddrReactorBase( GenericDhcpServiceReactor ):
   notifierTypeName = '*'

   @Tac.handler( 'hostname' )
   @Tac.handler( 'ipAddr' )
   def handleChange( self, *args ):
      pass

class ReservationMacAddrIpv4Reactor( ReservationMacAddrReactorBase ):
   notifierTypeName = 'DhcpServer::ReservationMacAddrIpv4'

class ReservationMacAddrIpv6Reactor( ReservationMacAddrReactorBase ):
   notifierTypeName = 'DhcpServer::ReservationMacAddrIpv6'

class MatchCriteriaModeConfigReactorBase( GenericDhcpServiceReactor ):
   notifierTypeName = '*'

   @Tac.handler( 'l2Info' )
   @Tac.handler( 'hostMacAddress' )
   @Tac.handler( 'vendorId' )
   @Tac.handler( 'vendorClass' )
   @Tac.handler( 'circuitIdRemoteIdHexOrStr' )
   @Tac.handler( 'duid' )
   @Tac.handler( 'remoteIdHexOrStr' )
   @Tac.handler( 'matchAny' )
   def handleChange( self, *args ):
      pass

class MatchCriteriaSubModeConfigReactor( MatchCriteriaModeConfigReactorBase ):
   notifierTypeName = 'DhcpServer::MatchCriteriaSubModeConfig'

class MatchCriteriaModeConfigReactor( MatchCriteriaModeConfigReactorBase ):
   notifierTypeName = 'DhcpServer::MatchCriteriaModeConfig'

   def __init__( self, config, service ):
      MatchCriteriaModeConfigReactorBase.__init__( self, config, service )
      self.matchCriteriaSubModeConfigReactor_ = Tac.collectionChangeReactor(
         self.config_.subMatchCriteria, MatchCriteriaSubModeConfigReactor,
         reactorArgs=( self.service_, ) )

class ClientClassConfigReactorBase( GenericDhcpServiceReactor ):
   notifierTypeName = '*'

   def __init__( self, config, service ):
      GenericDhcpServiceReactor.__init__( self, config, service )
      self.handlePrivateOptionConfig()
      self.handleArbitraryOptionConfig()
      self.handleMatchCriteriaConfig()

   @Tac.handler( 'matchCriteriaModeConfig' )
   def handleMatchCriteriaConfig( self, *args ):
      if self.config_.matchCriteriaModeConfig:
         self.matchCriteriaModeConfigReactor = (
               MatchCriteriaModeConfigReactor( self.config_.matchCriteriaModeConfig,
                                               self.service_ ) )
      else:
         self.matchCriteriaModeConfigReactor = None

   @Tac.handler( 'privateOptionConfig' )
   def handlePrivateOptionConfig( self, *args ):
      raise NotImplementedError

   @Tac.handler( 'arbitraryOptionConfig' )
   def handleArbitraryOptionConfig( self, *args ):
      raise NotImplementedError

   def close( self ):
      self.service_.sync()
      GenericDhcpServiceReactor.close( self )


class ClientClassConfigV4Reactor( ClientClassConfigReactorBase ):
   notifierTypeName = 'DhcpServer::ClientClassConfig'

   @Tac.handler( 'privateOptionConfig' )
   def handlePrivateOptionConfig( self, *args ):
      if self.config_.privateOptionConfig:
         self.privateOptionConfigReactor = (
               PrivateOptionV4Reactor( self.config_.privateOptionConfig,
                                       self.service_, ) )
      else:
         self.privateOptionConfigReactor = None

   @Tac.handler( 'arbitraryOptionConfig' )
   def handleArbitraryOptionConfig( self, *args ):
      if self.config_.arbitraryOptionConfig:
         self.arbitraryOptionConfigReactor = (
               ArbitraryOptionV4Reactor( self.config_.arbitraryOptionConfig,
                                         self.service_, ) )
      else:
         self.arbitraryOptionConfigReactor = None

   @Tac.handler( 'defaultGateway' )
   @Tac.handler( 'dnsServers' )
   @Tac.handler( 'domainName' )
   @Tac.handler( 'leaseTime' )
   @Tac.handler( 'tftpServerOption66' )
   @Tac.handler( 'tftpServerOption150' )
   @Tac.handler( 'tftpBootFileName' )
   def handleChange( self, *args ):
      pass

class ClientClassConfigV6Reactor( ClientClassConfigReactorBase ):
   notifierTypeName = 'DhcpServer::ClientClass6Config'

   @Tac.handler( 'privateOptionConfig' )
   def handlePrivateOptionConfig( self, *args ):
      if self.config_.privateOptionConfig:
         self.privateOptionConfigReactor = (
               PrivateOptionV6Reactor( self.config_.privateOptionConfig,
                                       self.service_, ) )
      else:
         self.privateOptionConfigReactor = None

   @Tac.handler( 'arbitraryOptionConfig' )
   def handleArbitraryOptionConfig( self, *args ):
      if self.config_.arbitraryOptionConfig:
         self.arbitraryOptionConfigReactor = (
               ArbitraryOptionV6Reactor( self.config_.arbitraryOptionConfig,
                                         self.service_, ) )
      else:
         self.arbitraryOptionConfigReactor = None

   @Tac.handler( 'dnsServers' )
   @Tac.handler( 'domainName' )
   @Tac.handler( 'leaseTime' )
   @Tac.handler( 'tftpBootFileName' )
   def handleChange( self, *args ):
      pass

class RangeClientClassConfigReactorBase( GenericDhcpServiceReactor ):
   notifierTypeName = '*'

   @Tac.handler( 'assignedIp' )
   def handleAssignedIpChange( self, *args ):
      pass

class RangeClientClassConfigV4Reactor( RangeClientClassConfigReactorBase,
                                       ClientClassConfigV4Reactor ):
   notifierTypeName = 'DhcpServer::RangeClientClassConfig'

   def __init__( self, config, service ):
      RangeClientClassConfigReactorBase.__init__( self, config, service )
      ClientClassConfigV4Reactor.__init__( self, config, service )

class RangeClientClassConfigV6Reactor( RangeClientClassConfigReactorBase,
                                       ClientClassConfigV6Reactor ):
   notifierTypeName = 'DhcpServer::RangeClientClass6Config'

   def __init__( self, config, service ):
      RangeClientClassConfigReactorBase.__init__( self, config, service )
      ClientClassConfigV6Reactor.__init__( self, config, service )

class RangeV4ConfigReactor( GenericDhcpServiceReactor ):
   notifierTypeName = 'DhcpServer::RangeConfig'

   def __init__( self, config, service ):
      GenericDhcpServiceReactor.__init__( self, config, service )

      self.rangeClientClassReactor_ = Tac.collectionChangeReactor(
         self.config_.clientClassConfig, RangeClientClassConfigV4Reactor,
         reactorArgs=( self.service_, ) )

      self.service_.sync()

   def close( self ):
      self.service_.sync()
      GenericDhcpServiceReactor.close( self )

class SubnetV4ConfigReactor( GenericDhcpServiceReactor ):
   notifierTypeName = 'DhcpServer::SubnetConfig'

   def __init__( self, config, service ):
      GenericDhcpServiceReactor.__init__( self, config, service )
      self.service_.handleSubnetAdded( self.config_.subnetId )

      self.reservationsMacAddrReactor_ = Tac.collectionChangeReactor(
         self.config_.reservationsMacAddr, ReservationMacAddrIpv4Reactor,
         reactorArgs=( self.service_, ) )

      self.rangeConfigReactor_ = Tac.collectionChangeReactor(
         self.config_.rangeConfig, RangeV4ConfigReactor,
         reactorArgs=( self.service_, ) )

      self.subnetClientClassReactor_ = Tac.collectionChangeReactor(
         self.config_.clientClassConfig, ClientClassConfigV4Reactor,
         reactorArgs=( self.service_, ) )

   def close( self ):
      self.service_.handleSubnetRemoved( self.config_.subnetId )
      GenericDhcpServiceReactor.close( self )

   @Tac.handler( 'subnetName' )
   @Tac.handler( 'ranges' )
   @Tac.handler( 'dnsServers' )
   @Tac.handler( 'defaultGateway' )
   @Tac.handler( 'leaseTime' )
   @Tac.handler( 'tftpServerOption66' )
   @Tac.handler( 'tftpServerOption150' )
   @Tac.handler( 'tftpBootFileName' )
   @Tac.handler( 'reservationsMacAddr' )
   def handleChange( self, *args ):
      pass

class ServiceV4Reactor( GenericDhcpServiceReactor ):
   notifierTypeName = 'DhcpServer::Config'

   def __init__( self, config, service ):
      GenericDhcpServiceReactor.__init__( self, config, service )
      self.subnetConfigReactor_ = Tac.collectionChangeReactor(
         self.config_.subnetConfigIpv4, SubnetV4ConfigReactor,
         reactorArgs=( self.service_, ) )

      self.vendorOptionReactor_ = Tac.collectionChangeReactor(
         self.config_.vendorOptionIpv4, VendorOptionV4Reactor,
         reactorArgs=( self.service_, ) )

      self.clientClassConfigV4Reactor_ = Tac.collectionChangeReactor(
         self.config_.clientClassConfigIpv4, ClientClassConfigV4Reactor,
         reactorArgs=( self.service_, ) )

      self.handleGlobalPrivateOptionConfig()
      self.handleGlobalArbitraryOptionConfig()
      self.handleClearLeaseEntry()

   @Tac.handler( 'globalPrivateOptionIpv4' )
   def handleGlobalPrivateOptionConfig( self, *args ):
      if self.config_.globalPrivateOptionIpv4:
         self.globalPrivateOptionReactor = (
               PrivateOptionV4Reactor( self.config_.globalPrivateOptionIpv4,
                                       self.service_, ) )
      else:
         self.globalPrivateOptionReactor = None

   @Tac.handler( 'globalArbitraryOptionIpv4' )
   def handleGlobalArbitraryOptionConfig( self, *args ):
      if self.config_.globalArbitraryOptionIpv4:
         self.globalArbitraryOptionReactor = (
            ArbitraryOptionV4Reactor(
               self.config_.globalArbitraryOptionIpv4, self.service_ ) )
      else:
         self.globalArbitraryOptionReactor = None

   @Tac.handler( 'clearIdIpv4' )
   def handleClearConfig( self ):
      self.service_.handleClearAllLease( self.config_.clearIdIpv4 )

   @Tac.handler( 'clearLeaseIpv4' )
   def handleClearLeaseEntry( self ):
      if self.config_.clearLeaseIpv4:
         self.service_.handleClearLeaseEntry( self.config_.clearLeaseIpv4.leaseIp )

   def close( self ):
      self.subnetConfigReactor_.close()
      self.vendorOptionReactor_.close()
      self.clientClassConfigV4Reactor_.close()
      GenericDhcpServiceReactor.close( self )

# This class will react to any changes in the keyToTypeMap
# collection and calls sync() to update the kea config.
class PrivateOptionV6Reactor( OptionReactorBase ):
   notifierTypeName = 'DhcpServer::PrivateOptions6Config'

class ArbitraryOptionV6Reactor( OptionReactorBase ):
   notifierTypeName = 'DhcpServer::ArbitraryOptions6Config'

class RangeV6ConfigReactor( GenericDhcpServiceReactor ):
   notifierTypeName = 'DhcpServer::Range6Config'

   def __init__( self, config, service ):
      GenericDhcpServiceReactor.__init__( self, config, service )

      self.rangeClientClassConfigReactor_ = Tac.collectionChangeReactor(
         self.config_.clientClassConfig, RangeClientClassConfigV6Reactor,
         reactorArgs=( self.service_, ) )

      self.service_.sync()

   def close( self ):
      self.service_.sync()
      GenericDhcpServiceReactor.close( self )

class SubnetV6ConfigReactor( GenericDhcpServiceReactor ):
   notifierTypeName = 'DhcpServer::Subnet6Config'

   def __init__( self, config, service ):
      GenericDhcpServiceReactor.__init__( self, config, service )
      self.service_.handleSubnetAdded( self.config_.subnetId )

      self.reservationsMacAddrReactor_ = Tac.collectionChangeReactor(
         self.config_.reservationsMacAddr, ReservationMacAddrIpv6Reactor,
         reactorArgs=( self.service_, ) )

      self.rangeConfigReactor_ = Tac.collectionChangeReactor(
         self.config_.rangeConfig, RangeV6ConfigReactor,
         reactorArgs=( self.service_, ) )

      self.subnetClientClassConfigReactor_ = Tac.collectionChangeReactor(
         self.config_.clientClassConfig, ClientClassConfigV6Reactor,
         reactorArgs=( self.service_, ) )

   def close( self ):
      self.service_.handleSubnetRemoved( self.config_.subnetId )
      GenericDhcpServiceReactor.close( self )

   @Tac.handler( 'subnetName' )
   @Tac.handler( 'ranges' )
   @Tac.handler( 'dnsServers' )
   @Tac.handler( 'leaseTime' )
   @Tac.handler( 'tftpBootFileName' )
   @Tac.handler( 'reservationsMacAddr' )
   def handleChange( self, *args ):
      pass

class ServiceV6Reactor( GenericDhcpServiceReactor ):
   notifierTypeName = 'DhcpServer::Config'

   def __init__( self, config, service ):
      GenericDhcpServiceReactor.__init__( self, config, service )

      self.subnetConfigReactor_ = Tac.collectionChangeReactor(
         self.config_.subnetConfigIpv6, SubnetV6ConfigReactor,
         reactorArgs=( self.service_, ) )

      self.clientClassConfigV6Reactor_ = Tac.collectionChangeReactor(
         self.config_.clientClassConfigIpv6, ClientClassConfigV6Reactor,
         reactorArgs=( self.service_, ) )

      self.handleGlobalPrivateOptionConfig()
      self.handleGlobalArbitraryOptionConfig()
      self.handleClearLeaseEntry()

   @Tac.handler( 'globalPrivateOptionIpv6' )
   def handleGlobalPrivateOptionConfig( self, *args ):
      if self.config_.globalPrivateOptionIpv6:
         self.globalPrivateOptionReactor = (
               PrivateOptionV6Reactor( self.config_.globalPrivateOptionIpv6,
                                       self.service_, ) )
      else:
         self.globalPrivateOptionReactor = None

   @Tac.handler( 'globalArbitraryOptionIpv6' )
   def handleGlobalArbitraryOptionConfig( self, *args ):
      if self.config_.globalArbitraryOptionIpv6:
         self.globalArbitraryOptionReactor = (
            ArbitraryOptionV6Reactor(
               self.config_.globalArbitraryOptionIpv6, self.service_ ) )
      else:
         self.globalArbitraryOptionReactor = None

   @Tac.handler( 'clearIdIpv6' )
   def handleClearConfig( self ):
      self.service_.handleClearAllLease( self.config_.clearIdIpv6 )

   @Tac.handler( 'clearLeaseIpv6' )
   def handleClearLeaseEntry( self ):
      if self.config_.clearLeaseIpv6:
         self.service_.handleClearLeaseEntry( self.config_.clearLeaseIpv6.leaseIp )

   def close( self ):
      self.subnetConfigReactor_.close()
      self.clientClassConfigV6Reactor_.close()
      GenericDhcpServiceReactor.close( self )

class VrfManagerReactor( Tac.Notifiee ):
   notifierTypeName = 'DhcpServer::Config'

   def __init__( self, config, vrfManager ):
      self.config_ = config
      self.vrfManager_ = vrfManager
      Tac.Notifiee.__init__( self, self.config_ )

   @Tac.handler( 'disabled' )
   def handleDisable( self ):
      self.vrfManager_.handleDisable( self.config_.disabled )

   @Tac.handler( 'interfacesIpv4' )
   @Tac.handler( 'interfacesIpv6' )
   def handleChange( self, *args ):
      self.vrfManager_.handleIntfChange()

   @Tac.handler( 'debugLogPath' )
   def handleDebugLog( self ):
      self.vrfManager_.handleDebugLog( self.config_.debugLogPath )

class ManagerReactor( Tac.Notifiee ):
   notifierTypeName = 'DhcpServer::Config'

   def __init__( self, config, vrf, manager ):
      self.config_ = config
      self.vrf_ = vrf
      self.manager_ = manager
      Tac.Notifiee.__init__( self, self.config_ )
      self.handleDhcpServerMode()

   @Tac.handler( 'dhcpServerMode' )
   def handleDhcpServerMode( self ):
      mode = self.config_.dhcpServerMode
      self.manager_.handleDhcpServerMode( mode, self.vrf_ )

class VrfConfigReactor( ManagerReactor ):
   def __init__( self, config, manager ):
      vrfStr = config.name
      vrf = Tac.ValueConst( 'L3::VrfName', vrfStr )
      manager.configs_[ vrf ] = config
      manager.vrfStatusDir.vrfStatus.newMember( vrfStr )
      manager.status_[ vrf ] = manager.vrfStatusDir.vrfStatus[ vrfStr ]
      ns = vrf.nsName()
      manager.kniStatus_[ vrf ] = manager.shmemEm.doMount(
         f"kni/ns/{ns}/status", "KernelNetInfo::Status",
         Smash.mountInfo( 'keyshadow' ) )
      super().__init__( config, vrf, manager )

   def close( self ):
      self.manager_.handleDhcpServerMode( self.config_.dhcpServerModeDefault,
                                           self.vrf_ )
      del self.manager_.configs_[ self.vrf_ ]
      del self.manager_.status_[ self.vrf_ ]
      del self.manager_.vrfStatusDir.vrfStatus[ self.vrf_.value ]
      Tac.Notifiee.close( self )

class VrfConfigDirReactor( Tac.Notifiee ):
   '''
   Reacts to VrfConfigDir. Whenever a new VRF is added or removed, a corresponding
   ManagerReactor will be created or removed for the VRF.
   '''
   notifierTypeName = 'DhcpServer::VrfConfigDir'

   def __init__( self, vrfConfigDir, manager ):
      self.vrfConfigDir = vrfConfigDir
      self.manager_ = manager
      self.vrfConfigReactor_ = Tac.collectionChangeReactor(
            self.vrfConfigDir.vrfConfig, VrfConfigReactor,
            reactorArgs=( self.manager_, ) )

class StatusGlobalReactor( Tac.Notifiee ):
   notifierTypeName = 'DhcpServer::Status'

   def __init__( self, status, vrfStatusDir ):
      self.vrfStatusDir = vrfStatusDir
      self.vrf = status.name
      Tac.Notifiee.__init__( self, self.vrfStatusDir.vrfStatus[ self.vrf ] )
      self.handleGlobalFlag()

   @Tac.handler( 'ipv4ServerDisabled' )
   @Tac.handler( 'ipv6ServerDisabled' )
   @Tac.handler( 'dhcpSnoopingV4Runnable' )
   @Tac.handler( 'dhcpSnoopingV6Runnable' )
   def handleGlobalFlag( self ):
      v4Running = False
      v6Running = False
      v4Match = False
      v6Match = False
      for status in self.vrfStatusDir.vrfStatus.values():
         v4Running = v4Running or not status.ipv4ServerDisabled
         v6Running = v6Running or not status.ipv6ServerDisabled
         v4Match = v4Match or status.dhcpSnoopingV4Runnable
         v6Match = v6Match or status.dhcpSnoopingV6Runnable
      self.vrfStatusDir.ipv4ServerRunning = v4Running
      self.vrfStatusDir.ipv6ServerRunning = v6Running
      self.vrfStatusDir.dhcpSnoopingV4Runnable = v4Running and v4Match
      self.vrfStatusDir.dhcpSnoopingV6Runnable = v6Running and v6Match

class VrfStatusDirReactor( Tac.Notifiee ):
   '''
   Reacts to VrfStatusDir. Whenever a new VRF is added or removed, a corresponding
   StatusGlobalReactor will be created or removed for the VRF.
   '''
   notifierTypeName = 'DhcpServer::VrfStatusDir'

   def __init__( self, vrfStatusDir ):
      self.vrfStatusDir = vrfStatusDir
      self.vrfStatusReactor_ = Tac.collectionChangeReactor(
            self.vrfStatusDir.vrfStatus, StatusGlobalReactor,
            reactorArgs=( self.vrfStatusDir, ) )

class KniIntfReactor( Tac.Notifiee ):
   notifierTypeName = "KernelNetInfo::Root"

   def __init__( self, kniRoot, vrfManager ):
      self.kniRoot_ = kniRoot
      self.vrfManager_ = vrfManager
      Tac.Notifiee.__init__( self, self.kniRoot_ )

   @Tac.handler( 'interfaceByDeviceName' )
   def handleIntfDevName( self, key ):
      self.vrfManager_.handleIntfChange()

class KniReactor( Tac.Notifiee ):
   notifierTypeName = "KernelNetInfo::Status"

   def __init__( self, kniStatus, vrfManager ):
      self.kniStatus_ = kniStatus
      self.vrfManager_ = vrfManager
      Tac.Notifiee.__init__( self, self.kniStatus_ )

      self.kniIntfInfo = Tac.newInstance( 'KernelNetInfo::InterfaceInfo', kniStatus )
      self.vrfManager_.interfaceToKni_ = self.kniIntfInfo.root.interfaceByDeviceName
      self.kniIntfReactor_ = KniIntfReactor( self.kniIntfInfo.root, vrfManager )

      for a in self.kniStatus_.addr:
         self.handleAddr( a, doIntfChange=False )

      self.vrfManager_.handleIntfChange()

   @Tac.handler( 'addr' )
   def handleAddr( self, key, doIntfChange=True ):
      '''
      This helps us react to assigned/unassigned ip-address(es) in order to know
      if an interface is valid for kea configurations.

      @Inputs
        key (KernelNetInfo::IpKey) : reacted IP[v6] address key
        doIntfChange (Bool) : True if we want to call handleIntfChange
      '''
      qt0( "handleAddr for:", qv( key.addr ),
           "on interface:", qv( key.ifKey.ifIndex ) )

      ipState = self.kniStatus_.addr.get( key )
      if not ipState or not ipState.assigned:
         qt0( "Removing this address from the interface" )
         if key.ifKey in self.vrfManager_.interfaceToAddrs_:
            self.vrfManager_.interfaceToAddrs_[ key.ifKey ].discard( key )
            if not self.vrfManager_.interfaceToAddrs_[ key.ifKey ]:
               qt0( "No more addresses for this interface" )
               del self.vrfManager_.interfaceToAddrs_[ key.ifKey ]
      elif ipState.flags & IFA_F_TENTATIVE:
         qt0( "Waiting for DAD to finish" )
         return
      else:
         qt0( "Adding this address to the interface" )
         qt0( "ipState's flags: ", qv( ipState.flags ) )

         addrs = self.vrfManager_.interfaceToAddrs_.get( key.ifKey )
         if not addrs:
            qt0( "Starting empty ip address set" )
            addrs = set()
            self.vrfManager_.interfaceToAddrs_[ key.ifKey ] = addrs

         addrs.add( key )
         if ( key.addr.af == addrFamilyEnum.ipv6 and
              ( len( addrs ) <= 1 or not hasLinkLocalAddr( addrs ) ) ):
            qt0( "IPv6 interfaces need at least 2 valid addresses configured,"
                 " where at least 1 is a Link Local address" )
            return

      if doIntfChange:
         self.vrfManager_.handleIntfChange()

   def close( self ):
      self.kniIntfReactor_.close()
      Tac.Notifiee.close( self )

class IntfStatusLocalReactor( Tac.Notifiee ):
   notifierTypeName = "Interface::IntfStatusLocal"

   def __init__( self, intfStatusLocal, manager ):
      self.manager_ = manager
      self.intfStatusLocal_ = intfStatusLocal
      Tac.Notifiee.__init__( self, intfStatusLocal )
      self.handleNetNsName()

   @Tac.handler( 'netNsName' )
   def handleNetNsName( self ):
      self.manager_.handleIntfChange()

class IntfStatusAllReactor( Tac.Notifiee ):
   notifierTypeName = "Interface::IntfStatus"

   def __init__( self, intfStatus, manager ):
      self.manager_ = manager
      self.intfStatus_ = intfStatus
      Tac.Notifiee.__init__( self, intfStatus )
      self.handleDeviceName()

   @Tac.handler( 'deviceName' )
   def handleDeviceName( self ):
      self.manager_.handleIntfChange()

class DhcpRelayHelperConfigReactor( Tac.Notifiee ):
   notifierTypeName = "Ip::Helper::DhcpRelay::Config"

   def __init__( self, dhcpRelayHelperConfig, service ):
      self.service_ = service
      self.dhcpRelayHelperConfig_ = dhcpRelayHelperConfig
      Tac.Notifiee.__init__( self, dhcpRelayHelperConfig )
      self.handleIntfConfig( None )

   @Tac.handler( 'intfConfig' )
   def handleIntfConfig( self, _ ):
      self.service_.handleIntfChange()

   @Tac.handler( 'alwaysOn' )
   def handleAlwaysOn( self ):
      self.service_.handleIntfChange()
