#!/usr/bin/env python3
# Copyright (c) 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string
# pylint: disable=raise-missing-from

import Cell, EntityManager, Tac, Tracing
import collections, os, re, sys, struct

__defaultTraceHandle__ = Tracing.Handle("diags")
t0 = Tracing.trace0   

class DiagError( Exception ):
   def __init__(self, message, error=None):
      self.msg = message
      self.error = error
      Exception.__init__( self )

   def __str__(self):
      return "  {}{}".format( self.msg,
                              "\n  %s" % self.error if self.error is not None
                              else "" )

def getattribute( obj, attributeName ):
   """ replacement for getattr & hasattr using __getattribute___ 
       instead of getattr """
   try:
      attribute = object.__getattribute__( obj, attributeName )
   except AttributeError:
      return None
   return attribute

class EntityManagerFactory:
   """ http://code.activestate.com/recipes/66531/ """
   __shared_state = { "_entityManager" : None }

   def __init__( self ):
      self.__dict__ = self.__shared_state 

   def entityManager( self, sysname=None, sysdbsockname=None ):
      # pylint: disable-next=access-member-before-definition
      if self._entityManager is None:
         try:
            # pylint: disable-next=attribute-defined-outside-init
            self._entityManager = EntityManager.Sysdb( sysname, 
                                                       sysdbsockname=sysdbsockname )
         except EntityManager.MountError as e:
            raise DiagError( "Could not establish connection to Sysdb. " \
                                "Are you sure Sysdb.%s is running?" % sysname, e )
      return self._entityManager

   def __get__( self, instance, owner ):
      return self._entityManager

class Name:
   """ Descriptor to access an object's name.  By using a descriptor,
       we can still use object.name to retrieve the value  """
   def __get__( self, instance, owner ):
      try:
         return object.__getattribute__( instance, "_name" )
      except AttributeError:
         return None

class Parent:
   """ Descriptor to access an object's parent. """
   def __get__( self, instance, owner ):
      try:
         return object.__getattribute__( instance, "_parent" )
      except AttributeError:
         return None

def actionsAndTests( instance, owner, tag ):
   """ Get the actions and tests defined for a chip.  This can be called on 
       either an instance of a chip or the chip class itself.  If we're called 
       on the instance then we want to use object.__getattribute__ to avoid
       swizzling the chip object. """
   actions = []
   obj = instance if instance else owner
   attributes = dir( obj )
   for name in attributes:
      if name in ("actions", "tests"):  # avoid recursion
         continue
      if instance:
         attr = object.__getattribute__( instance, name )
      else:
         attr = getattr( owner, name )
      if hasattr( attr, tag ) and getattr( attr, tag ):
         actions += [ name ]
   actions.sort()
   return actions

class Actions:
   """ Descriptor for an object's actions. """
   def __get__( self, instance, owner ):
      return actionsAndTests( instance, owner, "action" )

class Tests:
   """ Descriptor for an object's test. """
   def __get__( self, instance, owner ):
      return actionsAndTests( instance, owner, "test" )

class SimulationVariables:
   """ Descriptor return dict of simulation variables if under simulation"""
   def __get__( self, instance, owner ):
      if instance is None:
         # This is a class variable (e.g. getChips). No simulation!
         return {}
      try:
         return object.__getattribute__( instance, "_simulation" )
      except AttributeError:
         if 'SIMULATION_PREFDL' in os.environ:
            simulation={}
            for line in os.environ[ 'SIMULATION_PREFDL' ].split( '\n' ):
               m = re.match( r"(\w+)\s*:\s*(.*)", line )
               if m:
                  simulation[m.group(1)] = m.group(2)
            # Append SIMULATION_CONFIG, if present
            # pylint: disable-next=eval-used
            simulation.update( eval( os.environ.get('SIMULATION_CONFIG','{}') ))
         else:
            simulation = {}
         instance._simulation = simulation
         return simulation

def action( func ):
   """ Public methods that can be accessed by diagnostics """
   func.action = True
   return func

def test( func ):
   """ Tests that can be invoked from the command line """
   func.test = True
   return func

def setup( func ):
   """ Methods that can be invoked prior to swizzling """
   func.setup = True
   return func

class DiagBase:
   def __init__( self, name, parent, description=None, sliceId=None ):
      self._name = name
      self._parent = parent
      self.description = description
      self.sliceId = None
      self._nets = {}

   simulation = SimulationVariables()
   name = Name()
   parent = Parent()
   actions = Actions()
   tests = Tests()
   entityManager = EntityManagerFactory()
   cellId = Cell.cellId()

   def __repr__( self ):
      cls = object.__getattribute__( self, "__class__" )
      try:
         magic = object.__getattribute__( self, "__magic__" )
      except AttributeError:
         magic = False
      return "<{} {}{}@0x{:x}>".format( self.name, cls.__name__, 
                                        "*" if magic else "", id( self ) )
      
   def __str__( self ):
      return self.__repr__()

class ArList( list, DiagBase ):
   """ The universal diags collection object.
       It's a list that automatically grows in size.  For elements with
       a 'name' attribute, the ArList looks like a dict; the name can be 
       used to retrieve the object """
 
   def __init__( self, name, parent ):
      DiagBase.__init__( self, name, parent )
      list.__init__( self )
      self.nameMap = {}

   def __setitem__( self, idx, data ):
      shortfall = idx - len( self )
      if shortfall >= 0:
         self.extend( [None] * (shortfall+1) )
      list.__setitem__( self, idx, data )
      try:
         name = object.__getattribute__( data, "name" )
         self.nameMap[ name ] = data
      except AttributeError:
         pass

   def __getitem__( self, idx ):
      """ fetch either by numeric index or by element name """
      if isinstance( idx, int ):
         return list.__getitem__( self, idx )
      # pylint: disable-next=consider-merging-isinstance
      elif ( isinstance( idx, bytes ) or
             isinstance( idx, str ) ):
         if idx in self.nameMap:
            return self.nameMap[ idx ]
      raise KeyError( idx )

class Component( DiagBase ):
   """ Componets can be connected to other components through nets
       (e.g. Chips, Connectors ) """

   def __init__( self, name, parent, description=None ):
      DiagBase.__init__( self, name, parent, description )

   def peers( self, busName=None ):
      if busName is None:
         p = {}
         # pylint: disable-next=unused-variable,redefined-argument-from-local
         for (busName, (busObj, address)) in self._nets:
            p[ busName ] = busObj.deviceMap() - {self}
         return p
      if busName not in self._nets:
         raise DiagError( f"Device {self} not connected to net {busName}",
                          "Known nets: " + ", ".join( list( self._nets ) ) )
      return self._nets[ busName ][ 0 ].deviceMap() - {self}

class Connector( Component ):
   """ Connector between components """

   def __init__( self, name, parent=None, description=None ):
      self._nets = {}
      self._peers = {}
      Component.__init__( self, name, parent, description )

   @setup
   def addNet( self, name, busObj ):
      self._nets[ name ] = busObj

   @setup
   def addPeer( self, busName, peerObj, peerBusName ):
      self._peers[ busName ] = (peerObj, peerBusName)

   def peer( self, busName ):
      return self._peers[ busName ]

   def net( self, busName ):
      return self._nets[ busName ]

class Chip( Component ):

   # pylint: disable-next=keyword-arg-before-vararg
   def __init__( self, name, parent=None, *args, **kwargs ):
      Component.__init__( self, name, parent )
      self.__magic__ = True
      self.__magicArgs__ = args
      self.__magicKwargs__ = kwargs

   def __getattribute__( self, attr ):
      """ initialize the chip if 
          (1) it is still magical and hasn't been initialized yet, 
          (2) not a hidden attribute, 
          (3) not a diag framework attribute (e.g. 'parent', 'name'...), 
              which are accessible even if the object is still magical,
          (4) entityManager is not None. 
          (5) it not tagged as a 'setup' method, which can be used when
              assembling the components (author is stating that the 
              method won't swizzle the object), and

          (4) is a weird case that can occur if __del__ tries to
              access an attribute (c.f. Tac.notifiee); there's no point 
              in swizzling."""

      diagFrameworkAttributes = [ 'name', 'parent', 'entityManager', 
                                  'actions', 'tests',
                                  'net', 'peers', 'simulation' ]
      attribute = object.__getattribute__( self, attr ) 
      if ( object.__getattribute__( self, "__magic__" ) and                        #1
           not attr.startswith("_") and                                            #2
           attr not in diagFrameworkAttributes and                                 #3
           object.__getattribute__( self, "entityManager" ) is not None and        #4
           not (hasattr( attribute, 'setup' ) and getattr( attribute, 'setup' ))): #5

         args = object.__getattribute__( self, "__magicArgs__" )
         kwargs = object.__getattribute__( self, "__magicKwargs__" )
         del self.__magicArgs__
         del self.__magicKwargs__
         self.__magic__ = False
         
         className = object.__getattribute__( self, "__class__" ).__name__
         t0( f"[SWIZZLE -> {className}.{attr} 0x{id(self):x}]" )
         object.__getattribute__( self, "initialize" )( *args, **kwargs )
      return attribute

   def initialize( self, *args, **kwargs ):
      """ Initialize a chip instance.  This call to this method is delayed
          until just before the first attribute in the chip is accessed; this 
          reduces the startup time for the system.

          When 'initialize' is called, self.name and self.parent have 
          already been set.

          This method typically retrieves the chip's hardware configuration
          from Sysdb and saves any ahams required to access the chip. """

      assert False, "%s.initialize() not implemented." % self.name

   @classmethod
   def getChips( cls, em ):
      """ Return a list of chip objects valid on this system """
      assert False, "%s.getChips() not implemented." % cls.__name__

class Block( DiagBase ):
   """ Blocks are logical groupings of chips """
   pass # pylint: disable=unnecessary-pass

class Pca( Block ):
   """ Printed Circuit Assembly 

       The 'refdesDevices' and 'refdesPartNumbers' maps are used to
       automatically extract device parameters and reference
       designators from the board netlist.  All devices matching the
       provided list will be added to the board's refdes database.
       refdesPartNumbers uses Arista part numbers from Agile to
       uniquely identify each chip type.  This is also the 'PN'
       property in the schematic.

       However, older netlists do not include this information.  For those 
       boards, 'refdesDevices' can be used, which uses the 'DEVICE' property.
       This property is less desirable; it is always defined but is used 
       inconsistently in schematics.

       Each is a Python dict.  The key is the DEVICE or PN attribute.
       The value is either the software name for the chip (a string) or,
       in the case of multiple chips of the same type, a function that 
       takes the PROPERTIES of the chip and returns the software name.
       The latter is used for PHYs.
       """

   refdesDevices = {}
   refdesPartNumbers = {}

   refdesMfgPartNumbers = {}  # TEMPORARY  (SEE refdes/config.py)

   def __init__( self, name, parent ):
      Block.__init__(self, name, parent )

class Product( DiagBase ):
   """ A product is a collection of PCAs.  Most likely an assembly (ASY). """
   def __init__( self, name, description ):
      DiagBase.__init__( self, name, None, description=description )  # no parent!

   def connect( self, connections ):
      for (connObjA, busNameA, connObjB, busNameB) in connections:
         connObjA.addPeer( busNameA, connObjB, busNameB )
         connObjB.addPeer( busNameB, connObjA, busNameA )

#---------------------

BusDevice = collections.namedtuple( 'BusDevice', 'device bus' )
# describes a device and a particular bus connection to that device

class SmbusHal( DiagBase ):
   def __init__( self, name, parent, devAddr ):
      """ parent: Smbus object """
      DiagBase.__init__( self, name, parent )
      self.devAddr = devAddr

   def read( self, address, length=1 ):
      device, bus = self.parent.masterDevice()
      ham = device.busHam( bus )
      return ham.read( self.devAddr, address, length )

   def write( self, address, data ):
      device, bus = self.parent.masterDevice()
      ham = device.busHam( bus )
      return ham.write( self.devAddr, address, data )

class Smbus( DiagBase ):
   def __init__( self, name, parent ):
      DiagBase.__init__( self, name, parent )
      self._devices = {}
      self._connectors = []

   def hal( self, devAddr ):
      """ Create a HAL for a device on the bus """
      return SmbusHal( self.name, self, devAddr )

   def addDevice( self, chip, address ):
      self._devices[ address ] = chip

   def addMasterDevice( self, chip ):
      self.addDevice( chip, "master" )

   def addConnector( self, busName, connector ):
      self._connectors.append( (busName, connector) )

   def deviceMap( self, ignore=set() ): # pylint: disable=dangerous-default-value
      """ Return all devices attached to this bus.  Recurse across connectors 
          to sibling buses using 'ignore' to avoid cycles. """
      
      devices = {   addr: BusDevice( device, self )  for addr, device 
                  in self._devices.items()  }
      connectors = [ (busName, connectorObj) for (busName, connectorObj)
                     in self._connectors
                     if connectorObj not in ignore ]
      for busName, connectorObj in connectors:
         (remoteConnector, remoteBusName) = connectorObj.peer( busName )
         remoteBus = remoteConnector.net( remoteBusName )
         ignore.add( connectorObj )
         remote = remoteBus.deviceMap( ignore )
         devices.update( remote )

      return devices

   def masterDevice( self ):
      devices = self.deviceMap()
      if "master" in devices:
         return devices[ "master" ]
      else:
         raise DiagError( "No master device defined for Smbus %s" % self.name )

#---------------------

def addDeviceToBus( bus, device, address ):
   class Bus:
      pass
   if isinstance( device, Connector ):
      raise DiagError( "Don't use addDeviceToBus for connectors" )
   
   bus.addDevice( device, address )
   if isinstance( bus, Smbus ):
      device.smbus = Bus()
      smbus = getattribute( device, "smbus" )
      smbus.bus = bus
      smbus.address = address
      smbus.hal = bus.hal( address )
   else:
      raise DiagError( "Unknown bus type" )

def addConnectorToBus( busObj, connectorObj, busName ):
   connectorObj.addNet( busName, busObj )
   busObj.addConnector( busName, connectorObj )

#---------------------
# General helper functions
#---------------------

def strToInt( s ):
   """ strToInt() converts Binary/Hex strings to Integer. 
   It takes in a string inputs (ie. '23', '0b23', '0x23')
   and returns an integer"""
   try:                   
      i = int( s )
   except ValueError:
      if s.startswith( "0b" ) or s.startswith( "0B" ):
         i = int( s[2:], 2 )
      else:
         i = int( s, 16 )
   return i

def transformStr( s, fn ):
   result = []
   length = len( s )
   if (length==0) or ((length % 2) != 0):
      sys.exit( "ERROR: Data string of wrong length")
   start = 0
   if( s[1]=='x' or s[1]=='X' ):
      start = 2
   for n in range( start, length, 2 ):
      try:
         i = int( s[n : n+2], 16 )
      except ValueError:
         sys.exit( "Wrong data string" )
      result.append( fn( i ) )
   return result

def dataStrToBytes( s ):
   return transformStr( s, lambda x: struct.pack( '>B', x ) )

def binary( integerValue, bytes=4 ): # pylint: disable=redefined-builtin
   """ String representation of an integer in binary, similar to 'hex' 
   Turns 0xc7 into 1100 0111 The optional bytes parameter 
   indicates how many bytes wide the value is."""
   return "".join( [ ( ( integerValue>>n &1 ) and "1" or "0" ) +
                       ( ( n % 4 == 0 ) and " " or "" )
                       for n in range( bytes*8-1,-1,-1 ) ] )

def ascii( value, size ): # pylint: disable=redefined-builtin
   """ ascii() returns chr() for size = 1. For size = 2, 
   works for value greater than 256 by splitting value to 2
   7 binary bits."""
   if size == 1:
      return chr( value )
   elif size == 2:
      return chr( value & 0xff ) + chr( (value >> 8) & 0xff )
   else: 
      sys.exit( 'ERROR: Invald size for ascii()' )
