# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Acons
import socket
import PyClientBase

class DmfMockHelperException( Exception ):
   pass

class DmfMockHelper:
   def __init__( self, sysname='ar' ):
      self.agentName = 'Sysdb'
      self.sock = None
      self.console = None
      self.arPrefix = f'/{sysname}/{self.agentName}'
      self.currentTcamProfile = None

   def __enter__( self ):
      agent = Acons.findAgentByName( self.agentName )
      sockname = agent[ 'sockname' ]
      self.sock = socket.socket( socket.AF_UNIX, socket.SOCK_STREAM )
      self.sock.connect( sockname )
      self.sock.setblocking( True )
      execMode = PyClientBase.Rpc.execModeThreadPerConnection
      serviceIdChar = PyClientBase.Rpc.serviceIdCharShell
      self.sock.sendall( execMode + serviceIdChar )
      _ = self.sock.recv( 8192 )
      self.console = Acons.RemotePythonConsole( self.sock, shell=True )
      return self

   def __exit__( self, exc_type, exc_value, exc_traceback ):
      self.sock.close()
      self.sock = None
      self.console = None

   def runCmd( self, cmd ):
      assert self.console is not None
      print( f"Running cmd {cmd}" )
      out = self.console.getConsoleOutput( cmd )
      if out and "Exception raised" in out[ 0 ]:
         raise DmfMockHelperException( '\n'.join( out ) )

   def changePath( self, entityPath ):
      self.runCmd( "cd " + entityPath )

   def setupDmfSupported( self, enable=True ):
      self.changePath( f'{self.arPrefix}/tapagg/hwstatus' )
      self.runCmd( f'_.dmfSupported = {enable}' )
      self.changePath( f'{self.arPrefix}/dmf/platform/feature/capabilities' )
      self.runCmd( '_.tcamFeatureTapAggPortSupported = True' )
      self.runCmd( '_.ready = True' )

   def setupTrafficPolicies( self ):
      self.changePath( f'{self.arPrefix}/trafficPolicies/hardware/status/interface' )
      self.runCmd( '_.hwInitialized = True' )
      self.runCmd( "_.ingressTrafficPolicySupportedForIntfType.add( "
                   "Tac.Value( "
                   "'TrafficPolicy::HwStatus::IntfIdTypeIdPair', 0, 0 ) )" )

   def setupChip( self ):
      self.changePath( f'{self.arPrefix}/hardware/entmib/fixedSystem' )
      self.runCmd( "_.newChip( 1, 11000, 'SwitchAsic-0' )" )
      self.changePath( 'chip/11000' )
      self.runCmd( "_.initStatus = 'ok'" )

   def setupLag( self, entityName='veos' ):
      self.changePath( f'{self.arPrefix}/hardware/lag/input/config' )
      self.runCmd( f"_.newEntity( 'Hardware::Lag::Input::Config', '{entityName}' )" )
      self.changePath( f'{entityName}/lagGroup' )
      self.runCmd( f"_.newMember( '{entityName}' )" )
      self.changePath( entityName )
      self.runCmd( '_.maxLagIntfs = 256' )
      self.runCmd( '_.maxPortsPerLagIntf = 64' )

   def setupFlexCounter( self, entityNames ):
      for entityName in entityNames:
         self.changePath( f'{self.arPrefix}/flexCounter/featureStatusDir' )
         self.runCmd( "status = _.newEntity( "
                     f"'FlexCounters::FeatureStatus', '{entityName}' )" )
         self.runCmd( 'status.state = True' )
         self.runCmd( "status.featureId = Tac.Type( "
                     f"'FlexCounters::FeatureId' ).{entityName}" )

   def init( self ):
      self.setupDmfSupported()
      self.setupTrafficPolicies()
      self.setupChip()
      self.setupLag()
      self.setupFlexCounter( [ "TapAggIngress", "IntfTrafficPolicyIngress" ] )

   def setZtnStatus( self, enable=True ):
      self.changePath( f'{self.arPrefix}/ztn/status' )
      self.runCmd( f'_.enabled = {enable}' )

   def setTcamProfileStatus( self, profileName ):
      self.changePath( f'{self.arPrefix}/hardware/tcam/profiles/status/'
                    'tcamProfileStatusDir/tcamProfileStatus' )
      if self.currentTcamProfile:
         self.runCmd( f"del _[ '{self.currentTcamProfile}' ]" )
      self.runCmd( f"_.newMember( '{profileName}' )" )
      self.changePath( profileName )
      self.runCmd( "_.tcamProfileNameVersionStatus = "
                   "Tac.Value( 'AleTcam::TcamProfileNameVersionStatus', "
                   f"'{profileName}', 1, 'success' )" )
      self.currentTcamProfile = profileName
