# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import fcntl
import struct
from collections import namedtuple
from time import time

import Tac
import QuickTrace
import Arnet
from Arnet import arnetPkt
import Arnet.NsLib
from Arnet.NsLib import DEFAULT_NS

ETH_P_IP = 0x0800
ETH_HLEN = 14

BUFFER_SIZE = 1500
SUPPLICANT_TIMEOUT = 300 # < inactivity time to expire suppData

WebAuthNs = 'webauthNs'
WebAuthWireIntfName = 'webauth'
WebAuthTunRedirIntfName = 'webauthTunRedir'
WebAuthTunFwdIntfName = 'webauthTunFwd'

warn = QuickTrace.trace0
trace = QuickTrace.trace1
bv = QuickTrace.Var

EthType = Tac.Type( 'Arnet::EthType' )
ArnetEthHdr = Tac.Type( 'Arnet::EthHdrWrapper' )
ArnetEth8021QHdr = Tac.Type( 'Arnet::Eth8021QHdrWrapper' )
ArnetIpHdr = Tac.Type( 'Arnet::IpHdrWrapper' )
PrivateTcpPorts = Tac.Type( "Arnet::PrivateTcpPorts" )
EthHeader = namedtuple( 'EthHeader', [ 'vlan', 'src', 'dst' ] )
IpGenAddr = Tac.Type( "Arnet::IpGenAddr" )
WebAllowlist = Tac.Type( "Dot1x::WebAllowlist" )
ReverseDnsEntries = Tac.Type( 'ReverseDns::ReverseDnsEntries' )

# Structure of supplicant-specific data that is not shared between threads:
SupplicantData = namedtuple( 'SupplicantData',
                             [ 'ipLastPkt', 'reverseDnsHandlers',
                               'resolutionHandler' ] )

class Dot1xL2ForwarderSm:
   '''
   This is the main class of Dot1xWebL2Forwarder, it has all the logic and
   decision-making.

   It gets the external sysdb-mounted entities as arguments, and expects the packet
   sending methods to be set after the intialization. This helps with testing, as
   we can better use mocks and exercise all the logic without needing the actual
   networking up, and improves robustness when virtualTime is used.
   '''

   def __init__( self, scheduler, dot1xConfig, netStatus, webAgentStatus, configReq,
                 dot1xWebIpEthHeaderTable ):
      self.scheduler = scheduler
      self.dot1xConfig = dot1xConfig
      self.webAgentStatus = webAgentStatus
      self.configReq = configReq
      self.dot1xWebIpEthHeaderTable = dot1xWebIpEthHeaderTable
      # Entries in the allowlists are tracked by internal structures that are not
      # in sysdb, so we clear them here to avoid leaks.
      # Traffic from the supplicants brings back the entries that are still relevant.
      # The DNS resolution is restored by ReverseDnsCacheSm transparently, there's no
      # delay on the first packet.
      self.webAgentStatus.allowlist.clear()
      self.webAgentStatus.allowedIpMatch.clear()
      if not self.webAgentStatus.reverseDnsEntries:
         self.webAgentStatus.reverseDnsEntries = tuple()
      self.reverseDnsEntries = self.webAgentStatus.reverseDnsEntries
      resolvConf = Tac.newInstance( 'ResolvConf::ResolvConf' )
      self.reverseDnsCacheSm = Tac.newInstance(
         'ReverseDns::ReverseDnsCacheSm', resolvConf, netStatus, scheduler,
         self.reverseDnsEntries, 255 )
      self.reverseDnsEntriesHandler = \
         ReverseDnsEntriesHandler( self.reverseDnsEntries, self.webAgentStatus )
      self.allowlist = self.webAgentStatus.allowlist
      self.redirFunc = lambda _: None
      self.fwdFunc = lambda _: None
      self.sendToSupplicantFunc = lambda _: None
      self.suppData = {}
      self.suppExpiration = {}
      self.captivePortalAllowlistConfigHandler = \
         CaptivePortalAllowlistConfigHandler( self, self.dot1xConfig )
      self.captivePortalClearResolutionsHandler = \
         CaptivePortalClearResolutionsHandler( self.webAgentStatus, self.configReq,
                                               self.suppData,
                                               self.reverseDnsCacheSm )
      self.timer = Tac.ClockNotifiee( handler=self.handleSupplicantsExpirationTimer )

   def renewSupplicant( self, suppAddr ):
      suppExpiration = Tac.now() + SUPPLICANT_TIMEOUT
      self.suppExpiration[ str( suppAddr ) ] = suppExpiration
      if suppExpiration < self.timer.timeMin:
         self.timer.timeMin = suppExpiration

   def handleSupplicantsExpirationTimer( self ):
      now = Tac.now()
      nextExpiration = None
      for suppAddr, expiration in list( self.suppExpiration.items() ):
         if expiration <= now:
            trace( 'Dot1xL2Forwarder: expiring supplicant', bv( suppAddr ),
                   'expiration', bv( expiration ), 'now', bv( now ) )
            del self.webAgentStatus.allowlist[ suppAddr ]
            if suppData := self.suppData.get( suppAddr ):
               suppData.reverseDnsHandlers.clear()
               suppData.resolutionHandler.clear()
               del self.suppData[ suppAddr ]
            del self.suppExpiration[ suppAddr ]
         if nextExpiration is None or expiration < nextExpiration:
            nextExpiration = expiration
      if nextExpiration is not None:
         self.timer.timeMin = nextExpiration

   def handlePacketFromSupplicant( self, pkt ):
      '''Handles an Arnet::Pkt coming from the supplicant, called by the EthPam'''
      ethhdr = ArnetEthHdr( pkt, 0 )
      mac = Arnet.EthAddr( ethhdr.src )
      self.renewSupplicant( mac )
      if not self.dot1xConfig.captivePortalAllowlist:
         # FQDN allowlist not configured, we can short-circuit and redirect pkt
         self.sendPktToRedirector( pkt )
         return
      # FQDN allowlist is enabled, let's check it
      iphdr = ArnetIpHdr( pkt, ethhdr.ipHdrOffset )
      webAllowlist = self.webAgentStatus.allowlist.newMember( mac )
      allowed = webAllowlist.allow.get( IpGenAddr( iphdr.dst ) )
      if allowed is True:
         # IP was already resolved and is allowed
         self.fwdFunc( pkt )
      elif allowed is False:
         # IP was already resolved and is not allowed - i.e. should be redirected
         self.sendPktToRedirector( pkt )
      else: # IP is not resolved
         suppData = self.suppData.setdefault( str( mac ),
                                              SupplicantData( {}, {}, {} ) )
         ipstr = iphdr.dst
         ip = IpGenAddr( ipstr )
         suppData.ipLastPkt[ ipstr ] = pkt
         if ipstr in suppData.reverseDnsHandlers:
            return
         handler = self.reverseDnsCacheSm.resolve( ip )
         suppData.reverseDnsHandlers[ ipstr ] = handler
         suppData.resolutionHandler[ ipstr ] = SupplicantResolutionHandler(
            self, mac, handler.entry, ip )

   def sendPktForward( self, pkt ):
      self.fwdFunc( pkt )

   def sendPktToRedirector( self, pkt ):
      '''
      Store the eth header in dot1xWebIpEthHeaderTableLocked and sends the L3 (IP)
      part of the packet to the redirector.
      '''
      ethhdr = ArnetEthHdr( pkt, 0 )
      dot1q = ArnetEth8021QHdr( pkt, ETH_HLEN )
      iphdr = ArnetIpHdr( pkt, ethhdr.ipHdrOffset )
      isnew = self.dot1xWebIpEthHeaderTable.setIpEthHeader(
         iphdr.src,
         EthHeader( vlan=dot1q.tagControlVlanId,
                    src=ethhdr.src, dst=ethhdr.dst ) )
      if isnew:
         trace( 'Dot1xL2Forwarder: new MAC to redirect, frame',
               'eth src', bv( ethhdr.src ), 'dst', bv( ethhdr.dst ),
               'vlan', bv( dot1q.tagControlVlanId ),
               'ip src', bv( iphdr.src ), 'dst', bv( iphdr.dst ) )
      self.redirFunc( pkt )

   def _hostnameMatchesCaptivePortalAllowlist( self, mac, ip, hostname ):
      for allowedFqdn in self.dot1xConfig.captivePortalAllowlist:
         if allowedFqdn.startswith( '*' ):
            # Wildcard: trim it and check if the tails match:
            if hostname.endswith( allowedFqdn[ 1 : ] ):
               trace( 'Dot1xL2Forwarder: handleHostnames of supplicant',
                      bv( mac ), 'for ip', bv( ip ), 'hostname',
                      bv( hostname ), 'matched wildcarded', bv( allowedFqdn ) )
               return allowedFqdn
         elif hostname == allowedFqdn:
            # Full match
            trace( 'Dot1xL2Forwarder: handleHostnames of supplicant',
                   bv( mac ), 'for ip', bv( ip ), 'hostname',
                   bv( hostname ), 'that matched exact',
                   bv( allowedFqdn ) )
            return allowedFqdn
      trace( 'Dot1xL2Forwarder: handleHostnames of supplicant',
             bv( mac ), 'for ip', bv( ip ), 'hostname', bv( hostname ),
             'did not match any allowed mask' )
      return None

   def handleResolution( self, mac, ip ):
      suppData = self.suppData.get( str( mac ) )
      if not suppData:
         # We no longer need this resolution
         return
      resolution = suppData.reverseDnsHandlers[ str( ip ) ].entry.resolution
      if not resolution.ready:
         return
      webAllowlist = self.webAgentStatus.allowlist.newMember( mac )
      for hostname in resolution.hostname:
         allowedFqdn = self._hostnameMatchesCaptivePortalAllowlist(
            mac, ip, hostname )
         if allowedFqdn:
            webAllowlist.allow[ ip ] = True
            if ip not in self.webAgentStatus.allowedIpMatch:
               self.webAgentStatus.allowedIpMatch[ ip ] = allowedFqdn
            pkt = suppData.ipLastPkt.pop( str( ip ), None )
            if pkt:
               trace( 'Dot1xL2Forwarder: matched, forwarding popped packet' )
               self.sendPktForward( pkt )
            return
      trace( 'Dot1xL2Forwarder: handleHostnames of supplicant',
             bv( mac ), 'for ip', bv( ip ),
             'with', bv( len( resolution.hostname ) ),
             'hostnames, none matched, redirecting' )
      webAllowlist.allow[ ip ] = False
      del self.webAgentStatus.allowedIpMatch[ ip ]
      pkt = suppData.ipLastPkt.pop( str( ip ), None )
      if pkt:
         trace( 'Dot1xL2Forwarder: no match, sending popped packet to redirector' )
         self.sendPktToRedirector( pkt )

   def handlePacketFromRedirector( self, pkt ):
      '''
      Handles an L3 (IP) packet coming from the redirector via tun.
      We fetch the supplicant header corresponding to the destination IP, prepend
      it to the packet to craft the full eth packet, and send that to the supplicant.
      '''
      iphdr = ArnetIpHdr( pkt, 0 )
      if iphdr.version == 6:
         # We can't handle IPv6 yet
         return
      ethHeader = self.dot1xWebIpEthHeaderTable.getIpEthHeader( iphdr.dst )
      if not ethHeader:
         warn( 'EthPam, no ethHeader for ip dst', bv( iphdr.dst ),
               'src', bv( iphdr.src ) )
         return
      pkt.newSharedHeadData = 18
      ethhdr = ArnetEthHdr( pkt, 0 )
      # Invert src and dst as we are answering:
      ethhdr.src = ethHeader.dst
      ethhdr.dst = ethHeader.src
      ethhdr.ethType = 'ethTypeDot1Q'
      dot1q = ArnetEth8021QHdr( pkt, ETH_HLEN )
      dot1q.tagControlPriority = 0
      dot1q.tagControlCfi = False
      dot1q.tagControlVlanId = ethHeader.vlan
      dot1q.ethType = 'ethTypeIp'
      self.renewSupplicant( ethhdr.dst )
      self.sendToSupplicantFunc( pkt )

   def removeDstIp( self, dstIp ):
      '''Removes all data corresponding to dstIp from all supplicants'''
      trace( 'Dot1xL2Forwarder.removeDstIp: removing ip', bv( dstIp ) )
      del self.webAgentStatus.allowedIpMatch[ dstIp ]
      for allowlist in self.webAgentStatus.allowlist.values():
         del allowlist.allow[ dstIp ]
      for supp in self.suppData.values():
         supp.ipLastPkt.pop( str( dstIp ), None )
         supp.reverseDnsHandlers.pop( str( dstIp ), None )
         supp.resolutionHandler.pop( str( dstIp ), None )

   def configFqdnAdd( self, addFqdn ):
      '''
      Handle addition of a FQDN bypass configuration entry as done by
      "captive-portal bypass WILDCARD_FQDN"

      We handle this by re-evaluating the resolution of all the IP addresses that
      we are currently redirecting to the captive portal. This is done by calling
      self.handleResolution, which uses the existing DNS result to do the appropriate
      update.
      '''
      trace( 'Dot1xL2Forwarder.configFqdnAdd: bypass for', bv( addFqdn ),
             'added to config, re-evaluating redirected IPs' )
      evalMacIps = []
      for mac, allowlist in self.webAgentStatus.allowlist.items():
         for ip, bypass in allowlist.allow.items():
            if not bypass:
               evalMacIps.append( ( mac, ip ) )
      for mac, ip in evalMacIps:
         self.handleResolution( mac, ip )
      trace( 'Dot1xL2Forwarder.configFqdnAdd: redirected IPs re-evaluated' )

   def configFqdnRemove( self, delFqdn ):
      '''
      Handle removal of a FQDN bypass configuration entry as done by
      "no captive-portal bypass WILDCARD_FQDN"

      We handle this by removing all IPs that have resolved to names that match
      the removed mask. The resolution result is held by ReverseDnsCacheSm, so that
      when we have an immediate decision (no DNS) and status update when we get a new
      packet for the removed IP.
      '''
      trace( 'Dot1xL2Forwarder.configFqdnRemove: bypass for', bv( delFqdn ),
             'removed from config, purging entries' )
      delIps = []
      for ip, fqdn in self.webAgentStatus.allowedIpMatch.items():
         if fqdn == delFqdn:
            delIps.append( ip )
      for delIp in delIps:
         self.removeDstIp( delIp )
      trace( 'Dot1xL2Forwarder.configFqdnRemove: entries of', bv( delFqdn ),
             'purged' )

   def finish( self ):
      self.webAgentStatus.allowedIpMatch.clear()
      self.webAgentStatus.allowlist.clear()
      self.suppData.clear()
      self.reverseDnsCacheSm.prune()
      self.reverseDnsCacheSm.close()
      self.webAgentStatus.reverseDnsEntries.entry.clear()
      del self.timer

class SupplicantResolutionHandler( Tac.Notifiee ):
   '''
   Thin Tac.Notifiee class that captures changes in hostnames and calls
   handleResolution with the appropriate arguments.
   '''
   notifierTypeName = 'ReverseDns::ReverseDnsEntry'

   def __init__( self, supplicantAllowlistManager, mac, reverseDnsEntry, ip ):
      self.supplicantAllowlistManager = supplicantAllowlistManager
      self.mac = mac
      self.reverseDnsEntry = reverseDnsEntry
      self.ip = ip
      Tac.Notifiee.__init__( self, self.reverseDnsEntry )
      if self.reverseDnsEntry.resolution.ready:
         # If we have a resolution on creation, call the reactor immediately:
         self.supplicantAllowlistManager.handleResolution(
            self.mac, self.ip )

   @Tac.handler( 'resolution' )
   def handleReverseResolution( self ):
      self.supplicantAllowlistManager.handleResolution( self.mac, self.ip )

class ReverseDnsEntriesHandler( Tac.Notifiee ):
   '''
   Thin Tac.Notifiee class that captures the removal of reverse DNS entries by
   the ReverseDns* machinery, and removes them from the status command output.
   '''
   notifierTypeName = 'ReverseDns::ReverseDnsEntries'

   def __init__( self, reverseDnsEntries, webAgentStatus ):
      self.reverseDnsEntries = reverseDnsEntries
      self.webAgentStatus = webAgentStatus
      Tac.Notifiee.__init__( self, self.reverseDnsEntries )

   @Tac.handler( 'entry' )
   def handleEntry( self, ip ):
      if ip not in self.reverseDnsEntries.entry:
         trace( 'Dot1xL2Forwarder: expiring bypass of', ip )
         del self.webAgentStatus.allowedIpMatch[ ip ]

class CaptivePortalAllowlistConfigHandler( Tac.Notifiee ):
   '''
   Dot1x::Config.captivePortalAllowlist reactor that clears the matching entries when
   an FQDN is removed from the configuration.
   '''
   notifierTypeName = 'Dot1x::Config'

   def __init__( self, supplicantAllowlistManager, dot1xConfig ):
      self.supplicantAllowlistManager = supplicantAllowlistManager
      self.dot1xConfig = dot1xConfig
      Tac.Notifiee.__init__( self, self.dot1xConfig )

   @Tac.handler( 'captivePortalAllowlist' )
   def handleCaptivePortalAllowlist( self, fqdn ):
      if fqdn in self.dot1xConfig.captivePortalAllowlist:
         self.supplicantAllowlistManager.configFqdnAdd( fqdn )
      else:
         self.supplicantAllowlistManager.configFqdnRemove( fqdn )

class CaptivePortalClearResolutionsHandler( Tac.Notifiee ):
   '''Handler for the clear captive-portal resolutions command'''
   notifierTypeName = 'Dot1x::ConfigReq'

   def __init__( self, webAgentStatus, configReq, suppData, reverseDnsCacheSm ):
      self.webAgentStatus = webAgentStatus
      self.configReq = configReq
      self.suppData = suppData
      self.reverseDnsCacheSm = reverseDnsCacheSm
      Tac.Notifiee.__init__( self, self.configReq )

   @Tac.handler( 'clearCaptivePortalResolutions' )
   def handleClearCaptivePortalResolution( self ):
      trace( 'handleClearCaptivePortalResolution start' )
      self.webAgentStatus.allowedIpMatch.clear()
      self.webAgentStatus.allowlist.clear()
      self.suppData.clear()
      self.reverseDnsCacheSm.prune()
      trace( 'handleClearCaptivePortalResolution done' )

def createTun( intfName, ns=None ):
   trace( 'Creating tun', bv( intfName ), 'ns', bv( ns ) )
   TUNSETIFF = 0x400454ca
   IFF_TUN = 0x0001
   IFF_NO_PI = 0x1000
   tunFd = os.open( '/dev/net/tun', os.O_RDWR )
   ifr = struct.pack( '16sH', intfName.encode(), IFF_TUN | IFF_NO_PI )
   fcntl.ioctl( tunFd, TUNSETIFF, ifr )
   runCmd( [ 'sysctl', '-w', f'net.ipv4.conf.{intfName}.forwarding=1' ] )
   if ns and ns != DEFAULT_NS:
      runCmd( [ 'ip', 'link', 'set', intfName, 'netns', ns ] )
   return tunFd

class TunBase( Tac.Notifiee ):
   notifierTypeName = 'Tac::FileDescriptor'

   def __init__( self, tunFd ):
      self.tunFd = tunFd
      self.tunFileDesc = Tac.newInstance( 'Tac::FileDescriptor', 'tunFwd' )
      self.tunFileDesc.descriptor = self.tunFd
      self.handlePktFunc = None
      Tac.Notifiee.__init__( self, self.tunFileDesc )

   def finish( self ):
      '''Close tun device explicitly; required in cohab tests'''
      self.tunFileDesc.descriptor = -1
      os.close( self.tunFd )

   def sendPkt( self, pkt ):
      ethhdr = ArnetEthHdr( pkt, 0 )
      l3pkt = pkt.bytesValue[ ethhdr.ipHdrOffset : ]
      os.write( self.tunFd, l3pkt )

   @Tac.handler( 'readableCount' )
   def handleReadableCount( self ):
      data = os.read( self.tunFd, BUFFER_SIZE )
      if not data or not self.handlePktFunc:
         return
      pkt = arnetPkt( data )
      # pylint: disable=not-callable
      self.handlePktFunc( pkt )

class TunFwd( TunBase ):
   def __init__( self ):
      tunFd = createTun( intfName=WebAuthTunFwdIntfName )
      runCmd( [ 'ip', 'link', 'set', WebAuthTunFwdIntfName, 'up' ] )
      TunBase.__init__( self, tunFd )

class TunRedir( TunBase ):
   def __init__( self ):
      tunFd = createTun( intfName=WebAuthTunRedirIntfName, ns=WebAuthNs )
      iptablesPref = [ 'iptables', '-t', 'nat' ]
      dot1xWebHttpPort = PrivateTcpPorts.dot1xWebHttpPort
      dot1xWebHttpsPort = PrivateTcpPorts.dot1xWebHttpsPort
      webauthIp = '127.1.1.3'
      cmds = [
         [ 'ip', 'link', 'set', WebAuthTunRedirIntfName, 'up' ],
         [ 'ip', 'addr', 'add', webauthIp + '/24', 'dev',
           WebAuthTunRedirIntfName ],
         [ 'ip', 'route', 'add', 'default', 'dev', WebAuthTunRedirIntfName ],
         [ 'sysctl', '-w',
           f'net.ipv4.conf.{WebAuthTunRedirIntfName}.route_localnet=1' ],
         iptablesPref + [ '-F', 'PREROUTING' ],
         iptablesPref + [ '-A', 'PREROUTING',
                          '-i', WebAuthTunRedirIntfName, '-p', 'tcp',
                          '--dport', 'https',
                          '-j', 'DNAT',
                          '--to-destination',
                          f'{webauthIp}:{dot1xWebHttpsPort}' ],
         iptablesPref + [ '-A', 'PREROUTING',
                          '-i', WebAuthTunRedirIntfName, '-p', 'tcp',
                          '-j', 'DNAT',
                          '--to-destination',
                          f'{webauthIp}:{dot1xWebHttpPort}' ],
      ]
      for cmd in cmds:
         runCmd( cmd, ns=WebAuthNs )
      TunBase.__init__( self, tunFd )

class EthPam( Tac.Notifiee ):
   notifierTypeName = "Arnet::EthDevPam"

   def __init__( self ):
      self.ethPam = Tac.newInstance( 'Arnet::EthDevPam', WebAuthWireIntfName )
      self.ethPam.ethProtocol = ETH_P_IP
      self.ethPam.enabled = True
      self.handlePktFunc = None
      Tac.Notifiee.__init__( self, self.ethPam )

   def finish( self ):
      self.ethPam.enabled = False

   def sendPkt( self, data ):
      self.ethPam.txPkt = data

   @Tac.handler( 'readableCount' )
   def handleReadableCount( self ):
      pkt = self.ethPam.rxPkt()
      if not pkt or not self.handlePktFunc:
         return
      self.handlePktFunc( pkt )

class Dot1xL2Forwarder:
   '''
   "Main" class of Dot1xL2Forwarder

   It receives the relevant sysdb-mounted paths and instantiates the internal
   entities that interface with the OS (tuns, pams).
   '''
   def __init__( self, dot1xConfig, netStatus, webAgentStatus, configReq,
                 dot1xWebIpEthHeaderTable ):
      trace( 'Dot1xL2Forwarder initializing' )
      now = time()
      self.dot1xConfig = dot1xConfig
      self.netStatus = netStatus
      self.webAgentStatus = webAgentStatus
      self.configReq = configReq
      self.allowlist = webAgentStatus.allowlist
      self.dot1xWebIpEthHeaderTable = dot1xWebIpEthHeaderTable
      setupWebauthNs()
      scheduler = Tac.Type( "Ark::TaskSchedulerRoot" ).findOrCreateScheduler()
      self.dot1xL2ForwarderSm = Dot1xL2ForwarderSm( scheduler,
                                                    self.dot1xConfig,
                                                    self.netStatus,
                                                    self.webAgentStatus,
                                                    self.configReq,
                                                    self.dot1xWebIpEthHeaderTable )
      self.tunRedir = TunRedir()
      self.dot1xL2ForwarderSm.redirFunc = self.tunRedir.sendPkt
      self.tunFwd = TunFwd()
      self.dot1xL2ForwarderSm.fwdFunc = self.tunFwd.sendPkt
      self.ethPam = EthPam()
      self.ethPam.handlePktFunc = self.dot1xL2ForwarderSm.handlePacketFromSupplicant
      self.dot1xL2ForwarderSm.sendToSupplicantFunc = self.ethPam.sendPkt
      self.tunRedir.handlePktFunc = \
         self.dot1xL2ForwarderSm.handlePacketFromRedirector
      trace( 'Dot1xL2Forwarder initialized, took', bv( time() - now ) )

   def finish( self ):
      '''Finish some entities explicitly; helps in cohab tests'''
      trace( 'Dot1xL2Forwarder finishing' )
      self.tunRedir.finish()
      self.tunFwd.finish()
      self.ethPam.finish()
      self.dot1xL2ForwarderSm.finish()
      self.tunRedir = None
      self.tunFwd = None
      self.ethPam = None
      self.dot1xL2ForwarderSm = None
      teardownWebauthNs()
      trace( 'Dot1xL2Forwarder finished' )

def name():
   ''' Call this to establish an explicit dependency on the Dot1xWeb
   agent executable, to be discovered by static analysis. '''
   return 'Dot1xL2Forwarder'

# This is used for netns setup in both EOS and stests:

def runCmd( cmd, ns=None, doRaise=True ):
   trace( 'Dot1xL2Forwarder: [', bv( ns ), '] $', bv( ' '.join( cmd ) ) )
   try:
      output = Arnet.NsLib.runMaybeInNetNs( ns, cmd, stdout=Tac.CAPTURE,
                                            stderr=Tac.CAPTURE )
      if output:
         trace( 'Dot1xL2Forwarder: >', bv( output.rstrip() ) )
   except Tac.SystemCommandError as e:
      if e.output:
         trace( 'Dot1xL2Forwarder: >', bv( e.output.rstrip() ) )
      warn( 'Dot1xL2Forwarder: ', bv( e ) )
      if doRaise:
         raise
      return False
   return True

def setupWebauthNs():
   '''
   Creates webauthNs if it doesn't exist.
   It might already be there if the agent restarts.

   The "runCmd" function gets ( cmd, ns=None ) as arguments, and returns True
   for success or False for error. cmd is a string.
   '''
   trace( 'setupWebauthNs start' )
   if not runCmd( [ 'true' ], ns=WebAuthNs, doRaise=False ):
      trace( 'Creating namespace', WebAuthNs )
      runCmd( [ 'ip', 'netns', 'add', WebAuthNs ], doRaise=False )
   else:
      trace( 'Namespace', WebAuthNs, 'exists, skipping creation' )
   trace( 'setupWebauthNs done' )

def teardownWebauthNs():
   trace( 'teardownWebauthNs start' )
   runCmd( [ 'ip', 'netns', 'delete', WebAuthNs ], doRaise=False )
   trace( 'teardownWebauthNs done' )
