# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliMatcher
import Tac
from Arnet import IpAddrFullCompiledRe, Ip6AddrCompiledRe, IpAddress
from socket import AF_INET6
from IpLibConsts import DEFAULT_VRF

def addQosDscpCommandClass( mode, setDscp, noDscp, tokenProto=None ):
   """
   Registers the 'qos dscp <dscpValue>' cli command.

   Parameters
   ----------
   mode       : cli mode
   setDscp    : call back for setting dscp - setDscp( mode, args ) args[ 'DSCP' ]
   noDscp     : call back for un-setting dscp noDscp( mode, args )
   tokenProto : str, optional prefix
                if provided, cli command - [no] <tokenProto> qos dscp <val>
                else - [no] qos dscp DSCP

   Returns
   -------
   None

   """
   cliData = { 'qos': 'Configure QoS parameters',
               'dscp': 'Set DSCP value in IP header',
               'DSCP': CliMatcher.IntegerMatcher( 0, 63,
                                             helpdesc='DSCP value between 0 and 63' )
             }

   if tokenProto:
      cliData[ 'PROTOCOL' ] = tokenProto
      cliSyntax = 'PROTOCOL qos dscp DSCP'
      cliNoOrDefaultSyntax = 'PROTOCOL qos dscp ...'
   else:
      cliSyntax = 'qos dscp DSCP'
      cliNoOrDefaultSyntax = 'qos dscp ...'

   class QosDscp( CliCommand.CliCommandClass ):
      syntax = cliSyntax
      noOrDefaultSyntax = cliNoOrDefaultSyntax
      data = cliData
      handler = setDscp
      noOrDefaultHandler = noDscp

   mode.addCommandClass( QosDscp )

def addDscpRule( ruleColl,
                 hostname, port, isSrc, vrf, tProto,
                 dscpValue, v6=False ):
   """
   Creates a dscp config rule based on the given parameters
   and adds it to the given rule collection.

   Parameters
   ----------
   ruleColl  : Rule collection to populate
   hostname  : string representation of the host name or ip address
   port      : integer, tcp or udp port
   isSrc     : bool, specifies the type of port
   vrf       : string, L3 vrf name
   tProto    : transport protocol 'tcp' or 'udp'
   dscpValue : integer, dscp value (range 0-63)
   v6        : bool, true if v6 rule else false

   Returns
   -------
   None

   """

   if not port or tProto not in ( 'tcp', 'udp' ):
      return

   ipv4Match = IpAddrFullCompiledRe.match( hostname )
   ipv6Match = Ip6AddrCompiledRe.match( hostname )

   ipAddr = IpAddress( '0.0.0.0' )
   ip6Addr = IpAddress( '::', addrFamily=AF_INET6 )
   if ipv4Match:
      try:
         ipAddr = IpAddress( hostname )
         v6 = False
      except ( ValueError, TypeError ):
         return
   elif ipv6Match and ipv6Match.group( 0 ) == hostname:
      try:
         ip6Addr = IpAddress( hostname, addrFamily=AF_INET6 )
         v6 = True
      except ( ValueError, TypeError ):
         return

   if not vrf:
      vrf = DEFAULT_VRF

   ruleConfig = Tac.newInstance( 'Mgmt::Dscp::RuleConfig',
                                 tProto, ipAddr, ip6Addr, port, isSrc, vrf,
                                 dscpValue, v6 )

   ruleColl[ ruleConfig ] = True
