#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=import-error
from EDTAccess import EDTCmd
from EDTPacketIO import EDTPacketIO

class EcnPlugin( EDTPacketIO ):
   def __init__( self, reset=False ):
      if not reset:
         super().__init__()
      self.rxIntf = None
      self.txIntf = None
      self.rxIpAddr = "1.0.0.1"
      self.txIpAddr = "2.0.0.1"
      self.dstIp = "2.0.0.2"
      self.randMac = "aa:aa:aa:aa:aa:aa"

   @EDTCmd
   def resetEcnPlugin( self ):
      '''Reset ECN plugin config'''
      self.resetDutConfig()
      self.__init__( reset=True )

   def configRxIntf( self ):
      if not self.rxIntf:
         self.rxIntf = self.allocRxIntf()
         # Configure basic routing attributes for rxIntf.
         cmds = [ f"int {self.rxIntf}", "no switchport",
                  f"ip address {self.rxIpAddr}/24", "mtu 9000" ]
         self.dutCli.configCmd( cmds )

   def configTxIntf( self ):
      if not self.txIntf:
         self.txIntf = self.allocTxIntf()
         # Configure basic routing attributes for txIntf.
         cmds = [ f"int {self.txIntf}", "no switchport",
                  f"ip address {self.txIpAddr}/24", "mtu 9000" ]
         # Configure ECN related attributes.
         # Because 64 is the minimum size of ip packets, set threshold to 63 bytes
         # to guarantee congestion.
         cmds += [ "tx-queue 4", "random-detect ecn minimum-threshold 63 bytes \
                   maximum-threshold 63 bytes", "random-detect ecn count" ]
         self.dutCli.configCmd( cmds )

   def configIpRouting( self ):
      cmds = [ "ip routing", f"arp {self.dstIp} {self.randMac} arpa" ]
      self.dutCli.configCmd( cmds )

   @EDTCmd
   def setupEcn( self ):
      '''Set up rx and tx intfs to send ECN packets'''
      self.configRxIntf()
      self.configTxIntf()
      self.configIpRouting()
      print( "rx: ", self.rxIntf, "tx: ", self.txIntf )

   @EDTCmd
   def sendEcnMarkedPkt( self ):
      '''Send an ECN marked packet through ethxmit'''
      options = self.ethxmitOptions( dstMac=self.systemMacAddr(),
                                     srcMac=self.randMac, dstIp=self.dstIp,
                                     srcIp=self.rxIpAddr )
      options += " --udp-sport=16705 --udp-dport=16705"
      # ToS is a 8 bit field in IP headers.
      # The first 6 bits is for DSCP, and the last two bits is reserved for ECN.
      # The number for tx-queue on tx interface is 4, which maps to 63 in QoS
      # DSCP-TC table. We use the last two bits 0x3 to indicate congestions in ECN,
      # hence the number 0x83.
      options += " --tos=0x83"
      self.doEthxmit( self.rxIntf, options=options )

   @EDTCmd
   def showEcnCounters( self, intf=None ):
      '''Turn on and show ECN counters on interfaces'''
      cmd = [ "hardware counter feature ecn out" ]
      self.dutCli.configCmd( cmd )
      if intf is None:
         intf = ""
      cmd = [ f"show qos interfaces {intf} ecn counters queue" ]
      self.dutCli.runCmds( cmd, showOutput=True )
