# Copyright (c) 2007-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

# The PythonImpl file contains code specific to the python implementation which can
# be deleted once C++ Etba is complete. Some of the functions/classes are used by
# both implementations (and will need to be moved later).
import os, Cell
import traceback
import Tac, Tracing, Agent, Arnet, Plugins, SharedMem, Smash
import EbraTestBridgeLib
from EbraTestBridgePythonImpl import * # pylint: disable=wildcard-import
import CEosHelper
from Ark import getPlatform
from SysConstants.if_arp_h import ARPHRD_ETHER
from SysConstants.if_ether_h import ETH_P_ALL
from SysConstants.if_ether_h import ETH_FCS_LEN
from SysConstants.if_vlan_h import VLAN_ETH_HLEN
from Toggles import EtbaDutToggleLib
import StageSysdbHelper
import EbraTestBridgeConstants
# e.g. IP_MCAST_BASE
from EbraTestBridgeConstants import * # pylint: disable-msg=wildcard-import
import QuickTrace

handle = Tracing.defaultTraceHandle()
t0 = handle.trace0
t4 = handle.trace4    # misc important events
t5 = handle.trace5    # Packet tx/rx, error level
t6 = handle.trace6    # Packet tx/rx, alert level
t7 = handle.trace7    # Packet tx/rx, normal level
t8 = handle.trace8    # Packet tx/rx, debug level
t8b = handle.trace3   # Packet tx/rx, super-debug level
t9 = handle.trace9    # MAC address table aging/flushing

qt0 = QuickTrace.trace0

ArfaModeType = Tac.Type( "Arfa::ArfaMode" )
PluginStartupStageType = Tac.Type( "Arfa::PluginStartupStage" )
MoveDetectionType = Tac.Type( "Arfa::MoveDetectionType" )
RedundancyMode = Tac.Type( 'Redundancy::RedundancyMode' )
Constants = Tac.Type( 'Arfa::Constants' )

def etbaTrace( *args ):
   t0( *args )
   qt0( *args )

# During a switchover, Etba agent will take care of platform and hardware
# agents that don't run on a namespace dut so that all switchover stages
# can be marked as complete
agentsAndStagesDict = {}
agentsAndStagesDict[ 'PlxPcie-system' ] = [ 'PCIEAcquired',
                                            'PcieConfigurable' ]
agentsAndStagesDict[ 'ForwardingAgent' ] = [ 'DmaReady',
                                             'HwSyncWaitNormal',
                                             'HwSyncWaitForSlowCards' ]
agentsAndStagesDict[ 'ElectionMgr' ] = [ 'SwitchoverReady' ]
agentsAndStagesDict[ 'Fru' ] = [ 'Fru-Plugins' ]
agentsAndStagesDict[ 'ModularSystem' ] = [ 'ModularSystem-switchover' ]
agentsAndStagesDict[ 'NorCalCard' ] = [ 'CardPowerAgents' ]
agentsAndStagesDict[ 'Pca9555' ] = [ 'CardPowerAgents' ]
agentsAndStagesDict[ 'Ucd9012-system' ] = [ 'CardPowerAgents' ]
agentsAndStagesDict[ 'MactAgent' ] = [ 'InitMacTable',
                                       'L2RibReady' ]
agentsAndStagesDict[ 'AleL3Agent' ] = [ 'PlatformLfibSync' ]

class StageProgressReactor:
   def __init__( self, progressDir, redundancyStatus ):
      self.progressDir_ = progressDir
      self.redundancyStatus_ = redundancyStatus
      self.sm_ = StageSysdbHelper.ProgressDirSm( self.progressDir_, self._handler )

   def _handler( self, instance, key ):
      t0( 'StageProgressReactor marking switchoverStage', key, 'as complete' )
      self.redundancyStatus_.switchoverStage[ key ] = True

# It's tempting to use the Tracing.traceSettingIs() API in your plugin,
# because it allows you to enable traces always - but it also
# *removes* tracing that someone else wants - so let's go through
# a few hoops to stop people from doing this.
realTraceSettingIs = Tracing.traceSettingIs

def traceSettingWrapper( trace ):
   caller = traceback.extract_stack()[ -2 ]
   # DesiredTracing.applyDesiredTracing() is the right API
   # EbraBridgeTestLib._runProductTests(), when a test fails,
   #    does some debugging tasks including setting tracing
   #    explicitly, but it sets tracing back properly.
   if caller[ 2 ] in ( 'applyDesiredTracing', '_runProductTests' ):
      return realTraceSettingIs( trace )
   raise Exception( "Don't call traceSettingIs(); use DesiredTracing instead "
                    "at:\n%s" %
         traceback.format_list( ( caller, ) )[ 0 ] )

def disableTraceSetting():
   Tracing.traceSettingIs = traceSettingWrapper

def enableTraceSetting():
   Tracing.traceSettingIs = realTraceSettingIs

def loadEbraTestBridgePlugins( arfaMode ):
   filenameFilter = None
   if 'SKIP_ETBA_PLUGINS' in os.environ:
      return

   if 'ETBATESTBRIDGE_PLUGINS' in os.environ:
      plugins = os.environ[ 'ETBATESTBRIDGE_PLUGINS' ].split( ',' )
   else:
      plugins = None

      if arfaMode == ArfaModeType.arfaOnlyMode:
         # We don't actually get here anymore, but too scared to completely remove it
         # right now (will be removed later).
         def supportedPlugin( f ):
            return False
         filenameFilter = supportedPlugin

   t0( 'loading EbraTestBridgePlugins, plugin list:', plugins )
   disableTraceSetting()
   Plugins.loadPlugins( 'EbraTestBridgePlugin', plugins=plugins,
                        context=PluginContext( arfaMode ),
                        filenameFilter=filenameFilter )
   enableTraceSetting()
   os.environ[ EbraTestBridgeConstants.pluginsLoadedEnvVarName ] = '1'

class EbraTestPortTrampoline:
   """ Allow python handlers to call into a port to send a packet.
   All packet modifications are ignored for now.
   """
   def __init__( self, port ):
      self.port_ = port

   def name( self ):
      # Method needed in MplsL3EvpnEtbaDecap.py module for a trace
      return self.port_.intfId

   @property
   def intfName_( self ):
      # Property required in decapHandlersTaccCallback() of
      # class EbraTestBridgeTrampoline
      return self.port_.intfId

   @property
   def intfStatus_( self ):
      return self.port_.intfStatus

   @property
   def intfConfig_( self ):
      return self.port_.intfConfig

   def port( self ):
      # Not part of the python API, but used to trampoline back into C++
      return self.port_

   def processFrame( self, pkt ):
      self.port_.processor.processIngressFrame( pkt, self.port_ )

   def sendFrame( self, data, srcMac, dstMac, srcPort, priority,
                  vlanId, vlanAction ):
      pkt = Tac.newInstance( "Arnet::Pkt" )
      pkt.stringValue = data
      _ = srcMac = dstMac

      assert not priority

      if srcPort is None:
         srcPort = self.intfName_

      if priority is None:
         priority = 0

      if vlanId is None:
         vlanId = Tac.Value( 'Bridging::VlanId', 1 )

      result = EbraTestBridgeLib.applyVlanChanges(
         pkt.stringValue, 0, vlanId, vlanAction )
      # The C++ port wants a pktCtx
      pkt = Tac.newInstance( "Arnet::Pkt" )
      pkt.stringValue = result
      pktCtx = Tac.Type( "Arfa::PacketContext" ).createPktWrapper( pkt )
      self.port_.sendFrame( pktCtx )

class EbraTestBridgeTrampoline:
   # Trampoline from old python API to new C++ API.
   def __init__( self, entityManager, smashEntityManager, root, etbaAgent,
                 ethIntfConfigDir, ethIntfStatusDir, inNamespace, arfaMode ):
      self.entityManager_ = entityManager
      self.smashEntityManager_ = smashEntityManager
      self.arfaRoot_ = root
      self.brStatus = etbaAgent.brStatus_
      self.brStatus_ = self.brStatus
      self.brConfig_ = etbaAgent.brConfig_
      self.bridgeMac_ = self.brConfig_.bridgeMacAddr
      self.ethIntfConfigDir_ = ethIntfConfigDir
      self.ethIntfStatusDir_ = ethIntfStatusDir
      self.inNamespace_ = inNamespace
      self.pythonPorts = {}
      self.etbaAgent_ = etbaAgent
      self.arfaMode_ = arfaMode

   def arfaMode( self ):
      return self.arfaMode_

   def inArfaMode( self ):
      return self.arfaMode_ == ArfaModeType.arfaOnlyMode

   def bridgeMac( self ):
      return self.bridgeMac_

   def vlanTagState( self, srcPortName, srcIsFabric, etherType, data, dstMacAddr,
                     highOrderVlanBits ):
      etbaPort = self.arfaRoot_.ports.port[ srcPortName ]
      pkt = Tac.newInstance( "Arnet::Pkt" )
      pkt.stringValue = data
      pktCtx = self.arfaRoot_.inputPipeline.setupPacketContext( pkt, etbaPort )
      vlanTagInfo = pktCtx.vlanInfo
      # Arfa does not support extended vlans yet
      assert highOrderVlanBits is None

      return ( vlanTagInfo.effectiveVlanId,
               None, # priority?
               False, # vlanTagNeedsUpdating?
               vlanTagInfo.isTagged,
               vlanTagInfo.routed )

   def addPort( self, port, arfaPort=None ):
      cPort = arfaPort or createCPythonPort( port )
      self.arfaRoot_.ports.port.addMember( cPort )
      self.pythonPorts[ port.intfName_ ] = port
      return cPort

   def delPort( self, portName ):
      port = self.pythonPorts.pop( portName, None )
      if port:
         port.close()

      del self.arfaRoot_.ports.port[ portName ]

   def macTableFlushPort( self, intfId ):
      plugin = self.arfaRoot_.arfaPlugins.plugin[ "CoreBridge" ]
      plugin.flushRequestInternal.hostTableFlushRequest[ intfId ] = (
            Tac.Type( 'Bridging::HostTableFlushRequest' )( vlanId=0, intf=intfId ) )

   def learnAddr( self, vlanId, macAddr, portName,
                  entryType='learnedDynamicMac', moves=1 ):
      fid = self.brConfig_.vidToFidMap.get( vlanId )
      if not fid:
         fid = vlanId
      stepPriority = Tac.Type( "Arfa::LearnBridgingStep" ).myPriority
      step = self.arfaRoot_.arfaPlugins.plugin[
         'CoreBridge' ].actionSteps.step[ stepPriority ]
      step.learnHelper.processPktValues( fid, vlanId, macAddr, portName, entryType,
                                         MoveDetectionType.moveDetectionStandard,
                                         0.0, "Python" )

   def inNamespace( self ):
      return self.inNamespace_

   def em( self ):
      return self.entityManager_

   def sEm( self ):
      return self.smashEntityManager_

   def deleteMacAddressEntry( self, vlanId, macAddr, entryType=None ):
      t9( 'Deleting MAC entry at vlan ', vlanId, ' mac ', macAddr )
      fid = self.brConfig_.vidToFidMap.get( vlanId )
      if not fid:
         fid = vlanId

      key = HostKey( fid, macAddr )
      if key in self.etbaAgent_.smashBrStatus_.smashFdbStatus:
         # It is possible that the address was already deleted by a flushing
         # operation.
         hostEntry = self.etbaAgent_.smashBrStatus_.smashFdbStatus[ key ]
         if hostEntry.entryType == 'configuredStaticMac' and \
            entryType != 'configuredStaticMac':
            t4( "Do not Flush static entry" )
         else:
            del self.etbaAgent_.smashBrStatus_.smashFdbStatus[ key ]
            # The python implemtation of rmoving from the aging table doesn't
            # make sense to me. Only delete from the aging table if we deleted
            # from the smash table
            hostKey = Tac.Value( "Bridging::HostKey", fid, macAddr )
            del self.arfaRoot_.arfaPlugins.plugin[
               'CoreBridge' ].agingTable.agingTable[ hostKey ]

   def processFrame( self, data, dc1, dc2, srcPortPython, tracePkt,
                     highOrderVlanBits=None ):
      assert not highOrderVlanBits # no support for extended vlans in the Arfa yet
      pkt = Tac.newInstance( "Arnet::Pkt" )
      pkt.stringValue = data
      port = self.arfaRoot_.ports.port[ srcPortPython.intfName_ ]
      self.arfaRoot_.packetProcessor.processIngressFrame( pkt, port )


   def destLookup( self, vlanId, macAddr ):
      assert False, "Hoping this isn't used right now as it has been converted"
      intfs = Tac.newInstance( "Arfa::InterfaceSet" )
      result = self.arfaRoot_.bridge.destLookup( vlanId, macAddr, intfs )
      return result, list( intfs.intf )

   def bridgeFrame( self, data, srcMac, dstMac, srcPort, trace ):
      """ Allow a plugin (e.g., FwdIntfEtba) to bridge a frame using the
      python API. """
      pkt = Tac.newInstance( "Arnet::Pkt" )
      pkt.stringValue = data
      # srcPort is a trampoline port, so port() is the C++ port.
      srcPort.processFrame( pkt )

   @property
   def port( self ):
      # Return a dict of trampolines for all the ports.  The use case here
      # is basically bridge.port[ 'Ethernet32' ].sendFrame()
      portDict = {}
      for ( p, v ) in self.arfaRoot_.ports.port.items():
         if p in self.pythonPorts:
            # For python ports return python object
            portDict[ p ] = self.pythonPorts[ p ]
         else:
            portDict[ p ] = EbraTestPortTrampoline( v )
      return portDict

class PortFactory:
   def __init__( self, entityManager, bridge ):
      self.entityManager_ = entityManager
      self.bridge_ = bridge
      smash = SharedMem.entityManager( entityManager.sysname(), False ).shmemDir
      self.intfCounters = EtbaIntfCounterHelper(
         smash.entity[ mountHelperTac.basePath ][ 'EtbaDut' ][ 'current' ] )
      self.kernelOperStateRoot = \
         Tac.root.newEntity( 'KernelOperState::Root', 'kernelOperStateRoot' )
      self.kernelOperStateRoot.kosm = ()
      self.kernelOperStateRoot.intfStatusDir = ( 'intfStatusDir', )
      self.kernelOperStateRoot.intfStatusLocalDir = ( 'intfStatusLocalDir', )
      self.kernelOperStateRoot.netNsCollection = ( "kns", )
      self.kernelOperStateRoot.netNsManager = (
         self.kernelOperStateRoot.netNsCollection, )
      self.kernelOperStateRoot.intfStatusOperStatusSm = (
         self.kernelOperStateRoot.intfStatusDir,
         self.kernelOperStateRoot.intfStatusLocalDir,
         self.kernelOperStateRoot.kosm,
         self.kernelOperStateRoot.netNsCollection )

   def newPort( self, intfConfig, intfStatus, intfXcvrStatus, intfStatusLocal,
                tapDevice, trapDevice ):
      em = self.entityManager_

      if not intfConfig:
         # This can happen for some types of remote ports, like Peer ports in
         # the presence of an Mlag, if Etba is restarted
         return None

      if intfStatus.tacType.fullTypeName == 'Interface::VlanIntfStatus':
         # This can happen if a startup-config is being loaded.
         return None

      if intfStatus.tacType.fullTypeName == 'Interface::EthPhyIntfStatus':
         intfId = intfStatus.intfId
         intfCounter = Tac.Value( 'Interface::EthIntfCounterSmash', intfId )
         self.intfCounters.allIntfCounters.counter.addMember( intfCounter )
         port = EbraTestPhyPort( self.bridge_, tapDevice, trapDevice, intfConfig,
                                 intfStatus, intfXcvrStatus, intfStatusLocal,
                                 self.intfCounters, self.entityManager_ )
         ethIntfCounterWriterStatus = em.entity(
               'interface/ethIntfCounter/writerStatus/EtbaDut' )
         ethIntfCounterWriterStatus.intf[ intfStatus.intfId ] = True

         self.kernelOperStateRoot.intfStatusDir.intfStatus.addMember( intfStatus )
      else:
         for ( typeName, klass ) in interfaceHandlers:
            if intfStatus.tacType.fullTypeName == typeName:
               port = klass( self.bridge_, tapDevice, trapDevice,
                             intfConfig, intfStatus )
               break
         else:
            raise Exception( "Can't add an %s to an EbraTestBridge" %
                             intfStatus.tacType.fullTypeName )
      return port

class EtbaFdbStatusReactor( Tac.Notifiee ):
   notifierTypeName = "Smash::Bridging::Status"

   def __init__( self, status, bridgeTrampoline ):
      Tac.Notifiee.__init__( self, status )
      self.bridgeTrampoline_ = bridgeTrampoline

   # Is immediate? If so, should it be a codef?
   @Tac.handler( 'smashFdbStatus' )
   def handleFdbStatus( self, key ):
      vlanId = key.fid
      macAddr = key.addr
      if key in self.notifier_.smashFdbStatus:
         val = self.notifier_.smashFdbStatus[ key ]
         portName = val.intf
         entryType = val.entryType
         for h in learningHandlers:
            # ignore result, what does it even mean?
            h( vlanId, macAddr, portName, entryType )
      else:
         for h in postAgingHandlers:
            h( self.bridgeTrampoline_, vlanId, macAddr )

class EbraBridgingConfigReactor( Tac.Notifiee ):
   notifierTypeName = "Bridging::Config"

   def __init__( self, etbaConfigReactor ):
      etbaTrace( "Creating EbraTestBridge" )
      Tac.Notifiee.__init__( self, etbaConfigReactor.etbaAgent_.brConfig_ )
      self.etbaConfigReactor_ = etbaConfigReactor

   @Tac.handler( 'bridgeMacAddr' )
   def handleBridgeMacAddr( self ):
      etbaTrace( "handleBridgeMacAddr" )
      if self.notifier_.bridgeMacAddr != '00:00:00:00:00:00':
         if self.etbaConfigReactor_.arfaMode_ == ArfaModeType.hybridMode:
            self.etbaConfigReactor_.createPorts()
         self.etbaConfigReactor_.handleComplete()
         # this should happen only once so we can clean up reactor now
         del self.etbaConfigReactor_.ebraBridgingConfigReactor

class EntityManagerReactor( Tac.Notifiee ):
   notifierTypeName = "Sysdb::EntityManager"

   def __init__( self, em, configReactor ):
      Tac.Notifiee.__init__( self, em )
      self.configReactor = configReactor

   @Tac.handler( 'locallyReadOnly' )
   def handleLocallyReadOnly( self ):
      if self.notifier_.locallyReadOnly:
         etbaTrace( "Locally read only set to true" )
         return
      self.configReactor.handleComplete()
      self.configReactor.emReactor = None

class EtbaConfigReactor( Tac.Notifiee ):
   notifierTypeName = "Bridging::Etba::Config"

   def __init__( self, etbaAgentObj, etbaConfig, etbaCliConfig, ethIntfConfigDir,
                 ethIntfStatusDir, ethIntfStatusLocalDir, allEthPhyIntfStatusDir,
                 xcvrStatusDir, entityManager, inNamespace,
                 ebraCounterConfig, ebraCounterStatus, smashBrStatus, arfaMode ):
      self.etbaAgent_ = etbaAgentObj
      self.etbaConfig_ = etbaConfig
      self.etbaCliConfig_ = etbaCliConfig
      self.ethIntfConfigDir_ = ethIntfConfigDir
      self.ethIntfStatusDir_ = ethIntfStatusDir
      self.ethIntfStatusLocalDir_ = ethIntfStatusLocalDir
      self.allEthPhyIntfStatusDir_ = allEthPhyIntfStatusDir
      self.intfXcvrStatusDir_ = xcvrStatusDir
      self.entityManager_ = entityManager
      self.inNamespace_ = inNamespace
      self.smashBrStatus_ = smashBrStatus
      self.arfaMode_ = arfaMode
      self.tapDevices = {}
      self.bridge_ = None
      self.arfaRoot_ = self.entityManager_.root().parent.entity[
         "Etba/Root" ]
      self.bridgeConfigSm_ = None
      self.fabricTapDevices_ = list() # pylint: disable=use-list-literal
      self.maxExtendedVlan_ = 0x4000
      self.numFabricTapDevices_ = self.maxExtendedVlan_ >> 12
      self.shim_ = None
      Tac.Notifiee.__init__( self, etbaConfig )
      self.hwcapabilities_ = self.entityManager_.entity( 'bridging/hwcapabilities' )
      self.routingHwStatus = self.entityManager_.entity(
            'routing/hardware/status' )
      self.routing6HwStatus = self.entityManager_.entity(
            'routing6/hardware/status' )
      # ECMP
      self.routingHwStatus.maxLogicalProtocolEcmp = 128
      self.routing6HwStatus.maxLogicalProtocolEcmp = 128

      # UCMP
      self.routingHwStatus.maxUcmp = 128
      self.routingHwStatus.ucmpSupported = True
      self.routing6HwStatus.maxUcmp = 128
      self.routing6HwStatus.ucmpSupported = True
      self.routing6HwStatus.ecmpSupported = True

      # Other ECMP properties
      # self.routingHwStatus.defaultNexthopGroupEcmp
      # self.routingHwStatus.dlbEcmpSupported
      # self.routingHwStatus.maxNexthopGroupEcmp
      # self.routingHwStatus.maxResilientEcmp
      # self.routingHwStatus.orderedEcmpSupported
      # self.routingHwStatus.resilientEcmpSupported

      self.routingHwStatus.pbrSharkCounterSupported = True

      self.noTrapDevice = False
      # Note this code is duplicated in EbraTestBridgePythonImpl:EbraTestPhyPort
      if CEosHelper.isCeos():
         if EtbaDutToggleLib.toggleCEosLabWithTrapEnabled():
            etbaTrace( 'Creating trap devices anyway in cEOS-lab' )
         else:
            etbaTrace( 'Skip creating the trapDevices in CEos' )
            self.noTrapDevice = True
      self.ebraCounterConfig_ = ebraCounterConfig
      self.ebraCounterStatus_ = ebraCounterStatus

      self.hwcapabilities_.vxlanSupported = True
      self.hwcapabilities_.vxlanRoutingSupported = True
      # Needed to unblock the CLI configuration guard in Etba for
      # Vxlan interface.
      self.hwcapabilities_.vxlanUnderlayMcastSupported = True
      self.hwcapabilities_.vxlanUnderlayIifSwitchSupported = True
      self.hwcapabilities_.vxlanMcastUnderlayHerSupported = True
      # Needed to unblock the CLI configuration guard in Etba
      # "switchport mode dot1q-tunnel"
      self.hwcapabilities_.dot1qTunnelSupported = True
      # "switchport trunk native vlan tag"
      self.hwcapabilities_.taggedNativeVlanSupported = True

      # Replicator to create EthPhyIntfStatus under the archer path
      # This SM moved here from "class Etba( Agent.Agent ):"
      # as part of review: http://reviewboard/r/240481
      self._episReplicatorSm = None

      self.ethtoolHelper = Tac.Type( "Arnet::EthtoolHelper" )

      # This has to be last because it may cause us to call into functions of
      # this instance
      self.ebraBridgingConfigReactor = EbraBridgingConfigReactor( self )
      self.ebraBridgingConfigReactor.handleBridgeMacAddr()

      self.emReactor = None

   def isPhyDevice( self, devName ):
      phyPrefix = os.environ.get( 'ETBA_PHY_PREFIX' )
      if phyPrefix is not None and devName.startswith( phyPrefix ):
         return True
      sysClassNet = "/sys/class/net"
      if os.path.isdir( os.path.join( sysClassNet, devName ) ):
         # pylint: disable-next=consider-using-with
         devType = int( open( os.path.join( sysClassNet, devName,
                                            "type" ) ).read() )
         if ( devType == ARPHRD_ETHER and
            os.path.exists( os.path.join( sysClassNet, devName, "device" ) ) ):
            return  True
      return False

   def getMtu( self, devname ):
      sysClassNet = "/sys/class/net"
      # pylint: disable-next=consider-using-with
      mtu = int( open( os.path.join( sysClassNet, devname,
                                     "mtu" ) ).read() )
      return mtu

   def createPorts( self ):
      etbaTrace( "createPorts" )

      if not self.notifier_.complete:
         etbaTrace( "createPorts returning due to not complete" )
         return

      sysname = self.entityManager_.sysname()

      etbaTrace( "Creating EbraTestBridge" )

      self.bridge_ = EbraTestBridge( sysname,
                                     self.entityManager_,
                                     self.inNamespace_,
                                     self.ethIntfConfigDir_,
                                     self.ethIntfStatusDir_,
                                     self.ethIntfStatusLocalDir_,
                                     self.allEthPhyIntfStatusDir_,
                                     self.ebraCounterConfig_,
                                     self.ebraCounterStatus_,
                                     self.maxExtendedVlan_,
                                     self.numFabricTapDevices_,
                                     self.smashBrStatus_,
                                     self.etbaAgent_.brStatus_ )

      portFactory = PortFactory( self.entityManager_, self.bridge_ )

      bridgeMac = self.entityManager_.entity( 'bridging/config' ).bridgeMacAddr

      for intfName in Arnet.sortIntf( self.ethIntfStatusDir_.intfStatus ):
         if intfName.startswith( "Management" ):
            continue

         # Sort the names so that the tap interfaces tend to come out in
         # order.  We set the MAC address to the bridge
         # MAC in case this becomes a routed port.  There's no harm
         # in having it set if it's not a routed port, since this mac
         # is never otherwise used.
         intfStatus = self.ethIntfStatusDir_.intfStatus[ intfName ]
         intfConfig = self.ethIntfConfigDir_.intfConfig.get( intfName )
         intfXcvrStatus = self.intfXcvrStatusDir_.intfXcvrStatus.get( intfName )
         if not intfXcvrStatus:
            intfXcvrStatus = \
               self.intfXcvrStatusDir_.intfXcvrStatus.newMember( intfName )
         # Look up the local EthIntfStatus, if one exists, generally
         # only true for physical management and front panel interfaces.
         intfStatusLocal = self.ethIntfStatusLocalDir_.intfStatusLocal.get(
            intfName )
         trapDevice = None
         if not self.noTrapDevice:
            trapDevFn = EbraTestBridgeLib.trapDevice

            trapDevice = trapDevFn( sysname, intfName, self.inNamespace_,
                                    hwAddr=bridgeMac )
            etbaTrace( "createPorts IntfName {} trapDevice {}".format( intfName,
                                        trapDevice.deviceName if trapDevice
                                                              else "(None)" ) )

         if trapDevice or intfName in self.etbaConfig_.extTapDevice:
            # Look up external tap device created for this interface.
            if not os.environ.get( 'ETBA_DEBUG' ):
               tapDevice = self.etbaConfig_.extTapDevice[ intfName ]
            else:
               # Setting ETBA_DEBUG=1 allows manual launch of Etba.
               # e.g. ETBA_DEBUG=1 sudo Etba
               # This creates tap interfaces inside the namespace.
               # The DUT will not be able to send/recv packets from other NS DUTs.
               # Injecting packets from the tap interfaces inside the ns,
               # e.g., into rtrmpls2-et1 after you "netns rtrmpls2".
               print( "Locally generating Tap interfaces for ", intfName )
               self.tapDevices[ intfName ] = \
                        EbraTestBridgeLib.tapDevice( sysname, intfName )
               tapDevice = Tac.newInstance( 'Bridging::Etba::Config::TapDevice' )
               tapDevice.fileno = self.tapDevices[ intfName ].fileno()
               tapDevice.name = self.tapDevices[ intfName ].name

            # We set the intf deviceName to the trap device, because that is
            # where we're going to send trapped packets.  We put the tap (not
            # trap) device name in the etbaStatus, because that is the device
            # that the test should use.
            if self.noTrapDevice:
               intfStatus.deviceName = tapDevice.name
               result = self.ethtoolHelper.disableAllOffloadsIfVeth( tapDevice.name )
               if not result:
                  print( "Error disabling offloads for", tapDevice.name )
            else:
               intfStatus.deviceName = trapDevice.deviceName
            # On vEOS Lab running on certain hypervisors jumbo frames are not
            # supported. Check the mtu setting of the physical device
            devname = tapDevice.name
            if getPlatform() == "veos" and self.isPhyDevice( devname ) :
               mtu = self.getMtu( devname )
               mtu += VLAN_ETH_HLEN + ETH_FCS_LEN
               if mtu < self.hwcapabilities_.maxFrameSize:
                  etbaTrace( "Setting max frame size lower : Device %s mtu %d" %
                      ( devname, mtu ) )
                  self.hwcapabilities_.maxFrameSize = mtu
         else:
            # virtual interface, e.g., LAG
            tapDevice = None

         port = portFactory.newPort( intfConfig, intfStatus,
                                     intfXcvrStatus, intfStatusLocal,
                                     tapDevice, trapDevice )
         if port is not None:
            self.bridge_.addPort( port )

      # Add multiple fabric (a.k.a cpu) tap interfaces as required
      for i in range( self.numFabricTapDevices_ ):
         self.fabricTapDevices_.append(
            EbraTestBridgeLib.fabricDevice(
               sysname, bridgeMac, self.inNamespace_, index=i ) )
      fabricTrapDevice = EbraTestBridgeLib.trapDevice( sysname, 'Cpu',
                                                       self.inNamespace_ )

      for dev in self.fabricTapDevices_:
         dev.ifconfig( "mtu", str( Constants.defaultMtu ) )

      port = EbraTestCpuPort(
         'Cpu', self.bridge_, self.fabricTapDevices_, fabricTrapDevice )
      self.bridge_.addPort( port )

   def _handleNewEtbaComplete( self ):
      # start a shim to populate Bridging::Status from changes in the Smash table
      # for the benefit of plugins that look at brStatus_ directly.  Once these
      # plugins are rewritten to use smash, we can eliminate this reactor and
      # the Bridging::Status altogether.
      self.shim_ = Tac.Type( 'Bridging::ReaderFdbStatusSm' )(
         self.etbaAgent_.brStatus_, self.etbaAgent_.smashBrStatus_ )

      # XXX TODO don't create a new smash entity manager here!
      trampoline = EbraTestBridgeTrampoline(
         self.entityManager_,
         SharedMem.entityManager( self.entityManager_.sysname(),
                                  self.entityManager_.isLocalEm() ),
         self.arfaRoot_, self.etbaAgent_, self.ethIntfConfigDir_,
         self.ethIntfStatusDir_, self.inNamespace_, self.arfaMode_ )


      self.etbaAgent_.fdbReactorForPythonPlugins = EtbaFdbStatusReactor(
         self.etbaAgent_.smashBrStatus_, trampoline )

      for f in bridgeInitHandlers:
         try:
            f( trampoline )
         except Exception as e:   # pylint: disable-msg=broad-except
            # Lots of noise here until we get the backwards-compatible
            # API going.
            print( f"Ignoring {e} in {f.__module__}.{f.__name__}" )

   @Tac.handler( 'complete' )
   def handleComplete( self ):
      etbaTrace( "handleComplete" )

      if not self.notifier_.complete:
         etbaTrace( "handleComplete returning due to not complete" )
         return

      isActive = self.entityManager_.redundancyStatus().mode == "active"
      if not isActive:
         etbaTrace( "handleComplete returning due to not active state" )
         return

      if self.entityManager_.locallyReadOnly():
         etbaTrace( "Entity manager is locallyReadOnly" )
         self.emReactor = EntityManagerReactor( self.entityManager_.cEntityManager(),
                                                self )
         return

      if not self.bridge_ and not self.arfaRoot_:
         etbaTrace( "handleComplete returning due to bridge not ready" )
         return

      if self.arfaMode_ == ArfaModeType.arfaOnlyMode:
         self._handleNewEtbaComplete()
      else:
         etbaStatus = self.entityManager_.entity( 'bridging/etba/status' )
         etbaStatus.initialized = False # To take care of Etba restart case

         self._episReplicatorSm = Tac.newInstance(
            "Interface::EthPhyIntfStatusDirCreatorAndReplicatorSm",
            self.entityManager_.root().entity[ "interface/status/eth/phy/slice" ],
            self.entityManager_.root().entity[
               "interface/archer/status/eth/phy/slice" ] )

         self.bridge_.createAgingNotifierReactor()
         self.bridge_.createCounterConfigReactor()
         self.bridge_.runPlugins()
         for v in self.bridge_.port.values():
            v.onActive()

         etbaStatus.baseFabricDeviceName = self.fabricTapDevices_[ 0 ].deviceName
         etbaStatus.initialized = True

# This class handles issues that arise when running on sso active and standby.
# Specifically, when running on active, there are many actions that need to be taken
# that cause a 'Write to read-only proxy' error when done on standby.
#
# This is a wrapper for the EtbaConfigReactor that has two parts. First, whether on
# active or standby, it creates the EtbaConfigRreactor object. That sets up the
# necessary state, but has been stripped of anything that writes to Sysdb. Next, only
# on active, it runs EtbaConfigReactor's handleComplete(), which sets up any state
# that writes to Sysdb.
class EtbaRedundancyStatusReactor( Tac.Notifiee ):
   notifierTypeName = "Redundancy::RedundancyStatus"

   def __init__( self, etbaAgentObj, etbaConfig, etbaCliConfig, ethIntfConfigDir,
                 ethIntfStatusDir, ethIntfStatusLocalDir, allEthPhyIntfStatusDir,
                 xcvrStatusDir, entityManager, inNamespace, redundancyStatus,
                 ebraCounterConfig, ebraCounterStatus, progressDir, smashBrStatus,
                 arfaMode ):
      Tac.Notifiee.__init__( self, redundancyStatus )
      self.etbaAgent_ = etbaAgentObj
      self.etbaConfig_ = etbaConfig
      self.etbaCliConfig_ = etbaCliConfig
      self.ethIntfConfigDir_ = ethIntfConfigDir
      self.ethIntfStatusDir_ = ethIntfStatusDir
      self.ethIntfStatusLocalDir_ = ethIntfStatusLocalDir
      self.allEthPhyIntfStatusDir_ = allEthPhyIntfStatusDir
      self.intfXcvrStatusDir_ = xcvrStatusDir
      self.entityManager_ = entityManager
      self.inNamespace_ = inNamespace
      self.protocol = redundancyStatus.protocol
      self.ebraCounterConfig_ = ebraCounterConfig
      self.ebraCounterStatus_ = ebraCounterStatus
      self.redundancyStatus_ = redundancyStatus
      self.progressDir_ = progressDir
      self.smashBrStatus_ = smashBrStatus
      self.smashEm_ = None
      self.sPR_ = None
      self.arfaMode_ = arfaMode
      self.updateReactors()

   @Tac.handler( 'mode' )
   def updateReactors( self ):
      sysMode = self.notifier_.mode
      etbaTrace( 'mode:', sysMode, 'protocol:', self.protocol )
      # If we are not on a modular system, mode will be always set to active, which
      # means that we always create the EtbaConfigReactor, which is the desired
      # behavior. We only care if we have the situation where the dut might be
      # running on standby mode, if so we don't want EtbaConfigReactor to run.

      firstTimeRun = not self.etbaAgent_.etbaConfigReactor_
      if firstTimeRun:
         etbaTrace( "creating EtbaConfigReactor" )
         self.etbaAgent_.etbaConfigReactor_ = EtbaConfigReactor( self.etbaAgent_,
                                                   self.etbaConfig_,
                                                   self.etbaCliConfig_,
                                                   self.ethIntfConfigDir_,
                                                   self.ethIntfStatusDir_,
                                                   self.ethIntfStatusLocalDir_,
                                                   self.allEthPhyIntfStatusDir_,
                                                   self.intfXcvrStatusDir_,
                                                   self.entityManager_,
                                                   self.inNamespace_,
                                                   self.ebraCounterConfig_,
                                                   self.ebraCounterStatus_,
                                                   self.smashBrStatus_,
                                                   self.arfaMode_ )

      if not firstTimeRun and sysMode == RedundancyMode.active:
         self.etbaAgent_.etbaConfigReactor_.handleComplete()
      elif sysMode == 'switchover':
         for agent, stages in agentsAndStagesDict.items():
            sysdbRoot = self.entityManager_.root()
            agentDir = Cell.path( 'stageAgentStatus/switchover/%s' % agent )
            agentStatus = sysdbRoot.entity[ agentDir ]
            for key in stages:
               t0( "EtbaRedundancyStatusReactor Marking stageAgentStatus", key,
                   "as complete" )
               switchoverStatusKey = Tac.Value( 'Stage::AgentStatusKey', agent, key,
                                                'default' )
               agentStatus.complete[ switchoverStatusKey ] = True
            self.sPR_ = StageProgressReactor( self.progressDir_,
                                              self.redundancyStatus_ )

class WaitForArfaMounts( Tac.Notifiee ):
   notifierTypeName = "Arfa::PluginStageStatus"

   def __init__( self, pythonAgent, stageStatus ):
      Tac.Notifiee.__init__( self, stageStatus )
      self.pythonAgent = pythonAgent
      self.handleStage()

   @Tac.handler( 'stage' )
   def handleStage( self ):
      stageVal = Tac.enumValue( PluginStartupStageType, self.notifier_.stage )
      stageMountsVal = Tac.enumValue( PluginStartupStageType, "stageMounts" )
      if stageVal <= stageMountsVal:
         etbaTrace( "Mounts not complete", self.notifier_.stage )
         return
      etbaTrace( "Arfa mounts complete", self.notifier_.stage )
      self.pythonAgent.doRestOfInit()

      self.pythonAgent.waitForArfaMounts = None

class Etba( Agent.Agent ):

   def __init__( self, entityManager, inNamespace=False ):
      self.inNamespace = inNamespace
      if 'NSNAME' in os.environ:
         self.inNamespace = True

      self.etbaConfigReactor_ = None
      self.redStatusReactor_ = None

      self.brConfig_ = None
      self.brStatus_ = None
      self.brVlanStatus_ = None
      self.brHwCapabilities_ = None
      self.routingHwStatusCommon_ = None
      self.routingHwStatus_ = None
      self.routing6HwStatus_ = None
      self.smashBrStatus_ = None
      self.progressDir_ = None
      self.arpSmash_ = None
      self.allVrfStatusGlobal_ = None
      self.allVrfStatusLocal_ = None
      self.vrfNameStatus_ = None
      self.l3IntfStatusDir_ = None
      self.l3Config_ = None
      self.smashEm_ = None
      self.etbaCliConfig_ = None
      self.etbaStatus_ = None
      self.arfaMode_ = None
      self.vrfSeqStatusDir_ = None # TODO this may not be needed anymore
      self.arfaAgent = None
      self.waitForArfaMounts = None
      self.redStatus_ = None

      self.cPluginCtx_ = None

      self._cpuUsageCollectorSm = None

      Agent.Agent.__init__( self, entityManager )

      Tac.activityManager.useEpoll = True

      setPythonIntoCppShimContext( PythonIntoCppShimContext() )

   def arfaMode( self ):
      return self.arfaMode_

   def inArfaMode( self ):
      return self.arfaMode_ == ArfaModeType.arfaOnlyMode

   def startArfaAgent( self ):
      cAgent = self._cAgent()
      if cAgent is None:
         cAgent = self.agentRoot_[ self.agentName ]
         assert cAgent is not None
      self.arfaAgent = Tac.newInstance( "Arfa::Agent", "Etba" )
      self.arfaAgent.entityManager = self.entityManager.cEntityManager()
      self.arfaAgent.arfaMode = self.arfaMode_
      self.arfaAgent.pythonInNamespace = True
      self.arfaAgent.inNamespace = self.inNamespace
      self.arfaAgent.doInit()
      if self.arfaMode_ == ArfaModeType.arfaOnlyMode:
         # We can't execv because we have the configuration cli command needs mounts
         # to determine what mode, so instead we just skip the rest of python init.
         print( "Skipping rest of python init as in Arfa (FastEtba) mode" )
         return
      loadEbraTestBridgePlugins( self.arfaMode_ )
      self.waitForArfaMounts = WaitForArfaMounts(
         self, self.arfaAgent.arfaStageStatus )

   def doInit( self, entityManager ):
      mountGroup = entityManager.mountGroup()
      # Need to mount the cliConfig first so we can detect which mode we are in
      # before loading plugins
      self.etbaCliConfig_ = mountGroup.mount( 'bridging/etba/cli/config',
                                              'Bridging::Etba::CliConfig', 'r' )
      self.etbaStatus_ = mountGroup.mount( 'bridging/etba/status',
                                           'Bridging::Etba::Status', 'w' )
      self.redStatus_ = mountGroup.mount( Cell.path( 'redundancy/status' ),
                                          "Redundancy::RedundancyStatus", "rf" )
      def _finish():
         # Need to set this to false ASAP in case we are changing modes via the cli.
         # This will stop the new ArfaState agent and prevent multi writer smash
         # table exception (or not depending on timing) but will eventually settle
         if self.redStatus_.mode == RedundancyMode.active:
            self.etbaStatus_.isArfaMode = False
         arfaMode = ( ArfaModeType.arfaOnlyMode
                      if EtbaDutToggleLib.toggleFastEtbaEnabled() else
                      ArfaModeType.hybridMode )
         if self.etbaCliConfig_.arfaMode != ArfaModeType.unknownMode:
            arfaMode = self.etbaCliConfig_.arfaMode
         self.arfaMode_ = arfaMode

         mode = ( "Arfa (FastEtba)" if arfaMode == ArfaModeType.arfaOnlyMode else
                 "Python Etba" )
         print( "In mode", mode, "because", arfaMode )
         self.startArfaAgent()

      mountGroup.close( _finish )

   def doRestOfInit( self ):
      arfaRoot = self.entityManager.root().parent.entity[ "Etba/Root" ]
      assert arfaRoot
      getPythonIntoCppShimContext().arfaRoot_ = arfaRoot

      entityManager = self.entityManager

      mountGroup = entityManager.mountGroup()

      mountGroup.mount( "interface/status/eth/phy/slice", "Tac::Dir", "ri" )
      mountGroup.mount( "interface/archer/status/eth/phy/slice",
                        "Tac::Dir", "wi" )

      mountGroup.mount( 'interface/counter', 'Tac::Dir', 'wi' ) # BUG6489 workaround
      mountGroup.mount( 'bridging/topology/config',
                        'Bridging::Topology::Config', 'w' )
      mountGroup.mount( 'bridging/topology/inst/etba',
                        'Bridging::Topology::Inst::Status', 'cw' )
      self.brConfig_ = mountGroup.mount( 'bridging/config',
                        'Bridging::Config', 'r' )
      mountGroup.mount( 'bridging/flush/request/cli',
                        'Bridging::HostTableFlushRequestDir', 'r' )
      mountGroup.mount( 'bridging/flush/reply/all',
                        'Bridging::HostTableFlushReplyDir', 'w' )
      self.brStatus_ = Tac.newInstance( 'Bridging::Status', 'brStatus' )
      self.brVlanStatus_ = mountGroup.mount( 'bridging/vlan/status',
                                             'Bridging::VlanStatusDir', 'r' )
      self.brHwCapabilities_ = mountGroup.mount( 'bridging/hwcapabilities',
                                                 'Bridging::HwCapabilities', 'w' )
      self.routingHwStatusCommon_ = mountGroup.mount(
         'routing/hardware/statuscommon',
         'Routing::Hardware::StatusCommon', 'w' )
      self.routingHwStatus_ = mountGroup.mount( 'routing/hardware/status',
                                                'Routing::Hardware::Status', 'w' )
      self.routing6HwStatus_ = mountGroup.mount( 'routing6/hardware/status',
                                                 'Routing6::Hardware::Status', 'w' )
      self.allVrfStatusGlobal_ = mountGroup.mount( 'ip/vrf/status/global',
                                                   'Ip::AllVrfStatusGlobal', 'r' )
      self.allVrfStatusLocal_ = mountGroup.mount( Cell.path( 'ip/vrf/status/local' ),
                                                  'Ip::AllVrfStatusLocal', 'r' )
      self.vrfNameStatus_ = mountGroup.mount( Cell.path( 'vrf/vrfNameStatus' ),
                                              'Vrf::VrfIdMap::NameToIdMapWrapper',
                                              'r' )
      self.l3IntfStatusDir_ = mountGroup.mount( 'l3/intf/status',
                                                'L3::Intf::StatusDir', 'r' )
      self.l3Config_ = mountGroup.mount( 'l3/config', 'L3::Config', 'r' )

      # This path is mounted using f flag to support switchover for RedSupDut which
      # is derived from EtbaDut.
      self.vrfSeqStatusDir_ = mountGroup.mount(
         Cell.path( "routing/sequence/vrf/status" ), "Tac::Dir", "wif" )

      ethIntfStatusLocalDir = mountGroup.mount( 
                               Cell.path( "interface/status/eth/local" ),
                                          "Interface::EthIntfStatusLocalDir", "wf" )
      mountGroup.mount( 'interface/status/eth/phy/slice',
                        'Tac::Dir', 'wi' )
      mountGroup.mount( 'interface/ethIntfCounter/writerStatus/EtbaDut',
                        'Interface::EthIntfCounterWriterStatus', 'cw' )
      mountGroup.mount( 'bridging/etba/status',
                        'Bridging::Etba::Status', 'w' )
      etbaConfig = mountGroup.mount( 'bridging/etba/config',
                                     'Bridging::Etba::Config', 'r' )
      ebraCounterConfig = mountGroup.mount(
         'bridging/etba/counter/ebra/config',
         'Bridging::Etba::RoutingHandlerCounterConfig', 'r' )
      ebraCounterStatus = mountGroup.mount(
         'bridging/etba/counter/ebra/status',
         'Bridging::Etba::RoutingHandlerCounterStatus', 'w' )
      mountGroup.mount( Cell.path( "interface/status/eth/phy/local" ),
                              "Interface::EthPhyIntfStatusLocalDir", "wf" )
      xcvrStatusDir = mountGroup.mount(
         "interface/archer/eth/xcvr/slice/%d/etba" % Cell.cellId(),
         "Interface::EthIntfXcvrStatusDirEntry", "wf" )
      mountGroup.mount(
         "interface/archer/status/eth/capabilities/input/slice/FixedSystem/xcvr",
         "Interface::Capabilities::InputCapabilitiesDir", "w" )

      # Paths for SSO completion
      for agent in agentsAndStagesDict:
         mountGroup.mount( Cell.path( 'stageAgentStatus/switchover/%s' % agent ),
                           "Stage::AgentStatus", "wcf" )
      self.progressDir_ = mountGroup.mount( Cell.path( 'stage/switchover/progress' ),
                                            "Stage::ProgressDir", "r" )
      mountGroup.mount( Cell.path( 'redundancy/status' ),
                        "Redundancy::RedundancyStatus", "rf" )

      maxNumIntfs = 128
      smashEm = SharedMem.entityManager( entityManager.sysname(),
                                          entityManager.isLocalEm() )
      self.smashEm_ = smashEm
      mountHelper = Tac.newInstance( 'Interface::EthIntfCounterMountHelper',
                                     smashEm )
      mountHelper.doMountWrite( 'EtbaDut', 'current', maxNumIntfs )

      self.smashBrStatus_ = smashEm.getEntity[ 'bridging/status' ]
      assert self.smashBrStatus_

      self.arpSmash_ = smashEm.doMount(
         'arp/status', "Arp::Table::Status", Smash.mountInfo( 'shadow' ) )

      self.createLocalEntity( "EthIntfConfigDir",
                              "Interface::EthIntfConfigDir",
                              "interface/config/eth/intf" )

      self.createLocalEntity( 'AllIntfStatusDir',
                              'Interface::AllIntfStatusDir',
                              'interface/status/all' )

      self.createLocalEntity( "EthIntfStatusDir",
                              "Interface::EthIntfStatusDir",
                              "interface/status/eth/intf" )

      self.createLocalEntity( "AllEthPhyIntfStatusDir",
                              "Interface::AllEthPhyIntfStatusDir",
                              "interface/status/eth/phy/all" )

      # IntfForwardingModel plugins for EthIntfPhy interfaces
      self.createLocalEntity( "ForwardingModelResolverState",
                              "Interface::LocalForwardingModelResolverState",
                              "interface/status/forwardingModelResolverState" )
      self.createLocalEntity( "ManagedIntfList", "Interface::IntfStatusPtrDir",
                              "interface/status/managedIntfList" )

      for f in agentInitHandlers:
         print( "Calling agentInit in", f.__module__ )
         f( entityManager )

      self.localEntitiesCreatedIs( True )

      ethIntfConfigDir = entityManager.getLocalEntity( "interface/config/eth/intf" )
      ethIntfStatusDir = entityManager.getLocalEntity( "interface/status/eth/intf" )
      allEthPhyIntfStatusDir = entityManager.getLocalEntity(
                                  "interface/status/eth/phy/all" )


      def _createRedundancyStatusReactor():
         etbaTrace( "creating EtbaRedundancyStatusReactor" )
         # EtbaRedundancyStatusReactor will react to redundancy/status.mode and
         # instantiate EtbaConfigReactor if we are in active mode
         self.redStatusReactor_ = EtbaRedundancyStatusReactor( self, etbaConfig,
                                                           self.etbaCliConfig_,
                                                           ethIntfConfigDir,
                                                           ethIntfStatusDir,
                                                           ethIntfStatusLocalDir,
                                                           allEthPhyIntfStatusDir,
                                                           xcvrStatusDir,
                                                           entityManager,
                                                           self.inNamespace,
                                                           self.redStatus_,
                                                           ebraCounterConfig,
                                                           ebraCounterStatus,
                                                           self.progressDir_,
                                                           self.smashBrStatus_,
                                                           self.arfaMode_ )

      def _finish():

         # BUG27740 - we can't guarantee that all of the inner mount group
         # finish functions have been called at this point. We have to
         # return to the activity loop before continuing. Otherwise we
         # can end up calling into functions registered by plugin that
         # tries to access state that has been mounted but not converted
         # to an Entity from an EntityFuture yet
         Tac.Deferral( _createRedundancyStatusReactor )

         self._cpuUsageCollectorSm = Tac.newInstance(
               "CpuUsage::CpuUsageCollectorSm", "Etba" )

      mountGroup.close( _finish )

def main():
   # Custom toggle overrides just for Etba in agent mode
   import ArTogglesPyAgent # pylint: disable=import-outside-toplevel
   ArTogglesPyAgent.pyParseToggleOverridesFile( '/tmp/etba-override' )

   QuickTrace.initialize( "Etba.qt" )
   container = Agent.AgentContainer( [ Etba ] )
   container.addOption( "--inNamespace", action="store_true", default=False,
                        help="Etba is running in an isolated environment",
                        agentClass=Etba )
   container.parseOptions()
   container.runAgents()
