# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Tac
import Tracing
import subprocess
import Ethernet
from Arnet import Ip6Addr
from Arnet.PktParserTestLib import parsePktStr, findHeader
from Arnet.IpTestLib import ip6AddrFromMac
from SysConstants.if_ether_h import ETH_P_ARP, ETH_P_IPV6
from SysConstants.in_h import IPPROTO_ICMPV6
from EbraTestBridgeLib import macAddrsAsStrings
import MlagMountHelper

# pkgdeps library IpLib

handle = Tracing.Handle( 'EbraTestBridge' )
t2 = handle.trace2
t8 = handle.trace8
t0 = handle.trace0

# Global mlagStatus
mlagStatusMount = None

def lookupPrimaryVlanId( bridge, vlanId ):
   vlanConfig = bridge.brConfig.vlanConfig.get( vlanId )
   if vlanConfig is not None and vlanConfig.primaryVlan != 0 and \
                                 vlanConfig.vlanType != 'normal':
      primaryVlanId = vlanConfig.primaryVlan
      primaryVlanConfig = bridge.brConfig.vlanConfig.get( primaryVlanId )
      if primaryVlanConfig is not None and primaryVlanConfig.eligibleToBePrimary():
         # This is a secondary VLAN, so see if it's mapped to the SVI for the
         # primary VLAN.
         sviMapping = bridge.brConfig.sviMapping.get( primaryVlanId )
         if sviMapping is None or sviMapping.defaultSviMapping or \
                                  vlanId in sviMapping.secondaryVlan:
            return primaryVlanId
   return vlanId

def masterOfVrmac( bridge, vlanId, macAddr ):
   if vlanId == 0:
      # can't have a running virtual router, definitely not master
      return False

   # If this is a secondary VLAN that is mapped to the primary SVI, then we use the
   # SVI corresponding to the primary VLAN.
   vlanId = lookupPrimaryVlanId( bridge, vlanId )

   intfId = bridge.intfVlanTable.vlanIdToIntfId( vlanId )
   vrmac = Tac.Value( 'Routing::Fhrp::VirtualRouterMac',
                      intfId,
                      macAddr )
   return ( vrmac in bridge.virtualRouterMacStatus_.vrrpVirtualMac or
            vrmac in bridge.virtualRouterMacStatus_.subIntfVrrpMacCount or
            macAddr == bridge.virtualRouterMacStatus_.varpVirtualMac )

# If we're the master on this vid, then include the Cpu in the floodset.
def floodsetIncludesCpu( bridge, vlanId, dstMacAddr, data ):
   if masterOfVrmac( bridge, vlanId, dstMacAddr ):
      t8( 'Fhrp mac found', vlanId, dstMacAddr )
      return True
   else:
      t8( 'Fhrp mac missed', vlanId, dstMacAddr )
      return None

def isArpPkt( data ):
   arph = None
   pkt, headers, _ = parsePktStr( data )
   dot1qHdr = findHeader( headers, "EthDot1QHdr" )
   if dot1qHdr:
      ethType = dot1qHdr.typeOrLen
      arph_offset = 18
   else:
      ethHdr = findHeader( headers, "EthHdr" )
      ethType = ethHdr.typeOrLen
      arph_offset = 14
   if ethType == ETH_P_ARP:
      arph = Tac.newInstance( "Arnet::EthArpIpHdrWrapper", pkt, arph_offset )
   return ( arph, arph_offset )

def addArpEntry( sha, spa, intf, vrf ):
   if vrf == 'default':
      cmd = [ "sudo" ]
   else:
      cmd = [ "sudo", "ip", "netns", "exec", "ns-" + vrf ]

   cmd += [ "ip", "neigh", "replace", spa, "lladdr", sha, "nud", "reachable",
         "dev", intf ]

   t8( "Fhrp adding ARP %s => %s to cache on %s in VRF %s" % ( spa, sha, intf,
      vrf ) )
   try:
      subprocess.check_output( cmd )
   except subprocess.CalledProcessError as e:
      t0( " cmd %s retCode %s output %s" % ( e.cmd, e.returncode, e.output ) )


def isIcmp6Pkt( data ):
   ( ip6h, icmp6h, payloadOff ) = ( None, None, 0 )
   pkt, headers, _ = parsePktStr( data )
   dot1qHdr = findHeader( headers, "EthDot1QHdr" )
   if dot1qHdr:
      ethType = dot1qHdr.typeOrLen
      off = 18
   else:
      ethHdr = findHeader( headers, "EthHdr" )
      ethType = ethHdr.typeOrLen
      off = 14
   if ethType == ETH_P_IPV6:
      ip6h = Tac.newInstance( "Arnet::Ip6HdrWrapper", pkt, off )
      if ip6h and ip6h.nextHeader == IPPROTO_ICMPV6:
         icmp6h = Tac.newInstance( "Arnet::Icmp6HdrWrapper", pkt, off + 40 )
         payloadOff = off + 40 + 4
   return ( pkt, ip6h, icmp6h, payloadOff )

# Put the CPU's MAC address in place of the VR MAC so the CPU routes the packet
def rewritePacketOnTheWayToTheCpu( bridge, routed, vlanId, dstMacAddr, data,
                                   srcPortName ):
   if vlanId:
      evid = Tac.Value( 'Bridging::ExtendedVlanId', vlanId )
      if evid.inExtendedRange():
         return ( None, None )

   if routed and masterOfVrmac( bridge, vlanId, dstMacAddr ):
      t8( 'Fhrp mac missed', vlanId, dstMacAddr )
      shouldRewrite = bridge.fhrpEtbaConfig.rewriteDmac
      intfName = bridge.intfVlanTable.vlanIdToIntfId( vlanId )
      ipIntfStatus = bridge.ipIntfStatus.get( intfName )
      if ipIntfStatus and \
         ( ipIntfStatus.activeVirtualAddrWithMask.address != "0.0.0.0" ):
         # for "ip address virtual", Vxlan controller ARP rewrite feature expects
         # ARP pkts with dmac as varp mac, so do not change.
         ( arph, _ ) = isArpPkt( data )
         if arph:
            t8( 'Skipping rewrite for ARP pkts with dmac as virtual Mac' )
            if arph.opcode == 2: # ARP Reply
               # if we are not replacing dmac as bridge mac then we need to
               # add the ARP entry if this is a ARP response destined to virtual mac,
               # as IP stack of router namespace DUT is not going to process
               # the ARP response
               addArpEntry( arph.senderEthAddr, arph.senderIpAddr, intfName.lower(),
                     ipIntfStatus.vrf )
            shouldRewrite = False

      vmac = bridge.virtualRouterMacStatus_.varpVirtualMac
      ip6IntfStatus = bridge.ip6IntfStatus.get( intfName )
      if ip6IntfStatus and ip6IntfStatus.useVirtualAddr and dstMacAddr == vmac:
         ( pkt, ip6h, icmp6h, _ ) = isIcmp6Pkt( data )
         if icmp6h and icmp6h.typeNum == "icmp6TypeNeighborAdvertisement":
            t8( 'Skipping rewrite for NA pkts with dmac as virtual Mac' )
            if ip6h.dst == Ip6Addr( ip6AddrFromMac( vmac ) ):
               # rewrite the virtual link local destIp to avoid icmp6 redirect.
               ip6h.dst = Ip6Addr( ip6AddrFromMac( bridge.bridgeMac_ ) )
               icmp6h.icmp6LenU32 = ip6h.payloadLen
               icmp6h.ipSrc = ip6h.src
               icmp6h.ipDst = ip6h.dst
               icmp6h.checksum = 0
               computedChecksum = icmp6h.computedChecksum
               icmp6h.checksum = computedChecksum
               changedData = pkt.stringValue
               return ( changedData, dstMacAddr )
            else:
               shouldRewrite = False

      if not shouldRewrite:
         t8( 'Skipping rewrite of dest mac' )
         return ( None, None )

      t8( 'Rewriting dest mac', dstMacAddr, bridge.bridgeMac_ )
      packetMac = Ethernet.convertMacAddrToPackedString( bridge.bridgeMac_ )
      changedData = packetMac + data[ 6 : ]
      return ( changedData, bridge.bridgeMac_ )
   else:
      return ( None, None )

def postBridging( bridge, srcMac, destMac, vlanId,
                  data, srcPortName, destIntf, finalIntfs ):
   intfName = bridge.intfVlanTable.vlanIdToIntfId( vlanId )
   ipIntfStatus = bridge.ipIntfStatus.get( intfName )
   vmac = bridge.virtualRouterMacStatus_.varpVirtualMac
   if ( ipIntfStatus and
        ipIntfStatus.activeVirtualAddrWithMask.address != "0.0.0.0" and
        vmac != '00:00:00:00:00:00' ):
      # for "ip address virtual", replace the smac of outgoing ARP packets
      # with varp mac if it's configured
      ( arph, arph_offset ) = isArpPkt( data )
      if ( arph and srcMac == bridge.bridgeMac_ ):
         shaOffset = arph_offset + 8
         t8( 'Rewriting smac %s to %s in outgoing ARP' % ( srcMac, vmac ) )
         smac = Ethernet.convertMacAddrToPackedString( vmac )
         changedData = data[ 0 : 6 ] + smac + data[ 12 : shaOffset ] + smac + \
                       data[ shaOffset + 6 : ]
         return ( None, changedData, vmac, destMac )

   ip6IntfStatus = bridge.ip6IntfStatus.get( intfName )
   if ip6IntfStatus and ip6IntfStatus.useVirtualAddr and vmac != "00:00:00:00:00:00":
      # for "ipv6 address virtual", replace the smac of outgoing NDP packets
      # with varp mac if it's configured
      ( pkt, ip6h, icmp6h, nd_offset ) = isIcmp6Pkt( data )
      if ip6h and icmp6h and srcMac == bridge.bridgeMac_:
         vmacBytes = Ethernet.convertMacAddrToPackedString( vmac )

         if icmp6h.typeNum == "icmp6TypeNeighborSolicitation":
            t8( 'Rewriting smac %s to %s in outgoing NS' % ( srcMac, vmac ) )

            if ip6h.src.isZero:
               # DAD smac is not used by receiver, no need to change to vmac
               return ( None, None, None, None )

            if ip6h.src.isLinkLocal:
               ip6h.src = Ip6Addr( ip6AddrFromMac( vmac ) )

            optOff = nd_offset + 20
            if ( optOff + 8 ) <= pkt.bytes and \
                  pkt.rawByte[ optOff ] == 1 and pkt.rawByte[ optOff + 1 ] == 1:
               t8( 'found NS source link address option at offset %d' % optOff )
               for i in range( 6 ):
                  pkt.rawByte[ optOff + 2 + i ] = vmacBytes[ i ]

            icmp6h.icmp6LenU32 = ip6h.payloadLen
            icmp6h.ipSrc = ip6h.src
            icmp6h.ipDst = ip6h.dst
            icmp6h.checksum = 0
            computedChecksum = icmp6h.computedChecksum
            icmp6h.checksum = computedChecksum
            changedData = pkt.stringValue
            changedData = changedData[ : 6 ] + vmacBytes + changedData[ 12 : ]
            return ( None, changedData, vmac, destMac )

         elif icmp6h.typeNum == "icmp6TypeNeighborAdvertisement":
            t8( 'Rewriting smac %s to %s in outgoing NA' % ( srcMac, vmac ) )
            nA = Tac.newInstance( "Arnet::Icmp6NeighborAdvertisementHdrWrapper",
                                  pkt, nd_offset )
            if nA.targetIp6Addr.isLinkLocal:
               return ( None, None, None, None )

            if ip6h.src.isLinkLocal:
               ip6h.src = Ip6Addr( ip6AddrFromMac( vmac ) )
            optOff = nd_offset + 20
            if ( optOff + 8 ) <= pkt.bytes and \
                  pkt.rawByte[ optOff ] == 2 and pkt.rawByte[ optOff + 1 ] == 1:
               t8( 'found NA tgt link address option at offset %d' % optOff )
               for i in range( 6 ):
                  pkt.rawByte[ optOff + 2 + i ] = vmacBytes[ i ]

            icmp6h.icmp6LenU32 = ip6h.payloadLen
            icmp6h.ipSrc = ip6h.src
            icmp6h.ipDst = ip6h.dst
            icmp6h.checksum = 0
            computedChecksum = icmp6h.computedChecksum
            icmp6h.checksum = computedChecksum
            changedData = pkt.stringValue
            changedData = changedData[ : 6 ] + vmacBytes + changedData[ 12 : ]
            return ( None, changedData, vmac, destMac )

         elif icmp6h.typeNum == "icmp6TypeRouterAdvertisement":
            t8( 'Rewriting smac %s to %s in outgoing RA' % ( srcMac, vmac ) )
            if ip6h.src.isLinkLocal:
               ip6h.src = Ip6Addr( ip6AddrFromMac( vmac ) )
            optOff = nd_offset + 12
            if ( optOff + 8 ) <= pkt.bytes and \
                  pkt.rawByte[ optOff ] == 1 and pkt.rawByte[ optOff + 1 ] == 1:
               t8( 'found RA source link address option at offset %d' % optOff )
               for i in range( 6 ):
                  pkt.rawByte[ optOff + 2 + i ] = vmacBytes[ i ]

            icmp6h.icmp6LenU32 = ip6h.payloadLen
            icmp6h.ipSrc = ip6h.src
            icmp6h.ipDst = ip6h.dst
            icmp6h.checksum = 0
            computedChecksum = icmp6h.computedChecksum
            icmp6h.checksum = computedChecksum
            changedData = pkt.stringValue
            changedData = changedData[ : 6 ] + vmacBytes + changedData[ 12 : ]
            return ( None, changedData, vmac, destMac )

   return ( None, None, None, None )

def ipv6VirtAddrLLRedirect( bridge, finalIntfs, srcPort=None, dropReasonList=None,
                   vlanId=None, dstMacAddr=None, data=None ):

   if not dstMacAddr or 'Cpu' not in finalIntfs:
      return

   evid = Tac.Value( 'Bridging::ExtendedVlanId', vlanId )
   if not evid.inDot1qRange():
      return

   intfName = bridge.intfVlanTable.vlanIdToIntfId( vlanId )
   ip6IntfStatus = bridge.ip6IntfStatus.get( intfName )
   if ip6IntfStatus and ip6IntfStatus.useVirtualAddr:
      # for "ipv6 address virtual", for a NS packet:
      # 1. drop if the smac peer link mac and NS target is global addr
      # 2. redirect to Vx1 if NS tgt/dst is virtual link local
      ( pkt, ip6h, icmp6h, nd_offset ) = isIcmp6Pkt( data )
      if ip6h and icmp6h and icmp6h.typeNum == "icmp6TypeNeighborSolicitation":
         srcMacAddr = macAddrsAsStrings( data )[ 0 ]
         vmac = bridge.virtualRouterMacStatus_.varpVirtualMac
         vLL = Ip6Addr( ip6AddrFromMac( vmac ) )
         nS = Tac.newInstance( "Arnet::Icmp6NeighborSolicitationHdrWrapper",
                               pkt, nd_offset )
         if ip6h.src.isZero:
            # suppress DAD frame from peer for tgt global address
            if not nS.targetIp6Addr.isLinkLocal:
               if srcMacAddr == mlagStatusMount.peerMacAddr or \
                  mlagStatusMount.peerLinkIntf and \
                  srcPort.name() == mlagStatusMount.peerLinkIntf.intfId:
                  t8( 'discarding global IP DAD from mlag peer link' )
                  del finalIntfs[ 'Cpu' ]
                  return

         elif vmac != "00:00:00:00:00:00" and nS.targetIp6Addr == vLL:
            # redirect NS tgt virtual link local on front panel to vxlan port
            # instead of Cpu 
            if not mlagStatusMount.peerLinkIntf or \
               srcPort.name() != mlagStatusMount.peerLinkIntf.intfId:
               t8( 'redirect NS for virtual link local from Cpu to VxlanSwFwd' )
               del finalIntfs[ 'Cpu' ]
               if bridge.vxlanPort_.get( 'Vxlan1' ) and \
                  bridge.vxlanPort_[ 'Vxlan1' ].name() not in finalIntfs:
                  t8( 'redirecting NS for virtual link local: add to vxlan port' )
                  finalIntfs.setdefault( bridge.vxlanPort_[ 'Vxlan1' ].name(), None )
            else:
               t8( 'no s/w handling for NS tgt virtual link local received from'
                    'mlag peerlink' )
               del finalIntfs[ 'Cpu' ]
               if bridge.vxlanPort_.get( 'Vxlan1' ) and \
                  bridge.vxlanPort_[ 'Vxlan1' ].name() not in finalIntfs:
                  del finalIntfs[ bridge.vxlanPort_[ 'Vxlan1' ].name() ]
            return

def bridgeInit( bridge ):
   t2( 'Fhrp bridgeInit' )
   bridge.virtualRouterMacStatus_ = bridge.em().entity( 'routing/fhrp/vrMacStatus' )

   fhrpEtbaConfig = bridge.em().entity( 'routing/fhrp/etba/config' )
   bridge.fhrpEtbaConfig = fhrpEtbaConfig
   bridge.brConfig = bridge.em().entity( 'bridging/config' )
   bridge.intfVlanTable = Tac.newInstance( 'IntfVlanId::Table', 'table' )
   bridge.intfVlanSm = Tac.newInstance( 'IntfVlanId::ConfigSm',
      bridge.brConfig, None, bridge.intfVlanTable )
   bridge.ipIntfStatus = bridge.em().entity( 'ip/status' ).ipIntfStatus
   bridge.ip6IntfStatus = bridge.em().entity( 'ip6/status' ).intf

def agentInit( em ):
   t2( 'Fhrp agentInit' )
   em.mount( 'routing/fhrp/vrMacStatus',
             'Routing::Fhrp::VirtualRouterMacStatus', 'r' )
   em.mount( 'routing/fhrp/etba/config',
             'Routing::Fhrp::Etba::Config', 'r' )
   em.mount( 'bridging/config',
             'Bridging::Config', 'r' )
   mg = em.activeMountGroup()
   Tac.Type( "Ira::IraIpStatusMounter" ).doMountEntities( mg.cMg_, True, True )

   global mlagStatusMount
   # Mount mlag/status, Mlag::Status and its dependent paths
   mlagStatusMount = MlagMountHelper.mountMlagStatus( mg )

def Plugin( ctx ):
   ctx.registerBridgeInitHandler( bridgeInit )
   ctx.registerAgentInitHandler( agentInit )
   ctx.registerFloodsetIncludesCpuHandler( floodsetIncludesCpu )
   ctx.registerRewritePacketOnTheWayToTheCpuHandler(
      rewritePacketOnTheWayToTheCpu )
   ctx.registerPostBridgingHandler( postBridging )
   ctx.registerPacketReplicationHandler( ipv6VirtAddrLLRedirect )
