# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from collections import defaultdict
import Arnet
from Arnet import (
   IpGenAddr,
   IntfId,
)
from GenericReactor import GenericReactor
from ForwardingHelper import (
      ARP_SMASH_DEFAULT_VRF_ID,
      Af,
      getNhgSize,
   )
from MplsEtbaLib import (
      getIntfVlan,
      getL2Intf,
   )
from TypeFuture import TacLazyType
from IpLibConsts import DEFAULT_VRF
import Tac
import Tracing
from Toggles.RoutingLibToggleLib import toggleFibGenMountPathEnabled

# pkgdeps: library IraEtba

handle = Tracing.Handle( 'MplsEtbaNhgHardwareStatusSm' )
t8 = handle.trace8

inArfaMode = None

AdjType = TacLazyType( 'Smash::Fib::AdjType' )
DestIpIntf = TacLazyType( 'Routing::NexthopGroup::DestIpIntf' )
DynamicTunnelIntfId = TacLazyType( 'Arnet::DynamicTunnelIntfId' )
EthAddr = TacLazyType( 'Arnet::EthAddr' )
FecHwProgrammingState = TacLazyType( 'Routing::Hardware::FecHwProgrammingState' )
FecId = TacLazyType( 'Smash::Fib::FecId' )
FecIdIntfId = TacLazyType( 'Arnet::FecIdIntfId' )
IpAddr = TacLazyType( 'Arnet::IpAddr' )
LfibViaType = TacLazyType( 'Mpls::LfibViaType' )
MplsLabel = TacLazyType( 'Arnet::MplsLabel' )
MplsLabelAction = TacLazyType( 'Arnet::MplsLabelAction' )
NexthopGroupAckInfo = TacLazyType( 'Routing::Hardware::NexthopGroupAckInfo' )
NexthopGroupIntfIdType = TacLazyType( 'Arnet::NexthopGroupIntfId' )
NhgId = TacLazyType( 'Routing::NexthopGroup::NexthopGroupId' )
NhgL2Via = TacLazyType( 'Routing::Hardware::NexthopGroupL2Via' )
NhgType = TacLazyType( 'Routing::NexthopGroup::NexthopGroupType' )
NhgEntryKey = TacLazyType( 'Routing::NexthopGroup::NexthopGroupEntryKey' )
NhgVia = Tac.Type( 'Routing::Hardware::NexthopGroupVia' )
RoutingNexthopGroupType = TacLazyType( 'Routing::NexthopGroup::NexthopGroupType' )
TunnelId = TacLazyType( 'Tunnel::TunnelTable::TunnelId' )
TunnelType = TacLazyType( 'Tunnel::TunnelTable::TunnelType' )
NhFecIdType = Tac.Type( 'Routing::NexthopStatusFecIdType' )
PICK = TacLazyType( "Routing::Hardware::PlatformIndependentCounterKey" )
CounterKeyType = Tac.Type( "Routing::Hardware::CounterKeyType" )
NhgEntryL2L3Info = Tac.Type( "Arfa::NexthopGroupEntryL2L3Info" )
EntryCounterType = Tac.Type( 'Routing::NexthopGroup::EntryCounterType' )
IpGenAddrType = Tac.Type( 'Arnet::IpGenAddr' )
IpGenPrefix = TacLazyType( 'Arnet::IpGenPrefix' )
IntfIdType = Tac.Type( 'Arnet::IntfId' )
VlanIntfId = TacLazyType( 'Arnet::VlanIntfId' )
EgressCounterState = TacLazyType( "Routing::Hardware::EgressCounterState" )

class L2Hop:
   def __init__( self, vlanId_, macAddr_ ):
      '''
      Returns an L2Hop object with the vlanId and macAddr passed in.
      Note: The macAddr will automatically have L2Hop.macAddrToString() called on it
      and will assert if that value isn't a valid Arnet.EthAddr.
      '''
      self.vlanId = vlanId_
      self.macAddr = L2Hop.macAddrToString( macAddr_ )

   def __repr__( self ):
      return f"L2Hop( {self.vlanId}, '{self.macAddr}' )"

   def __eq__( self, other ):
      return ( isinstance( other, self.__class__ ) and
               self.__dict__ == other.__dict__ )

   def __ne__( self, other ):
      return not self.__eq__( other )

   def __hash__( self ):
      return hash( ( self.vlanId, self.macAddr ) )

   @staticmethod
   def macAddrToString( macAddr_ ):
      '''
      Returns the colon separated string version of the macAddr.
      If bool( macAddr ) == False, it defaults to returning a zero macAddr.
      '''
      macAddr = macAddr_ or EthAddr.ethAddrZero
      assert L2Hop.isMacAddrValid( macAddr ), f'Invalid macAddr {macAddr_}'

      # Convert to a EthAddr object to convert all string representations to colon
      # separated version. Also need guard to prevent a double conversion to EthAddr.
      # pylint: disable-next=isinstance-second-argument-not-valid-type
      if not isinstance( macAddr, EthAddr ):
         macAddr = Arnet.EthAddr( macAddr )
      return str( macAddr )

   @staticmethod
   def isMacAddrValid( macAddr ):
      'Returns True if macAddr can successfully convert to Arnet.EthAddr'
      try:
         Arnet.EthAddr( str( macAddr ) )
         return True
      except IndexError:
         return False

   @staticmethod
   def isMacAddrNonZero( macAddr ):
      'Returns True if macAddr is a valid EthAddr and is not 0.0.0'
      return ( L2Hop.isMacAddrValid( macAddr ) and
               bool( Arnet.EthAddr( str( macAddr ) ) ) )

   @staticmethod
   def isVlanIdValid( vlanId ):
      return isinstance( vlanId, int ) and vlanId > 0

   def isValid( self ):
      'Returns True if the both the macAddr and vlanId are non-zero'
      return ( L2Hop.isMacAddrNonZero( self.macAddr ) and
               L2Hop.isVlanIdValid( self.vlanId ) )

class NhgAdjSmHelper:
   '''
   This is a helper object that contains all of the relevant mappings needed for
   the NhgAdjSms. It will also facilitate the adding of unresolved dstIps to the
   NhVrfConfig.nexthop collection for processing by the Ira L3 Nexthop Resolver.

   Lastly, it also will also contain references to the bridging config/status as
   well as the arpSmash in order to allow the other SMs to externally update the
   nhg adjacencies based on changes in resolved L3 or L2 information.

   Example mappings:

   config = {
      1.1.1.1: {
         'nhg1': set([ 0, 1 ]),
      }
      2222::: {
         'nhg2': set([ 2 ])
      }
   }
   status = {
      1.1.1.1: NhgVia() {
                  hop: 10.0.0.2,
                  l3Intf: Ethernet1,
                  l2Via: L2NhgVia() {
                            vlanId: 1006
                            l2Intf: 'Ethernet1'
                            macAddr: '00:00:00:01:00:02'
                         }
               }
   }
   l3HopToDstIp = {
      10.0.0.2: set([ 1.1.1.1, 1.2.3.4 ]),
      20.0.0.2: set([ 2.2.2.2 ])
      2001::2: set([ 2222:: ])
   }
   l2HopToL3Hop = {
      ( 1006, 00:00:00:01:00:02 ): set([ 10.0.0.2, 1001::2 ])
      ( 1007, 00:01:00:02:00:03 ): set([ 20.0.0.2, 2001::2 ])
   }
   nhgAdjSm = {
      'nhg1': NhgAdjSm()
   }
   nhgName = {
      1: 'nhg1',
      2: 'nhg2'
   }

   :type brConfig: Bridging::Config
   :type brStatus: Bridging::Status
   :type nhVrfConfig: Routing::NexthopVrfConfig
   :type arpSmash: Arp::Table::Status
   :type proactiveArpNexthopConfig: Arp::ProactiveArp::ClientConfig
   '''
   def __init__( self, brConfig, brStatus, nhVrfConfig, arpSmash,
                 proactiveArpNexthopConfig, nhgPickAllocator=None ):
      self._traceName = 'NhgAdjSmHelper'
      self.nhVrfConfig = nhVrfConfig
      self._brConfig = brConfig
      self._brStatus = brStatus
      self._arpSmash = arpSmash
      self._proactiveArpNexthopConfig = proactiveArpNexthopConfig

      self.config = defaultdict( lambda: defaultdict( set ) )
      self.status = {}
      self.l3HopToDstIp = defaultdict( set )
      self.l2HopToL3Hop = defaultdict( set )
      self.adjSm = {}
      self.nhgPickAllocator = nhgPickAllocator

      # Need to clear the nhVrfConfig since it might contain old nexthop entries
      # that weren't cleared (e.g. if agent crashed)
      nhVrfConfig.nexthop.clear()

   def formDstIpString( self, dstIp, linkLocalL3Intf ):
      assert isinstance( dstIp, IpGenAddrType )
      if linkLocalL3Intf:
         assert isinstance( linkLocalL3Intf, IntfIdType )

      dstIpAsString = dstIp.stringValue
      if dstIp.isV6LinkLocal:
         intfIdAsString = linkLocalL3Intf.stringValue
         dstIpAsString = f'{dstIpAsString}%{intfIdAsString}'
      return dstIpAsString

   def splitDstIp( self, dstIpAsString ):
      if '%' in dstIpAsString:
         dstIp = IpGenAddr( dstIpAsString.split( '%' )[ 0 ] )
         intfId = IntfId( dstIpAsString.split( '%' )[ 1 ] )
         return dstIp, intfId
      else:
         return IpGenAddr( dstIpAsString ), None

   def configEntryIs( self, dstIp, nhgName, index, linkLocalL3Intf=None ):
      # We only add to nhVrfConfig for entries with global IPs.
      # These entries are processed by the Ira L3 Nexthop Resolver
      # which is unnecessary for link-local
      if not dstIp.isV6LinkLocal:
         nhce = Tac.newInstance( 'Routing::NexthopConfigEntry', dstIp )
         self.nhVrfConfig.addNexthop( nhce )

      self.configEntryGetNhgEntries(
            dstIp, nhgName, linkLocalL3Intf=linkLocalL3Intf ).add( index )

   def linkLocalCleanupUnusedDstIp( self, dstIp, dstIpAsString, linkLocalL3Intf ):
      func = self._traceName + '.linkLocalCleanupUnusedDstIp:'
      t8( func, 'Clearing maps of unconfigured dstIp ', dstIpAsString )
      oldNhgVia = self.statusEntryGet( dstIp, linkLocalL3Intf=linkLocalL3Intf )
      self.statusEntryDel( dstIp, linkLocalL3Intf=linkLocalL3Intf )
      oldL3Hop = oldNhgVia.hop
      self.l3HopToDstIpDel( oldL3Hop, dstIp, linkLocalL3Intf=linkLocalL3Intf )
      if ( oldL3Hop not in self.l3HopToDstIp or
           dstIpAsString not in self.l3HopToDstIp[ oldL3Hop ] ):
         # Delete L2Hop to L3Hop entry if dstIp%intf combination is no longer used.
         # Link-locals can share the same L3Hop with different L2Vias
         oldL2Hop = L2Hop( oldNhgVia.l2Via.vlanId, oldNhgVia.l2Via.macAddr )
         self.l2HopToL3HopDel( oldL2Hop, oldL3Hop )
      # Clear old proactive arp
      oldProactiveArpEntry = Tac.ValueConst(
            'Arp::ProactiveArp::Entry', oldNhgVia.hop, oldNhgVia.l3Intf )
      del self._proactiveArpNexthopConfig.request[ oldProactiveArpEntry ]

   def configEntryDel( self, dstIp, nhgName, entry, linkLocalL3Intf=None ):
      dstIpAsString = self.formDstIpString( dstIp, linkLocalL3Intf )
      # Delete unconfigured entry
      self.config[ dstIpAsString ][ nhgName ].remove( entry )

      remainingEntries = self.configEntryGetNhgEntries(
            dstIp, nhgName, linkLocalL3Intf=linkLocalL3Intf )
      nhgs = self.configEntryGetNhgs( dstIp, linkLocalL3Intf=linkLocalL3Intf )
      # If no one else cares about this dstIp, remove the dstIp entirely from
      # both the config map and nhVrfConfig
      if not remainingEntries:
         if len( nhgs ) == 1:
            # Delete no longer configured dstIp from map
            del self.config[ dstIpAsString ]
            if dstIp.isV6LinkLocal:
               # Delete link-local via from status since it is no longer
               # used in any entries.
               # This only needs to be handled here for link-local IPs since
               # the NhgNhBacklogSm handles the deletion for globals
               self.linkLocalCleanupUnusedDstIp(
                     dstIp, dstIpAsString, linkLocalL3Intf )
            else:
               # We only add to nhVrfConfig for entries with global IPs.
               # These entries are processed by the Ira L3 Nexthop Resolver
               # which is unnecessary for link-local
               del self.nhVrfConfig.nexthop[ dstIp ]
         else:
            # Delete nhg with no matching entries from map
            del self.config[ dstIpAsString ][ nhgName ]

   def configEntryGetNhgEntries( self, dstIp, nhgName, linkLocalL3Intf=None ):
      dstIpAsString = self.formDstIpString( dstIp, linkLocalL3Intf )
      return self.config[ dstIpAsString ][ nhgName ]

   def configEntryGetNhgs( self, dstIp, linkLocalL3Intf=None ):
      dstIpAsString = self.formDstIpString( dstIp, linkLocalL3Intf )
      return self.config[ dstIpAsString ]

   def statusEntryIs( self, dstIp, nhgVia, linkLocalL3Intf=None ):
      assert isinstance( nhgVia, NhgVia )
      dstIpAsString = self.formDstIpString( dstIp, linkLocalL3Intf )
      self.status[ dstIpAsString ] = nhgVia

   def statusEntryDel( self, dstIp, linkLocalL3Intf=None ):
      dstIpAsString = self.formDstIpString( dstIp, linkLocalL3Intf )
      if dstIpAsString in self.status:
         del self.status[ dstIpAsString ]

   def statusEntryGet( self, dstIp, linkLocalL3Intf=None ):
      dstIpAsString = self.formDstIpString( dstIp, linkLocalL3Intf )
      if dstIpAsString in self.status:
         return self.status[ dstIpAsString ]
      return None

   def l3HopToDstIpIs( self, l3Hop, dstIp, linkLocalL3Intf=None ):
      func = self._traceName + '.l3HopToDstIpIs:'
      if l3Hop.isAddrZero or dstIp.isAddrZero:
         msg = f"{func} Can't add invalid l3Hop {l3Hop} dstIp {dstIp}"
         assert False, msg

      t8( func, 'Adding new l3Hop', l3Hop, 'dstIp', dstIp,
         'linkLocalL3Intf', linkLocalL3Intf )

      dstIpAsString = self.formDstIpString( dstIp, linkLocalL3Intf )
      self.l3HopToDstIp[ l3Hop ].add( dstIpAsString )

   def l3HopToDstIpDel( self, l3Hop, dstIp, linkLocalL3Intf=None ):
      func = self._traceName + '.l3HopToDstIpDel:'
      l3HopToDstIp = self.l3HopToDstIp
      dstIpAsString = self.formDstIpString( dstIp, linkLocalL3Intf )

      if l3Hop not in l3HopToDstIp or dstIpAsString not in l3HopToDstIp[ l3Hop ]:
         t8( func, "Skip deleting non-existent l3Hop", l3Hop, 'dstIp',
             dstIpAsString )
         return

      t8( func, "Deleting l3Hop", l3Hop, 'dstIp', dstIpAsString )

      l3HopToDstIp[ l3Hop ].remove( dstIpAsString )
      if not l3HopToDstIp[ l3Hop ]:
         del l3HopToDstIp[ l3Hop ]

   def l2HopToL3HopIs( self, l2Hop, l3Hop ):
      func = self._traceName + '.l2HopToL3HopIs:'
      assert isinstance( l2Hop, L2Hop )
      assert isinstance( l3Hop, IpGenAddrType )

      if not l2Hop.isValid():
         msg = f"{func} Can't add invalid l2Hop {l2Hop} l3Hop {l3Hop}"
         assert False, msg

      t8( func, 'Adding new l2Hop', l2Hop, 'l3hop', l3Hop )
      self.l2HopToL3Hop[ l2Hop ].add( l3Hop )

   def l2HopToL3HopDel( self, l2Hop, l3Hop ):
      func = self._traceName + '.l2HopToL3HopDel:'
      assert isinstance( l2Hop, L2Hop )
      assert isinstance( l3Hop, IpGenAddrType )

      l2HopToL3Hop = self.l2HopToL3Hop
      if l2Hop not in l2HopToL3Hop or not l3Hop in l2HopToL3Hop[ l2Hop ]:
         t8( func, "Skip deleting non-existent l2Hop", l2Hop, 'hop', l3Hop )
         return

      t8( func, "Deleting l2Hop", l2Hop, 'hop', l3Hop )
      l2HopToL3Hop[ l2Hop ].remove( l3Hop )
      if not l2HopToL3Hop[ l2Hop ]:
         del l2HopToL3Hop[ l2Hop ]

   def arpEntry( self, l3Hop, intf ):
      'Retrieves an ARP or ND entry from the arpSmash table given an l3 intf and hop'

      arpKey = Tac.Value( 'Arp::Table::ArpKey', ARP_SMASH_DEFAULT_VRF_ID,
                          l3Hop, intf )
      afGetFunc = {
            Af.ipv4: self._arpSmash.arpEntry.get,
            Af.ipv6: self._arpSmash.neighborEntry.get,
         }
      return afGetFunc[ l3Hop.af ]( arpKey )

   def updateViaL2Info( self, dstIp, macAddr, vlanId=None, linkLocalL3Intf=None ):
      '''
      Given the new mac addr, the L3 info is copied from the previous status entry's
      via, while the other L2 info is retrieved and copied into a new NhgVia
      '''
      func = self._traceName + '.updateViaL2Info:'
      t8( func, 'Updating the NhgVia L2 info for dstIp', dstIp )

      # Get old L3 info
      oldVia = self.statusEntryGet( dstIp, linkLocalL3Intf=linkLocalL3Intf )

      l3Intf = oldVia.l3Intf
      l3Hop = oldVia.hop
      route = oldVia.route

      # Get new L2 info
      if not vlanId:
         # getIntfVlan returns None instead of 0, but NhgL2Via expects a U32
         vlanId = getIntfVlan( self._brConfig, l3Intf ) or 0
      if L2Hop.isMacAddrNonZero( macAddr ):
         l2Intf = getL2Intf( self._brStatus, vlanId, macAddr ) if vlanId else ''
         if not l2Intf and not VlanIntfId.isVlanIntfId( l3Intf ):
            # If not configured as a vlan, a RoutedPort l2Intf should match l3Intf
            # BUG819603 - there is no fdbStatus for RoutedPorts
            l2Intf = l3Intf
      nhgL2Via = NhgL2Via() if not macAddr else NhgL2Via( vlanId, l2Intf, macAddr )
      nhgVia = NhgVia( route, l3Hop, l3Intf, nhgL2Via )

      self.statusEntryIs( dstIp, nhgVia, linkLocalL3Intf=IntfId( l3Intf ) )

   # Updates all the nhg entries in the config map using the corresponding status via
   def updateAdjEntries( self, dstIp, linkLocalL3Intf=None ):
      func = self._traceName + '.updateAdjEntries:'
      nhgs = self.configEntryGetNhgs( dstIp, linkLocalL3Intf=linkLocalL3Intf )
      if not nhgs:
         dstIpAsString = self.formDstIpString( dstIp, linkLocalL3Intf )
         t8( func, 'No nhgs using dstIp ', dstIpAsString )
         return
      via = self.statusEntryGet( dstIp, linkLocalL3Intf=linkLocalL3Intf )
      if not via:
         via = NhgVia()
      for nhg in nhgs:
         for index in nhgs[ nhg ]:
            self.adjSm[ nhg ].adjViaIs( index, via )

   def updateProactiveArpNexthop( self, oldNhgVia, nhgVia ):
      func = 'NhgAdjSmHelper.updateProactiveArpNexthop:'
      assert isinstance( oldNhgVia, NhgVia )
      assert isinstance( nhgVia, NhgVia )

      # Remove the old L3 hop if it's no longer in the l3HopToDstIp map
      isOldL3InfoValid = not oldNhgVia.hop.isAddrZero and oldNhgVia.l3Intf
      if isOldL3InfoValid and oldNhgVia.hop not in self.l3HopToDstIp:
         t8( func, 'Deleting old L3 hop', oldNhgVia.hop )
         oldEntry = Tac.ValueConst( 'Arp::ProactiveArp::Entry',
                                    oldNhgVia.hop, oldNhgVia.l3Intf )
         del self._proactiveArpNexthopConfig.request[ oldEntry ]

      # Add new L3 hop if it's valid
      isL3InfoValid = not nhgVia.hop.isAddrZero and nhgVia.l3Intf
      if isL3InfoValid:
         failMsg = f"Valid L3 Hop {nhgVia.hop} should be in L3 map"
         assert nhgVia.hop in self.l3HopToDstIp, failMsg

         t8( func, 'Adding new l3Hop', nhgVia.hop )
         entry = Tac.ValueConst( 'Arp::ProactiveArp::Entry',
                                 nhgVia.hop, nhgVia.l3Intf )
         request = Tac.ValueConst( 'Arp::ProactiveArp::Request', entry )
         self._proactiveArpNexthopConfig.addRequest( request )

   def updateMappings( self, oldNhgVia, nhgVia, dstIp, linkLocalL3Intf=None ):
      func = self._traceName + '.updateMappings: '
      assert isinstance( oldNhgVia, NhgVia )
      assert isinstance( nhgVia, NhgVia )

      # Remove old mappings for the oldNhgVia
      if not oldNhgVia.hop.isAddrZero:
         oldL3Hop = oldNhgVia.hop
         oldL2Hop = L2Hop( oldNhgVia.l2Via.vlanId, oldNhgVia.l2Via.macAddr )
         t8( func, 'Removing old NHG via L3Hop', oldL3Hop,
                   'L2Hop', oldL2Hop, 'from mappings' )
         self.l3HopToDstIpDel( oldL3Hop, dstIp, linkLocalL3Intf=linkLocalL3Intf )
         if oldL3Hop not in self.l3HopToDstIp:
            self.l2HopToL3HopDel( oldL2Hop, oldL3Hop )

      # Add new entries if resolved and update the status map
      if not nhgVia.hop.isAddrZero:
         l3Hop = nhgVia.hop
         t8( func, 'Adding resolved L3Hop', l3Hop )
         self.l3HopToDstIpIs( l3Hop, dstIp, linkLocalL3Intf=linkLocalL3Intf )

         l2Hop = L2Hop( nhgVia.l2Via.vlanId, nhgVia.l2Via.macAddr )
         if l2Hop.isValid():
            t8( func, 'Adding resolved L2Hop', l2Hop )
            self.l2HopToL3HopIs( l2Hop, nhgVia.hop )
         self.statusEntryIs( dstIp, nhgVia, linkLocalL3Intf=linkLocalL3Intf )
      else:
         self.statusEntryDel( dstIp, linkLocalL3Intf=linkLocalL3Intf )

class NhgBridgingStatusSm( Tac.Notifiee ):
   '''
   Reacts to updates to learned hosts in the Sysdb BridgingStatus by updating the L2
   info of the NhgVias that use both the vlanId and macAddr of the learned host. It
   will also trigger an update to all relevent NHG adj entries.
   '''
   notifierTypeName = 'Bridging::Status'

   def __init__( self, nhgAdjSmHelper, brConfig, brStatus ):
      self._traceName = 'NhgBridgingStatusSm'
      self._helper = nhgAdjSmHelper
      self._brConfig = brConfig
      self._brStatus = brStatus
      self._fdbReactor = {}
      Tac.Notifiee.__init__( self, brStatus )
      for vlanId in self._brStatus.fdbStatus:
         self.handleFdbStatus( vlanId )

   def updateViaL2Intf( self, dstIp, macAddr, vlanId_, linkLocalL3Intf=None ):
      self._helper.updateViaL2Info(
            dstIp, macAddr, vlanId=vlanId_, linkLocalL3Intf=linkLocalL3Intf )

   def handleLearnedHost( self, reactor, macAddr ):
      func = self._traceName + '.handleLearnedHost:'
      fdbStatus = reactor.notifier()
      vlanId = fdbStatus.fid
      l2Hop = L2Hop( vlanId, macAddr )

      if l2Hop not in self._helper.l2HopToL3Hop:
         t8( func, 'L2Hop', l2Hop, 'not found in l2HopToL3Hop map' )
         return

      t8( func, 'Updating the L2Intf for all NHG adj entries using L2Hop', l2Hop )
      for l3Hop in self._helper.l2HopToL3Hop[ l2Hop ]:
         for dstIpAsString in self._helper.l3HopToDstIp[ l3Hop ]:
            dstIp, linkLocalL3Intf = self._helper.splitDstIp( dstIpAsString )
            if not self.shouldUpdateL2Via( dstIp, linkLocalL3Intf, vlanId ):
               # Link-local addresses that have interfaces that don't match
               # the vlanId should not have their L2 info updated.
               # The L2 info depends on the interface configured
               t8( func, 'dstIp ', dstIpAsString,
                   ' is link-local with a routed port. skipping updating L2Via' )
               continue
            self.updateViaL2Intf(
                  dstIp, macAddr, vlanId, linkLocalL3Intf=linkLocalL3Intf )
            self._helper.updateAdjEntries( dstIp, linkLocalL3Intf=linkLocalL3Intf )

   def shouldUpdateL2Via( self, dstIp, linkLocalL3Intf, vlanId ):
      if not linkLocalL3Intf:
         # Is a global dstIp, should update
         return True
      compareVlanId = getIntfVlan( self._brConfig, linkLocalL3Intf ) or 0
      # Only update L2Via if vlanId matches that of the interface
      return compareVlanId == vlanId

   @Tac.handler( 'fdbStatus' )
   def handleFdbStatus( self, vlanId ):
      '''
      Reacts to changes in fdbStatus entries by creating/deleting reactors to monitor
      learnedHost entries for that specific fdbStatus.
      '''
      func = self._traceName + '.handleFdbStatus:'
      fdbStatus = self._brStatus.fdbStatus.get( vlanId )
      if not fdbStatus:
         if vlanId in self._fdbReactor:
            t8( func, 'Deleting fdbStatus reactor with vlanId', vlanId )
            del self._fdbReactor[ vlanId ]
      else:
         t8( func, 'Creating fdbStatus reactor with vlanId', vlanId )
         self._fdbReactor[ vlanId ] = GenericReactor( fdbStatus, [ 'learnedHost' ],
                                                     self.handleLearnedHost )
         for macAddr in fdbStatus.learnedHost:
            self.handleLearnedHost( self._fdbReactor[ vlanId ], macAddr )

class NhgArpSm( Tac.Notifiee ):
   '''
   Reacts to ARP/ND updates by updating the L2 info of the NhgVias that use the L3Hop
   of the ARP/ND entry, as well as trigger an update to all relevant NHG adj entries.
   '''
   notifierTypeName = 'Arp::Table::Status'

   def __init__( self, nhgAdjEtbaHelper, arpSmash ):
      self._traceName = 'NhgArpSm'
      self._helper = nhgAdjEtbaHelper
      self._arpSmash = arpSmash
      Tac.Notifiee.__init__( self, arpSmash )
      for arpKey in arpSmash.arpEntry:
         self.handleArpEntry( arpKey )
      for ndKey in arpSmash.neighborEntry:
         self.handleNeighborEntry( ndKey )

   def getVia( self, l3Hop ):
      'Returns a NhgVia with the given L3Hop'
      func = self._traceName + '.getVia:'
      if l3Hop not in self._helper.l3HopToDstIp:
         t8( func, 'l3Hop not in l3HopToDstIp map, returning None' )
         return None

      # Get arbitrary dstIp from set since any entry will have the same l2 info
      dstIpAsString = next( iter( self._helper.l3HopToDstIp[ l3Hop ] ) )
      nhgVia = self._helper.status[ dstIpAsString ]
      return nhgVia

   def updateL2HopToL3Hop( self, oldNhgVia, l3Hop ):
      func = self._traceName + '.updateL2HopToL3Hop:'
      t8( func, 'Updating l2HopToL3Hop with new l3hop', l3Hop )
      newNhgVia = self.getVia( l3Hop )
      newL2Hop = L2Hop( newNhgVia.l2Via.vlanId, newNhgVia.l2Via.macAddr )
      oldL2Hop = L2Hop( oldNhgVia.l2Via.vlanId, oldNhgVia.l2Via.macAddr )

      self._helper.l2HopToL3HopDel( oldL2Hop, oldNhgVia.hop )
      if newL2Hop.isValid():
         self._helper.l2HopToL3HopIs( newL2Hop, l3Hop )

   def updateNhgViasAndL2HopToL3Hop( self, arpKey, arpEntry ):
      l3Hop = arpKey.addr
      oldNhgVia = self.getVia( l3Hop )
      # OldNhgVia will only exist if this l3Hop is used by any NHG entries
      if oldNhgVia:
         for dstIpAsString in self._helper.l3HopToDstIp[ l3Hop ]:
            macAddr = arpEntry.ethAddr if arpEntry else None
            dstIp = IpGenAddr( dstIpAsString )
            self._helper.updateViaL2Info( dstIp, macAddr )
            self._helper.updateAdjEntries( dstIp )
         self.updateL2HopToL3Hop( oldNhgVia, l3Hop )

   def updateNhgViasLinkLocal( self, arpKey, arpEntry ):
      l3Hop = arpKey.addr
      # Get old nhgVia. For link-local entries, dstIp == l3Hop
      dstIp = l3Hop
      l3Intf = IntfId( arpKey.intfId )
      oldNhgVia = self._helper.statusEntryGet( dstIp, linkLocalL3Intf=l3Intf )
      if oldNhgVia:
         macAddr = arpEntry.ethAddr if arpEntry else None
         self._helper.updateViaL2Info( dstIp, macAddr, linkLocalL3Intf=l3Intf )
         self._helper.updateAdjEntries( dstIp, linkLocalL3Intf=l3Intf )
         # update l2HopToL3Hop
         newNhgVia = self._helper.statusEntryGet( dstIp, linkLocalL3Intf=l3Intf )
         newL2Hop = L2Hop( newNhgVia.l2Via.vlanId, newNhgVia.l2Via.macAddr )
         oldL2Hop = L2Hop( oldNhgVia.l2Via.vlanId, oldNhgVia.l2Via.macAddr )

         self._helper.l2HopToL3HopDel( oldL2Hop, oldNhgVia.hop )
         if newL2Hop.isValid():
            self._helper.l2HopToL3HopIs( newL2Hop, l3Hop )

   @Tac.handler( 'arpEntry' )
   def handleArpEntry( self, arpKey ):
      func = self._traceName + '.handleArpEntry:'
      t8( func, 'Handling arp entry for L3Hop', arpKey.addr )
      arpEntry = self._arpSmash.arpEntry.get( arpKey )
      self.updateNhgViasAndL2HopToL3Hop( arpKey, arpEntry )

   @Tac.handler( 'neighborEntry' )
   def handleNeighborEntry( self, ndKey ):
      func = self._traceName + '.handleNeighborEntry:'
      t8( func, 'Handling neighbor entry for L3Hop', ndKey.addr )
      ndEntry = self._arpSmash.neighborEntry.get( ndKey )
      l3Hop = ndKey.addr
      if l3Hop.isV6LinkLocal:
         # We cannot iterate through all the dstIps in l3HopToDstIp in the link-local
         # case because, unlike global dstIps, depending on the interfaces
         # configured, link-local entries are not guaranteed to share the same
         # L2 info
         self.updateNhgViasLinkLocal( ndKey, ndEntry )
      else:
         self.updateNhgViasAndL2HopToL3Hop( ndKey, ndEntry )

class NhgNhBacklogSm( Tac.Notifiee ):
   notifierTypeName = 'Routing::NexthopBacklog'

   def __init__( self, fwdHelper, nhgAdjSmHelper, nhrStatus, brConfig, brStatus,
                 backlog ):
      self._traceName = 'NhgNhBacklogSm'
      self._fwdHelper = fwdHelper
      self._helper = nhgAdjSmHelper
      self._nhrStatus = nhrStatus
      self._backlog = backlog
      self._brConfig = brConfig
      self._brStatus = brStatus
      Tac.Notifiee.__init__( self, backlog )
      self.handleBacklogUpdateCount()

   def getValidViaInfo( self, nhse, fec ):
      '''
      Iterates through all FEC vias and returns the hop/intf pair for the first via
      with: a valid intf, MPLS null label, no NHG id and no tunnelId.
      '''
      for via in fec.via.values():
         # If the relevant fields are valid, return the hop and intfId
         if( via.mplsLabel == MplsLabel.null and
             not via.nexthopGroupId and
             not via.tunnelId and
             via.intfId ):
            hop = IpGenAddr( str( via.hop ) )
            hop = nhse.addr if hop.isAddrZero else hop
            assert not hop.isAddrZero, 'Both FEC via and NHSE are zero addrs'
            return hop, via.intfId
      return None, None

   def createNhgVia( self, nhse, fec ):
      '''
      Using the information stored in the Ira NexthopStatusEntry (nhse) and the fec
      for some dstIp, this method will create and return a NhgVia.
      '''
      func = self._traceName + '.createNhgVia:'

      # Get the route and the L3Hop based on whether the fec via has a valid hop
      route = nhse.route
      l3Hop, l3Intf = self.getValidViaInfo( nhse, fec )
      if not l3Hop:
         t8( func, 'No unlabeled IP routes found, returning blank NHG via' )
         return NhgVia()

      # Fetch the L2 info if available and create the NhgL2Via
      arpEntry = self._helper.arpEntry( l3Hop, l3Intf )
      if arpEntry:
         macAddr = arpEntry.ethAddr
         vlanId = getIntfVlan( self._brConfig, l3Intf ) or 0
         l2Intf = getL2Intf( self._brStatus, vlanId, macAddr ) if vlanId else ''
         if not l2Intf and not VlanIntfId.isVlanIntfId( l3Intf ):
            # If not configured as a vlan, l2Intf should match l3Intf
            l2Intf = l3Intf
         nhgL2Via = NhgL2Via( vlanId, l2Intf, macAddr )
      else:
         nhgL2Via = NhgL2Via()

      nhgVia = NhgVia( route, l3Hop, l3Intf, nhgL2Via )

      t8( func, 'Created nhgVia for route', route, 'fecId', fec.fecId )
      return nhgVia

   def handleBacklogNexthop( self, dstIp ):
      '''
      Performs three tasks:
      1) Processes the nexthop by pulling the resolved L3 information from the FIB
         and updating the L3 map, as well as the L2 map if any of the L2 information
         has already been resolved for the L3 hop.
      2) Updates the status map with the newly created NhgVia and triggers an update
         to all NHG adj entries using this dstIp.
      3) Updates the proactive ARP nexthop config collection by removing unresolved
         L3 hops and adding newly resolved L3 hops.
      '''
      func = self._traceName + '.handleBacklogNexthop:'

      # IPv6 Link-Local L3 resolution is handled in NhgAdjSm, not by NhgNhBacklogSm
      if dstIp.isV6LinkLocal:
         return

      # There should not be an update if the dstIp isn't being used by some NHG
      if str( dstIp ) not in self._helper.config:
         t8( func, 'Ignoring nexthop update since not in NHG config map' )
         return

      nhgVia = NhgVia()
      nhse = self._nhrStatus.vrf[ DEFAULT_VRF ].nexthop.get( dstIp )
      if nhse:
         fecId = nhse.fecId
         fec = self._fwdHelper.resolveFecId( fecId )
         # Although the route may be resolved (NHSE is present), there may be a
         # transient state where the nhse.fecId is not present in FIB. So,
         # check this
         if fec:
            nhgVia = self.createNhgVia( nhse, fec )

      oldNhgVia = self._helper.statusEntryGet( dstIp )
      if not oldNhgVia:
         oldNhgVia = NhgVia()
      self._helper.updateMappings( oldNhgVia, nhgVia, dstIp )
      self._helper.updateAdjEntries( dstIp )
      self._helper.updateProactiveArpNexthop( oldNhgVia, nhgVia )

   @Tac.handler( 'backlogUpdateCount' )
   def handleBacklogUpdateCount( self ):
      '''
      At least one new entry has shown up in the backlog nexthops. Iterate through
      and process each one.
      '''
      func = self._traceName + '.handleBacklogUpdateCount:'
      for dstIp in self._backlog.backlog:
         t8( func, 'Processing backlog nexthop', str( dstIp ) )
         self.handleBacklogNexthop( dstIp )
         del self._backlog.backlog[ dstIp ]

class NhgAdjSm( Tac.Notifiee ):
   '''
   Reacts to any changes in the nexthopGroupConfigEntry's destinationIp collection.
   Updates to the nhgAdj vias are done externally through the NhgAdjSmHelper by the
   other SMs (such as the NhgNhBacklogSm).

   This does not need to handle the case where the nexthopGroupConfig is deleted,
   since this will also cause the associated FEC to disappear, which will trigger the
   cleanup of this SM via the NhgAdjManagerSm.
   '''
   notifierTypeName = 'Routing::NexthopGroup::NexthopGroupConfigEntry'

   def __init__( self, nhgAdjSmHelper, nexthopGroup, nhgAdj, nhrStatus, brConfig,
                brStatus ):
      self._traceName = f'NhgAdjSm[ {nexthopGroup.name} ]'
      self._nexthopGroup = nexthopGroup
      self._nhgName = nexthopGroup.name
      self._helper = nhgAdjSmHelper
      self._nhgAdj = nhgAdj
      self._nhrStatus = nhrStatus
      self._brConfig = brConfig
      self._brStatus = brStatus

      # A local map of index -> dstIp needs to be maintained in the case that actual
      # nexthopGroup entries (or the entire entity) is deleted. In the latter case,
      # all revelant entries can be quickly deleted by iterating through the
      # map, otherwise you'd need to search through every dstIp in the configMap.
      self._currNhgEntries = {}
      self._nhgAdj.size = getNhgSize( self._nexthopGroup )
      t8( self._traceName + ':',
          'Initializing SM. NhgAdj size set to', self._nhgAdj.size )
      Tac.Notifiee.__init__( self, self._nexthopGroup )

      # Add newly created adjSm to helper's adjSm mapping
      self._helper.adjSm[ self._nhgName ] = self

      for index in range( self._nhgAdj.size ):
         self.handleDstIp( index, isInit=True )
      # Counters are always on in Arfa and no notion of any counters being inactive
      if inArfaMode:
         self._nhgAdj.entryCounterState = EgressCounterState.allCountersProgrammed

   def cleanup( self ):
      func = self._traceName + '.cleanup:'
      t8( func, 'Removing nhg adj' )
      for index in list( self._currNhgEntries ):
         self.entryDel( index )

   def entryDel( self, index ):
      func = self._traceName + '.entryDel:'
      t8( func, 'Deleting nhgAdj via', index )

      # Delete the config entry and local entry, and the adj entry if it exists
      dstIpAsString = self._currNhgEntries[ index ]
      dstIp, intfId = self._helper.splitDstIp( dstIpAsString )

      self._helper.configEntryDel( dstIp, self._nhgName, index,
                                   linkLocalL3Intf=intfId )
      del self._currNhgEntries[ index ]
      nhgEntryKey = NhgEntryKey( self._nhgAdj.nhId, index )
      self.deletePickAndCounterKey( nhgEntryKey )
      del self._nhgAdj.via[ index ]

   def currNhgEntriesIs( self, index, dstIp, linkLocalL3Intf=None ):
      dstIpAsString = self._helper.formDstIpString( dstIp, linkLocalL3Intf )
      self._currNhgEntries[ index ] = dstIpAsString

   def reallocateEntryCounter( self, index ):
      nhgEntryKey = NhgEntryKey( self._nhgAdj.nhId, index )
      if index in self._nexthopGroup.entry:
         dstIpAndIntfId = self._nexthopGroup.entry[ index ].destinationIpIntf
      else:
         dstIpAndIntfId = DestIpIntf()
      dstIp = dstIpAndIntfId.destIp
      intfId = IntfId( dstIpAndIntfId.intfId )
      nhgVia = self._helper.statusEntryGet( dstIp, linkLocalL3Intf=intfId )
      if nhgVia:
         self.allocatePickAndCounterKey( nhgEntryKey, nhgVia )
         self._nhgAdj.via[ index ] = nhgVia

   # Attributes like SIP/TTL that affect unshared counters and are
   # configurable for non-MPLS type NHGs would also need to relloacate
   # counters for when we support those nhg types
   def reallocateAllEntryCounters( self ):
      # Reallocate PICK for nhg counters because of counter type change
      for index in range( self._nhgAdj.size ):
         self.reallocateEntryCounter( index )

   @Tac.handler( 'entryCounterType' )
   def handleEntryCounterType( self ):
      if not inArfaMode:
         return
      func = self._traceName + '.handleEntryCounterType:'
      t8( func, 'entry counter type changed to',
          self._nexthopGroup.entryCounterType )
      self.reallocateAllEntryCounters()

   @Tac.handler( 'mplsLabelStackInternal' )
   def handleMplsLabelStack( self, index ):
      if not inArfaMode:
         return
      self.reallocateEntryCounter( index )

   @Tac.handler( 'sourceIp' )
   def handleSourceIp( self ):
      if not inArfaMode:
         return
      self.reallocateAllEntryCounters()

   @Tac.handler( 'configuredVersion' )
   def handleConfiguredVersion( self ):
      self._nhgAdj.ackInfo = NexthopGroupAckInfo(
         FecHwProgrammingState.allProgrammed,
         self._nexthopGroup.configuredVersion,
         0 # PdVersionId can be set to 0 as this is only necessary for AleMpls
      )

   @Tac.handler( 'readyNotification' )
   def handleReadyNotification( self ):
      self._nhgAdj.readyNotification = self._nexthopGroup.readyNotification

   @Tac.handler( 'destinationIpIntfInternal' )
   def handleDstIp( self, index, isInit=False ):
      func = self._traceName + '.handleDstIp:'
      if not isInit:
         # Need to manually calculate the new size since it can completely change
         # E.g. NHG can have entries 0 and 6 populated. If 6 gets deleted, size goes
         #      from 7 down to 1 since entries 1-5 are unpopulated (zero addresses)
         newSize = getNhgSize( self._nexthopGroup )
         if self._nhgAdj.size != newSize:
            t8( func, 'Nhg size changed from', self._nhgAdj.size, 'to', newSize )
            self._nhgAdj.size = newSize

      if index in self._nexthopGroup.entry:
         dstIpAndIntfId = self._nexthopGroup.entry[ index ].destinationIpIntf
      else:
         dstIpAndIntfId = DestIpIntf()
      dstIp = dstIpAndIntfId.destIp
      intfId = IntfId( dstIpAndIntfId.intfId )

      # Three cases to handle:
      # 1) Entry was deleted, array size decreased and the dstIp entry doesn't exist
      # 2) Entry was simply deleted, which shows up as a zero address dstIp
      # 3) Entry changed to a different dstIp, which means old dstIp must be deleted
      if not dstIp.isAddrZero:
         oldDstIp = self._currNhgEntries.get( index )
         if oldDstIp:
            t8( func, 'DstIp for entry', index, 'changed from', oldDstIp, 'to',
                dstIp )
            self.entryDel( index )
         else:
            t8( func, 'Entry', index, 'has new dstIp', dstIp )

         self.currNhgEntriesIs( index, dstIp, linkLocalL3Intf=intfId )
         self._helper.configEntryIs( dstIp, self._nhgName, index,
                                     linkLocalL3Intf=intfId )

         # Populate the adj via with whatever might be in status.
         # Only need to do this for global dstIps since this will happen later in
         # handleLinkLocalL3Resolution for link-local dstIps
         if not dstIp.isV6LinkLocal:
            nhgVia = self._helper.statusEntryGet( dstIp, linkLocalL3Intf=intfId )
            self.adjViaIs( index, nhgVia )
      elif not isInit and index in self._currNhgEntries:
         # It seems it's possible to react to a spurious zero addr dstIp entry, which
         # may not be present in the config
         t8( func, 'NhgConfig entry', index, 'deleted.' )
         self.entryDel( index )

      # Update mappings for IPv6 Link-Local Address.
      # Global IPs need help from the Ira L3 Nexthop Resolver for L3 resolution.
      # This is unnecessary (and link-local IPs will not have FIB entries),
      # so we update the mappings and the adj with the L3 resolution here, in-line
      if dstIp.isV6LinkLocal:
         self.handleLinkLocalL3Resolution( dstIp, intfId )

   def handleLinkLocalL3Resolution( self, dstIp, l3Intf ):
      func = self._traceName + '.handleLinkLocalL3Resolution:'
      t8( func, 'Handling L3 resolution for link-local dstIp' )
      # 'route' in NhgVia should be empty in link-local case
      route = IpGenPrefix()
      # l3Hop is the same as dstIp in link-local case
      l3Hop = dstIp
      # Resolve L2 Info
      arpEntry = self._helper.arpEntry( l3Hop, l3Intf )
      if arpEntry:
         macAddr = arpEntry.ethAddr
         vlanId = getIntfVlan( self._brConfig, l3Intf ) or 0
         l2Intf = getL2Intf( self._brStatus, vlanId, macAddr ) if vlanId else ''
         if not l2Intf and not VlanIntfId.isVlanIntfId( l3Intf ):
            # If not configured as a vlan (l3Intf is a RoutedPort),
            # l2Intf should match l3Intf
            l2Intf = l3Intf
         nhgL2Via = NhgL2Via( vlanId, l2Intf, macAddr )
      else:
         nhgL2Via = NhgL2Via()

      # Create new NhgVia
      nhgVia = NhgVia( route, l3Hop, l3Intf, nhgL2Via )

      # Update Mappings
      oldNhgVia = self._helper.statusEntryGet( dstIp, linkLocalL3Intf=l3Intf )
      if not oldNhgVia:
         oldNhgVia = NhgVia()
      self._helper.updateMappings( oldNhgVia, nhgVia, dstIp, linkLocalL3Intf=l3Intf )

      self._helper.updateAdjEntries( dstIp, linkLocalL3Intf=l3Intf )
      self._helper.updateProactiveArpNexthop( oldNhgVia, nhgVia )

   def allocatePickAndCounterKey( self, nhgEntryKey, nhgVia ):
      if not inArfaMode:
         return

      counterType = (
         CounterKeyType.nhgCounterUnshared if
         self._nexthopGroup.entryCounterType == EntryCounterType.unshared else
         CounterKeyType.nhgCounterShared )

      entry = self._nexthopGroup.entry[ nhgEntryKey.nhgEntryIndex ]
      dstIpIntf = entry.destinationIpIntf
      configIntf = dstIpIntf.intfId

      pick = self._helper.nhgPickAllocator.maybeCreatePick(
         nhgEntryKey.nhgId,
         nhgEntryKey.nhgEntryIndex,
         nhgVia,
         self._nexthopGroup.type,
         self._nexthopGroup.ttl,
         self._nexthopGroup.sourceIp,
         dstIpIntf.destIp,
         entry.mplsLabelStack,
         self._nexthopGroup.greKeyType,
         counterType,
         configIntf )

      nhgVia.counterIndex = PICK( pick ).counterIndex()

   def deletePickAndCounterKey( self, nhgEntryKey ):
      if not inArfaMode:
         return
      self._helper.nhgPickAllocator.deletePickForNhgEntry( nhgEntryKey.nhgId,
                                                           nhgEntryKey.nhgEntryIndex,
                                                           False )

   def adjViaIs( self, index, nhgVia ):
      nhgEntryKey = NhgEntryKey( self._nhgAdj.nhId, index )
      if nhgVia:
         self.allocatePickAndCounterKey( nhgEntryKey, nhgVia )
         self._nhgAdj.via[ index ] = nhgVia
      else:
         del self._nhgAdj.via[ index ]

class NhgAdjManagerSm:
   '''
   This SM contains reactors for: both the V4 and V6 forwardingStatuses, the
   system LFIB, the tunnel FIB, the NHG Config entities in Sysdb and the NHG Smash
   entities. A NhgAdjSm is created only when a route is created to an existing NHG.
   In other words, the equation for creating the NhgAdjSm is:

   NHG Smash Entry (NhgId, NhgName) + NHG Sysdb Entry (NhgName)
      - Created when a NHG is configured

   as well as any of the following FIB entries pointing to the NHG:
   - NHG FEC (NhgId)
   - NHG tunnel FEC (TunnelId) + Tunnel FIB entry (TunnelId, NhgId )
   - NHG LFIB route (NhgId )

   Conversely, if any of the above entries are deleted, the corresponding NhgAdjSm
   will also get deleted (assuming it exists).
   '''
   def __init__( self, fwdHelper, nhgAdjSmHelper, nhgHwDefaultVrfStatus,
                 lfib, nhrStatus, brConfig, brStatus ):
      self._traceName = 'NhgAdjManagerSm'
      t8( self._traceName + ':', 'Initializing SM' )
      self._fwdHelper = fwdHelper
      self._helper = nhgAdjSmHelper
      self._nhgHwDefaultVrfStatus = nhgHwDefaultVrfStatus
      # Keeps track of programmed NHGs
      self._nhgIdToName = {}
      self._nhgNameToId = {}
      # For NHG tunnel routes (FECs <-> Tunnels <-> NHGs)
      self._tunnelIdToNhgId = defaultdict( set )
      self._nhgIdToTunnelId = defaultdict( set )
      self._tunnelIdToFecId = defaultdict( set )
      self._fecIdToTunnelId = defaultdict( set )
      # For static NHG routes (FECs <-> NHGs)
      self._fecIdToNhgId = defaultdict( set )
      self._nhgIdToFecId = defaultdict( set )
      # For MPLS NHG routes (LFIB routes <-> NHGs)
      self._nhgIdToRouteKey = defaultdict( set )
      self._routeKeyToNhgId = defaultdict( set )
      # Entities to react to
      self._fs = self._fwdHelper.forwardingStatus_
      self._fs6 = self._fwdHelper.forwarding6Status_
      self._tunnelFib = self._fwdHelper.tunnelFib_
      self._fsGen = self._fwdHelper.forwardingGenStatus_
      self._lfib = lfib
      self.iraEtbaRoot = Tac.root.newEntity( 'Ira::IraEtbaRoot', 'ira-etba-root' )
      if not self.iraEtbaRoot.nexthopGroupConfig:
         self.iraEtbaRoot.nexthopGroupConfig = Tac.newInstance(
            "Routing::NexthopGroup::Config" )
      self._nhgConfig = Tac.root.get( 'ira-etba-root' ).nexthopGroupConfig
      self._nhgEntryStatus = fwdHelper.nhgEntryStatus_
      # The python version of the SM is almost ready to be removed
      self._nhgRegistrar = Tac.newInstance( "Arfa::NhgRegistrar", "" )
      # Reactors
      self._fsReactor = GenericReactor( self._fs, [ 'fec' ], self.handleFec )
      self._fs6Reactor = GenericReactor( self._fs6, [ 'fec' ], self.handleFec )
      self._tunnelFibReactor = GenericReactor( self._tunnelFib,
                                              [ 'entry' ],
                                              self.handleTunnelFibEntry )
      self._fsGenReactor = GenericReactor( self._fsGen, [ 'fec' ], self.handleFec )
      self._nhgSmashReactor = GenericReactor( self._nhgEntryStatus,
                                              [ 'nexthopGroupEntry' ],
                                              self.handleNhgSmashEntry )
      self._nhgConfigReactor = GenericReactor( self._nhgConfig, [ 'nexthopGroup' ],
                                               self.handleNhgConfigEntry )
      self._nhgRegistrarReactor = GenericReactor( self._nhgRegistrar, [ 'data' ],
                                                  self.handleNhgRegistrarEntry )

      self._lfibTrackingStatus = Tac.newInstance( 'Mpls::LfibTrackingStatus',
                                                  'NhgAdjManager-TrackingStatus' )
      self._lfibTrackingSm = Tac.newInstance(
            'Mpls::LfibTrackingSm',
            self._lfib,
            self._lfibTrackingStatus,
            Tac.Type( 'Mpls::LfibTrackingMode' ).shadowRouteKeys_,
            'NhgAdjManager-TrackingSm' )
      self._mplsLfibRouteReactor = GenericReactor( self._lfibTrackingStatus,
                                                   [ 'routeVersion' ],
                                                   self.handleMplsLfibRoute )
      self._nhrStatus = nhrStatus
      self._brConfig = brConfig
      self._brStatus = brStatus

      # Walk the FIB and LFIB and create NhgAdjSms for existing NHGs
      for fecKey in self._fs.fec:
         self.handleFec( self._fsReactor, fecKey )
      for fecKey in self._fs6.fec:
         self.handleFec( self._fs6Reactor, fecKey )
      for fecKey in self._fsGen.fec:
         self.handleFec( self._fsGenReactor, fecKey )
      for routeKey in self._lfib.lfibRoute:
         self.handleMplsLfibRoute( self._mplsLfibRouteReactor, routeKey )

   def __del__( self ):
      # Delete all reactors. It is possible to hit a ReferenceError (weakly bound
      # method called on a dead object) after the object is marked as dead otherwise.
      self._fsReactor = None
      self._fs6Reactor = None
      self._tunnelFibReactor = None
      self._fsGenReactor = None
      self._nhgSmashReactor = None
      self._nhgConfigReactor = None
      self._mplsLfibRouteReactor = None

   def _createNhgAdjSm( self, nhgId=None, nhgName=None ):
      '''
      If an MPLS NHG config is found in Sysdb using the provided info, this will
      create the NHG adj, the NhgAdjSm and trigger the update for the helper maps.
      '''
      func = self._traceName + '._createNhgAdjSm:'
      savedNhgId = self._nhgNameToId.get( nhgName )
      if savedNhgId and savedNhgId != nhgId:
         # NhgId is always increasing. Due to how the SMs are currently setup, we
         # can be notified about the old nhgId value. So if we have a newer one,
         # ignore the older one
         if savedNhgId > nhgId:
            t8( func, "Saved id", savedNhgId, "is greater than one passed in",
                nhgId, "Skipping updates" )
            return
         t8( func, "Updating", nhgName, "from old id", savedNhgId, "to new id",
             nhgId )
         # Nhg was deleted and recreated with the same name, update maps to use new
         # nhgId
         self._nhgNameToId[ nhgName ] = nhgId
         del self._nhgIdToName[ savedNhgId ]
         self._nhgIdToName[ nhgId ] = nhgName
         return

      nhgConfigEntry = self._fwdHelper.getNhgFromSysdb( nhgId=nhgId,
                                                        nhgName=nhgName )
      if nhgConfigEntry.type != RoutingNexthopGroupType.mpls:
         t8( func, 'Unsupported NHG type', nhgConfigEntry.type )
         return

      keyStr = f'NhgId: {nhgId}, NhgName: {nhgName}'
      t8( func, 'Creating NhgAdjSm with', keyStr )
      adj = self._nhgHwDefaultVrfStatus.nexthopGroupAdjacency.newMember( nhgId )
      _ = NhgAdjSm( self._helper, nhgConfigEntry, adj, self._nhrStatus,
                       self._brConfig, self._brStatus )

      self._nhgNameToId[ nhgConfigEntry.name ] = nhgId
      self._nhgIdToName[ nhgId ] = nhgConfigEntry.name

   def _delNhgAdjSm( self, nhgId=None, nhgName=None ):
      '''
      If both (possibly derived) nhgId and nhgName exist in helper mappings, the
      NhgAdjSm will be deleted via the helper and the corr. adj will be removed.
      '''
      keyStr = f'NhgId: {nhgId}, NhgName: {nhgName}'
      func = self._traceName + '._delNhgAdjSm:'
      if not nhgId and not nhgName:
         assert False, f"{func} Can't delete invalid NHG with {keyStr}"

      savedNhgId = self._nhgNameToId.get( nhgName )
      savedNhgName = self._nhgIdToName.get( nhgId )

      if savedNhgName is not None and nhgName is not None:
         # This shouldn't be possible
         assert savedNhgName == nhgName, "NhgId was reused"

      savedKeyStr = f'NhgId: {savedNhgId}, NhgName: {savedNhgName}'

      self._nhgNameToId.pop( nhgName, None )
      # NhgIds are always increasing, so if we receive an older (lower) nhgId for a
      # Nhg, we only want to update the mappings for it. If the Nhg is actually
      # deleted, we will get a notificatino about that with the correct nhgId
      if nhgId and savedNhgId and nhgId < savedNhgId:
         t8( func, "Removing mapping to to oldNhgId", nhgId, "for", nhgName )
         self._nhgIdToName.pop( nhgId, None )
      else:
         t8( func, 'Deleting NhgAdjSm with with input references', keyStr,
          'and saved references', savedKeyStr )
         self._nhgIdToName.pop( savedNhgId, None )
         helperSm = self._helper.adjSm.pop( nhgName, None )
         if helperSm:
            helperSm.cleanup()

         if nhgId and nhgId in self._nhgHwDefaultVrfStatus.nexthopGroupAdjacency:
            del self._nhgHwDefaultVrfStatus.nexthopGroupAdjacency[ nhgId ]
         if ( savedNhgId and
              savedNhgId in self._nhgHwDefaultVrfStatus.nexthopGroupAdjacency ):
            del self._nhgHwDefaultVrfStatus.nexthopGroupAdjacency[ savedNhgId ]

   def handleEntry( self, nhgId=None, nhgName=None ):
      '''
      This method is called whenever an entry in the FIBs or the NHG Sysdb/Smash
      collections is added or deleted.

      If all entries exist, the NhgAdjSm will be created. If any entry is missing
      instead, it is assumed that an entry has been removed and a deletion will
      be attempted.
      '''
      func = self._traceName + '.handleEntry:'
      if not nhgId and not nhgName:
         t8( func, 'Neither NhgId:', nhgId, 'nor NhgName:', nhgName, 'are valid' )
         return

      t8( func, 'Parameters NhgId:', nhgId, 'NhgName:', nhgName )
      # Ensure there is a NhgSmashEntry. This should be checked first since it's the
      # only entity with both the nhgId and nhgName, and both are needed
      nhgSmashEntry = self._fwdHelper.getNhgSmashEntry( nhgId=nhgId,
                                                        nhgName=nhgName )
      if not nhgSmashEntry:
         keyStr = f'NhgId: {nhgId}, NhgName: {nhgName}'
         t8( func, 'No NHG Smash Entry found with', keyStr )
         self._delNhgAdjSm( nhgId=nhgId, nhgName=nhgName )
         return
      nhgId = nhgId or nhgSmashEntry.nhgId
      nhgName = nhgName or nhgSmashEntry.key.nhgName()

      t8( func, 'After lookups NhgId:', nhgId, 'NhgName:', nhgName )
      # Ensure there is a proper FIB reference to the NHG
      tunnelIds = self._nhgIdToTunnelId.get( nhgId, set() )
      isNhgTunnelRoute = tunnelIds.intersection( list( self._tunnelIdToFecId ) )
      isStaticNhgRoute = self._nhgIdToFecId.get( nhgId )
      isMplsNhgRoute = self._nhgIdToRouteKey.get( nhgId )
      isRegistered = self._nhgRegistrar.data.get( nhgName )
      if ( not isStaticNhgRoute and not isNhgTunnelRoute and not isMplsNhgRoute and
           not isRegistered ):
         t8( func, 'No valid route found referencing NhgId:', nhgId )
         self._delNhgAdjSm( nhgId=nhgId, nhgName=nhgName )
         return

      # Ensure there is a NHG Config Entry in Sysdb wih this nhgName
      nhgConfigEntry = self._fwdHelper.getNhgFromSysdb( nhgName=nhgName )
      if not nhgConfigEntry:
         keyStr = f'NhgId: {nhgId}, NhgName: {nhgName}'
         t8( func, 'No NHG Config Entry found with', keyStr )
         self._delNhgAdjSm( nhgId=nhgId, nhgName=nhgName )
         return

      self._createNhgAdjSm( nhgId=nhgId, nhgName=nhgName )

   def handleMplsLfibRoute( self, reactor, routeKey ):
      func = self._traceName + '.handleMplsLfibRoute:'

      route = self._lfib.lfibRoute.get( routeKey )

      def removeNhgIdMappings( nhgId, routeKey ):
         self._routeKeyToNhgId[ routeKey ].remove( nhgId )
         if not self._routeKeyToNhgId[ routeKey ]:
            del self._routeKeyToNhgId[ routeKey ]
         assert routeKey in self._nhgIdToRouteKey[ nhgId ]
         self._nhgIdToRouteKey[ nhgId ].remove( routeKey )
         if not self._nhgIdToRouteKey[ nhgId ]:
            del self._nhgIdToRouteKey[ nhgId ]

      # Route was deleted
      if route is None:
         if routeKey in self._routeKeyToNhgId:
            oldNhgIds = set( self._routeKeyToNhgId[ routeKey ] )
            for nhgId in oldNhgIds:
               removeNhgIdMappings( nhgId, routeKey )
               self.handleEntry( nhgId=nhgId )
         return

      viaSetKey = route.viaSetKey

      # Route was added and is mplsIp adjacency type
      viaSet = self._lfib.viaSet.get( viaSetKey )

      # Mark route to be processed later when a corresponding adjacency is found
      if viaSet is None:
         t8( func, 'LFIB route', routeKey, 'has no existing via set', viaSetKey,
             'so processing is deferred until the via set is added.' )
         return

      if not viaSet.hasViaType( LfibViaType.viaTypeMplsIp ):
         return

      vias = []
      for vk in viaSet.viaKey.values():
         via = self._lfib.mplsVia.get( vk )
         if via is None:
            t8( func, 'LFIB route', routeKey, 'has no existing via', vk,
                'so processing is deferred until the via is added.' )
            return
         vias.append( via )

      # Adjacency exists, so process route
      oldNhgIds = self._routeKeyToNhgId.get( routeKey, set() ).copy()
      for via in vias:
         if not FecIdIntfId.isNexthopGroupIdIntfId( via.intf ):
            continue
         nhgId = FecIdIntfId.intfIdToNexthopGroupId( via.intf )
         t8( func, 'Handling NHG route', routeKey, 'with NhgId', nhgId )
         self._routeKeyToNhgId[ routeKey ].add( nhgId )
         self._nhgIdToRouteKey[ nhgId ].add( routeKey )
         oldNhgIds.discard( nhgId )
         self.handleEntry( nhgId=nhgId )

      # Remove the stale nhgIds
      for oldNhgId in oldNhgIds:
         removeNhgIdMappings( oldNhgId, routeKey )
         self.handleEntry( nhgId=oldNhgId )

   def handleFec( self, reactor, fecKey ):
      fecId = Tac.Value( 'Smash::Fib::FecId', fecKey )
      func = self._traceName + '.handleFec:'
      if ( fecId.adjType() != AdjType.fibV4Adj and
           fecId.adjType() != AdjType.fibV6Adj and
           not toggleFibGenMountPathEnabled() ):
         return

      def removeNhgIdMappings( nhgId, fecKey ):
         t8( func, 'Deleting mapping for nhgId:', nhgId, ', fecId:', fecKey )
         self._fecIdToNhgId[ fecKey ].remove( nhgId )
         if not self._fecIdToNhgId[ fecKey ]:
            del self._fecIdToNhgId[ fecKey ]
         self._nhgIdToFecId[ nhgId ].remove( fecKey )
         if not self._nhgIdToFecId[ nhgId ]:
            del self._nhgIdToFecId[ nhgId ]

      def removeTunnelIdMappings( tunnelId, fecKey ):
         t8( func, 'Deleting mapping for tunnelId:', tunnelId, ', fecId:', fecKey )
         self._fecIdToTunnelId[ fecKey ].remove( tunnelId )
         if not self._fecIdToTunnelId[ fecKey ]:
            del self._fecIdToTunnelId[ fecKey ]
         self._tunnelIdToFecId[ tunnelId ].remove( fecKey )
         if not self._tunnelIdToFecId[ tunnelId ]:
            del self._tunnelIdToFecId[ tunnelId ]

      # Update all mappings pertaining to fecId, nhgId, and tunnelId, as well as
      # create the nhgAdjSm if possible
      if toggleFibGenMountPathEnabled():
         fec = self._fsGen.fec.get( fecId )
      else:
         fec = self._fs.fec.get( fecId ) or self._fs6.fec.get( fecId )
      fecVias = fec.via if fec else {}
      oldNhgIds = self._fecIdToNhgId.get( fecKey, set() ).copy()
      oldTunnelIds = self._fecIdToTunnelId.get( fecKey, set() ).copy()
      for i in fecVias:
         nhgId = None
         viaIntfId = fecVias[ i ].intfId
         if DynamicTunnelIntfId.isDynamicTunnelIntfId( viaIntfId ):
            # Only update local maps if it's actually a NHG tunnel
            tunnelId = DynamicTunnelIntfId.tunnelId( viaIntfId )
            if TunnelId( tunnelId ).tunnelType() == TunnelType.nexthopGroupTunnel:
               t8( func, 'Handling NHG tunnel FEC:', fecKey, 'with tunnelId:',
                   tunnelId )
               self._fecIdToTunnelId[ fecKey ].add( tunnelId )
               self._tunnelIdToFecId[ tunnelId ].add( fecKey )
            # Builds the tunnelId <-> nhgId mappings and tries to create the nhgAdjSm
            self.handleTunnelFibEntry( self._tunnelFibReactor, tunnelId )
            oldNhgIds -= self._tunnelIdToNhgId.get( tunnelId, set() )
            oldTunnelIds.discard( tunnelId )
         elif NexthopGroupIntfIdType.isNexthopGroupIntfId( viaIntfId ):
            nhgId = NexthopGroupIntfIdType.nexthopGroupId( viaIntfId )
            t8( func, 'Handling NHG FEC', fecKey, 'with NhgId', nhgId )
            self._fecIdToNhgId[ fecKey ].add( nhgId )
            self._nhgIdToFecId[ nhgId ].add( fecKey )
            self.handleEntry( nhgId=nhgId )
            oldNhgIds.discard( nhgId )
         else:
            continue

      # Remove the stale nhgIds from the fecId <-> nhgId mappings and
      # deletes the corresponding nhgAdjSms
      for oldNhgId in oldNhgIds:
         removeNhgIdMappings( oldNhgId, fecKey )
         self.handleEntry( nhgId=oldNhgId )

      # Remove the stale tunnelIds from the fecId <-> tunnelId mappings and deletes
      # the corresponding nhgAdjSms
      for oldTunnelId in oldTunnelIds:
         removeTunnelIdMappings( oldTunnelId, fecKey )
         for nhgId in self._tunnelIdToNhgId.get( oldTunnelId, set() ):
            self.handleEntry( nhgId=nhgId )

   def handleTunnelFibEntry( self, reactor, tunnelId ):
      func = self._traceName + '.handleTunnelFibEntry:'
      if TunnelId( tunnelId ).tunnelType() != TunnelType.nexthopGroupTunnel:
         return

      def removeNhgIdMappings( nhgId, tunnelId ):
         t8( func, 'Deleting mapping for nhgId:', nhgId, ', tunnelId:', tunnelId )
         self._tunnelIdToNhgId[ tunnelId ].remove( nhgId )
         if not self._tunnelIdToNhgId[ tunnelId ]:
            del self._tunnelIdToNhgId[ tunnelId ]
         self._nhgIdToTunnelId[ nhgId ].remove( tunnelId )
         if not self._nhgIdToTunnelId[ nhgId ]:
            del self._nhgIdToTunnelId[ nhgId ]

      oldNhgIds = self._tunnelIdToNhgId.get( tunnelId, set() ).copy()
      tunnelFibEntry = self._tunnelFib.entry.get( tunnelId )
      if tunnelFibEntry and tunnelFibEntry.tunnelVia:
         assert len( tunnelFibEntry.tunnelVia ) <= 1, (
                  'ECMP for NHG tunnels not supported' )
         nhgId = NexthopGroupIntfIdType.nexthopGroupId(
               tunnelFibEntry.tunnelVia[ 0 ].intfId )
         t8( func, 'Found NHG tunnel entry:', tunnelId, 'with nhgId:', nhgId )
         self._nhgIdToTunnelId[ nhgId ].add( tunnelId )
         self._tunnelIdToNhgId[ tunnelId ].add( nhgId )
         self.handleEntry( nhgId=nhgId )
         oldNhgIds.discard( nhgId )
      else:
         t8( func, 'NHG tunnel entry:', tunnelId, 'was deleted' )

      # Remove the stale nhgIds from the tunnelId <-> nhgId mappings and deletes the
      # corresponding nhgAdjSms
      for oldNhgId in oldNhgIds:
         removeNhgIdMappings( oldNhgId, tunnelId )
         self.handleEntry( nhgId=oldNhgId )

   def handleNhgSmashEntry( self, reactor, nhgKey ):
      nhgId = None
      nhgName = nhgKey.nhgName()
      entry = self._nhgEntryStatus.nexthopGroupEntry.get( nhgKey )
      if entry:
         nhgId = entry.nhgId
      self.handleEntry( nhgId=nhgId, nhgName=nhgName )

   def handleNhgConfigEntry( self, reactor, nhgName ):
      self.handleEntry( nhgName=nhgName )

   def handleNhgRegistrarEntry( self, reactor, nhgName ):
      self.handleEntry( nhgName=nhgName )

class NhgL3ResolverSm( Tac.Notifiee ):
   '''
   Responsible for creating the FibMonitorHelperSm once the default nhVrfStatus shows
   up in Sysdb.
   '''
   notifierTypeName = 'Routing::NexthopStatus'

   def __init__( self, nhrStatus, fs, fs6, fsGen, backlog, scheduler ):
      self._traceName = 'NhgL3ResolverSm:'
      self._nhrStatus = nhrStatus
      self._backlog = backlog
      self._scheduler = scheduler
      self._fs = fs
      self._fs6 = fs6
      self._fsGen = fsGen
      self._fibMonitorHelperSm = None
      Tac.Notifiee.__init__( self, self._nhrStatus )
      for vrfName in self._nhrStatus.vrf:
         self.handleVrf( vrfName )

   @Tac.handler( 'vrf' )
   def handleVrf( self, vrfName ):
      if vrfName != DEFAULT_VRF:
         return

      vrf = self._nhrStatus.vrf.get( vrfName )
      if vrf:
         self._fibMonitorHelperSm = Tac.newInstance( 'Routing::FibMonitorHelperSm',
                                                    'etba',
                                                    vrf,
                                                    self._backlog,
                                                    self._fsGen,
                                                    self._scheduler )
         t8( self._traceName, 'Created new fibMonitorHelperSm for vrf', vrfName )
      else:
         self._fibMonitorHelperSm = None
         t8( self._traceName, 'Deleted fibMonitorHelperSm for vrf', vrfName )

class MplsNhgHardwareStatusSm:
   '''
   This SM simply instantiates and encapsulates everything pertaining to the creation
   of NexthopGroup Adjacencies in Sysdb. Refer to AID4354 for implementation details.
   '''

   def __init__( self,
                 # outputs
                 routingHwStatus,
                 nhgHwDefaultVrfStatus,
                 nhrConfig,
                 proactiveArpNexthopConfig,
                 # inputs
                 nhrStatus,
                 brConfig,
                 brStatus,
                 fs,
                 fs6,
                 fsGen,
                 arpSmash,
                 fwdHelper,
                 lfib,
                 nhgPickAllocator=None,
                 em=None,
                 resolverRoot=None,
                 smashBrStatus=None,
                 startArex=False ):
      if startArex:
         return

      self.routingHwStatus = routingHwStatus

      nhVrfConfig = nhrConfig.newVrf( DEFAULT_VRF, NhFecIdType.fecId )
      nhgAdjSmHelper = NhgAdjSmHelper( brConfig, brStatus, nhVrfConfig, arpSmash,
                                       proactiveArpNexthopConfig,
                                       nhgPickAllocator )

      scheduler = Tac.Type( 'Ark::TaskSchedulerRoot' ).findOrCreateScheduler()
      backlog = Tac.newInstance( 'Routing::NexthopBacklog' )
      self.nhgL3ResolverSm = NhgL3ResolverSm( nhrStatus, fs, fs6, fsGen, backlog,
                                              scheduler )
      self.nhgNhBacklogSm = NhgNhBacklogSm( fwdHelper, nhgAdjSmHelper, nhrStatus,
                                            brConfig, brStatus, backlog )

      self.nhgArpSm = NhgArpSm( nhgAdjSmHelper, arpSmash )
      self.nhgBridgingStatusSm = NhgBridgingStatusSm( nhgAdjSmHelper, brConfig,
                                                      brStatus )
      self.nhgAdjManagerSm = NhgAdjManagerSm( fwdHelper, nhgAdjSmHelper,
                                              nhgHwDefaultVrfStatus, lfib,
                                              nhrStatus, brConfig, brStatus )

factoryInstances = []

def mplsNhgHardwareStatusSmFactory( bridge ):
   em = bridge.em()
   defaultVrf = DEFAULT_VRF
   path = 'routing/hardware/nexthopgroup/status/'
   nhgHwDefaultVrfStatus = em.entity( path ).vrfStatus[ defaultVrf ]

   global inArfaMode
   inArfaMode = bridge.inArfaMode()
   nhgPickAllocator = None
   emArg = None
   smashBrStatus = None
   resolverRoot = None
   if inArfaMode:
      smashBrStatus = bridge.sEm().getEntity[ "bridging/status" ]
      emArg = em
      resolverRoot = (
         bridge.arfaRoot_.arfaPlugins.plugin[ 'CoreRouting' ].resolverRoot )

   factory = MplsNhgHardwareStatusSm(
         routingHwStatus=em.entity( 'routing/hardware/status' ),
         nhgHwDefaultVrfStatus=nhgHwDefaultVrfStatus,
         nhrConfig=bridge.nhrConfig,
         proactiveArpNexthopConfig=bridge.proactiveArpNhgNexthopConfig,
         nhrStatus=bridge.nhrStatus,
         brConfig=bridge.brConfig,
         brStatus=bridge.brStatus,
         fs=bridge.forwardingStatus_,
         fs6=bridge.forwarding6Status_,
         fsGen=bridge.forwardingGenStatus_,
         arpSmash=bridge.arpSmash_,
         fwdHelper=bridge.fwdHelper,
         lfib=bridge.lfib_,
         nhgPickAllocator=nhgPickAllocator,
         em=emArg,
         resolverRoot=resolverRoot,
         smashBrStatus=smashBrStatus,
         startArex=inArfaMode )

   factoryInstances.append( factory )
   return factory
