#!/usr/bin/env python3
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

"""
We implement hooks so that EbraTestBridge calls us before sending
a packet out an interface or trapping it to the CPU. 
In each case, we want to consult the FlowTable(s) and, if we have a
match, take the appropriate actions.
"""

import Tac, Tracing
import struct, types
import Arnet.PktParserTestLib, if_ether_arista
import EbraTestBridge, EbraTestBridgePort, IpUtils # pylint: disable=unused-import
import EbraTestBridgeLib
import Toggles.OpenFlowToggleLib

handle = Tracing.Handle( "OpenFlowEtbaPlugin" )

t0 = handle.trace0
t1 = handle.trace1
t2 = handle.trace2
t3 = handle.trace3

sendReason = Tac.Type( 'OpenFlowTable::OutputControllerReason' )

def agentInit( em ):
   "Etba needs access to the config and flow table(s)."
   t0( "agentInit" )
   EbraTestBridge.EbraTestPhyPort.normalSendFrame_ = \
       EbraTestBridge.EbraTestPhyPort.sendFrame
   EbraTestBridge.EbraTestPhyPort.sendFrame = sendFrame
   em.mount( "openflow/status", "OpenFlow::Status", "r" )
   em.mount( "openflow/hwstatus", "OpenFlowTable::HwStatus", "w" )
   em.mount( "openflow/openflowhwconfig", "OpenFlowTable::OpenFlowHwConfig", "r" )
   em.mount( "openflow/countercontrol", "OpenFlowTable::FlowCtrControl", "r" )

maxTableEntries = 1000

def bridgeInit( bridge ):
   "Make it easier to access status and flow table(s)."
   t0( "bridgeInit" )
   bridge.openFlowHwConfig_ = bridge.em().entity( 'openflow/openflowhwconfig' )
   bridge.openFlowHwStatus_ = bridge.em().entity( 'openflow/hwstatus' )
   bridge.counterControl_ = bridge.em().entity( 'openflow/countercontrol' )
   vms = Tac.Value( "OpenFlowTable::VxlanMatchSet" )
   mf0 = Tac.Value( "OpenFlowTable::MatchFieldSet" )
   mf0.inIntf = True
   mf0.ethSrc = True
   mf0.ethDst = True
   mf0.vlanId = True
   mf0.vlanPri = True
   mf0.ethType = True
   mf1 = Tac.Value( "OpenFlowTable::MatchFieldSet" )
   mf1.inIntf = True
   mf1.ethSrc = True
   mf1.ethDst = True
   mf1.vlanId = True
   mf1.vlanPri = True
   mf1.ethType = True
   mf1.ipSrc = True
   mf1.ipDst = True
   mf1.ipTos = True
   mf1.ipProto = True
   mf1.l4Src = True
   mf1.l4Dst = True
   mf1.icmpType = True
   mf1.icmpCode = True
   a0 = Tac.Value( "OpenFlowTable::ActionSet" )
   a0.outputNormal = True
   a0.ingrMirror = True
   a0.egrMirror = True
   a1 = Tac.Value( "OpenFlowTable::ActionSet" )
   a1.outputIntf = True
   a1.outputInIntf = True
   a1.outputNormal = False
   a1.outputFlood = True
   a1.outputAll = True
   a1.outputController = True
   a1.outputLocal = False
   a1.outputQueue = True
   a1.ingrMirror = True
   a1.egrMirror = True
   a1.setVlanId = True
   a1.setVlanPri = True
   a1.setEthSrc = True
   a1.setEthDst = True
   a1.setIpSrc = True
   a1.setIpDst = True
   a1.setIpTos = True
   a1.setL4Src = True
   a1.setL4Dst = True
   a1.setIcmpType = True
   a1.setIcmpCode = True
   td0 = bridge.openFlowHwStatus_.newSupportedTableDesc( "bindModeInterface" )
   td0.newTableDesc( "tableProfileFullMatch", mf1, vms, False, mf1, vms, a1,
                     maxTableEntries )
   td0.newTableDesc( "tableProfileL2Match", mf0, vms, False, mf1, vms, a1,
                     maxTableEntries )
   td1 = bridge.openFlowHwStatus_.newSupportedTableDesc( "bindModeVlan" )
   td1.newTableDesc( "tableProfileFullMatch", mf1, vms, False, mf1, vms, a1,
                     maxTableEntries )
   td1.newTableDesc( "tableProfileL2Match", mf0, vms, False, mf1, vms, a1,
                     maxTableEntries )
   td2 = bridge.openFlowHwStatus_.newSupportedTableDesc( "bindModeMonitor" )
   td2.newTableDesc( "tableProfileFullMatch", mf1, vms, False, mf1, vms, a0,
                     maxTableEntries )
   td2.newTableDesc( "tableProfileL2Match", mf0, vms, False, mf1, vms, a0,
                     maxTableEntries )
   bridge.openFlowHwStatus_.openFlowSupported = True
   bridge.openFlowHwStatus_.groupsSupported = True
   t0( "openFlowHwConfig = %s" % bridge.openFlowHwConfig_ )
   t0( "openFlowHwStatus = %s" % bridge.openFlowHwStatus_ )
   bridge.openFlowTable_ = {}
   bridge.openFlowIntfStats_ = {}
   bridge.openFlowRecircPort_ = None
   bridge.groupTable = {}
   bridge.openFlowHwConfigReactor_ = HwConfigReactor( bridge )
   bridge.openFlowCtrControlReactor_ = \
       FlowCtrControlReactor( bridge, bridge.openFlowHwConfigReactor_ )
   bridge.normalProcessFrame_ = bridge.processFrame
   bridge.processFrame = types.MethodType( processFrame, bridge )

def headerToMatch( ingress, data ): # pylint: disable=inconsistent-return-statements
   "Extract 10-tuple from packet header and return a Flow specifier."
   t0( "headerToMatch" )
   t0( "extracting packet (size=%s)" % len( data ) )
   ( p, headers, offset ) = Arnet.PktParserTestLib.parsePktStr( data )
   t0( "headers = %s" % headers )
   t0( "offset = %s" % offset )
   ethHdr = Arnet.PktParserTestLib.findHeader( headers, "EthHdr" )
   if not ethHdr:
      t0( "no ethHdr?" )
      return
   dot1qHdr = Arnet.PktParserTestLib.findHeader( headers, "EthDot1QHdr")
   ipHdr = Arnet.PktParserTestLib.findHeader( headers, "IpHdr" )
   udpHdr = Arnet.PktParserTestLib.findHeader( headers, "UdpHdr" )
   tcpHdr = Arnet.PktParserTestLib.findHeader( headers, "TcpHdr" )
   icmpHdr = Arnet.PktParserTestLib.findHeader( headers, "IcmpHdr" )
   snapHdr = Arnet.PktParserTestLib.findHeader( headers, "EthSnapHdr" )

   match = Tac.Value( "OpenFlowTable::Match" )
   matched = Tac.Value( "OpenFlowTable::MatchFieldSet" )
   match.inIntf[ ingress ] = True
   matched.inIntf = True
   t0( "ethHdr = %s" % ethHdr )
   match.ethSrc = ethHdr.src
   match.ethSrcMask = "ff:ff:ff:ff:ff:ff"
   matched.ethSrc = True
   match.ethDst = ethHdr.dst
   match.ethDstMask = "ff:ff:ff:ff:ff:ff"
   matched.ethDst = True
   match.ethType = ethHdr.typeOrLen
   matched.ethType = True
   if dot1qHdr:
      t0( "dot1qHdr = %s" % dot1qHdr )
      match.ethType = dot1qHdr.typeOrLen
      matched.ethType = True
      match.vlanId = dot1qHdr.tagControlVlanId
      match.vlanIdMask = 0x0fff
      matched.vlanId = True
      match.vlanPri = dot1qHdr.tagControlPriority
      matched.vlanPri = True
   elif p.tciPresent:
      match.vlanId = p.tciVlanId
      matched.vlanId = True
      match.vlanPri = p.tciPriority
      matched.vlanPri = True
   if snapHdr:
      t0( "snapHdr = %s" % snapHdr )
      match.ethType = snapHdr.localCode
      matched.ethType = True
   if ipHdr:
      t0( "ipHdr = %s" % ipHdr )
      match.ipSrc = ipHdr.src
      match.ipSrcMask = "255.255.255.255"
      matched.ipSrc = True
      match.ipDst = ipHdr.dst
      match.ipDstMask = "255.255.255.255"
      matched.ipDst = True
      match.ipTos = ipHdr.tos
      matched.ipTos = True
      match.ipProto = ipHdr.protocol
      matched.ipProto = True
      match.ttl = ipHdr.ttl
      match.ttlMask = 0xff
      matched.ttl = True
   if udpHdr:
      t0( "udpHdr = %s" % udpHdr )
      match.l4Src = udpHdr.srcPort
      matched.l4Src = True
      match.l4Dst = udpHdr.dstPort
      matched.l4Dst = True
   if tcpHdr:
      t0( "tcpHdr = %s" % tcpHdr )
      match.l4Src = tcpHdr.srcPort
      matched.l4Src = True
      match.l4Dst = tcpHdr.dstPort
      matched.l4Dst = True
   if icmpHdr:
      t0( "icmpHdr = %s" % icmpHdr )
      match.icmpType = icmpHdr.type
      matched.icmpType = True
      match.icmpCode = icmpHdr.code
      matched.icmpCode = True
   match.matched = matched
   return match

def macToNumber( mac ):
   return int( mac.replace( ':', '' ), 16 )

def ipToNumber( ip ):
   hexn = ''.join( "%02X" % int( i ) for i in ip.split( '.' ) )
   return int( hexn, 16 )

def matchInFlowTable( pkt, flowTable, nativeVlanId ):
   "Check whether a packet's flow spec matches in a flow table."
   #t0( "matchInFlowTable: pkt: %s" % pkt )
   t = sorted(
      flowTable.items(),
      key=lambda i: i[1].flowEntry.priority,
      reverse=True )
   t0( "matchInFlowTable: pkt=%s flowTable=%s" % (pkt, t) )
   for flowName, flow in t: # pylint: disable=too-many-nested-blocks
      m = flow.flowEntry.match
      #t0( "matchInFlowTable: flowEntry.match: %s" % m )
      if m.matched.inIntf \
            and ( len( [ intf for intf in pkt.inIntf if intf in m.inIntf ] ) == 0 ):
         continue
      if m.matched.ethSrc:
         mask = macToNumber( m.ethSrcMask )
         pktSrc = macToNumber( pkt.ethSrc )
         flowSrc = macToNumber( m.ethSrc )
         if mask & pktSrc != mask & flowSrc:
            continue
      if m.matched.ethDst:
         mask = macToNumber( m.ethDstMask )
         pktDst = macToNumber( pkt.ethDst )
         flowDst = macToNumber( m.ethDst )
         if mask & pktDst != mask & flowDst:
            continue
      if m.matched.vlanId:
         vlanId = m.vlanId
         if m.vlanId == 0:
            vlanId = nativeVlanId
            if pkt.vlanId != vlanId:
               continue
         else:
            if pkt.vlanId & m.vlanIdMask != vlanId & m.vlanIdMask:
               continue
      if m.matched.vlanPri:
         if pkt.vlanPri != m.vlanPri:
            continue
      if m.matched.ethType:
         if pkt.ethType != m.ethType:
            continue
         if m.ethType == 0x0800: # IP
            if m.matched.ipSrc:
               mask = ipToNumber( m.ipSrcMask )
               pktIp = ipToNumber( pkt.ipSrc )
               flowIp = ipToNumber( m.ipSrc )
               if mask & pktIp != mask & flowIp:
                  continue
            if m.matched.ipDst:
               mask = ipToNumber( m.ipDstMask )
               pktIp = ipToNumber( pkt.ipDst )
               flowIp = ipToNumber( m.ipDst )
               if mask & pktIp != mask & flowIp:
                  continue
            # Only match on dscp, ignore ECN bits.
            if m.matched.ipTos and ( pkt.ipTos & 0xfc != m.ipTos & 0xfc ):
               continue
            if m.matched.ipProto:
               if pkt.ipProto != m.ipProto:
                  continue
               if m.ipProto in [ 6, 17 ]: # TCP or UDP
                  if m.matched.l4Src and pkt.l4Src != m.l4Src:
                     continue
                  if m.matched.l4Dst and pkt.l4Dst != m.l4Dst:
                     continue
               if m.ipProto == 1: # ICMP
                  if m.matched.icmpType and pkt.icmpType != m.icmpType:
                     continue
                  if m.matched.icmpCode and pkt.icmpCode != m.icmpCode:
                     continue
            if m.matched.ttl:
               if pkt.ttl & m.ttlMask != m.ttl & m.ttlMask:
                  continue
      return flowName, flow

def handleOutputs( config, actions, srcPort ):
   outputSet = set()
   if actions.enabled.outputIntf:
      outputSet.update( set( actions.outputIntf.keys() ) )
   if actions.enabled.outputInIntf:
      outputSet.update( { srcPort } )
   if actions.enabled.outputNormal:
      pass # TODO
   if actions.enabled.outputFlood:
      outputSet.update( set( config.flowInterface.keys() ) - { srcPort } )
   if actions.enabled.outputAll:
      outputSet.update( set( config.flowInterface.keys() ) - { srcPort } )
   if actions.enabled.outputController:
      outputSet.update( { "Controller" } )
   if actions.enabled.outputLocal:
      pass # TODO
   if actions.enabled.outputQueue:
      pass # queue number is ignored, all output packets are counted as queue 0
   if actions.enabled.ingrMirror:
      outputSet.update( set( actions.ingrMirrorIntf.keys() ) )
      # FIXME: should copy the unmodified packet
   if actions.enabled.egrMirror:
      outputSet.update( set( actions.egrMirrorIntf.keys() ) )
   return outputSet

def handleData( actions, data, vlanId, nativeVlanId ):
   t0( "handleData" )
   # pylint: disable-next=unused-variable
   p, headers, offset = Arnet.PktParserTestLib.parsePktStr( data )
   t0( "original packet size=%s headers=%s" % (len( data ), headers) )
   ethHdr = Arnet.PktParserTestLib.findHeader( headers, "EthHdr" )
   if not ethHdr:
      return p.stringValue
   dot1qHdr = Arnet.PktParserTestLib.findHeader( headers, "EthDot1QHdr" )
   if actions.enabled.setVlanId or actions.enabled.setVlanPri:
      vlanPri = dot1qHdr.tagControlPriority
      vlanAct = EbraTestBridgeLib.PKTEVENT_ACTION_REPLACE
      if actions.enabled.setVlanId:
         if actions.setVlanId == 0:
            vlanId = nativeVlanId
         else:
            vlanId = actions.setVlanId
      if actions.enabled.setVlanPri:
         vlanPri = actions.setVlanPri
      t0( "applyVlanChanges pri=%s id=%s act=%s" % (vlanPri, vlanId, vlanAct) )
      data = EbraTestBridgeLib.applyVlanChanges( data, vlanPri, vlanId, vlanAct )
   p, headers, offset = Arnet.PktParserTestLib.parsePktStr( data )
   t0( "packet after vlan actions size=%s headers=%s" % (len( data ), headers) )
   ethHdr = Arnet.PktParserTestLib.findHeader( headers, "EthHdr" )
   if ethHdr:
      if actions.enabled.setEthSrc:
         ethHdr.src = actions.setEthSrc
      if actions.enabled.setEthDst:
         ethHdr.dst = actions.setEthDst
   data = p.stringValue
   p, headers, offset = Arnet.PktParserTestLib.parsePktStr( data )
   t0( "packet after eth actions size=%s headers=%s" % (len( data ), headers) )
   ipHdr = Arnet.PktParserTestLib.findHeader( headers, "IpHdr" )
   if ipHdr:
      if actions.enabled.setIpSrc:
         ipHdr.src = actions.setIpSrc
      if actions.enabled.setIpDst:
         ipHdr.dst = actions.setIpDst
      if actions.enabled.setIpTos:
         ipHdr.tos = actions.setIpTos
      ipHdr.checksum = 0
      ipHdr.checksum = ipHdr.computedChecksum
   data = p.stringValue
   p, headers, offset = Arnet.PktParserTestLib.parsePktStr( data )
   t0( "packet after ip actions size=%s headers=%s" % (len( data ), headers) )
   udpHdr = Arnet.PktParserTestLib.findHeader( headers, "UdpHdr" )
   if udpHdr:
      if actions.enabled.setL4Src:
         udpHdr.srcPort = actions.setL4Src
      if actions.enabled.setL4Dst:
         udpHdr.dstPort = actions.setL4Dst
      if udpHdr.checksum != 0:
         # ipv6/udp packet will not have the ipHdr initialized.
         if ipHdr:
            udpHdr.ipSrc = ipHdr.src
            udpHdr.ipDst = ipHdr.dst
         udpHdr.checksum = 0
         udpHdr.checksum = udpHdr.computedChecksum
   tcpHdr = Arnet.PktParserTestLib.findHeader( headers, "TcpHdr" )
   if tcpHdr:
      if actions.enabled.setL4Src:
         tcpHdr.srcPort = actions.setL4Src
      if actions.enabled.setL4Dst:
         tcpHdr.dstPort = actions.setL4Dst
      # ipv6/tcp packet will not have the ipHdr initialized.
      if ipHdr:
         tcpHdr.ipSrc = ipHdr.src
         tcpHdr.ipDst = ipHdr.dst
         tcpHdr.tcpLen = ipHdr.totalLen - ipHdr.headerBytes
         tcpHdr.checksum = 0
         tcpHdr.checksum = tcpHdr.computedChecksum
   return p.stringValue, vlanId

def _vlanTagPresent( data ):
   if data[ 12 ] == 0x81 and data[ 13 ] == 0x00:
      t0( data )
      t0( "Vlan tag present" )
      return True
   t0( "Vlan tag NOT present" )
   return False

def vlanIdFromTag( data ):
   if _vlanTagPresent( data ):
      hi = ( data[ 14 ] & 0x0f ) << 8
      lo = data[ 15 ]
      t0( "Vlan tag =%d" % ( hi | lo ) )
      return ( hi | lo ) # pylint: disable=superfluous-parens
   return 0

def addOrReplaceVlanIdInTag( data, vlanId ):
   vlanIdHi = ( vlanId >> 8 ) & 0x0f
   vlanIdLo = vlanId & 0xff
   t0( "Vlan tag =%d - %d" % ( vlanIdHi, vlanIdLo ) )
   if _vlanTagPresent( data ):
      currentVlanIdHi = data[ 14 ] & 0xf0
      t0( "current VlanId %d" % currentVlanIdHi )
      return ( data[ :14 ] +
               struct.pack( ">B", currentVlanIdHi | vlanIdHi ) +
               struct.pack( ">B", vlanIdLo ) +
               data[ 16: ] )
   else:
      return ( data[ :12 ] +
               b'\x81' +
               b'\x00' +
               struct.pack( ">B", vlanIdHi ) +
               struct.pack( ">B", vlanIdLo ) +
               data[ 12: ] )

def processOpenFlowFrame( bridge, srcPort, vlanId, tagged, dstMacAddr, data ):

   """If a packet is from an OpenFlow-controlled VLAN (and port),
      modify and forward it according to the appropriate flow table."""
   t0( "processOpenFlowFrame: data-type %s" % type( data ) )
   if srcPort.name() == "Cpu":
      srcIntfName = "OpenFlowRouter"
   else:
      srcIntfName = srcPort.name()

   if bridge.openFlowHwConfig_.bindMode == "bindModeInterface":
      if not tagged:
         vlanId = 0

   t0( "processOpenFlowFrame: srcPort=%s vlanId=%s ethDst=%s len(data)=%s" % (
         srcIntfName, vlanId, dstMacAddr, len( data ) ) )

   if srcIntfName == "OpenFlowRouter":
      rvlans = { v[ 1 ]: v[ 0 ] for v in
                 bridge.openFlowHwConfig_.routedVlan.items() }
      if vlanId in rvlans:
         t0( "Replacing vlanId %s (%s) with %s" % (
               vlanId, vlanIdFromTag( data ), rvlans[ vlanId ]) )
         vlanId = rvlans[ vlanId ]

   # Simulate VLAN tagging on ingress
   data = addOrReplaceVlanIdInTag( data, vlanId )

   # Ignore packets that aren't for OpenFlow
   if vlanId not in bridge.openFlowHwConfig_.flowVlan:
      t0( "Non-OpenFlow vlan %s - ignoring packet" % vlanId )
      return False

   if srcIntfName not in bridge.openFlowHwConfig_.flowInterface:
      t0( "Interface %s not in %s - ignoring packet" % (
            srcIntfName, list( bridge.openFlowHwConfig_.flowInterface ) ) )
      return False

   # Look up and process
   pktMatch = headerToMatch( srcIntfName, data )
   if not pktMatch:
      return True # drop the packet

   matchFlow = matchInFlowTable( pktMatch, bridge.openFlowTable_,
                                 bridge.openFlowHwConfig_.nativeVlan )
   if not matchFlow:
      t0( "Didn't match - ignoring packet!" )
      return False

   flowName, flow = matchFlow
   t0( "Matched packet! flow = %s, actions = %s" %
       (flowName, flow.flowEntry.actions) )

   # Update statistics
   flow.packetCount += 1
   # XXX Should this include the (possibly added) default
   # or native vlan tag? We may need to adjust this.
   flow.byteCount += len( data )
   flow.lastMatchTime = Tac.now()

   # Perform actions
   outputIntfs = handleOutputs(
      bridge.openFlowHwConfig_, flow.flowEntry.actions, srcIntfName )
   t0( "returning outputIntfs = %s" % outputIntfs )
   data, vlanId = handleData( flow.flowEntry.actions, data, vlanId,
                              bridge.openFlowHwConfig_.nativeVlan )

   t0( "processOpenFlowFrame: data = %s" % Tracing.HexDump( data ) )

   for intf in outputIntfs:
      if intf == "Controller":
         p = bridge.openFlowRecircPort_ if srcPort.name() == "Cpu" else srcPort
         if not p:
            continue
         t0( "Sending packet to controller srcPort=%s" % p.name() )
         etbaHeader = struct.pack( "!6s6sH",
                                   b"",
                                   b"",
                                   if_ether_arista.ETH_P_ARISTA_OPENFLOW_ETBA )
         if ( flow.flowEntry.actions.outputControllerReason ==
              sendReason.noMatchingFlow ):
            reason = if_ether_arista.OPENFLOW_COPY_REASON_NO_MATCH
         elif ( flow.flowEntry.actions.outputControllerReason ==
                sendReason.invalidTtl ):
            reason = if_ether_arista.OPENFLOW_COPY_REASON_INVALID_TTL
         else:
            reason = if_ether_arista.OPENFLOW_COPY_REASON_ACTION
         openFlowHeader = struct.pack( "!BBBBL", 0, 0, 0, reason, 0 )
         p.trapFrame( etbaHeader + openFlowHeader + data )
      else:
         vlan = vlanId
         vlanAction = None
         if intf == "OpenFlowRouter":
            p = bridge.port[ "Cpu" ]
            rvlans = dict( bridge.openFlowHwConfig_.routedVlan.items() )
            if vlan in rvlans:
               t0( "Replacing vlan %s (%s) with %s for OpenFlowRouter" % (
                     vlan, vlanIdFromTag( data ), rvlans[ vlan ]) )
               vlan = rvlans[ vlan ]
               vlanAction = EbraTestBridgeLib.PKTEVENT_ACTION_REPLACE
         elif bridge.openFlowHwConfig_.bindMode == "bindModeInterface":
            p = bridge.port[ intf ]
            if vlan == 0:
               t0( "Stripping vlan %s for %s" % (vlan, intf) )
               vlanAction = EbraTestBridgeLib.PKTEVENT_ACTION_REMOVE
         else:
            p = bridge.port[ intf ]
            c = bridge.brConfig_.switchIntfConfig[ intf ]
            if c.switchportMode != "trunk" or c.nativeVlan == vlan:
               t0( "Stripping vlan %s for %s" % (vlan, intf) )
               vlanAction = EbraTestBridgeLib.PKTEVENT_ACTION_REMOVE
         t0( "Sending packet to %s" % p.name() )
         p.sendFrame( data, pktMatch.ethSrc, dstMacAddr, srcPort.name(),
                      pktMatch.vlanPri, vlan, vlanAction )
         try:
            intfStats = bridge.openFlowIntfStats_[ intf ]
            intfStats.packetCount += 1
            intfStats.byteCount += len( data )
         except KeyError:
            pass

   if bridge.openFlowHwStatus_.bindMode == "bindModeMonitor":
      return False
   else:
      return True

def processFrame( bridge, data, srcMacAddr, dstMacAddr, srcPort, tracePkt,
                  highOrderVlanBits=None ):
   t0( "processFrame: data-type %s" % type( data ) )
   etherType = ( data[ 12 ] << 8 ) | data[ 13 ]
   t0( "processFrame: srcMacAddr=%s dstMacAddr=%s srcPort=%s etherType=0x%04x" % (
         srcMacAddr, dstMacAddr, srcPort.name(), etherType ) )
   if srcPort.name() == "Cpu":
      srcIntfName = "OpenFlowRouter"
   else:
      srcIntfName = srcPort.name()
   if srcIntfName in bridge.openFlowHwConfig_.flowInterface:
      # pylint: disable-next=unused-variable
      vlanId, priority, vlanTagNeedsUpdating, tagged, routed = bridge.vlanTagState(
         srcPort.name(), srcPort.name() == "Cpu", etherType, data, dstMacAddr,
         highOrderVlanBits )
      if ( srcIntfName != "OpenFlowRouter" or
           vlanId in bridge.openFlowHwConfig_.routedVlan or
           vlanId in bridge.openFlowHwConfig_.routedVlan.values() ):
         if processOpenFlowFrame( bridge, srcPort, vlanId, tagged, dstMacAddr,
                                  data ):
            return None
   c = bridge.brConfig_.switchIntfConfig.get( srcIntfName )
   if c and c.nativeVlan == 0 and c.switchportMode == "access":
      # This should never happen, but when it does, it causes
      # EbraTestBridge.processFrame to get vlanId == 0 from vlanTagState
      # and pass it to Mroute.route, which blows up in vlanIdToIntfId
      return None
   return bridge.normalProcessFrame_( data, srcMacAddr, dstMacAddr, srcPort,
                                      tracePkt, highOrderVlanBits )

def sendFrame( port, data, srcMacAddr, dstMacAddr, srcPortName,
               priority, vlanId, vlanAction ):
   t0( "sendFrame: port=%s srcMacAddr=%s dstMacAddr=%s vlanId=%s vlanAction=%s" % (
         port.name(), srcMacAddr, dstMacAddr, vlanId, vlanAction ) )
   if port == port.bridge_.openFlowRecircPort_:
      rvlans = dict( port.bridge_.openFlowHwConfig_.routedVlan.items() )
      vlanId = vlanIdFromTag( data ) or 0
      if vlanId not in rvlans:
         t0( "no routed vlan for packet vlan %d" % vlanId )
         return
      rVlanId = rvlans[ vlanId ]
      t0( "replacing packet vlan %d with routed vlan %d" % (vlanId, rVlanId) )
      data = addOrReplaceVlanIdInTag( data, rVlanId )
      t0( "sending packet to cpu port" )
      port.bridge_.port[ "Cpu" ].sendFrame( data, None, None, None,
                                            None, None, None )
      return
   port.normalSendFrame_( data, srcMacAddr, dstMacAddr, srcPortName,
                          priority, vlanId, vlanAction )

class FlowEntryReactor( Tac.Notifiee ):
   notifierTypeName = "OpenFlowTable::FlowEntry"

   def __init__( self, flowEntry, hwConfigReactor, flowName ):
      t0( "FlowEntryReactor.__init__" )
      self.hwConfigReactor = hwConfigReactor
      self.flowName = flowName
      Tac.Notifiee.__init__( self, flowEntry )

   @Tac.handler( "changeCount" )
   def handle( self ):
      t0( "FlowEntryReactor.handle" )
      self.hwConfigReactor.handleFlowEntry( self.flowName )

class GroupEntryReactor( Tac.Notifiee ):
   notifierTypeName = "OpenFlowTable::GroupEntry"

   def __init__( self, groupEntry, hwConfigReactor, groupName ):
      t0( "GroupEntryReactor.__init__" )
      self.hwConfigReactor = hwConfigReactor
      self.groupName = groupName
      Tac.Notifiee.__init__( self, groupEntry )

   @Tac.handler( "changeCount" )
   def handle( self ):
      t0( "GroupEntryReactor.handle" )
      self.hwConfigReactor.handleGroupEntry( self.groupName )


class HwConfigReactor( Tac.Notifiee ):
   notifierTypeName = "OpenFlowTable::OpenFlowHwConfig"

   def __init__( self, bridge ):
      t0( "HwConfigReactor.__init__" )
      self.bridge = bridge
      self.enabled = Toggles.OpenFlowToggleLib.toggleOpenFlowMultiTableModeEnabled()
      self.updateCounterActivity = Tac.ClockNotifiee( self.updateCounters )
      self.pollFlowStateActivity = Tac.ClockNotifiee( self.pollFlowState )
      Tac.Notifiee.__init__( self, self.bridge.openFlowHwConfig_ )
      self.handleBindMode()
      self.handleTableProfile()

   def updateCounters( self ):
      t0( "HwConfigReactor.updateCounters" )
      now = Tac.now()
      for name, flow in self.bridge.openFlowTable_.items():
         t0( "Updating flow: %s" % flow )
         try:
            flowStats = self.bridge.openFlowHwStatus_.flowStats[ name ]
         except KeyError:
            continue
         flowStats.byteCount += flow.byteCountDelta
         flow.byteCountDelta = 0
         flowStats.packetCount += flow.packetCountDelta
         flow.packetCountDelta = 0
      for intf, s in self.bridge.openFlowIntfStats_.items():
         t0( "Updating intf: %s" % intf )
         try:
            queueStats = self.bridge.openFlowHwStatus_.intfQueueStats[
               intf ].queueStats[ 0 ]
         except KeyError:
            t0( "KeyError Updating intf: %s" % intf )
            continue
         queueStats.byteCount += s.byteCount
         s.byteCount = 0
         queueStats.packetCount += s.packetCount
         s.packetCount = 0
      self.bridge.openFlowHwStatus_.lastFlowCounterUpdateTime = now
      self.bridge.openFlowHwStatus_.lastIntfQueueCounterUpdateTime = now
      self.handleCounterUpdateTime()

   def pollFlowState( self ):
      now = Tac.now()
      totalPacketCountDelta = 0
      defaultPacketCountDelta = 0
      for name, flow in self.bridge.openFlowTable_.items():
         flow.packetCountDelta += flow.packetCount
         totalPacketCountDelta += flow.packetCount
         if name == "__default__":
            defaultPacketCountDelta += flow.packetCount
         flow.packetCount = 0
         flow.byteCountDelta += flow.byteCount
         flow.byteCount = 0
         if name not in self.bridge.openFlowHwStatus_.flowStats:
            continue
         stats = self.bridge.openFlowHwStatus_.flowStats[ name ]
         if ( stats.createdTime > 0 ) and ( flow.flowEntry.hardTimeout > 0 ):
            if stats.createdTime + flow.flowEntry.hardTimeout < now:
               t0( "flow %s expired due to hard timeout" % name )
               del self.bridge.openFlowTable_[ name ]
               stats.removedTime = now
               stats.status = "flowExpiredHard"
               stats.changeCount = 0xffffffffffffffff
               continue
         if flow.flowEntry.idleTimeout > 0:
            if flow.lastMatchTime + flow.flowEntry.idleTimeout < now:
               t0( "flow %s expired due to idle timeout" % name )
               del self.bridge.openFlowTable_[ name ]
               stats.removedTime = now
               stats.status = "flowExpiredIdle"
               stats.changeCount = 0xffffffffffffffff
               continue
      self.bridge.openFlowHwStatus_.totalPacketCount += totalPacketCountDelta
      self.bridge.openFlowHwStatus_.defaultPacketCount += defaultPacketCountDelta
      self.pollFlowStateActivity.timeMin = Tac.now() + 2

   def setEnabled( self, enabled ):
      t0( "HwConfigReactor.setEnabled %s" % enabled )
      self.enabled = enabled
      for name in self.bridge.openFlowHwConfig_.flowEntry:
         self.handleFlowEntry( name )
      for intf in self.bridge.openFlowHwConfig_.flowInterface:
         self.handleFlowInterface( intf )
      self.handleCounterUpdateTime()
      if self.enabled:
         self.pollFlowStateActivity.timeMin = Tac.now()
      else:
         self.pollFlowStateActivity.timeMin = Tac.endOfTime

   @Tac.handler( "bindMode" )
   def handleBindMode( self ):
      t0( "HwConfigReactor.handleBindMode" )
      self.bridge.openFlowHwStatus_.bindMode = self.bridge.openFlowHwConfig_.bindMode
      for name in self.bridge.openFlowHwConfig_.flowEntry:
         self.handleFlowEntry( name )

   @Tac.handler( "tableProfile" )
   def handleTableProfile( self ):
      t0( "HwConfigReactor.handleTableProfile" )
      self.bridge.openFlowHwStatus_.tableProfile = \
          self.bridge.openFlowHwConfig_.tableProfile
      for name in self.bridge.openFlowHwConfig_.flowEntry:
         self.handleFlowEntry( name )

   @Tac.handler( "groupEntry" )
   def handleGroupEntry( self, groupId ):
      t0( "HwConfigReactor.handleGroupEntry id=%d" % groupId )
      t0( "Group table: ", self.bridge.groupTable )
      try:
         groupEntry = self.bridge.openFlowHwConfig_.groupEntry[ groupId ]
      except KeyError:
         t0( "groupEntry %d deleted in hwConfig" % groupId )
         groupEntry = None
         if groupId in self.bridge.openFlowHwStatus_.groupStats:
            t0( "Deleting groupStats for group id %d" % groupId )
            del self.bridge.openFlowHwStatus_.groupStats[ groupId ]
         if groupId in self.bridge.groupTable:
            t0( "Deleting entry %d from group table" % groupId )
            del self.bridge.groupTable[ groupId ]
      if groupEntry:
         if groupId not in self.bridge.groupTable:
            t0( "Creating reactor for groupEntry id=%d" % groupId )
            groupEntryReactor = GroupEntryReactor( groupEntry, self, groupId )
            self.bridge.groupTable[ groupId ] = Group( groupEntry,
                                                       groupEntryReactor )
         if not groupEntry.ready:
            t0( "groupEntry not yet ready" )
            return
         if groupId not in self.bridge.openFlowHwStatus_.groupStats:
            t0( "Creating groupStats for group id %d" % groupId )
            self.bridge.openFlowHwStatus_.newGroupStats( groupId, "groupCreated" )
            self.bridge.openFlowHwStatus_.groupStats[ groupId ].changeCount = \
                  groupEntry.changeCount
         else:
            if groupEntry.deleted:
               t0( "Group entry %d marked as deleted" % groupId )
               if groupId in self.bridge.groupTable:
                  t0( "Deleting entry %d from group table" % groupId )
                  del self.bridge.groupTable[ groupId ]
               self.bridge.openFlowHwStatus_.groupStats[ groupId ].status = \
                     "groupDeleted"
               self.bridge.openFlowHwStatus_.groupStats[ groupId ].changeCount = \
                     groupEntry.changeCount
            else:
               t0( "Modifying group %d" % groupId )
               self.bridge.openFlowHwStatus_.groupStats[ groupId ].changeCount = \
                     groupEntry.changeCount

   @Tac.handler( "flowEntry" )
   def handleFlowEntry( self, name ):
      t0( "HwConfigReactor.handleFlowEntry name=%s" % name )
      if name == "__default__":
         if name in self.bridge.openFlowHwConfig_.flowEntry:
            if not self.enabled:
               self.setEnabled( True )
         else:
            if self.enabled:
               self.setEnabled( False )
      try:
         flowEntry = self.bridge.openFlowHwConfig_.flowEntry[ name ]
      except KeyError:
         flowEntry = None
      if self.enabled and flowEntry and not flowEntry.deleted:
         self.addOrModifyFlowEntry( name, flowEntry )
      else:
         self.removeFlowEntry( name, flowEntry )

   def addOrModifyFlowEntry( self, name, flowEntry ):
      t0( "HwConfigReactor.addOrModifyFlowEntry name=%s" % name )
      if name in self.bridge.openFlowTable_:
         t0( "replacing flow %s" % name )
      else:
         t0( "adding flow %s" % name )
      status = "flowCreated"
      m = flowEntry.match
      if self.bridge.openFlowHwStatus_.tableProfile == "l2-match":
         if ( m.matched.ipSrc or m.matched.ipDst or m.matched.ipTos or
              m.matched.ipProto or m.matched.l4Src or m.matched.ipDst ):
            t0( "rejecting flow match on l3/l4 fields in l2-match mode" )
            status = "flowRejectedBadMatch"
      if m.matched.ethType and m.ethType == 0x0806:
         if m.matched.ipSrc or m.matched.ipDst or m.matched.ipProto:
            t0( "rejecting flow matching arp ethType and ipSrc or ipDst or ipProto")
            status = "flowRejectedBadMatch"
      a = flowEntry.actions
      if self.bridge.openFlowHwStatus_.bindMode == "bindModeMonitor":
         if ( a.enabled.setVlanId or a.enabled.setVlanPri or a.enabled.setEthSrc or
              a.enabled.setEthDst or a.enabled.setIpSrc or a.enabled.setIpDst or
              a.enabled.setIpTos or a.enabled.setL4Src or a.enabled.setL4Dst or
              a.enabled.setIcmpType or a.enabled.setIcmpCode or
              a.enabled.outputIntf or a.enabled.outputInIntf or
              a.enabled.outputFlood or a.enabled.outputAll or
              a.enabled.outputController or a.enabled.outputLocal or
              a.enabled.outputQueue ):
            t0( "rejecting flow with set or output flow action in monitor mode" )
            status = "flowRejectedBadAction"
         if not a.enabled.outputNormal:
            t0( "rejecting entry without output normal action in monitor mode" )
            status = "flowRejectedBadAction"
      if len( self.bridge.openFlowTable_ ) > maxTableEntries:
         status = "flowRejectedHwTableFull"
      if status == "flowCreated" and name not in self.bridge.openFlowTable_:
         flowEntryReactor = FlowEntryReactor( flowEntry, self, name )
         self.bridge.openFlowTable_[ name ] = Flow( flowEntry, flowEntryReactor )
      if name in self.bridge.openFlowHwStatus_.flowStats:
         stats = self.bridge.openFlowHwStatus_.flowStats[ name ]
         stats.status = status
         stats.changeCount = flowEntry.changeCount
      else:
         self.bridge.openFlowHwStatus_.newFlowStats( name, status, Tac.now(), 0,
                                     flowEntry.changeCount, True, "config",
                                     "notTwoStep" )

   def removeFlowEntry( self, name, flowEntry ):
      t0( "HwConfigReactor.removeFlowEntry name=%s" % name )
      flow = None
      if name in self.bridge.openFlowTable_:
         t0( "removing flow %s" % name )
         flow = self.bridge.openFlowTable_[ name ]
         del self.bridge.openFlowTable_[ name ]
      if not self.enabled or not flowEntry:
         # When the flow entry goes away, remove flow stats
         del self.bridge.openFlowHwStatus_.flowStats[ name ]
         return
      # As long as the flow entry still exists, keep flow stats
      # around, with status = flowDeleted
      if name in self.bridge.openFlowHwStatus_.flowStats:
         stats = self.bridge.openFlowHwStatus_.flowStats[ name ]
         stats.removedTime = Tac.now()
         stats.status = "flowDeleted"
         if flow:
            stats.byteCount += flow.byteCountDelta
            flow.byteCountDelta = 0
            stats.packetCount += flow.packetCountDelta
            flow.packetCountDelta = 0
         stats.changeCount = flowEntry.changeCount
      else:
         self.bridge.openFlowHwStatus_.newFlowStats(
            name, "flowDeleted", 0, Tac.now(), flowEntry.changeCount, True,
            "notTwoStep" )

   @Tac.handler( "flowInterface" )
   def handleFlowInterface( self, intf ):
      t0( "HwConfigReactor.handleFlowInterface %s" % intf )
      port = self.bridge.port.get( intf ) # pylint: disable=unused-variable
      if self.enabled and intf in self.bridge.openFlowHwConfig_.flowInterface:
         if intf not in self.bridge.openFlowIntfStats_:
            self.bridge.openFlowIntfStats_[intf] = IntfStats()
         self.bridge.openFlowHwStatus_.newIntfQueueStats( intf ).newQueueStats( 0 )
      else:
         if intf in self.bridge.openFlowIntfStats_:
            del self.bridge.openFlowIntfStats_[intf]
         del self.bridge.openFlowHwStatus_.intfQueueStats[intf]

   @Tac.handler( "bindVlan" )
   def handleBindVlan( self, vlan ):
      t0( "HwConfigReactor.handleBindVlan %s" % vlan )
      if self.enabled and vlan in self.bridge.openFlowHwConfig_.bindVlan:
         self.bridge.openFlowHwStatus_.bindVlan[vlan] = True
      else:
         del self.bridge.openFlowHwStatus_.bindVlan[vlan]

   @Tac.handler( "bindInterface" )
   def handleBindInterface( self, intf ):
      t0( "HwConfigReactor.handleBindInterface %s" % intf )
      if self.enabled and intf in self.bridge.openFlowHwConfig_.bindInterface:
         self.bridge.openFlowHwStatus_.bindInterface[intf] = True
      else:
         del self.bridge.openFlowHwStatus_.bindInterface[intf]

   @Tac.handler( "recircInterface" )
   def handleRecircInterface( self ):
      intf = self.bridge.openFlowHwConfig_.recircInterface
      rvlans = dict( self.bridge.openFlowHwConfig_.routedVlan )
      t0( "HwConfigReactor.handleRecircInterface %s" % intf )
      self.bridge.openFlowHwStatus_.routedVlan.clear()
      self.bridge.openFlowRecircPort_ = self.bridge.port.get( intf )
      if self.bridge.openFlowRecircPort_:
         for vlan, rvlan in rvlans.items():
            self.bridge.openFlowHwStatus_.routedVlan[vlan] = rvlan
      for vlan in list( self.bridge.openFlowHwStatus_.routedVlan ):
         if not self.bridge.openFlowRecircPort_ or vlan not in rvlans:
            del self.bridge.openFlowHwStatus_.routedVlan[vlan]

   @Tac.handler( "routedVlan" )
   def handleRoutedVlan( self, vlan ):
      self.handleRecircInterface()

   @Tac.handler( "intfQueueCounterUpdateTime" )
   def handleCounterUpdateTime( self ):
      last1 = self.bridge.openFlowHwStatus_.lastFlowCounterUpdateTime
      nxt1 = self.bridge.counterControl_.flowCounterUpdateTime
      last2 = self.bridge.openFlowHwStatus_.lastIntfQueueCounterUpdateTime
      nxt2 = self.bridge.openFlowHwConfig_.intfQueueCounterUpdateTime
      t0( "HwConfigReactor.handleCounterUpdateTime "
          "last1=%s next1=%s last2=%s next2=%s" % (last1, nxt1, last2, nxt2) )
      if self.enabled and nxt1 > last1:
         self.updateCounterActivity.timeMin = nxt1
      elif self.enabled and nxt2 > last2:
         self.updateCounterActivity.timeMin = nxt2
      else:
         self.updateCounterActivity.timeMin = Tac.endOfTime

class FlowCtrControlReactor( Tac.Notifiee ):
   notifierTypeName = "OpenFlowTable::FlowCtrControl"

   def __init__( self, bridge, hwConfigReactor ):
      t0( "FlowCtrControlReactor.__init__" )
      self.bridge = bridge
      self.hwConfigReactor = hwConfigReactor
      Tac.Notifiee.__init__( self, self.bridge.counterControl_ )

   @Tac.handler( "flowCounterUpdateTime" )
   def handle( self ):
      t0( "FlowCtrControlReactor.handle" )
      self.hwConfigReactor.handleCounterUpdateTime()

class Flow:
   def __init__( self, flowEntry, flowEntryReactor ):
      self.flowEntry = flowEntry
      self.flowEntryReactor = flowEntryReactor
      self.byteCount = 0
      self.byteCountDelta = 0
      self.packetCount = 0
      self.packetCountDelta = 0
      self.lastMatchTime = Tac.now()

class Group:
   def __init__( self, groupEntry, groupEntryReactor ):
      self.groupEntry = groupEntry
      self.groupEntryReactor = groupEntryReactor

class IntfStats:
   def __init__( self ):
      self.byteCount = 0
      self.packetCount = 0

def Plugin( ctx ):
   "Install OpenFlow hooks into Etba."
   t0( "Plugin init" )
   ctx.registerBridgeInitHandler( bridgeInit )
   ctx.registerAgentInitHandler( agentInit )
   t0( "Plugin init complete" )
