#!/usr/bin/env python3
# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-in

import struct, time
from if_ether_arista import ETH_P_ARISTA_PTP
import Tac, Tracing
from PtpPacketLib import ptpPacketType, ptpTwoStepFlag, ptpHdrOffset
import EbraTestBridgeConstants

handle = Tracing.Handle( 'EbraTestBridgePtp' )
t2 = handle.trace2
t3 = handle.trace3
t6 = handle.trace6
t7 = handle.trace7
t8 = handle.trace8

ptpCapability = None
ptpStatus = None
ptpConfig = None

noMatch = ( False, False, False )
forward = ( True, False, False )
trap = ( True, EbraTestBridgeConstants.TRAP_ANYWAY, False )
mirror = ( True, False, True )

MessageTypeNumber = Tac.Type( "Ptp::MessageTypeNumber" )
PTP_SYNC_MSG_TYPE = Tac.enumValue( MessageTypeNumber, "messageSync" )

def trapLookupHandler( bridge, routed, vlanId, destMac, data, srcPortName ):
   if ptpConfig.ptpMode == "ptpDisabled":
      t8( "PTP not trapping because it's disabled" )
      return noMatch

   packetType = ptpPacketType( data )
   if packetType is None:
      t8( "PTP not trapping because packet isn't PTP" )
      return noMatch

   if ptpConfig.ptpMode == "ptpBoundaryClock" or \
      ptpConfig.ptpMode == "ptpOneStepBoundaryClock":
      t3( "PTP trapping because of boundary clock mode" )
      return trap
   elif ptpConfig.ptpMode == "ptpGeneralized":
      t3( "PTP trapping because of gPTP clock mode" )
      return trap
   elif ptpConfig.ptpMode == "ptpEndToEndTransparentClock":
      if packetType == "messageSync" or packetType == "messageDelayReq":
         t3( "PTP mirroring to CPU" )
         return mirror
      elif packetType == "messageFollowUp" or packetType == "messageDelayResp":
         t3( "PTP trapping to CPU" )
         return trap
      else:
         t3( "PTP forwarding packet normally" )
         return forward
   elif ptpConfig.ptpMode == "ptpPeerToPeerTransparentClock":
      if packetType == "messageSync":
         t3( "PTP mirroring to CPU" )
         return mirror
      elif packetType in ( "messageFollowUp",
                           "messageDelayReq",
                           "messageDelayResp",
                           "messagePDelayReq",
                           "messagePDelayResp",
                           "messagePDelayRespFollowup" ):
         t3( "PTP trapping to CPU" )
         return trap
      else:
         t3( "PTP forwarding packet normally" )
         return forward
   else:
      t3( "Unknown PTP Mode!" )
   return noMatch

def rewriteHandler( bridge, routed, vlanId, destMac, data, srcPortName ):
   if ptpPacketType( data ) != None: # pylint: disable=singleton-comparison
      # Not a great timestamp for real PTP purposes, but should be sufficient
      # for protocol verification.
      # This timestamp is used to calculate the real ingress timestamp in
      # PtpPacketUtil using the following math:
      # ingressTs = Treal + ( Tmono - T0mono )
      # The timestamp below takes the place of Tmono due to which Tac.now() or some
      # function that returns monotonic time needs to be used.
      # The math is used to prevent the real timestamp to go backwards due to NTP
      # adjustments.
      timestamp = int( Tac.now() * 1e9 )
      t7( "Adding timestamp", hex( timestamp ) )
      ptpHeader = struct.pack( "!HQ", ETH_P_ARISTA_PTP, timestamp )
      # Stick the header between the DA/SA and the rest of the packet.
      data = data[ 0 : 12 ] + ptpHeader + data[ 12 : ]
      return ( data, destMac )
   else:
      t8( "PTP not rewriting packet" )
      return ( None, None )

def createOriginTimestamp( timestamp ):
   # Convert timestamp to seconds for 48 MSB of originTimestamp
   # Using 3 16-bit numbers to denote the 48-bit number
   originTimestampSecs = int( timestamp / 1e9 )
   originTimestampSecs1 = ( originTimestampSecs >> 32 ) & 0xffff
   originTimestampSecs2 = ( originTimestampSecs >> 16 ) & 0xffff
   originTimestampSecs3 = originTimestampSecs & 0xffff

   # Get Remainder of timestamp as nsecs
   originTimestampNsecs = int( timestamp % 1e9 )

   # Get OriginTimestamp that comprises of
   # 48-bit number denoting seconds ( 3 16-bit numbers )
   # 32-bit number denoting nsecs
   originTimestamp = struct.pack( "!HHHL", originTimestampSecs1,
                                  originTimestampSecs2, originTimestampSecs3,
                                  originTimestampNsecs )
   return originTimestamp

def egressRewrite( bridge, data, srcMacAddr, dstMacAddr, srcPortName, priority,
                   vlanId, vlanAction ):
   if ptpConfig.ptpMode != "ptpOneStepBoundaryClock":
      return None

   if ptpPacketType( data ) == PTP_SYNC_MSG_TYPE and not ptpTwoStepFlag( data ):
      t7( "Rewriting originTimestamp and correctionField" )
      offset = ptpHdrOffset( data )
      # This needs to be real time because this timestamp is used to generate the
      # originTimestamp field in the sync packets of 1-step BC.
      timestamp = int( time.time() * 1e9 )

      originTimestamp = createOriginTimestamp( timestamp )

      # Fixing correctionField to 1000 ( totally arbitrary )
      correction = 1000
      correctionField = struct.pack( "!Q", correction )
      # replace 8 byte correction field in PTP header
      # correction field is at offset 8 in PTP header
      header = data[ offset : offset + 8 ] + correctionField + \
               data[ offset + 16 : offset + 34 ]

      # Replace PTP Header and originTimestamp
      # PTP header is 34 bytes and originTimestamp is 10 bytes
      newData = data[ : offset ] + header + originTimestamp + data[ offset + 44 : ]
      return newData

   return None

def bridgeInit( bridge ):
   t2( 'Ptp bridgeInit' )

   ptpCapability.ptpBoundaryClockSupported = True
   ptpCapability.ptpE2ETransparentClockSupported = True
   ptpCapability.ptpP2PTransparentClockSupported = True
   ptpCapability.ptpGptpClockSupported = True
   ptpCapability.ptpOrdinaryMasterClockSupported = True
   ptpCapability.ptpSupported = True
   # TODO: This is only partial support as it enables PTP agent in etba dut
   # to send PTP packets with designated mac addresses of G8275.1, snooping packets
   # support has not been added yet.
   ptpCapability.ptpG82751BCSupported = True

   ptpCapability.ptpOneStepBoundaryClockSupported = True

   # We need to explicitly set clockQuality here for etba dut, especially the value
   # of clock accuracy to be 0x30, because in a few ptp ptests, like Ptp/PtpSmoke.py
   # , Ptp/GPtpTriangleTest.py and Ptp/GPtpCompetingMastersTest.py, the assumption
   # that namespace dut (like walta) has better clockQuality than Sand and Strata
   # dut has been made. This assumption is used in test to determine which dut should
   # be master when priority1 is in tie.
   # A better way is to change those tests and explicitly set the priority1, now we
   # fix this with this workaround.
   ptpStatus.clockQuality = Tac.Value(
         'Ptp::ClockQuality',
         248,
         0x30,
         Tac.Value( 'Arnet::NetU16', 0xFFFF )
         )

   for port in bridge.port:
      if not port.startswith( 'Ethernet' ):
         continue
      intfStatus = ptpStatus.newIntfStatus( port,
                   Tac.newInstance( 'Ptp::LinuxTsOptions' ) )
      intfStatus.twoStepClock = True
      intfStatus.timestampMultiplier = 1
      intfStatus.timestampDivider = 1
      intfStatus.timestampAtPacketEnd = False
      intfStatus.hwReadyForPacket = True

def agentInit( em ):
   t2( 'Ptp agentInit' )
   global ptpConfig, ptpStatus, ptpCapability
   ptpConfig = em.mount( 'ptp/hwConfig', 'Ptp::HwConfig', 'rO' )
   ptpCapability = em.mount( 'ptp/hwCapability/etba', 'Ptp::HwCapability', 'wc' )
   ptpStatus = em.mount( 'ptp/hwStatus/etba', 'Ptp::HwStatus', 'wc' )

def Plugin( ctx ):
   ctx.registerBridgeInitHandler( bridgeInit )
   ctx.registerAgentInitHandler( agentInit )
   ctx.registerTrapLookupHandler( trapLookupHandler,
                                  priority=ctx.TRAP_PRIORITIES[ "PTP" ] )
   ctx.registerRewritePacketOnTheWayToTheCpuHandler( rewriteHandler )
   ctx.registerEgressRewriteHandler( egressRewrite )
