# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Tac
import Tracing
import SharedMem
import Smash
from Arnet import (
   PktParserTestLib,
   IntfId,
)
from Arnet.EthTestLib import EthHdrSize
from Arnet.MplsLib import (
   constructMplsHeader,
)
from EbraTestBridgePort import EbraTestPort
from ForwardingHelper import Af
from MplsEtbaLib import (
   computeFlowOrEntropyLabel,
   ecmpHash,
   ELI,
   getTunnelEntry,
   removeImpNullFromLabelStack,
)
from EbraTestBridgePythonImpl import getPythonIntoCppShimContext
from MplsEtbaForwardingHelper import mplsEtbaForwardingHelperFactory
from SysConstants.if_ether_h import ETH_P_MPLS_UC
from TypeFuture import TacLazyType
from EbraTestBridgeLib import (
   PKTEVENT_ACTION_ADD,
   PKTEVENT_ACTION_REMOVE,
   PKTEVENT_ACTION_REPLACE,
   PKTEVENT_ACTION_NONE,
   applyVlanChanges,
)
import FibUtils

handle = Tracing.Handle( 'EtbaVpls' )
t0 = handle.trace0
t2 = handle.trace2
t8 = handle.trace8

PseudowireIntfId = TacLazyType( 'Arnet::PseudowireIntfId' )
FlowLabelType = TacLazyType( 'Pseudowire::FlowLabelType' )
gIntfStatusDirReactor = None
gAllIntfStatusDir = None

def debugPrint( msg, portName, pkt ):
   t8( f'{msg} {portName}' )
   t8( pkt )

def getGroupKeyFromShadow( bridge, intfId ):
   return bridge.vplsLabelToIntfIdSm.stream.shadowMap.shadow.get( intfId )

# The Etba agent loops thru sysdb path interface/status/eth/intf
# (of type Interface::EthIntfStatusDir). This is a collection of EthIntfStatus
# pointers of the various interface types.
# The registerInterfaceHandler() is used to register this callback class for
# EthIntfStatus that has the specific type of VplsEthIntfStatus, representing a PW
# port
class VplsPseudowirePort( EbraTestPort ):
   '''Simulates a VPLS Pseudowire port '''
   def __init__( self, bridge, tapDevice, trapDevice, intfConfig, intfStatus ):
      assert tapDevice is None
      assert trapDevice is None
      assert intfConfig is None
      assert intfStatus is not None
      assert PseudowireIntfId.isPwInstanceId( intfStatus.intfId )
      self.intfStatus = intfStatus
      self.bridge = bridge
      t2( "create port ", intfStatus.intfId )
      EbraTestPort.__init__( self, bridge, intfConfig, intfStatus )
      self.counter = bridge.vplsIntfCounters.intfCounter[ intfStatus.intfId ]

   def resolveTunnel( self, bridge, tunnel, ethHdr ):
      tunnelId = Tac.Type( "Tunnel::TunnelTable::TunnelId" )( tunnel )
      if not tunnelId.isValid():
         t0( f'TunnelId {tunnelId} is invalid' )
         return None
      tunEnt = getTunnelEntry( bridge.tunnelTables, bridge.pfs, ethHdr, tunnelId )
      if tunEnt is None:
         t0( f'No tunnel entry for tunnelId {tunnelId}' )
         return None
      if not tunEnt.via:
         t0( f'No via for tunnelId {tunnelId}' )
         return None
      via = ecmpHash( tunEnt.via, ethHdr.src, ethHdr.dst )
      if not isinstance( via, Tac.Type( 'Tunnel::TunnelTable::MplsVia' ) ):
         t0( f'Via for tunnelId {tunnelId} is not of type Mpls' )
         return None
      viaHop = via.nexthop
      viaIntfId = via.intfId
      viaLabels = via.labels
      labelStack = []
      for idx in range( viaLabels.calculatedStackSize ):
         labelStack.insert( 0, viaLabels.calculatedLabelStack( idx ) )

      if ELI in labelStack:
         entropyLabel = computeFlowOrEntropyLabel( None, ethHdr, None, None )
         eliIndexes = [ i + 1 for i, x in enumerate( labelStack ) if x == ELI ]
         for idx in eliIndexes:
            labelStack[ idx ] = entropyLabel

      if not labelStack:
         t0( f'LabelStack not present for tunnelId {tunnelId}' )
         return None
      return viaHop, viaIntfId, labelStack

   def constructMplsEthHeader( self, srcMac, dstMac ):
      ethHeader = Tac.newInstance( 'Arnet::Pkt' )
      ethHeader.newSharedHeadData = EthHdrSize
      ethHdr = Tac.newInstance( 'Arnet::EthHdrWrapper', ethHeader, 0 )
      ethHdr.src = srcMac
      ethHdr.dst = dstMac
      ethHdr.typeOrLen = ETH_P_MPLS_UC
      return ethHeader.stringValue

   def rewriteVlanTag( self, data, vlanTag ):
      '''This function takes a ethernet frame (which may or may not already have a 
      .1Q tag) and a vlanTag (Ark::TristateU16) as input. Based on whether a valid 
      vlanTag is passed in, it would decide what to do with the ethernet frame. If 
      a valid tag is passed, it would write that value as a .1Q tag (by replacing 
      the one that the frame already has or creating a new one, if the frame does 
      not have one). If a valid tag is not passed, it would actively remove the 
      pre-existing tag, if one exists.'''
      ( __, headers, __ ) = PktParserTestLib.parsePktStr( data )
      innerDot1QHdr = PktParserTestLib.findHeader( headers, 'EthDot1QHdr', 0 )
      vlanAction = PKTEVENT_ACTION_NONE
      if innerDot1QHdr and vlanTag.isSet:
         vlanAction = PKTEVENT_ACTION_REPLACE
      elif innerDot1QHdr and not vlanTag.isSet:
         vlanAction = PKTEVENT_ACTION_REMOVE
      elif vlanTag.isSet:
         vlanAction = PKTEVENT_ACTION_ADD
      else:
         t8( "No innerDot1q and no rewrite vlan. No action needed." )

      t8( f"Vlan Action: {vlanAction}" )
      if vlanTag.isSet:
         t8( f"tag={vlanTag.value}" )

      return applyVlanChanges( data, 0, vlanTag.value, vlanAction )

   def getTunnelFrameAndEgressIntf( self, bridge, data, vlanId, srcPortName ):
      ##
      # Get details about the PW interface first. Get encap tunnelId it goes over.
      ##
      intfId = self.intfStatus.intfId
      groupKey = getGroupKeyFromShadow( bridge, intfId )

      if not groupKey:
         return self.dropPkt( msg=f'Unknown intf {intfId}' )
      inst = bridge.vplsPwForwardingStatus.instance.get( groupKey.instanceName )
      if not inst:
         return self.dropPkt(
               msg='Instance not found for intf {intf} and inst {inst}'.
               format( intf=intfId, inst=groupKey.instanceName ) )
      vlanInfo = inst.vlan.get( vlanId )
      if not vlanInfo:
         return self.dropPkt(
               msg='Invalid vlanId {v} for instance {inst} intf {intf}'.
               format( v=vlanId, intf=intfId, inst=groupKey.instanceName ) )
      group = inst.pseudowireGroup.get( groupKey.groupName )
      if not group:
         return self.dropPkt(
            msg='Group not found for intf {intf} inst {inst} group {g}'.
            format( intf=intfId, inst=groupKey.instanceName, g=groupKey.groupName ) )
      pw = group.pseudowire.get( intfId )
      if not pw:
         return self.dropPkt(
               msg=f'Pseudowire not found for intf {intfId}' )

      ##
      # Resolve the encapTunnelId
      ##
      ( pkt, headers, _ ) = PktParserTestLib.parsePktStr( data )
      ethHdr = PktParserTestLib.findHeader( headers, 'EthHdr' )

      t2( 'IntfId {i}: Going to resolve tunnelId {t}'.format(
                           i=intfId, t=pw.encapTunnel ) )
      tunnelInfo = self.resolveTunnel( bridge, pw.encapTunnel, ethHdr )
      if tunnelInfo is None:
         return self.dropPkt(
                  msg='IntfId {i}: Tunnel not resolved tunnelId {t}'.
                  format( i=intfId, t=pw.encapTunnel ) )

      ##
      # Construct the frame with mpls labels
      ##
      viaHop, viaIntf, labelStack = tunnelInfo
      if viaHop.af not in ( Af.ipv4, Af.ipv6 ):
         return self.dropPkt( msg=f'Invalid AF tunnelInfo {viaHop.af}' )

      helper = bridge.fwdHelper
      l2Hop = helper.getPopSwapResolvedNexthopInfo( str( viaHop ), viaIntf )

      resolved, hopMac = l2Hop[ 0 : 2 ]
      if not resolved:
         return self.dropPkt(
                  msg='Unable to resolve MAC for {h} {i}'.
                  format( h=str( viaHop ), i=viaIntf ) )

      removeImpNullFromLabelStack( labelStack )
      labelStack.append( pw.encapLabel )
      flowLabelPresent = FlowLabelType( pw.flowLabelType ).transmit
      if flowLabelPresent:
         flowLabel = computeFlowOrEntropyLabel( pkt, ethHdr, None, None )
         labelStack.append( flowLabel )

      # Packet format to be constructed in tunnelFrame
      #  ---------------------------------------
      # | ETHER-1 [ no DOT1Q added ]            | <== Outer ether header
      #  ---------------------------------------
      # | PW-LABEL [FLOW-LABEL] [CONTROL-WORD]  | <== mpls header
      #  ---------------------------------------
      # | ETHER-2 [ DOT1Q-2 ]                   | <== Inner ether header
      #  ---------------------------------------
      # | PAYLOAD                               |
      #  ---------------------------------------

      # TODO: handle tc
      mplsHdrBytes = constructMplsHeader( labelStack, tc=0,
                                          flowLabelPresent=flowLabelPresent )

      outerEthHdr = self.constructMplsEthHeader( bridge.bridgeMac_, hopMac )

      controlWordStr = b'\x00' * 4 if pw.controlWord else b''

      # Figure out if the packet has arrived from a dot1qTunnel port. If it has
      # then any .1Q tag present is treated as a customer-tag and is conveyed
      # as-is.
      switchIntfConfig = bridge.brConfig_.switchIntfConfig.get( srcPortName )
      if switchIntfConfig and switchIntfConfig.switchportMode == 'dot1qTunnel':
         # Packet is from dot1qTunnel port. We need to preserve any c-tags.
         # If required add the normalized tag on top
         if vlanInfo.normalizedTag.isSet:
            data = applyVlanChanges( data, 0, vlanInfo.normalizedTag.value,
                                     PKTEVENT_ACTION_ADD )
      else:
         # The packet is from a switchport or trunk port. So either it would not
         # have any tag at all, or it would have the s-tag on the top.
         # Adjust the inner ether header to either include or not include a tag
         # based on configuration. If a tag is included, the normalized tag is
         # used.
         data = self.rewriteVlanTag( data, vlanInfo.normalizedTag )

      tunnelFrame = outerEthHdr + \
                    mplsHdrBytes + \
                    controlWordStr + \
                    data
      ##
      # Find out the Egress interface
      ##
      egressIntf = viaIntf

      # When egress L3 intf is an SVI port, we need to find out the underlying phys
      # port by looking up the DMAC.
      if Tac.Type( 'Arnet::VlanIntfId' ).isVlanIntfId( viaIntf ):
         # Outer DMAC lookup on the vlan-id
         outerVlanId = Tac.Type( 'Arnet::VlanIntfId' ).vlanId( viaIntf )
         macVlanPair = Tac.Value( "Bridging::MacVlanPair", hopMac, outerVlanId )
         hostAndSource = bridge.l2RibHostOutput.host.get( macVlanPair )
         egressIntf = None
         if hostAndSource and hostAndSource.host.intfIsStored:
            egressIntf = hostAndSource.host.intf

      t2( f'Tunnel frame to be sent to egress intf {egressIntf}' )
      return ( tunnelFrame, egressIntf )

   def shouldForward( self, srcPortName ):
      # In case this is locally generated allow forwarding
      if not srcPortName:
         return True

      # Convert srcPortName to IntfId
      srcIntfId = IntfId( srcPortName )

      # Not coming from another PW, allow forwarding
      if not PseudowireIntfId.isPwInstanceId( srcIntfId ):
         return True

      # Check if IntfId exists in bridge.vplsIntfStatusDir
      srcIntfStatus = self.bridge.vplsIntfStatusDir.intfStatus.get( srcIntfId )

      # If not present, allow forwarding
      # (should we prevent other type PW from forwarding??)
      if not srcIntfStatus:
         return True

      # If present, get the groupKey for the PW
      srcGroupKey = getGroupKeyFromShadow( self.bridge, srcIntfId )
      if not srcGroupKey:
         self.dropPkt( msg=f'Unknown intf {srcIntfId}' )
         return False

      # Get the groupKey for the current port
      intfId = self.intfStatus.intfId
      groupKey = getGroupKeyFromShadow( self.bridge, intfId )
      if not groupKey:
         # Maybe possible if the interface has gone away, but Etba dut has not
         # caught up yet. Lets drop the packet
         self.dropPkt( msg=f'Unknown self intf {intfId}' )
         return False

      # If the groupKeys dont match, then allow forwarding. Split horizon is only
      # enforced within pseudowires in the same VPLS group.
      if groupKey != srcGroupKey:
         return True

      # if they match, get the group config
      inst = self.bridge.vplsPwForwardingStatus.instance.get( groupKey.instanceName )
      if not inst:
         self.dropPkt(
               msg='Instance {inst} not found for intf {src} and self intf {intf}'.
               format( inst=groupKey.instanceName, src=srcIntfId, intf=intfId ) )
         return False

      group = inst.pseudowireGroup.get( groupKey.groupName )
      if not group:
         self.dropPkt(
            msg='Inst {inst}: Group {g} not found for intf {src} self intf {intf}'.
                format( inst=groupKey.instanceName, g=groupKey.groupName,
                        src=srcIntfId, intf=intfId ) )
         return False

      # if group config is split-horizon deny forwarding
      if group.splitHorizon:
         self.dropPkt(
            msg='Intf {src} and self intf {intf} are one split-horizon group {g}'.
                format( g=groupKey.groupName, src=srcIntfId, intf=intfId ) )
         return False

      return True

   def sendFrame( self, data, srcMacAddr, dstMacAddr, srcPortName,
                  priority, vlanId, vlanAction ):
      debugPrint( 'SRC: ', srcPortName, data )
      t8( "send frame into ", self.intfStatus.intfId )

      if not self.shouldForward( srcPortName ):
         t0( f"Not Forwarding from this srcPort {srcPortName}" )
         return

      ( tunnelFrame, egressIntf ) = self.getTunnelFrameAndEgressIntf( self.bridge,
                                                                      data, vlanId,
                                                                      srcPortName )
      if not egressIntf or not tunnelFrame:
         return

      egressPort = self.bridge_.port.get( egressIntf )
      if egressPort:
         debugPrint( 'SENDING INTO: ', egressPort.name(), tunnelFrame )
         # TODO: BUG551672
         # The egressPort is the L2 port out of which we have to send the pkt
         # If the L3 intf happens to be an SVI, the actual L2 port would be one of
         # the switch ports. It is also possible that the actual L2 port could be
         # a trunk port. In such cases, calling sendFrame() with
         # PKTEVENT_ACTION_NONE will not work. The outer ether header should have
         # a .1Q tag corresponding to the vlan of the SVI (L3 intf from above)
         egressPort.sendFrame( tunnelFrame, None, None, None, None, None,
                               PKTEVENT_ACTION_NONE )
      else:
         self.dropPkt(
            msg=f'Egress port not found for intf {egressIntf}' )
         return

   def dropPkt( self, msg=None ):
      self.counter.txDrop += 1
      t0( msg )
      return ( None, None )

   def trapFrame( self, data ):
      # We must never trap any frame or deal with it.
      pass

class PwIntfStatusDirReactor( Tac.Notifiee ):
   notifierTypeName = 'Pseudowire::VplsEthIntfStatusDir'

   def __init__( self, bridge ):
      t8( "PwIntfStatusDirReactor: init" )
      self.vplsIntfStatusDir = bridge.vplsIntfStatusDir
      self.bridge = bridge
      Tac.Notifiee.__init__( self, self.vplsIntfStatusDir )
      for i in self.vplsIntfStatusDir.intfStatus:
         self.handleIntfStatus( i )

   @Tac.handler( 'intfStatus' )
   def handleIntfStatus( self, key ):
      intfStatus = self.vplsIntfStatusDir.intfStatus.get( key )
      t8( f"PwIntfStatusDirReactor: handleIntfStatus {key}" )
      if intfStatus:
         t8( f"PwIntfStatusDirReactor: create port for {key}" )
         self.bridge.vplsIntfCounters.intfCounter.newMember( intfStatus.intfId )
         port = VplsPseudowirePort( self.bridge, None, None, None, intfStatus )
         self.bridge.addPort( port )
      else:
         t8( f"PwIntfStatusDirReactor: delete port for {key}" )
         self.bridge.delPort( key )
         del self.bridge.vplsIntfCounters.intfCounter[ key ]

def handleAgentInit( em ):
   '''Mount needed entities here'''
   t2( "Agent Init" )
   em.mount( 'vpls/pseudowire/forwardingStatus',
             'Pseudowire::VplsForwardingStatus', 'r' )
   em.mount( 'interface/status/eth/pseudowire',
             'Pseudowire::VplsEthIntfStatusDir', 'r' )
   em.mount( 'bridging/input/config/pseudowire',
             'Bridging::Input::Config', 'r' )
   em.mount( 'vpls/pseudowire/etba/counters',
             'Pseudowire::VplsEtbaIntfCounterDir', 'w' )
   em.mount( 'bridging/flush/request/vpls',
             'Bridging::HostTableFlushRequestDir', 'r' )
   em.mount( 'bridging/flush/reply/all',
             'Bridging::HostTableFlushReplyDir', 'w' )
   em.mount( 'bridging/hwcapabilities',
             'Bridging::HwCapabilities', 'w' )

def handleBridgeInit( bridge ):
   t2( "Bridge Init" )
   em = bridge.em()
   shmemEm = SharedMem.entityManager( sysdbEm=em )
   # Mount LFIB from smash and set it into bridge
   bridge.pwLfib = shmemEm.doMount( 'mpls/protoLfibInputDir/pseudowire',
                                    'Mpls::LfibStatus',
                                    Smash.mountInfo( 'keyshadow' ) )
   bridge.l2RibHostOutput = shmemEm.doMount( "bridging/l2Rib/hostOutput",
                                             "L2Rib::HostOutput",
                                             Smash.mountInfo( 'keyshadow' ) )
   bridge.pfs = shmemEm.doMount( 'forwarding/srte/status',
                                 'Smash::Fib::ForwardingStatus',
                                 FibUtils.forwardingStatusInfo( 'reader' ) )

   bridge.fwdHelper = mplsEtbaForwardingHelperFactory( bridge )

   # Get forwardingStatus and set it into bridge
   bridge.vplsPwForwardingStatus = em.entity( 'vpls/pseudowire/forwardingStatus' )
   bridge.vplsIntfStatusDir = em.entity( 'interface/status/eth/pseudowire' )
   bridge.bridgingInputConfig = em.entity( 'bridging/input/config/pseudowire' )
   bridge.vplsIntfCounters = em.entity( 'vpls/pseudowire/etba/counters' )
   bridgingHwCapabilities = em.entity( 'bridging/hwcapabilities' )
   bridgingHwCapabilities.vplsSupported = True

   # Instantiate IntfStatusDir reactor and create ports
   global gIntfStatusDirReactor
   gIntfStatusDirReactor = PwIntfStatusDirReactor( bridge )
   bridge.vplsFlushReactor_ = \
         Tac.newInstance( 'Arfa::ArfaBridgeMacFlush',
                          'VplsFlushReactor',
                          em.entity( 'bridging/flush/request/vpls' ),
                          None,
                          em.entity( 'bridging/flush/reply/all' ),
                          bridge.smashBrStatus_,
                          getPythonIntoCppShimContext().counters(),
                          getPythonIntoCppShimContext().agingNotifier() )

   # Instantiate an instance of the VplsLabelToIntfId sm so that
   # we have a handy way to map intfIds back to the VplsForwardingStatusPseudowire
   smControl = Tac.newInstance( 'Arx::SmControl' )
   bridge.vplsLabelToIntfIdSm = Tac.newInstance(
      'PseudowireAgent::VplsLabelToIntfIdArx',
      bridge.vplsPwForwardingStatus,
      Tac.Type( 'PseudowireAgent::VplsLabelToIntfId' )(),
      smControl )
   global gAllIntfStatusDir
   gAllIntfStatusDir = em.getLocalEntity( 'interface/status/all' )

def lookupPwDecap( bridge, label ):
   intfId = bridge.vplsLabelToIntfIdSm.labelToIntfId.intfId.get(
                         Tac.Type( 'Arnet::MplsLabel' )( label ) )
   if not intfId:
      t0( f'No intfId for Label {label}' )
      return ( None, None, None, False )
   groupKey = getGroupKeyFromShadow( bridge, intfId )
   if not groupKey:
      t0( f'No groupKey for Label {label}' )
      return ( None, None, None, True )
   inst = bridge.vplsPwForwardingStatus.instance.get( groupKey.instanceName )
   if not inst:
      t0( f'No inst for Label {label}' )
      return ( None, None, None, True )
   group = inst.pseudowireGroup.get( groupKey.groupName )
   if not group:
      t0( f'No group for Label {label}' )
      return ( None, None, None, True )
   pw = group.pseudowire.get( intfId )
   if not pw:
      t0( f'No PW for Label {label}' )
      return ( None, None, None, True )
   return ( bridge.port.get( intfId ), pw, inst, False )

def decapMplsPwFrame( bridge, fwdStatusInst, data, mplsHdr, controlWord,
                      flowLabel=False ):
   # The mplsHdr.offset points to PW-LABEL.
   innerPktOffset = mplsHdr.offset + 4
   if flowLabel:
      innerPktOffset += 4
   if controlWord:
      innerPktOffset += 4

   innerData = data[ innerPktOffset : ]
   ( _, headers, _ ) = PktParserTestLib.parsePktStr( innerData )
   innerEth = PktParserTestLib.findHeader( headers, 'EthHdr' )
   if not innerEth:
      t0( f'No inner Ether header found Label {mplsHdr.label}' )
      return None

   if not fwdStatusInst.vlan:
      t0( 'No vlan configured for instance {} Label {}'.format(
                     fwdStatusInst.name, mplsHdr.label ) )
      return None

   innerDot1QHdr = PktParserTestLib.findHeader( headers, 'EthDot1QHdr', 0 )
   tag = None
   if innerDot1QHdr:
      tag = innerDot1QHdr.tagControlVlanId

   if not tag:
      # If we have multiple vlans, we need the tag to distinguish which vlan to
      # switch in. Otherwise it is not a valid packet
      if len( fwdStatusInst.vlan ) > 1:
         t0( 'No tag in pkt, '
             'more than 1 vlans ({}) configured for instance {} Label {}'.format(
                  len( fwdStatusInst.vlan ), fwdStatusInst.name, mplsHdr.label ) )
         return None
      # If the vlan tag is present, then we need the pkt to carry it as well.
      v = next( iter( fwdStatusInst.vlan ) )
      if not v or fwdStatusInst.vlan[ v ].normalizedTag.isSet:
         t0( 'No tag in pkt, vlanId {} configured with tag Instance {} Label {}'.
             format( v, fwdStatusInst.name, mplsHdr.label ) )
         return None
      return innerEth.pkt.stringValue

   # At this point a tag is present. Check if there is only 1 vlan without a
   # requirement of a tag configured in this VPLS instance
   if len( fwdStatusInst.vlan ) == 1:
      v = next( iter( fwdStatusInst.vlan ) )
      if v and not fwdStatusInst.vlan[ v ].normalizedTag.isSet:
         # This means we have a c-tag in our hands. We just convey it as-is
         t0( 'C-Tag present in pkt {}, vlanId {} Instance {} Label {}'.
             format( tag, v, fwdStatusInst.name, mplsHdr.label ) )
         return innerEth.pkt.stringValue

   vlanId = None
   for v in fwdStatusInst.vlan:
      status = fwdStatusInst.vlan[ v ]
      if status.normalizedTag.isSet and tag == status.normalizedTag.value:
         vlanId = v
         break

   if not vlanId:
      t0( 'Unknown tag in pkt {} Instance {} Label {}'.
          format( tag, fwdStatusInst.name, mplsHdr.label ) )
      return None
   modifiedPkt = applyVlanChanges( innerEth.pkt.stringValue, 0, vlanId,
                                   PKTEVENT_ACTION_REPLACE )
   return modifiedPkt

# This is called prior to bridging the frame. This allows us to change the data and
# source port.
def handleTunnelPkt( bridge, dstMacAddr, data, srcPort ):
   '''Send into L2 switching pipeline if the packet happens to be coming
   from a PW'''

   retVal = ( None, None, False, None )

   if not bridge.pwLfib or not bridge.pwLfib.lfibRoute:
      # Optimization to skip processing below when no pseudowires are
      # present
      return retVal

   bridgeMacAddr = bridge.bridgeMac()
   if dstMacAddr != bridgeMacAddr:
      t0( "dstMacAddr", dstMacAddr, " is different from bridgeMac", bridgeMacAddr )
      # On ceos duts, the MAC address set on interfaces is not the bridgeMac
      # (unlike on namespace and physical duts). So we need to find out if the
      # dest MAC address of the pkt matches that of the interface MAC on the
      # source port. If they match, then we can accept such packets for further
      # considerations.
      srcIntf = None
      if srcPort and srcPort.intfStatus_:
         srcIntf = srcPort.intfStatus_.intfId
      if not srcIntf:
         t0( "src PORT does not have an Intf" )
         return retVal

      intfStatus = gAllIntfStatusDir.intfStatus.get( srcIntf )
      if not intfStatus:
         t0( "src PORT", srcIntf, " does not have a status" )
         return retVal

      intfMac = intfStatus.routedAddr
      if dstMacAddr != intfMac:
         t0( "dstMacAddr is different from src port MAC:", intfMac )
         return retVal

   # Expected packet format:
   #  ---------------------------------------
   # | ETHER-1 [ DOT1Q-1 ]                   |
   #  ---------------------------------------
   # | PW-LABEL [FLOW-LABEL] [CONTROL-WORD]  |
   #  ---------------------------------------
   # | ETHER-2 [ DOT1Q-2 ]                   |
   #  ---------------------------------------
   # | PAYLOAD                               |
   #  ---------------------------------------

   # Parse the packet and extract all the MPLS headers. Since we need to find out
   # if a control-word exists and if it is a PwAch for VCCV, we need to parse
   # beyond the Mpls headers.
   ( _, headers, _ ) = PktParserTestLib.parsePktStr( data, parseBeyondMpls=True )

   ethHdr = PktParserTestLib.findHeader( headers, 'EthHdr' )
   if not ethHdr:
      t0( "Cannot find ethHdr" )
      return retVal

   mplsHdr = PktParserTestLib.findHeader( headers, "MplsHdr" )
   if not mplsHdr:
      t0( "Cannot find MplsHdr" )
      return retVal

   if mplsHdr.ttl <= 1:
      t0( 'ttl=', mplsHdr.ttl )
      # We cannot drop packets with an expiring TTL as it will break
      # mpls traceroute.
      return retVal

   pwEtbaPort, fwdStatusPw, fwdStatusInst, drop = \
                               lookupPwDecap( bridge, mplsHdr.label )
   if drop:
      t0( f'Drop pkt with label {mplsHdr.label}' )
      return( False, False, drop, None )

   if not pwEtbaPort or not fwdStatusPw or not fwdStatusInst:
      t0( f'No port or fwdStatus for label {mplsHdr.label}' )
      return retVal

   # Check if this has a VCCV control-word and if so trap the frame
   # Note: This also handles standby PW
   pwAchHdr = PktParserTestLib.findHeader( headers, 'PwAchHdr' )
   if pwAchHdr and fwdStatusPw.controlWord:
      t0( "PCH pkt trapping to CPU", pwEtbaPort.intfStatus.operStatus )
      srcPort.trapFrame( data )
      return( None, None, True, None )

   if pwEtbaPort and \
      pwEtbaPort.intfStatus.operStatus != 'intfOperUp' and \
      pwEtbaPort.intfStatus.linkStatus != 'linkUp':
      t0( "Dropping, port operStatus ", pwEtbaPort.intfStatus.operStatus,
          " linkStatus ", pwEtbaPort.intfStatus.linkStatus )
      return( None, None, True, None )

   flowLabel = FlowLabelType( fwdStatusPw.flowLabelType ).receive
   innerData = decapMplsPwFrame( bridge, fwdStatusInst, data, mplsHdr,
                                 fwdStatusPw.controlWord, flowLabel=flowLabel )
   if not innerData:
      t0( f'Inner data not found for pkt with label {mplsHdr.label}' )
      return( False, False, True, None )

   t0( f'handleTunnelPkt returning {pwEtbaPort.name()} drop {drop}' )
   # Return the modified packet with the new srcPort as pwEtbaPort
   if not drop:
      debugPrint( 'RECVD ON: ', pwEtbaPort.name(), innerData )
   return ( innerData, pwEtbaPort, drop, None )

def Plugin( ctx ):
   t2( "Etba Vpls plugin registering" )

   if ctx.inArfaMode():
      # We dont support the C++ Etba implementation yet
      return

   # Register a callback class to create Etba interfaces for each PW intfStatus
   ctx.registerInterfaceHandler( 'Interface::EthPhyIntfStatus',
                                 VplsPseudowirePort )

   # callback for doInit() of Etba agent
   ctx.registerAgentInitHandler( handleAgentInit )

   # Etba mounts have completed. Install other mounts
   ctx.registerBridgeInitHandler( handleBridgeInit )

   # Decap Mpls header send packet into L2 switching pipeline
   ctx.registerPreTunnelHandler( handleTunnelPkt )
