# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import QuickTrace
import Tac
import Tracing

from Arnet.PktParserTestLib import parseDataPacket
from EbraTestBridgePlugin import FwdIntfEtba
from EbraTestBridgePlugin.FwdIntfEtba import FwdIntfDevice

handle = Tracing.Handle( 'EbraTestBridgeVrfRouteLeak' )
t1 = handle.trace1
t2 = handle.trace2

qtInitialized = False

def qTrace():
   '''Initializes QuickTrace once, return the module itself'''
   global qtInitialized
   if not qtInitialized:
      QuickTrace.initialize( 'etba-routingHandlers.qt' )
      qtInitialized = True
   return QuickTrace

def qv( *args ):
   qTrace().Var( *args )

def qt0( *args ):
   qTrace().trace0( *args )

leakedHandler = 'leakedRouteHandler:'

def handleIpRoute( bridge, pkt, ethHdr, ethDot1QHdr, ipHdr, vrfName ):
   noMatch = ( None, None )
   func = 'handleIpRoute:'
   isV6 = isinstance( ipHdr, Tac.Type( 'Arnet::Ip6HdrWrapper') )

   # Check that ip routing is also enabled
   if isV6:
      routingInfo = bridge.routing6VrfInfo_.get( vrfName )
   else:
      routingInfo = bridge.routingVrfInfo_.get( vrfName )

   if not routingInfo or not routingInfo.routing:
      af = 'ipv6' if isV6 else 'ipv4'
      t1( func, af, 'routing is disabled' )
      qt0( leakedHandler, qv( af ), 'routing is disabled' )
      return noMatch

   dstIp = str( ipHdr.dst )
   nexthopInfo = bridge.fwdHelper.getResolvedNexthopInfo( dstIp, intf=None,
                                                          vrfName=vrfName )
   if not nexthopInfo.resolved:
      t1( func, 'cannot resolve', ipHdr.dst )
      qt0( leakedHandler, 'cannot resolve', qv( ipHdr.dst ) )
      return noMatch

   if nexthopInfo.labelStack:
      t1( func, 'mpls encap, should be handled separately', ipHdr.dst )
      qt0( leakedHandler, 'mpls encap, should be handled separately',
           qv( ipHdr.dst ) )
      return noMatch

   # Check TTL value
   # handleIpRoute must not decrement the TTL; the TTL in the frame at this point
   # is already the oTtl.  The frames are delivered to handleIpRoute via the
   # fwd0 interface, and so the oTtl calculation has already been performed by
   # the kernel during the kernel routing action.  This also means that TTL=1
   # is valid at this point in the packet handling code.
   if isV6:
      if ipHdr.hopLimit < 1:
         t1( func, 'Dropping packet with ttl of', ipHdr.hopLimit )
         qt0( leakedHandler, 'Dropping packet with ttl of', qv( ipHdr.hopLimit ) )
         return noMatch
      oTtl = ipHdr.hopLimit
      ipHdr.hopLimit = oTtl
      ipEthType = 'ethTypeIp6'
   else:
      if ipHdr.ttl < 1:
         t1( func, 'dropping packet with ttl of', ipHdr.ttl )
         qt0( leakedHandler, 'dropping packet with ttl of', qv( ipHdr.ttl ) )
         return noMatch
      oTtl = ipHdr.ttl
      ipHdr.ttl = oTtl
      ipEthType = 'ethTypeIp'

   # recompute the ip header checksum
   if not isV6:
      ipHdr.checksum = 0
      ipHdr.checksum = ipHdr.computedChecksum

   newEthType = ipEthType

   ethHdr.dst = nexthopInfo.dstMac
   ethHdr.src = bridge.bridgeMac_
   if ethDot1QHdr:
      ethDot1QHdr.ethType = newEthType
   else:
      ethHdr.ethType = newEthType

   newData = pkt.stringValue

   return( newData, nexthopInfo.intf )

def leakedRouteHandler( self, data ):
   'Leaked route handler to be registered by the fwd0 device'
   t2( leakedHandler, 'Processing pkt' )
   noMatch = ( None, None )

   # this routing handler only works in unified FEC mode
   if self.bridge.fecModeStatus.fecMode != 'fecModeUnified':
      qt0( leakedHandler, 'Not in unified FEC mode' )
      return noMatch

   # Retrieve and validate packet info
   pktInfo = parseDataPacket( data )
   if not pktInfo.ethHdr:
      t1( leakedHandler, 'No L2 Hdr in frame, drop' )
      qt0( leakedHandler, 'No L2 Hdr in frame, drop' )
      return noMatch

   if pktInfo.mplsHdr:
      t1( leakedHandler, 'MPLS frame, drop' )
      qt0( leakedHandler, 'MPLS frame, drop' )
      return noMatch

   ipHdr = pktInfo.ipHdr or pktInfo.ip6Hdr
   if not ipHdr:
      t1( leakedHandler, 'Non-IP frame, drop' )
      qt0( leakedHandler, 'Non-IP frame, drop' )
      return noMatch

   # get the vrfName from fwd0 device
   routeMatch = handleIpRoute( self.bridge,
                               pktInfo.pkt,
                               pktInfo.ethHdr,
                               pktInfo.ethDot1QHdr,
                               ipHdr,
                               self.vrfName )
   return routeMatch

def Plugin( ctx ):
   FwdIntfDevice.registerFwdIntfRoutingHandler( leakedRouteHandler )
   # FWD Interface (depends on forwardingBridgeInit)
   FwdIntfEtba.pluginHelper( ctx )
