# Copyright (c) 2007-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=attribute-defined-outside-init
# pylint: disable=consider-using-f-string
# pylint: disable=superfluous-parens
# pylint: disable=consider-using-in

import errno
import os
import weakref
import Tac
import Tracing
import Ethernet
import SharedMem
from collections import namedtuple
from Intf import SyncMtu
from EbraTestBridgePort import EbraTestPortBase, EbraTestPort
from SysConstants.if_ether_h import ETH_P_PAUSE, ETH_P_8021Q, ETH_P_ALL
from Toggles import EtbaDutToggleLib
from EbraTestBridgeLib import (
   HANDLER_PRIO_HIGHEST,
   HANDLER_PRIO_NORMAL,
   HANDLER_PRIO_LOWEST,
   PKTEVENT_ACTION_NONE,
   PKTEVENT_ACTION_REMOVE,
   PKTEVENT_ACTION_ADD,
   PKTEVENT_ACTION_REPLACE,
   applyVlanChanges,
   macAddrsAsStrings,
)
from TypeFuture import TacLazyType
import EbraTestBridgeLib
import importlib
import CEosHelper

EthAddr = Tac.Type( "Arnet::EthAddr" )
IntfId = Tac.Type( "Arnet::IntfId" )
VlanIdOrAnyOrNone = Tac.Type( "Bridging::VlanIdOrAnyOrNone" )
Constants = Tac.Type( "Arfa::Constants" )

pythonIntoCppShimContext = None

def getPythonIntoCppShimContext():
   return pythonIntoCppShimContext

def setPythonIntoCppShimContext( obj ):
   global pythonIntoCppShimContext
   pythonIntoCppShimContext = obj

try:
   # We're not using "import Arnet.PacketFormat" to prevent a
   # dependency on scapy to be generated.
   PacketFormat = importlib.import_module('Arnet.PacketFormat').PacketFormat
except ImportError:
   # PacketFormat uses scapy, which is GPL'd. In order to avoid distributing it
   # to customers (in vEOS specifically), use Tracing.HexDump if PacketFormat
   # is not available
   PacketFormat = Tracing.HexDump

__defaultTraceHandle__ = Tracing.Handle( "EbraTestBridge" )
t0 = Tracing.trace0
t4 = Tracing.trace4    # misc important events
t5 = Tracing.trace5    # Packet tx/rx, error level
t6 = Tracing.trace6    # Packet tx/rx, alert level
t7 = Tracing.trace7    # Packet tx/rx, normal level
t8 = Tracing.trace8    # Packet tx/rx, debug level
t8b = Tracing.trace3   # Packet tx/rx, super-debug level
t9 = Tracing.trace9    # MAC address table aging/flushing

mountHelperTac = TacLazyType( "Interface::EthIntfCounterMountHelper" )
EtbaIntfCounterHelper = TacLazyType( 'Etba::EtbaIntfCounterHelper' )
HostKey = TacLazyType( "Bridging::HostKey" )
SmashFdbStatus = TacLazyType( "Smash::Bridging::SmashFdbStatus" )
RoutingHandlerCounter = TacLazyType( 'Bridging::Etba::RoutingHandlerCounter' )
EthTypesApi = TacLazyType( "Interface::EthTypesApi" )
BridgingDecisionStep = TacLazyType( "Arfa::BridgingDecisionStep" )
ArfaModeType = Tac.Type( "Arfa::ArfaMode" )

# the maximum number of fdbStatus entries that an etba dut will handle. Note that
# setting this higher will require more memory for each etba dut regardless of
# whether the entries are used
ETBA_SMASH_SIZE = 16 * 1024

# pylint: disable-next=wrong-import-position
from EbraTestBridgeConstants import * # pylint: disable-msg=wildcard-import

def createCPythonPort( pythonPort ):
   import _Tac # pylint: disable=import-outside-toplevel
   # pylint: disable=protected-access

   portNotifier = Tac.newInstance( "Arfa::ArfaPythonPortNotifier" )
   pythonPort.arfaPortReactor = ArfaPortReactor( portNotifier, pythonPort )

   cPort = Tac.newInstance( "Arfa::ArfaPythonPort",
                            pythonPort.intfName_, None, None,
                            portNotifier )
   if pythonPort.intfName_ == "Cpu":
      cPort.isCpuPort = True

   if "MplsTrunk" in pythonPort.intfName_:
      cPort.isMplsTrunkPort = True

   if ( pythonPort.intfName_.startswith( "Mpls" ) or
        pythonPort.intfName_.startswith( "Vxlan" ) ):
      cPort.isVirtualValue = True

   return cPort

class PythonIntoCppShimContext:
   """
   To ease the transition from python Etba to Arfa, python plugins will be calling
   C++ code. This context contains some global data that is needed to do this.
   """
   def __init__( self ):
      self.arfaRoot_ = None

   def addPort( self, arfaPythonPort ):
      self.arfaRoot_.ports.port.addMember( arfaPythonPort )

   def delPort( self, intfId ):
      assert intfId in self.arfaRoot_.ports.port
      del self.arfaRoot_.ports.port[ intfId ]

   def getPort( self, intfId, portMustExist=True ):
      port = None
      if intfId:
         port = self.arfaRoot_.ports.port.get( str( intfId ) )
      assert port or not portMustExist
      return port

   def plugin( self, pluginName ):
      assert self.arfaRoot_
      return self.arfaRoot_.arfaPlugins.plugin.get( pluginName )

   def createPkt( self, data ):
      pkt = Tac.newInstance( "Arnet::Pkt" )
      pkt.stringValue = data
      return pkt

   def pktCtx( self, data, srcPort, portMustExist=True ):
      pkt = self.createPkt( data )
      cPort = self.getPort( srcPort, portMustExist=portMustExist )
      return self.arfaRoot_.inputPipeline.setupPacketContext( pkt, cPort )

   def bCtx( self, pktCtx ):
      pipeline = self.arfaRoot_.arfaPlugins.plugin[ 'CoreBridge' ].pipeline
      return pipeline.setupBridgeContext( pktCtx )

   def rCtx( self, pktCtx, l3SrcIntfId, vrfName="default" ):
      rCtx = Tac.newInstance( "Arfa::RoutingContext", pktCtx, l3SrcIntfId,
                              vrfName )
      return rCtx

   def maybeDecapPacket( self, data, srcPort, vrfName="default" ):
      t7( 'maybeDecapPacket input: srcPort - %s data - %s' %
            ( srcPort, data ) )
      pktCtx = self.pktCtx( data, srcPort )
      decapPktInfo = Tac.newInstance( "Arfa::PythonDecapPacketInfo" )
      pktCtx.pktInfo.addMember( decapPktInfo )
      rCtx = self.rCtx( pktCtx, srcPort )
      routingArfaPlugin = self.arfaRoot_.arfaPlugins.plugin[ 'CoreRouting' ]
      vrfRoot = routingArfaPlugin.routingVrfRoots.vrfRoot[ vrfName ]
      decapDemuxer = routingArfaPlugin.decapDemuxer
      decapDemuxer.maybeDecapPacket( rCtx, vrfRoot )
      chgData = rCtx.pktCtx.parsedPkt.pkt.stringValue
      dstMacAddr = rCtx.pktCtx.parsedPkt.dstMacAddr
      t7( 'maybeDecapPacket output: dstMacAddr - %s data - %s' %
            ( dstMacAddr, chgData ) )
      return ( chgData, dstMacAddr )

   def routeFrame( self, data, srcIntfId ):
      pktCtx = self.pktCtx( data, srcIntfId )
      self.arfaRoot_.inputPipeline.routeFrame( pktCtx, "default", srcIntfId )

   @staticmethod
   def callBridgingDecisionStep( step, bCtx ):
      t7( "Calling step", step.name )
      stepOutput = Tac.newInstance( "Arfa::BridgingStepDecisions" )

      step.decision( stepOutput, bCtx )

      for decision in stepOutput.decisions.values():
         decision.applyDecision( bCtx.bOut )
         bCtx.bOut.appliedDecision.enq( decision )

      return bCtx.bOut

   def callBridgingDecisionStepPktCtx( self, step, pktCtx ):
      return self.callBridgingDecisionStep( step, self.bCtx( pktCtx ) )

   def callBridgingDecisionSuppressionStepPktCtx( self, step, pktCtx, outIntfs ):
      bCtx = self.bCtx( pktCtx )
      intfSet = Tac.newInstance( "Arfa::InterfaceSet" )

      for intf in outIntfs:
         intfSet.intf.add( intf )

      bCtx.bOut.outputIntfs = intfSet

      return self.callBridgingDecisionStep( step, bCtx )

   @staticmethod
   def callBridgingActionStep( step, bCtx ):
      stepOutput = Tac.newInstance( "Arfa::BridgingStepDecisions" )
      bpCtx = Tac.newInstance( "Arfa::BridgingActionPipelineContext",
                               stepOutput,
                               bCtx )

      step.decision( bpCtx )

      for decision in stepOutput.decisions.values():
         decision.applyDecision( bCtx.bOut )

      return bCtx.bOut

   def callBridgingActionStepPktCtx( self, step, pktCtx ):
      return self.callBridgingActionStep( step, self.bCtx( pktCtx ) )

   def callOutputPipelineStepLookup( self, step, pktCtx, srcPortName, dstPortName ):
      srcPort = self.getPort( srcPortName )
      dstPort = self.getPort( dstPortName )
      oCtx = Tac.newInstance(
         "Arfa::OutputPipelineContext", pktCtx, srcPort, dstPort, None )
      return self.callOutputPipelineStep( step, oCtx )

   @staticmethod
   def callOutputPipelineStep( step, oCtx ):
      return step.process( oCtx )

   @staticmethod
   def callInputPipelineStep( step, pktCtx ):
      return step.process( pktCtx )

   def counters( self ):
      return self.arfaRoot_.counters

   def agingNotifier( self ):
      return self.arfaRoot_.arfaPlugins.plugin[ 'CoreBridge' ].agingNotifier

class ArfaBridgingDecisionShimBase:
   def __init__( self, step ):
      self.step = step
      self.__class__.__name__ = self.__class__.__name__ + ":" + step.name

      stepTypeValid = False
      for parent in type( self.step ).__mro__:
         if parent == BridgingDecisionStep:
            stepTypeValid = True
            break

      assert stepTypeValid, "Step does not inherit from BridgingDecisionStep"

   def shim( self ):
      return getPythonIntoCppShimContext()

   def bCtx( self, routed, vlanId, data, srcPortName ):
      pktCtx = self.shim().pktCtx( data, srcPortName,
                                   portMustExist=srcPortName is not None )
      pktCtx.destIsFabric = routed
      bCtx = self.shim().bCtx( pktCtx )

      if not bCtx.maybeVlanId:
         # If there is no vlanId in the packet, that means the packet is not tagged.
         # But we do know the source port is a switchport because of where this is
         # called from in bridgeFrame(). So we need to emulate the vlanInfo being
         # constructed properly.
         pktCtx.curTags.effectiveVlanId = vlanId
         bCtx = self.shim().bCtx( pktCtx )

      bVlanId = bCtx.maybeVlanId

      # For private vlans, we are called with vlanId == fid
      if bVlanId != vlanId:
         fid = bCtx.fid
         assert vlanId == fid
         vlanInfo = Tac.Value( "Arfa::VlanInfo", "notTagged", vlanId, False )
         pktCtx.vlanInfo = vlanInfo
         bCtx = self.shim().bCtx( pktCtx )
         bVlanId = bCtx.maybeVlanId

      assert bVlanId == vlanId

      return bCtx


class ArfaShimFloodsetIncludesCpuHandler( ArfaBridgingDecisionShimBase ):
   # floodsetIncludesCpuHandlers
   def __call__( self, i1, i2, i3, data ):
      pktCtx = self.shim().pktCtx( data, None, portMustExist=False )
      bOut = self.shim().callBridgingDecisionStepPktCtx( self.step, pktCtx )
      result = None
      for dec in bOut.appliedDecision.values():
         if dec.name == "OutputIntfs":
            if "Cpu" in dec.addIntfs.intf:
               result = True
            elif "Cpu" in dec.removeIntfs.intf:
               result = False
               break
         elif dec.name == "ReplaceIntfs":
            result = "Cpu" in dec.newIntfs.intf
            break

      t7( "Returning", result, "for", self.step.name, "in floodset includes cpu" )
      return result

class ArfaShimPacketReplicationHandler( ArfaBridgingDecisionShimBase ):
   # packetReplicationHandlers
   def __call__( self, _, finalIntfs, srcPort=None, dropReasonList=None, vlanId=None,
                 dstMacAddr=None, data=None, pktCtx=None ):
      # The pktCtx is not created properly if data is None or not an ethernet packet
      if not data:
         t5( 'ArfaShimPacketReplicationHandler called with no data.'
             'Returning without calling DecisionStep %s' % self.step.name )
         return
      intfId = srcPort.name() if srcPort else None
      if not pktCtx:
         pktCtx = self.shim().pktCtx( data, intfId, portMustExist=False )
      bCtx = self.shim().bCtx( pktCtx )

      intfSet = Tac.newInstance( "Arfa::InterfaceSet" )
      for intf in finalIntfs:
         intfSet.intf.add( intf )
      bCtx.bOut.outputIntfs = intfSet

      dropDecision = Tac.newInstance(
         "Arfa::BridgingStepDecisionDrop", "drop", bool( dropReasonList ) )
      bCtx.bOut.dropDecision = dropDecision

      bOut = self.shim().callBridgingDecisionStep( self.step, bCtx )

      for intf in list( finalIntfs ):
         if intf not in bOut.outputIntfs.intf:
            del finalIntfs[ intf ]

      for intf in bOut.outputIntfs.intf:
         if intf not in finalIntfs:
            finalIntfs[ intf ] = PKTEVENT_ACTION_NONE

class ArfaShimTrapLookupHandler( ArfaBridgingDecisionShimBase ):
   # trapLookupHandlers
   def __init__( self, step, trueTrapValue ):
      ArfaBridgingDecisionShimBase.__init__( self, step )
      # The trap lookup handlers have two different values they can return to cause a
      # packet to be trapped, True or EbraTestBridgeConstants.TRAP_ANYWAY
      # This value is what we will return when the pipeline returns true for trap.
      # Hopefully, one of these converted handlers won't return both.
      self.trueTrapValue = trueTrapValue

   def __call__( self, i1, routed, vlanId, i2, data, srcPortName ):
      if vlanId > 4094:
         # Arfa does not perform any bridging actions on extended vlans
         return False, False, False
      t8( "Trap input", routed, vlanId )
      bCtx = self.bCtx( routed, vlanId, data, srcPortName )

      bOut = self.shim().callBridgingDecisionStep( self.step, bCtx )

      match = bOut.trap()
      trapReturn = self.trueTrapValue if bOut.trap() else False
      cpToCpu = False
      if bOut.trap():
         cpToCpu = bOut.trapDecision.continueProcessingAfterTrap

      if cpToCpu:
         # Python Etba is weird. cpToCpu means trap to Cpu. In Arfa, it means
         # route the packet. So here if the Arfa says trap and continue after
         # the trap, that means for python Etba to not trap, but copyToCpu
         trapReturn = False

      result = match, trapReturn, cpToCpu
      t7( "Returning", result, "for", self.step.name, "in trap lookup" )
      return result

class ArfaShimLearningHandlers( ArfaBridgingDecisionShimBase ):
   # learningHandlers
   def __call__( self, pktCtx ):
      bOut = self.shim().callBridgingDecisionStepPktCtx( self.step, pktCtx )
      result = "learn" if bOut.learn() else "dontLearn"
      t7( "Returning", result, "for", self.step.name, "in learning" )
      return result

class ArfaShimDestLookupHandlers( ArfaBridgingDecisionShimBase ):
   def __call__( self, i1, vlanId, i2, data ):
      if vlanId > 4094:
         # Arfa does not perform any bridging actions on extended vlans
         return False, None
      assert data
      bCtx = self.bCtx( False, vlanId, data, None )

      bOut = self.shim().callBridgingDecisionStep( self.step, bCtx )

      intfDecisionsApplied = any(
         "Intfs" in dec.name for dec in bOut.appliedDecision.values() )

      result = False, None
      if intfDecisionsApplied:
         result = True, list( bOut.outputIntfs.intf )

      t7( "Returning", result, "for", self.step.name, "in dest lookup" )
      return result

# Return type for EbraTestBridge.bridgeFrame
BridgeFrameInfo = namedtuple( 'BridgeFrameInfo', [
   'vlanInfo',
   'data',
   'egressIntfs',
   ] )

# EbraTestBridge routing handler counters
ebraRoutingHandlerCounter = {}
ebraRoutingHandlerCounter[ "overall" ] = RoutingHandlerCounter()

# FwdIntfDevice routing handler counters
fwdIntfRoutingHandlerCounter = {}

# Interface plugins (e.g., Lag) register themselves in this list with an
# intfStatus type name and an EbraTestPort subclass.
interfaceHandlers = []

# Callback function that can be registered from a plugin to determine the action
# that needs to be performed on the packet right when it enters on the front panel
# interface. Return values could be 'permit', 'deny' or 'trap'. If 'deny' is
# returned, the packet is dropped and subsequent stages are not invoked (including
# bridging). 'trap' would result in the packet getting trapped to cpu (subsequent
# stages are skipped) and 'permit'/any other return value any other return value
# would cause the packet to proceed to the subsequent stages in the pipeline
preBridgingHandler = None

# Callback function that can be registered from a plugin that decrypts the packets
# on links that have macsec enabled.
macsecDecryptHandler = None

# Callback function that can be registered from a plugin that encrypts the packets
# before transmit on links that have macsec enabled.
macsecEncryptHandler = None

# Callback function that can be registered from a plugin to determine the action
# that needs to be performed on the packet right when it enters on the front panel
# interface and before bridging the frame. Return values could be (False, False)
# (chgData, chgSrcPort) if expected Tunnel header was found to be stripped.
# In any case packet to proceed to the subsequent stages in the pipeline.
preTunnelHandler = []
preTunnelHdlrPrioMap = { k: [] for k in range( HANDLER_PRIO_HIGHEST,
                                               HANDLER_PRIO_LOWEST ) }

# Destination lookup (e.g., IgmpSnooping) plugins register themselves in this
# list with a function to perform the lookup.  All registered functions
# are called, in arbitrary [Plugin registration] order, and return
# a tuple of ( matched, intfList ) -- if a plugin returns True in
# matched, then no further plugins are consulted.  If no plugin returns
# True, then the fdb for the VLAN is consulted.
destLookupHandlers = []

# Floodset-Includes-Cpu plugins register themselves with a function
# that is called to determine whether a given packet should be flooded
# to the CPU.  If the Cpu interface is in the floodset (meaning there is
# an SVI for that VLAN), then if the packet is not broadcast, the
# Floodset-Includes-Cpu plugins are called in arbitrary order, and
# return False (don't send this packet to the Cpu), True (do), or
# None (I don't have an opinion about sending this packet to the Cpu).
floodsetIncludesCpuHandlers = []

# Rewrite-Packet-On-The-Way-To-The-Cpu plugins register themselves with a
# function that is called just before a packet is sent to the CPU. These
# plugins get the chance to modify the packet, for instance to rewrite the
# destination MAC address, so that the CPU handles the packet properly.
# The plugin returns a copy of the packet so that destinations beside the
# CPU do not see the modifications.
# The plugins are called in arbitrary order, and return ( data, dstMac )
# if they have made a modification to either of those values (or want to
# prevent any later plugins from making a modification).
# The "metadata" destination MAC address is passed as a separate argument
# and so can be modified separately from the address written into the packet.
rewritePacketOnTheWayToTheCpuHandlers = []

# Packet replication plugins register themselves with a function that is
# called to modify the outgoing list of interfaces. The function is
# given the source port and a list of output interfaces.  Examples include
# port mirroring and Mlag constrained flooding.
packetReplicationHandlers = []

# Trap lookup (e.g., IgmpSnooping) plugins register themselves in this
# list with a function to perform the lookup.  This hook is for things
# that redirect packets to the cpu or force a copy of the packet to be
# sent to the cpu.  All registered functions are called, in arbitrary
# [Plugin registration] order, and return a tuple of ( matched, trap,
# copyToCpu ) -- only 1 plugin is expected to return true in matched,
# If no plugin returns True, then the packet is forwarded normally.
# Typically, the value of variable trap would be True for trapping or
# False for not trapping, but in some special case, like Ptp, it could
# also be a constant TRAP_ANYWAY, which is used when stpState is
# stpDiscarding, but we want to trap it anyway. This is because, though
# most protocol agents listen on the port-channel interfaces for their
# respective protocols on lags for packets, a subset like PTP agent
# listens on the individual member interfaces instead of the port-channel
# interface itself, but for Etba the packets don't make it to the member
# interfaces as the stpState is marked as stpDiscarding on the member
# interfaces, and stpForwarding on the port-channel itself.
trapLookupHandlers = []

# Sflow plugin registers in this list
# with a function to preform sampling if necessary.
# We currently are interested in the following packets:
#  - Arrived on an interface that is not the Fabric.
#  - Is not being trapped to the CPU.
#  - Has no reason to be dropped by hardware.
sflowHandlers = []

# Post bridging handlers will  register here .Plugin is called just before sending
# the frame out of the interface ( pkts not destined to CPU ) and can modify
# contents of Ethernet frame, including smac and/or dmac.
# This plugin is currrently used by FHRP to modify outgoing ARP packets with smac
# as varp mac if outgoing interface belongs to an unnumbered SVI/Vlan
postBridgingHandlers = []

# Egress rewrite handlers will register here. The handler is called just before
# sending the frame on the egress intf. It is called right before the macsec encrypt
# handler. The handler can be used to modify the packet right before it leaves the
# namespace dut.
egressRewriteHandlers = []

# Routing plugin ( e.g., Mroute ) register themselves in this list
# with a function to perform the routing. All registered functions are
# called succesively in no particular order, the packets are then bridged
ebraRoutingHandlers = []
preTunnelHandlerBpfFilter = {}

# Plugins may need to initialize themselves at bridge-init or agent-init
# time.
bridgeInitHandlers = []
agentInitHandlers = []

# using for bridgeInitHandlers for now, because agent may try
# initialize these plugins at more than one place. This list will be
# used to guard against such multiple initializations
pluginsInitialized = []

# Learning handlers are invoked at learnAddr() time.  For example, if there
# is an active peerLink (in an Mlag configuration)
learningHandlers = []
postAgingHandlers = []

# During a switchover, Etba agent will take care of platform and hardware
# agents that don't run on a namespace dut so that all switchover stages
# can be marked as complete
agentsAndStagesDict = {}
agentsAndStagesDict[ 'PlxPcie-system' ] = [ 'PCIEAcquired',
                                            'PcieConfigurable' ]
agentsAndStagesDict[ 'ForwardingAgent' ] = [ 'DmaReady',
                                             'HwSyncWaitNormal',
                                             'HwSyncWaitForSlowCards' ]
agentsAndStagesDict[ 'ElectionMgr' ] = [ 'SwitchoverReady' ]
agentsAndStagesDict[ 'Fru' ] = [ 'Fru-Plugins' ]
agentsAndStagesDict[ 'ModularSystem' ] = [ 'ModularSystem-switchover' ]
agentsAndStagesDict[ 'NorCalCard' ] = [ 'CardPowerAgents' ]
agentsAndStagesDict[ 'Pca9555' ] = [ 'CardPowerAgents' ]
agentsAndStagesDict[ 'Ucd9012-system' ] = [ 'CardPowerAgents' ]
agentsAndStagesDict[ 'MactAgent' ] = [ 'InitMacTable' ]
agentsAndStagesDict[ 'AleL3Agent' ] = [ 'PlatformLfibSync' ]

maxExtendedVlanSupported = 0x4000

class PluginContext:
   TRAP_PRIORITIES = TRAP_PRIORITIES

   def __init__( self, arfaMode ):
      self.arfaMode_ = arfaMode

   def arfaMode( self ):
      return self.arfaMode_

   def inArfaMode( self ):
      return self.arfaMode_ == ArfaModeType.arfaOnlyMode

   def registerBridgeInitHandler( self, func ):
      bridgeInitHandlers.append( func )

   def registerAgentInitHandler( self, func ):
      agentInitHandlers.append( func )

   def registerInterfaceHandler( self, typeName, klass ):
      interfaceHandlers.append( ( typeName, klass ) )

   def registerPreBridgingHandler( self, func ):
      global preBridgingHandler
      preBridgingHandler = func

   def registerPreTunnelHandler( self, func, hdlrPrio=HANDLER_PRIO_NORMAL,
                                 bpfFilterInfo=None ):
      if ( hdlrPrio == HANDLER_PRIO_HIGHEST and
         preTunnelHdlrPrioMap[ HANDLER_PRIO_HIGHEST ] ):
         assert False, "Multiple handlers are not allowed at HANDLER_PRIO_HIGHEST"
      preTunnelHdlrPrioMap[ hdlrPrio ].append( func )
      hdlrlist = []
      for prio in preTunnelHdlrPrioMap:
         hdlrlist.extend( preTunnelHdlrPrioMap[ prio ] )

      if self.arfaMode() == ArfaModeType.hybridMode:
         assert bpfFilterInfo is None
      else:
         assert bpfFilterInfo is not None
         bpfFilterKey, bpfFilterString = bpfFilterInfo
         assert bpfFilterKey not in preTunnelHandlerBpfFilter
         preTunnelHandlerBpfFilter[ bpfFilterKey ] = ( bpfFilterString, func )
      # As the EbraTestBridge file imports this one, we can't create a new list
      del preTunnelHandler[ : ]
      preTunnelHandler.extend( hdlrlist )

   def registerDestLookupHandler( self, func ):
      destLookupHandlers.append( func )

   def registerFloodsetIncludesCpuHandler( self, func ):
      floodsetIncludesCpuHandlers.append( func )

   def registerRewritePacketOnTheWayToTheCpuHandler( self, func ):
      rewritePacketOnTheWayToTheCpuHandlers.append( func )

   def registerPacketReplicationHandler( self, func ):
      packetReplicationHandlers.append( func )

   def registerPostBridgingHandler( self, func ):
      postBridgingHandlers.append( func )

   def registerTrapLookupHandler( self, func,
                                  priority=TRAP_PRIORITIES[ "DEFAULT" ] ):
      # Higher priority number means higher priority
      trapLookupHandlers.append( ( priority, func ) )

   def registerLearningHandler( self, func ):
      learningHandlers.append( func )

   def registerPostAgingHandler( self, func ):
      postAgingHandlers.append( func )

   def registerSflowHandler( self, func ):
      sflowHandlers.append( func )

   def registerRoutingHandler( self, func ):
      ebraRoutingHandlers.append( func )
      ebraRoutingHandlerCounter[ func.__name__ ] = RoutingHandlerCounter()

   def registerMacsecDecryptHandler( self, func ):
      global macsecDecryptHandler
      macsecDecryptHandler = func

   def registerMacsecEncryptHandler( self, func ):
      global macsecEncryptHandler
      macsecEncryptHandler = func

   def registerEgressRewriteHandler( self, func ):
      egressRewriteHandlers.append( func )

# learning handlers are invoked in learnAddr(), for example to
# disable learning for peer links in an Mlag configuration.
# A learning handler should be invoked with ( vlanId, macAddr, portName )
# and should return either a string or None.

def invokeLearningHandlers( vlanId, macAddr, portName, entryType, pktCtx=None ):
   result = []
   for h in learningHandlers:
      r = None
      # It would be too hard to construct the Arfa types without the packet data,
      # so if the pktCtx was passed in, and it is the special Arfa shim, call it
      if isinstance( h, ArfaShimLearningHandlers ):
         if pktCtx:
            r = h( pktCtx )
      else:
         r = h( vlanId, macAddr, portName, entryType )
      if( r is not None ):
         result.append( r )
   return result

def invokePostAgingHandlers( ctx, vlanId, macAddr ):
   for h in postAgingHandlers:
      h( ctx, vlanId, macAddr )

def isIEEEReserved( addr ):
   return addr[ :15 ] == '01:80:c2:00:00:'

def isIEEELinkConstrained( addr ):
   return ((addr[ :16 ] == '01:80:c2:00:00:0') or (addr == PVST_BPDU_ADDR))

# Starting in Linux 4.5, if the interface of a tun/tap device is down,
# the kernel returns EIO on writes instead of silently dropping the
# packet.

def writeToTunTapDevice( fd, buf ):
   try:
      os.write( fd, buf )
   except OSError as e:
      if e.errno != errno.EIO:
         raise

class VlanStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Bridging::VlanStatusDir'
   def __init__( self, bridgingVlanStatusDir, bridge, entityManager ):
      assert( bridge )
      self.bridge_ = bridge
      self.entityManager_ = entityManager
      Tac.Notifiee.__init__( self, bridgingVlanStatusDir )

   def getFid( self, vlanId ):
      if 0 < vlanId < 4095:
         fid = self.bridge_.brConfig_.vidToFidMap.get( vlanId )
      else:
         fid = None
      if not fid:
         fid = vlanId
      return fid

   @Tac.handler( 'vlanStatus' )
   def handleVlanStatus( self, vlanId ):
      fid = self.getFid( vlanId )
      if vlanId in self.notifier_.vlanStatus:
         # vlan being added, add existing configured hosts, if any
         t0( 'handleVlanStatus Up, vlanId %d ' % vlanId )
         fdbConfig = self.bridge_.brConfig_.fdbConfig.get( fid )
         if fdbConfig:
            for key in fdbConfig.configuredHost:
               hostEntry = fdbConfig.configuredHost.get( key )
               if hostEntry:
                  t0( "handleVlanStatus: adding %d/%s to %s" %
                     ( vlanId, key, hostEntry.intf ) )
                  self.bridge_.learnAddr( vlanId, key,
                     hostEntry.intf, hostEntry.entryType )
         return

      isActive = self.entityManager_.redundancyStatus().mode == "active"

      if isActive:
         # Vlan is down, flush all learnedHosts
         for lh in self.bridge_.smashBrStatus_.smashFdbStatus.values():
            if lh.fid == fid:
               t9( "handleVlanStatus flushing macAddr %s/%s" %
                     ( str( lh.address ), lh.entryType ) )
               self.bridge_.deleteMacAddressEntry( vlanId, str( lh.address ),
                                                   lh.entryType )

class FdbConfigReactor( Tac.Notifiee ):
   notifierTypeName = 'Bridging::FdbConfig'

   def __init__( self, fdbConfig, bridge ):
      Tac.Notifiee.__init__( self, fdbConfig )
      self.bridge_ = bridge
      self.vlanId = fdbConfig.fid
      for key in fdbConfig.configuredHost:
         self.handleConfiguredHost( key )

   @Tac.handler( 'configuredHost' )
   def handleConfiguredHost( self, key ):
      hostEntry = self.notifier_.configuredHost.get( key )
      if hostEntry:
         t0( "handleConfiguredHost: adding %d/%s to %s" %
               ( self.vlanId, key, hostEntry.intf ) )
         self.bridge_.learnAddr( self.vlanId, key,
                                 hostEntry.intf, hostEntry.entryType )
      else:
         t0( "handleConfiguredHost: removing %d/%ss" %
               ( self.vlanId, key ) )
         self.bridge_.deleteMacAddressEntry( self.vlanId, key,
                                             'configuredStaticMac' )

def isPhyControl( srcMac, dstMac, etherType, data ):
   return dstMac == '01:1c:73:ff:ff:ff' and etherType == 0xd28b \
         and data[ 0 : 4 ] == b'\xe1\xba\x00\x00'

class AgingNotifierReactor( Tac.Notifiee ):
   notifierTypeName = "Arfa::AgingNotifier"

   def __init__( self, notifier, bridge ):
      Tac.Notifiee.__init__( self, notifier )
      self.bridge_ = bridge

   @Tac.handler( 'agedCount' )
   def handleAged( self ):
      entry = self.notifier_.aged
      invokePostAgingHandlers( self.bridge_, entry.fid, entry.address )

class EbraTestPhyPortIntfStatusReactor( Tac.Notifiee ):
   '''Synchronizes the configured MTU with the interface if
   it becomes a routed port.'''
   notifierTypeName = 'Interface::EthIntfStatus'
   def __init__( self, intfStatus, intfStatusLocal, intfConfig ):
      Tac.Notifiee.__init__( self, intfStatus )
      self.intfConfig_ = intfConfig
      self.intfStatusLocal_ = intfStatusLocal

   @Tac.handler( 'forwardingModel' )
   def handleForwardingModel( self ):
      # Synchronize the MTU with the kernel and the status if the forwarding
      # model becomes routed.
      SyncMtu.syncMtu( self.notifier_, self.intfStatusLocal_, self.intfConfig_ )

class EbraTestPhyPortIntfConfigReactor( Tac.Notifiee ):
   '''Controls IntfStatus::linkState based on IntfConfig::enabled,
   and synchronizes the configured MTU with the interface when it
   changes.'''
   notifierTypeName = 'Interface::EthPhyIntfConfig'

   def __init__( self, intfConfig, intfStatus, intfStatusLocal, ebraTestPhyPort ):
      Tac.Notifiee.__init__( self, intfConfig )
      self.intfStatus_ = intfStatus
      self.intfStatusLocal_ = intfStatusLocal
      self.ebraTestPhyPort = ebraTestPhyPort
      self.handleEnabledStateLocal()
      self.handleLinkModeLocal()
      self.handleMtu()

   @Tac.handler( 'enabledStateLocal' )
   def handleEnabledStateLocal( self ):
      # Note that we react to 'enabledStateLocal' because neither 'enabled' nor
      # 'enabledState' are notifying.
      self.ebraTestPhyPort.updateLinkStatus()

   @Tac.handler( 'linkModeLocal' )
   def handleLinkModeLocal( self ):
      self.ebraTestPhyPort.updateSpeedDuplex()

   @Tac.handler( 'mtu' )
   def handleMtu( self ):
      # Synchronize the MTU for L3 interfaces.
      SyncMtu.syncMtu( self.intfStatus_, self.intfStatusLocal_, self.notifier_ )

class PktReader( Tac.Notifiee ):
   notifierTypeName = "Arnet::EthDevPam"

   def __init__( self, pam, handler ):
      self.pam = pam
      Tac.Notifiee.__init__( self, pam )
      self.handler = handler
      self.counter = 0

   @Tac.handler( "readableCount" )
   def readHandler( self ):
      t0( 'readHandler called %s ' % self.pam.readableCount )
      pkt = self.pam.rxPkt()
      if pkt:
         self.counter += 1
         self.handler( pkt.bytesValueMaybeWithVlan )

class ArfaPortReactor( Tac.Notifiee ):
   notifierTypeName = "Arfa::ArfaPythonPortNotifier"

   def __init__( self, notifier, port ):
      Tac.Notifiee.__init__( self, notifier )
      self.port = port

   @Tac.handler( 'sendFrameCount' )
   def handleSendFrame( self ):
      self.port.sendFrame(
         self.notifier_.pkt.bytesValue,
         self.notifier_.srcAddr,
         self.notifier_.dstAddr,
         self.notifier_.srcPort,
         0,
         self.notifier_.outgoingVlanId,
         "none",  # Vlan actions were already taken in the Arfa caller
      )

      self.notifier_.pkt = None
      self.notifier_.srcAddr = EthAddr()
      self.notifier_.dstAddr = EthAddr()
      self.notifier_.srcPort = IntfId()
      self.notifier_.outgoingVlanId = VlanIdOrAnyOrNone()

   @Tac.handler( 'trapCount' )
   def handleTrapFrame( self ):
      self.port.trapFrame( self.notifier_.pkt.bytesValue )
      self.notifier_.pkt = None

class EbraTestPhyPort( EbraTestPort ):
   """Simulates a physical port."""

   def __init__( self, bridge, tapDevice, trapDevice, intfConfig,
                 intfStatus, intfXcvrStatus,
                 intfStatusLocal, intfCounters, entityManager ):
      """Initialize the port to its default state."""
      EbraTestPort.__init__( self, bridge, intfConfig, intfStatus )
      self.noTrapDevice = False
      self.pktReader = None
      # Note this code is duplicated in EbraTestBridge:EtbaConfigReactor
      # It would probably be fine to just set "noTrapDevice = trapDevice is None"
      # but the python code is going away so it's not worth trying to clean up
      if CEosHelper.isCeos():
         if EtbaDutToggleLib.toggleCEosLabWithTrapEnabled():
            t0( 'Creating trap devices anyway in cEOS-lab' )
         else:
            t0( 'Skip creating the trapDevices in CEos' )
            self.noTrapDevice = True

      # pylint is confused by the value returned by Tac.nonConst.
      # pylint: disable=maybe-no-member, E1103
      if tapDevice.fileno == 0:
         # Bind to the interface ourselves (this is used in the vEOS
         # environment where we don't already have a socket set up
         # for us)
         EthDevPam = Tac.Type( 'Arnet::EthDevPam' )
         pam =  EthDevPam( tapDevice.name )
         pam.ethProtocol = ETH_P_ALL
         pam.maxRxPktData = Constants.defaultMtu
         pam.separateDot1QHdr = True
         pam.enabled = True

         self.pktReader = PktReader( pam, self._tapReadHandler )
         tapDevice = Tac.nonConst( tapDevice )
         tapDevice.fileno = pam.sd
         if self.noTrapDevice:
            trapDevice = tapDevice
         self.tapFile_ = None

      self.tapDevice_ = tapDevice
      self.trapDevice_ = trapDevice

      self.processFrame_ = None

      self.phyIsUp = True
      self.prevIntfConfigEnabled = False

      self.intfCounters_ = intfCounters

      self.bridge_ = bridge
      self.intfConfig_ = intfConfig
      self.intfStatus_ = intfStatus
      self.intfXcvrStatus_ = intfXcvrStatus
      self.intfStatusLocal_ = intfStatusLocal

   def onActive( self ):
      t0( "onActive EbraTestPhyPort" )
      if not self.pktReader:
         self.tapFile_ = Tac.File( self.tapDevice_.fileno, self._tapReadHandler,
                                   self.tapDevice_.name, readBufferSize=16000 )
      if not self.noTrapDevice:
         self.trapFile_ = Tac.File( self.trapDevice_, self._trapDeviceReadHandler,
                                    self.trapDevice_.name, readBufferSize=16000 )


      self.intfPhyConfigReactor_ = EbraTestPhyPortIntfConfigReactor(
         self.intfConfig_, self.intfStatus_, self.intfStatusLocal_, self )
      self.intfPhyStatusReactor_ = EbraTestPhyPortIntfStatusReactor(
         self.intfStatus_, self.intfStatusLocal_, self.intfConfig_ )
      self.operStatusReactor = Tac.newInstance( "EthIntf::OperStatusReactor",
                                                self.intfStatus_.intfId,
                                                self.intfStatus_,
                                                self.intfXcvrStatus_)

      self.intfXcvrStatus_.xcvrPresence = 'xcvrPresent'
      self.intfXcvrStatus_.xcvrType = 'EbraTestPhyPort'
      if self.intfStatus_.speedEnum == 'speedUnknown':
         self.intfStatus_.speedEnum = 'speed1Gbps'
      self.intfStatus_.duplex = 'duplexFull'
      self.intfStatus_.addr = self.intfStatus_.burnedInAddr
      if CEosHelper.isCeos():
         # In cEOS, the environment gives us the
         # interface MAC address.
         self.intfStatus_.routedAddr = self.intfStatus_.addr
      else:
         self.intfStatus_.routedAddr = self.bridge_.bridgeMac()


   def sendPhyControl( self ):
      raw = EbraTestBridgeLib.rawPhyControlPacket( self.intfConfig_.enabled,
               src=Ethernet.convertMacAddrToPackedString( self.intfStatus_.addr ) )
      writeToTunTapDevice( self.tapDevice_.fileno, raw )

   def updateLinkStatus( self ):
      # The link state is dependent on admin state and on the link
      # status obtained from the phy control protocol.
      linkStatus = 'linkUp' if (self.intfConfig_.enabled and self.phyIsUp) \
                   else 'linkDown'
      t4( 'Set link state to', linkStatus, 'for', self.intfName_ )
      if self.prevIntfConfigEnabled != self.intfConfig_.enabled:
         # Send out a PhyControl frame to let the other end know about
         # the admin status of this port
         self.sendPhyControl()
      self.prevIntfConfigEnabled = self.intfConfig_.enabled
      self.intfStatus_.linkStatus = linkStatus

   def updateSpeedDuplex( self ):
      speed = EthTypesApi.linkModeToEthSpeed( self.intfConfig_.linkMode )
      duplex = EthTypesApi.linkModeToEthDuplex( self.intfConfig_.linkMode )
      if speed != 'speedUnknown':
         self.intfStatus_.speed = speed
         self.intfStatus_.duplex = duplex

   def setProcessFrame( self, func ):
      """Hook frame input for this port"""
      self.processFrame_ = func

   def _handlePhyControl( self, data ):
      """Handle a phy control frame.  This can be used to bring up or
      down the link by sending in a packet.  See AID6280"""
      self.phyIsUp = ( data[ 18 ] != 0 )
      self.updateLinkStatus()

   def _countIn( self, intf, octets, dstMacAddr ):
      if Ethernet.isBroadcast( dstMacAddr ):
         self.intfCounters_.broadcastIn( intf, octets )
      elif Ethernet.isMulticast( dstMacAddr ):
         self.intfCounters_.multicastIn( intf, octets )
      else:
         self.intfCounters_.unicastIn( intf, octets )

   def _countOut( self, intf, octets, dstMacAddr ):
      if Ethernet.isBroadcast( dstMacAddr ):
         self.intfCounters_.broadcastOut( intf, octets )
      elif Ethernet.isMulticast( dstMacAddr ):
         self.intfCounters_.multicastOut( intf, octets )
      else:
         self.intfCounters_.unicastOut( intf, octets )

   def _tapReadHandler( self, data ):
      ( srcMacAddr, dstMacAddr ) = macAddrsAsStrings( data )
      etherType = ( data[ 12 ] << 8 ) | data[ 13 ]
      if isPhyControl( srcMacAddr, dstMacAddr, etherType, data[ 14: ] ):
         t8( "got phy control frame!", srcMacAddr, dstMacAddr, etherType )
         self._handlePhyControl( data )
         return

      if self.intfConfig_.enabled:
         t8( "Received frame on %s" % self.intfName_, PacketFormat( data ) )
         bad = False
         size = len( data ) + 4 # Add 4 bytes for CRC
         # We emulate Tahoe for now and round the mtu up to the nearest
         # multiple of 4.
         # intfStatus_.mtu is the L3 MTU, so we add lower layer headers
         # to come up with the L2 MTU.
         mtu = ( self.intfStatus_.l2Mtu if self.intfStatus_.l2Mtu else
                 Constants.defaultMtu )
         if (mtu % 4) != 0:
            mtu += (4 - (mtu % 4))
         if size > mtu:
            bad = True

         self._countIn( self.intfName_, len( data ) + 4, dstMacAddr )

         if etherType == ETH_P_PAUSE:
            t5( 'Discarding MAC control frame on %s' % self.intfName_ )
         elif bad:
            t5( 'Discarding bad frame (size %d) on %s' % ( size, self.intfName_ ) )
         else:
            processFrame = self.processFrame_ or self.bridge_.processFrame
            processFrame( data, srcMacAddr, dstMacAddr, self, False )
      else:
         t8( "Discarding frame received on disabled port %s" % self.intfName_ )

   def sendFrame( self, data, srcMacAddr, dstMacAddr, srcPortName,
                  priority, vlanId, vlanAction ):
      if self.intfConfig_.enabled:
         data = applyVlanChanges( data, priority, vlanId, vlanAction )

         for func in egressRewriteHandlers:
            chgData = func( self, data, srcMacAddr, dstMacAddr, srcPortName,
                            priority, vlanId, vlanAction )
            if chgData:
               data = chgData

         if macsecEncryptHandler is not None:
            chgData = macsecEncryptHandler( self, data )
            if chgData:
               data = chgData

         t8b( "Sending frame out %s" % self.intfName_, PacketFormat( data ) )
         writeToTunTapDevice( self.tapDevice_.fileno, data )

         if not dstMacAddr and not srcMacAddr:
            ( _, dstMacAddr ) = macAddrsAsStrings( data )
         self._countOut( self.intfName_, len( data ) + 4, dstMacAddr )
      else:
         t8( "Refusing to transmit on disabled port %s" % self.intfName_ )

   def _trapDeviceReadHandler( self, data ):
      if not self.noTrapDevice:
         finalIntfs = { self.intfName_: None }
         t8( "trapping frame output on %s" % self.intfName_, PacketFormat( data ) )
         for func in packetReplicationHandlers:
            func( self.bridge_, finalIntfs=finalIntfs )
         for intfName in finalIntfs:
            self.bridge_.port[ intfName ].sendFrame(
               data, None, None, None, None, None, PKTEVENT_ACTION_NONE )

   def trapFrame( self, data ):
      if not self.noTrapDevice:
         t8( "trapping frame input on %s" % self.intfName_, PacketFormat( data ) )
         os.write( self.trapDevice_.fileno(), data )

class EbraTestCpuPortTapDeviceHandler:
   """wrap the fabricIndex in the Tac.File read handler so that we can pass
   it back to the bridge"""
   def __init__( self, fabricIndex, cpuPort, device ):
      self.fabricIndex_ = fabricIndex
      self.cpuPort_ = cpuPort
      self.device_ = device
      self.tapFile = Tac.File( self.device_, self.readHandler,
                               self.device_.name, readBufferSize=16000 )

   def readHandler( self, data ):
      ( srcMacAddr, dstMacAddr ) = macAddrsAsStrings( data )
      self.cpuPort_.bridge_.processFrame( data, srcMacAddr, dstMacAddr,
                                          self.cpuPort_, False,
                                          highOrderVlanBits=self.fabricIndex_ )

class EbraTestCpuPort( EbraTestPortBase ):
   """Simulates the interface from the switch chips to the CPU/fabric"""

   def __init__( self, intfName, bridge, tapDevices, trapDevice ):
      super().__init__( intfName, bridge )
      assert isinstance( tapDevices, list )
      self.tapDevices_ = tapDevices
      self.trapDevice_ = trapDevice
      self.tapHandlers_ = dict() # pylint: disable=use-dict-literal

   def _tapReadHandlerN( self, fabricIndex, data ):
      ( srcMacAddr, dstMacAddr ) = macAddrsAsStrings( data )
      self.bridge_.processFrame( data, srcMacAddr, dstMacAddr, self,
                                 False, highOrderVlanBits=fabricIndex )
   def onActive( self ):
      t0( "onActive EbraTestCpuPort" )

      # pylint: disable-next=consider-using-enumerate
      for i in range( len( self.tapDevices_ ) ):
         dev = self.tapDevices_[ i ]
         self.tapHandlers_[ i ] = EbraTestCpuPortTapDeviceHandler( i, self, dev )

      t8( "trap", self.trapDevice_.name, self.trapDevice_.fileno() )
      self.trapFile_ = Tac.File( self.trapDevice_, self._trapDeviceReadHandler,
                                 self.trapDevice_.name, readBufferSize=16000 )

   def _trapDeviceReadHandler( self, data ):
      t8( "trapping frame 'output' on Cpu", PacketFormat( data ) )
      writeToTunTapDevice( self.tapDevices_[ 0 ].fileno(), data )

   def sendFrame( self, data, srcMacAddr, dstMacAddr, srcPortName,
                  priority, vlanId, vlanAction ):
      t8( "Sending frame out %s" % self.intfName_, PacketFormat( data ) )

      evid = Tac.Value( 'Bridging::ExtendedVlanId', vlanId )
      fabricIndex = 0
      if not evid.inDot1qRange():
         vlanId = evid.lowOrderVlanId()
         fabricIndex = evid.highOrderBits()
         t8( 'rewriting extended vlan ID %d to %d w/ fabricId %d' %
             ( evid.value, vlanId, fabricIndex ) )

      data = applyVlanChanges( data, priority, vlanId, vlanAction )
      assert fabricIndex < len( self.tapDevices_ )
      writeToTunTapDevice( self.tapDevices_[ fabricIndex ].fileno(), data )

   def trapFrame( self, data ):
      t8( "trapping frame 'input' on Cpu", PacketFormat( data ) )
      os.write( self.trapDevice_.fileno(), data )

class EbraTestBridge:

   def __init__( self, name, entityManager, inNamespace, ethIntfConfigDir,
                 ethIntfStatusDir, ethIntfStatusLocalDir,
                 allEthPhyIntfStatusDir, ebraCounterConfig, ebraCounterStatus,
                 maxExtendedVlan, numFabricIntfs, smashBrStatus, bridgingStatus ):
      self.name_ = name
      self.entityManager_ = entityManager
      self.shmemEntityManager_ = SharedMem.entityManager( entityManager.sysname(),
                                                          False )
      self.inNamespace_ = inNamespace
      self.ethIntfConfigDir_ = ethIntfConfigDir
      self.ethIntfStatusDir_ = ethIntfStatusDir
      self.ethIntfStatusLocalDir_ = ethIntfStatusLocalDir
      self.allEthPhyIntfStatusDir_ = allEthPhyIntfStatusDir
      self.ebraCounterConfig_ = ebraCounterConfig
      self.ebraCounterStatus_ = ebraCounterStatus
      self.maxExtendedVlan_ = maxExtendedVlan
      self.numFabricIntfs_ = numFabricIntfs
      self.brConfig_ = entityManager.entity( 'bridging/config' )
      self.bridgeMac_ = entityManager.entity( 'bridging/config' ).bridgeMacAddr
      self.brStatus_ = bridgingStatus
      self.brVlanStatus_ = entityManager.entity( 'bridging/vlan/status' )
      self.brHwCapabilities_ = entityManager.entity( 'bridging/hwcapabilities' )
      self.flush_ = entityManager.entity( 'bridging/flush/request/cli' )
      self.flushReplyDir_ = entityManager.entity( 'bridging/flush/reply/all' )
      self.topoConfig = entityManager.entity( 'bridging/topology/config' )
      self.topoInstStatus = entityManager.entity( 'bridging/topology/inst/etba' )
      self.etbaConfig = entityManager.entity( 'bridging/etba/config' )
      self.port = {}
      self.agingInterval_ = 15 # Check for aged hosts every 15 seconds
      self.smashBrStatus_ = smashBrStatus

      self.arfaMode_ = ArfaModeType.hybridMode

      # start a shim to populate Bridging::Status from changes in the Smash table
      # for the benefit of plugins that look at brStatus_ directly.  Once these
      # plugins are rewritten to use smash, we can eliminate this reactor and
      # the Bridging::Status altogether.
      self.shim_ = Tac.Type( 'Bridging::ReaderFdbStatusSm' )( self.brStatus_,
                                                              self.smashBrStatus_ )

      self.agingTable = {}

      # Vxlan datapath learning
      self.datapathLearningMode_ = False

      # Per-packet context, may be used by plugins to transmit information to a
      # later stage in the pipeline.
      self.packetContext = {}

      self.hostAgingActivity_ = Tac.ClockNotifiee( self.hostAging )
      self.hostAgingActivity_.timeMin = Tac.now() + self.agingInterval_
      self.bridge_ = weakref.proxy( self )

      self.fdbConfigReactor_ = Tac.collectionChangeReactor(
         self.brConfig_.fdbConfig, FdbConfigReactor, reactorArgs=( self.bridge_, ) )

      self.vlanStatusReactor_ = VlanStatusReactor( self.brVlanStatus_, self.bridge_,
                                                   self.entityManager_ )

      self.brHwCapabilities_.dot1qTunnelSupported = True
      self.brHwCapabilities_.taggedNativeVlanSupported = True
      self.brHwCapabilities_.pvstSupported = True
      self.brHwCapabilities_.backupInterfacesSupported = True
      self.brHwCapabilities_.privateVlansSupported = True
      self.brHwCapabilities_.vxlanSupported = True
      self.brHwCapabilities_.vxlanRoutingSupported = True
      self.brHwCapabilities_.overlayEcmpSupported = True
      self.brHwCapabilities_.igmpSnoopingSupported = True
      # Needed to unblock the CLI configuration guard in Etba for
      # Vxlan interface.
      self.brHwCapabilities_.vxlanUnderlayMcastSupported = True
      self.brHwCapabilities_.vxlanUnderlayIifSwitchSupported = True
      self.brHwCapabilities_.vxlanMcastUnderlayHerSupported = True
      self.brHwCapabilities_.ieeeReservedMacForwardAddrSupported = True
      # XXX Should be mounted read, and we need another mount point to
      # specify the hardware capabilities. Bug 15122 opened to track this.
      self.topoInstStatus.maximumTopologiesSupported = 63
      self.topoInstStatus.hardwareConfigured = True
      self.brHwCapabilities_.staticMcastSupported = True
      self.brHwCapabilities_.ndProxySupported = True
      self.brHwCapabilities_.tunnelIntfBumCountersSupported = True
      self.brHwCapabilities_.tunnelIntfUmCountersSupported = False
      self.brHwCapabilities_.tunnelIntfErrorCountersSupported = True
      # Needed to unblock the CLI configuration guard in Etba for the
      # 'no mac address learning' command for VLANs
      self.brHwCapabilities_.vlanMacLearningSupported = True
      # Needed to unblock the CLI configuration guard in Etba for
      # 'no switchport mac address learning' command
      self.brHwCapabilities_.noMacLearningSupported = True

   # Functions to call on active

   def createAgingNotifierReactor( self ):
      self.agingNotifierReactor_ = AgingNotifierReactor(
                                 getPythonIntoCppShimContext().agingNotifier(),
                                 self )

   def createCounterConfigReactor( self ):
      t0( "creating EtbaCounterConfigReactor" )
      self.etbaCounterConfigReactor_ = (
         EbraTestBridgeLib.EtbaCounterConfigReactor( self.ebraCounterConfig_,
                                                     self.ebraCounterStatus_,
                                                     ebraRoutingHandlerCounter ) )

   def runPlugins( self ):
      t0( "runPlugins" )
      for f in bridgeInitHandlers:
         if f not in pluginsInitialized:
            t0( "Initializing", f.__name__ )
            f( self.bridge_ )
            pluginsInitialized.append( f )

   def arfaMode( self ):
      return self.arfaMode_

   def inArfaMode( self ):
      return self.arfaMode_ == ArfaModeType.arfaOnlyMode

   def name( self ):
      return self.name_

   def em( self ):
      return self.entityManager_

   def sEm( self ):
      return self.shmemEntityManager_

   def inNamespace( self ):
      return self.inNamespace_

   def bridgeMac( self ):
      return self.bridgeMac_
   
   def addPort( self, port, arfaPort=None ):
      assert port.name() not in self.port
      self.port[ port.name() ] = port
      cPort = arfaPort or createCPythonPort( port )
      getPythonIntoCppShimContext().addPort( cPort )

   def delPort( self, intf ):
      if intf in self.port:
         self.port[ intf ].close()
         del self.port[ intf ]
         getPythonIntoCppShimContext().delPort( intf )

   def forwardIeeeReservedMac( self, mac ):
      forwardAll = self.brConfig_.ieeeReservedForwarding.get(
                        self.brHwCapabilities_.ieeeReservedMacForwardAddrAll )
      return forwardAll or self.brConfig_.ieeeReservedForwarding.get( mac )

   def vlanTagState( self, srcPortName, srcIsFabric, etherType, data, dstMacAddr,
                     highOrderVlanBits ):
      """Returns a tuple:
      ( vlanId, priority, vlanTagNeedsUpdating, tagged, routed )
      """
      dot1qIgnoreTag = False
      routed = False

      if highOrderVlanBits is not None:
         assert highOrderVlanBits < self.numFabricIntfs_

      t7( ( 'vlanTagState: srcPortName: %s, srcIsFabric: %s, etherType: %s, ' +
            'dstMacAddr: %s, highOrderVlanBits: %s' ) %
          ( srcPortName, srcIsFabric, etherType, dstMacAddr, highOrderVlanBits ) )

      if not srcIsFabric:
         switchIntfConfig = self.brConfig_.switchIntfConfig.get( srcPortName )

         if switchIntfConfig is None:
            return ( None, None, None, None, None )

         untaggedVlan = switchIntfConfig.nativeVlan

         # If packet comes from dot1qTunnel, ignore 802.1q tag
         if switchIntfConfig.switchportMode == 'dot1qTunnel':
            dot1qIgnoreTag = True
         elif switchIntfConfig.switchportMode == 'routed':
            routed = True
            dot1qIgnoreTag = True

      # frames from the fabric should always be tagged.

      # Calculate the VLAN ID of the frame.
      vlanTagNeedsUpdating = False
      if etherType == ETH_P_8021Q and not dot1qIgnoreTag:
         vlanTag = ( ( data[ 14 ] << 8 ) |
                     data[ 15 ] )
         vlanId = vlanTag & 0x0fff
         priority = ( vlanTag & 0xe000 ) >> 13
         if vlanId == 0:
            t7( '802.1q priority tag present, with priority', priority )
            assert not srcIsFabric
            vlanId = untaggedVlan
            tagged = False
            vlanTagNeedsUpdating = True
         else:
            t7( '802.1q tag present, with VLAN ID', vlanId, ', priority', priority )
            tagged = True
      else:
         vlanId = None if srcIsFabric else untaggedVlan
         t7( 'No 802.1q tag present, initial vlan id is', vlanId )
         # Drop untagged frames on the fabric interface by setting the
         # vlanId to None.  We don't really expect these, but some
         # linux process sends untagged IPv6 packets to this
         # interface.
         priority = 0
         tagged = False

      if vlanId and highOrderVlanBits:
         evid = Tac.Value( 'Bridging::ExtendedVlanId', vlanId )
         # function might get called twice, so let's ignore the high
         # order bits if they already match
         if evid.highOrderBits() != highOrderVlanBits:
            # can't set it to a new value if we've already got an
            # extended-range vlan
            assert evid.highOrderBits() == 0

            lowVlanId = vlanId
            vlanId = ( highOrderVlanBits << 12 ) | vlanId
            assert vlanId < self.maxExtendedVlan_
            t7( 'prepending %d to vlanId %d (0x%x), yielding %d (0x%x)' %
                ( highOrderVlanBits, lowVlanId, lowVlanId, vlanId, vlanId ) )

      return( vlanId, priority, vlanTagNeedsUpdating, tagged, routed )

   def vlanAction( self, srcIntf, dstIntf, vlanId, destIsFabric,
                   etherType, vlanTagNeedsUpdating ):
      vlanAction = PKTEVENT_ACTION_NONE
      dot1qIgnoreTag = False

      if destIsFabric:
         sendTagged = True
      else:
         if srcIntf != 'Cpu':
            srcSwitchIntfConfig = self.brConfig_.switchIntfConfig.get( srcIntf )
            if ( srcSwitchIntfConfig is not None and
                 srcSwitchIntfConfig.switchportMode == 'dot1qTunnel' ):
               dot1qIgnoreTag = True
         dstSwitchIntfConfig = self.brConfig_.switchIntfConfig.get( dstIntf )
         sendTagged = ( dstSwitchIntfConfig is not None and
                        dstSwitchIntfConfig.switchportMode == 'trunk' and
                        vlanId != dstSwitchIntfConfig.nativeVlan )

      if etherType == ETH_P_8021Q and not dot1qIgnoreTag:
         if not sendTagged:
            vlanAction = PKTEVENT_ACTION_REMOVE
         elif vlanTagNeedsUpdating:
            vlanAction = PKTEVENT_ACTION_REPLACE
      elif sendTagged:
         vlanAction = PKTEVENT_ACTION_ADD
      return vlanAction

   def processFrame( self, data, srcMacAddr, dstMacAddr, srcPort, tracePkt,
                     highOrderVlanBits=None ):
      """Processes a frame received on one of the simulated ports, performing zero
      or more of the following actions:

      -  Queueing the frame for transmission to the CPU. XXX We don't do this yet.
      -  Sending the frame out one or more simulated ports.
      -  Learning the location of the MAC address that sent the frame.
      -  Re-writing the MAC address and/or VLAN tag of the frame.

      If tracePkt is True, this will process the frame as usual, but instead of
      sending the packets out the determined egress interfaces, it will simply return
      that list of egress interfaces without sending anything.

      highOrderVlanBits is used to indicate the higher order bits (more than 12)
      of a Bridging::ExtendedVlanId that do not fit in the 802.1q header in
      the packet data
      """

      if highOrderVlanBits is not None:
         assert highOrderVlanBits < self.numFabricIntfs_

      self.packetContext = {}
      srcPortName = srcPort.name()
      t8( 'Processing frame with srcMacAddr: %s, dstMacAddr: %s, srcPort: %s, '
          'length: %d, highOrderVlanBits: %s' %
          ( srcMacAddr, dstMacAddr, srcPortName, len( data ), highOrderVlanBits ) )

      if preBridgingHandler is not None:
         action = preBridgingHandler( self, data, srcMacAddr, dstMacAddr, srcPort )
         if action == 'trap':
            t7( 'Trapping frame with source: %s, destination: %s, srcPort: %s '
                % ( srcMacAddr, dstMacAddr, srcPortName ) )
            srcPort.trapFrame( data )
            return None
         elif action == 'deny':
            t7( 'Dropping frame with source: %s, destination: %s, srcPort: %s '
                % ( srcMacAddr, dstMacAddr, srcPortName ) )
            return None

      if macsecDecryptHandler is not None:
         chgData = macsecDecryptHandler( self, data, srcPort )
         if chgData:
            data = chgData

      # Invoke Tunnel handler, which may change packet data and/or source port
      t8( "Invoke Tunnel handlers" )
      for preTunnelHdlr in preTunnelHandler:
         ( chgData, chgSrcPort, drop, chgHighOrderVlanBits ) = preTunnelHdlr(
            self, dstMacAddr, data, srcPort )
         if drop:
            t7( '%s consumed the frame' % preTunnelHdlr.__name__ )
            return None
         if chgData:
            t7( '%s changed data' % preTunnelHdlr.__name__ )
            data = chgData
            # Get the src and dst Mac from the modified packet
            ( srcMacAddr, dstMacAddr ) = macAddrsAsStrings( data )
         if chgSrcPort:
            t7( '{} changed srcPort to {}'.format( preTunnelHdlr.__name__,
                                               chgSrcPort.name() ) )
            srcPort = chgSrcPort
         if chgHighOrderVlanBits is not None:
            assert chgHighOrderVlanBits < self.numFabricIntfs_
            t7( '{} changed highOrderVlanBits to {}'.format( preTunnelHdlr.__name__,
                                                         chgHighOrderVlanBits ) )
            highOrderVlanBits = chgHighOrderVlanBits
      t8( "Tunnel handlers complete, srcPort: %s, length: %d, highVlan: %s"
          % ( srcPort.name(), len( data ), highOrderVlanBits ) )

      # Do normal bridging of packet in ingress vlan
      vlanInfo, data, _ = self.bridgeFrame( data, srcMacAddr, dstMacAddr, srcPort,
                                            False, highOrderVlanBits )
      t8( 'Bridging in ingress VLAN complete' )

      # Get the priority and tagging action since every copy of the packet
      # will need to be retagged
      ( vlanId, priority, _vlanTagNeedsUpdating, tagged, _routed ) = vlanInfo

      # This is a list of packets and the vlans they need to go out on
      routedPackets = []
      # Route the packet in these vlans
      for routingHandler in ebraRoutingHandlers:
         modifiedPktVlanTupleList =  routingHandler( self, vlanId, dstMacAddr, data )
         for ( modifiedPkt, vlans ) in modifiedPktVlanTupleList: 
            if modifiedPkt:
               t8( "Routed packet appended by %s in vlans", 
                     ( routingHandler.__name__, vlans ) )
               routedPackets.append( ( modifiedPkt, vlans ) )
               ebraRoutingHandlerCounter[ routingHandler.__name__ ].routed += 1
            else:
               ebraRoutingHandlerCounter[ routingHandler.__name__ ].ignored += 1
      if routedPackets:
         ebraRoutingHandlerCounter[ "overall" ].routed += 1
      else:
         ebraRoutingHandlerCounter[ "overall" ].ignored += 1
      vlanAction = PKTEVENT_ACTION_REPLACE if tagged else PKTEVENT_ACTION_ADD
      t8( 'Routing handlers complete, routed packets: %s' % routedPackets )

      bridgeFrameInfoList = []
      for ( modifiedPkt, vlans ) in routedPackets:
         # Get the src and dst Mac from the modified packet
         ( srcMacAddr, dstMacAddr ) = macAddrsAsStrings( modifiedPkt )
         for vlan in vlans:
            # For each packet rewrite the vlan id because it has to be tagged
            # since it comes from the cpu port
            newdata = applyVlanChanges( modifiedPkt, priority, vlan, 
                                        vlanAction )
            # Bridge the packet, and if tracePkt=True, get the egress intfs
            t8( 'Bridging post-routed frame with srcMac: %s, dstMac: %s in VLAN %d' \
                  % ( srcMacAddr, dstMacAddr, vlan ) )
            bridgeFrameInfo = self.bridgeFrame( newdata, srcMacAddr, dstMacAddr,
                                                self.port[ 'Cpu' ], tracePkt,
                                                highOrderVlanBits )
            bridgeFrameInfoList.append( bridgeFrameInfo )
      t8( 'processFrame return' )
      return bridgeFrameInfoList

   def bridgeFrame( self, data, srcMacAddr, dstMacAddr, srcPort, tracePkt,
                    highOrderVlanBits ):
      dropReasonList = []
      srcPortName = srcPort.name()
      srcIsFabric = srcPortName == 'Cpu'

      etherType = ( data[ 12 ] << 8 ) | data[ 13 ]
      t7( 'Bridging frame with source: %s, destination: %s, EtherType: 0x%04x, '
          'length: %d srcPort: %s, highOrderVlanBits: %s' %
          ( srcMacAddr, dstMacAddr, etherType, len( data ) + 4, srcPortName,
            highOrderVlanBits ) )
      if highOrderVlanBits is not None:
         assert highOrderVlanBits < self.numFabricIntfs_

      if not Ethernet.isUnicast( srcMacAddr ):
         t5( 'Warning: frame has multicast source address: %s' % srcMacAddr )

      vlanInfo = self.vlanTagState( srcPortName, srcIsFabric, etherType,
                                    data, dstMacAddr, highOrderVlanBits )
      ( vlanId, priority, vlanTagNeedsUpdating, tagged, routed ) = vlanInfo

      t8( 'Using VLAN ID', vlanId, ', priority', priority, ' tagged', tagged )
      if vlanId is None:
         return BridgeFrameInfo( vlanInfo, data, None )

      evid = Tac.Value( 'Bridging::ExtendedVlanId', vlanId )

      fid = None
      # In pvlans, multiple (secondary) vlans share same (primary vlan's)
      # fid/fdbConfig. So use fid instead of vlanId to get fdbConfig.
      if 0 < vlanId < 4095:
         fid = self.brConfig_.vidToFidMap.get( vlanId )

      if not fid:
         fid = vlanId

      if 0 < fid < 4095:
         fdbConfig = self.brConfig_.fdbConfig.get( fid )
      else:
         fdbConfig = None

      if evid.inDot1qRange():
         vlanConfig = self.brConfig_.vlanConfig.get( vlanId )
      else:
         vlanConfig = None

      if evid.inDot1qRange() and not isIEEELinkConstrained( dstMacAddr ):
         if fdbConfig:
            dropHostSrc = fdbConfig.configuredHost.get( srcMacAddr )

            if dropHostSrc and dropHostSrc.intf == '':
               t0( "dropping due to drop src mac" )
               dropReasonList.append( "drop src mac entry" )

            dropHostDst = fdbConfig.configuredHost.get( dstMacAddr )
            if dropHostDst and dropHostDst.intf == '':
               t0( "dropping due to drop dst mac" )
               dropReasonList.append( "drop dst mac entry" )
               return BridgeFrameInfo( vlanInfo, data, None )

         if not vlanConfig:
            t4( 'Dropping packet on bad vlan %s (no VC for bridging)' % vlanId )
            dropReasonList.append( 'badVlan' )
         elif srcPortName != 'Cpu':
            if srcPortName not in vlanConfig.intf:
               t4( 'bad ingress port %s on vlan %d' % ( srcPort, vlanId ) )
               dropReasonList.append( 'ingress vlan violation' )
               return BridgeFrameInfo( vlanInfo, data, None )
            elif vlanConfig.intf[ srcPortName ].allowedTrafficDirection == 'egress':
               t4( 'ingress traffic not allowed on port %s on vlan %d' % \
                      ( srcPort, vlanId ) )
               dropReasonList.append( 'allowedTrafficDirection vlan violation' )
               return BridgeFrameInfo( vlanInfo, data, None )

      # no STP for extended range vlans yet
      if evid.inDot1qRange() and not srcIsFabric:
         stpState = self.lookupStpState( vlanId, srcPortName )

         forwardingModel = 'unknown'
         if srcPortName in self.ethIntfStatusDir_.intfStatus:
            forwardingModel = self.ethIntfStatusDir_.\
                  intfStatus[ srcPortName ].forwardingModel
         if forwardingModel == "intfForwardingModelUnauthorized" and \
            dstMacAddr != IEEE_8021X_ADDR:
            t5( 'Got a non 802.1x packet when the intf forwarding '
                'model is unauthorized. Dropping packet. ' )
            dropReasonList.append( 'stpdiscarding' )
         if( (stpState == 'discarding') and \
             not isIEEELinkConstrained( dstMacAddr ) ):
            # packet will be discarded for vlanId == 0 signifying
            # an untagged non-control packet received on a trunk port with
            # native vlan tagging enabled
            t5( 'Dropping packet, input forwarding state discarding for ',
                f'address {srcMacAddr} on port {srcPortName}' )
            dropReasonList.append( 'stpdiscarding' )

         if( ( stpState == 'learning' or stpState == 'forwarding' ) and
               not isIEEELinkConstrained( dstMacAddr ) ):
            if( ( not self.datapathLearningMode_ ) and
                srcPortName.startswith( 'Vxlan' ) ):
               t9( 'Not learning on Vxlan : vlan %d, mac %s port %s' %
                   ( vlanId, srcMacAddr, srcPortName ) )
            elif( srcPortName.startswith( 'MplsTrunk' ) ):
               t9( 'Not learning on MplsTrunk1 : vlan %d, mac %s port %s' %
                   ( vlanId, srcMacAddr, srcPortName ) )
            else:
               # Learn the location of the source MAC address.
               entryType = 'learnedRemoteMac' if srcPortName.startswith( 'Vxlan' ) \
                     else 'learnedDynamicMac'
               pktCtx = getPythonIntoCppShimContext().pktCtx( data, srcPort )
               self.learnAddr( vlanId, srcMacAddr, srcPortName, entryType=entryType,
                               pktCtx=pktCtx )

            if( stpState == 'learning' ):
               t5( 'Dropping packet, input forwarding state learning for ',
                   f'address {srcMacAddr} on port {srcPortName}' )
               dropReasonList.append( 'stplearning' )
      else:
         if vlanId is None:
            t7( 'Source is fabric/cpu' )
            # drop untagged packets
            return BridgeFrameInfo( vlanInfo, data, None )

      # Check for packets that are to be trapped to the CPU.
      trap = False
      copyToCpu = False

      bestMatch = None
      maxTrapPriority = -1
      # Allow for extensions to force trap or copy to cpu actions here.
      for trapPriority, h in trapLookupHandlers:
         ( match, chgTrap, chgCopyToCpu ) = h( self, routed, vlanId, dstMacAddr,
                                               data, srcPortName )
         if match:
            if trapPriority > maxTrapPriority:
               # Use the trap handler override with the highest priority.
               maxTrapPriority = trapPriority
               bestMatch = match
               trap = chgTrap
               copyToCpu = chgCopyToCpu
            elif trapPriority == maxTrapPriority:
               dupMatches = [ bestMatch, match ]
               assert False, "Multiple trap handlers with the same priority " \
                     "not allowed: %s" % dupMatches

      if not bestMatch:
         if routed:
            trap = True
         elif ( isIEEELinkConstrained( dstMacAddr ) or
              dstMacAddr == IEEE_MICROBFD_ADDR ):
            if dstMacAddr == IEEE_PAUSE_ADDR:
               etherType = ( ( data[ 12 ] << 8 )
                             | data[ 13 ] )
               t7( 'dst mac is pause addr %x ' % etherType )
               if etherType == ETH_P_PAUSE or \
                     not self.forwardIeeeReservedMac( dstMacAddr ):
                  t5( 'Dropping frame sent to PAUSE address from address %s '
                      'on port %s' % ( srcMacAddr, srcPortName ) )
                  dropReasonList.append( 'destaddrpause' )
            elif self.forwardIeeeReservedMac( dstMacAddr ):
               trap = False
            elif dstMacAddr == IEEE_BPDU_ADDR:
               if self.topoConfig.trapStpBpdus:
                  t7( 'dst mac is ieee bpdu addr' )
                  trap = True
            elif dstMacAddr == PVST_BPDU_ADDR:
               if self.topoConfig.trapPvstBpdus:
                  t7( 'dst mac is pvst bpdu addr' )
                  trap = True
            else:
               trap = True
         else:
            # Fall through, and use default trap and copyToCpu values
            pass

      finalIntfs = {}
      destIsFabric = dstMacAddr == self.bridgeMac_

      # Handle for packet that needs to be terminated
      if destIsFabric:
         ( data, dstMacAddr ) = \
            getPythonIntoCppShimContext().maybeDecapPacket( data, srcPortName )

      if trap == TRAP_ANYWAY or ( not dropReasonList and trap ):
         t8( 'Trapping frame to CPU' )
         dataToSend = data
         for func in rewritePacketOnTheWayToTheCpuHandlers:
            ( chgData, chgDstMac ) = func( self, routed, vlanId,
                                           dstMacAddr, data, srcPortName )
            if chgData:
               t7( 'data changed by %s' % func.__name__ )
               dataToSend = chgData
               break
         t8( 'Rewrite packet on the way to the CPU handlers complete' )
         srcPort.trapFrame( dataToSend )
         dropReasonList.append( 'trap' )

      if not dropReasonList: # pylint: disable=too-many-nested-blocks
         if copyToCpu:
            t7( 'Copying frame to CPU' )
            srcPort.trapFrame( data )

         outputIntfs = []

         # Look up the location of the destination MAC address.
         if destIsFabric:
            # Make sure a vlan interface was created.
            if evid.inDot1qRange() and not vlanConfig:
               t4( 'Dropping packet on bad vlan %s (no VC for SVI)' % vlanId )
               dropReasonList.append( 'badVlan' )
            else:
               if( not evid.inDot1qRange() or
                   ( vlanConfig and 'Cpu' in vlanConfig.intf ) ):
                  # Great, send the frame to the kernel.
                  outputIntfs = [ 'Cpu', ]
               else:
                  t4( 'No vlan interface created' )
         else:
            destMatch = False
            for h in destLookupHandlers:
               ( destMatch, outputIntfs ) = h( self, vlanId, dstMacAddr, data )
               if destMatch:
                  t8( 'destLookupHandler', h, 'returned outputIntfs', outputIntfs )
                  break
            if not destMatch and vlanId:
               ( destMatch, outputIntfs ) = self.destLookup( vlanId, dstMacAddr )
               t8( 'destLookup returned', destMatch, outputIntfs )
            if not destMatch:
               t8( 'Frame has unknown destination address' )
               if not vlanConfig:
                  t4( 'Dropping packet on bad vlan %s (no VC for learn)' % vlanId )
                  dropReasonList.append( 'badVlan' )
                  outputIntfs = []
               else:
                  # In pvlans, the ports in isolated vlans are not egress allowed in
                  # their own vlans, so exclude those ports
                  outputIntfs = \
                    [i for i in vlanConfig.intf if ((i in self.port) and \
                    ( vlanConfig.intf[ i ].allowedTrafficDirection != 'ingress' )) ]
                  t8( 'outputIntfs is', outputIntfs )

            # We only want to include the Cpu in the floodset for
            # 1) broadcasts, 2) local multicasts, 3) other packets identified by
            # floodsetIncludesCpuHandlers or 4) there is a match for 'Cpu' in
            # outputIntfs[]
            if srcIsFabric:
               # If the srcPort is Cpu, then there is no need to figure out if
               # Cpu needs to be included in the floodset.
               cpuBelongs = False
            elif Ethernet.isIPv6Multicast( dstMacAddr ):
               cpuBelongs = True
               t8( "IPv6 Multicast Frame." )
            elif Ethernet.isBroadcast( dstMacAddr ) or (
               dstMacAddr >= IP_MCAST_BASE and # pylint: disable=chained-comparison
               dstMacAddr <= IP_MCAST_LOCAL_END ) or (
               destMatch and 'Cpu' in outputIntfs):
               cpuBelongs = True
            else:
               cpuBelongs = None
               if vlanId != 0:
                  for h in floodsetIncludesCpuHandlers:
                     cpuBelongs = h( self, vlanId, dstMacAddr, data )
                     if cpuBelongs is not None:
                        break
            # If it's there and doesn't belong, remove it.  If it's
            # not there (e.g., IgmpSnooping's destLookup handler
            # provided a constrained set of output interfaces),
            # add it.
            if 'Cpu' in outputIntfs:
               if not cpuBelongs:
                  outputIntfs.remove( 'Cpu' )
                  t7( 'Removing cpu from vlan', vlanId, 'dst', dstMacAddr )
            elif cpuBelongs:
               if vlanConfig is not None and 'Cpu' in vlanConfig.intf:
                  outputIntfs.append( 'Cpu' )
                  t7( 'Adding cpu for vlan', vlanId, 'dst', dstMacAddr )

         # Filter output ports on spanning tree state and remove the source port
         # if it is there. Also, do not send frame out non-local interfaces.
         for intf in outputIntfs:
            if intf not in self.port:
               continue
            if intf == 'Cpu':
               if intf != srcPortName:
                  finalIntfs.setdefault( intf, None )
            else:
               state = self.lookupStpState( vlanId, intf )
               if( intf in self.ethIntfStatusDir_.intfStatus ):
                  forwardingModel = self.ethIntfStatusDir_.\
                        intfStatus[ intf ].forwardingModel
                  t8( "interface", intf, "has stp state", state,
                      "and forwarding model", forwardingModel )
                  # Vxlan allows tunnel to tunnel forwarding via Vxlan interface when
                  # vtep-to-vtep bridging is enabled
                  if( ( (state == 'forwarding') and
                      ( forwardingModel in ( 'intfForwardingModelBridged',
                                            'intfForwardingModelRouted' ) ) ) and 
                      ( ( intf != srcPortName ) or
                      ( ( intf == srcPortName ) and ( 'Vxlan' in srcPortName ) and
                      self.port[ intf ].checkVtepToVtepBridging() ) ) ):
                     t8( "intf ", intf, "srcPortName ", srcPortName )
                     finalIntfs.setdefault( intf, None )

      if not trap and not srcIsFabric and not dropReasonList:
         for sflowHandler in sflowHandlers:
            sflowHandler( self, data, srcPort, finalIntfs )

      for func in packetReplicationHandlers:
         func( self, finalIntfs, srcPort, dropReasonList, vlanId, dstMacAddr,
               data=data )

      if tracePkt:
         t4( "PacktTracer: Return egress intfs and don't send the frames" )

      if dropReasonList:
         if tracePkt:
            t4( "PacketTracer: No egress intfs, dropReasonList:", dropReasonList )
         return BridgeFrameInfo( vlanInfo, data, None )

      t8( 'Host table or VID table indicates ports ', finalIntfs )
      # List of the interfaces that we actually tried to send the frame out of. For
      # the packet tracer, we want to get to most accurate representation of this
      # list of egress interfaces.
      egressIntfs = []
      for intf in finalIntfs: # pylint: disable=consider-using-dict-items
         sendToMacAddr = dstMacAddr
         dataToSend = data
         sendSrcMacAddr = srcMacAddr
         skipThisIntf = False
         if finalIntfs[ intf ] != None: # pylint: disable=singleton-comparison
            vlanAction = finalIntfs[ intf ]
         else:
            if vlanId != fid:
               if intf == 'Cpu' and evid.inDot1qRange():
                  vlanId = fid
                  vlanTagNeedsUpdating = True
            vlanAction = self.vlanAction( srcPortName, intf, vlanId, intf == 'Cpu',
                                          etherType, vlanTagNeedsUpdating )
         if intf == 'Cpu':
            for func in rewritePacketOnTheWayToTheCpuHandlers:
               ( chgData, chgDstMac ) = func( self, True, vlanId, dstMacAddr, data,
                                              srcPortName )
               if chgData or chgDstMac:
                  sendToMacAddr = chgDstMac
                  dataToSend = chgData
                  break
         else:
            for func in postBridgingHandlers:
               ( action, chgData, chgSrcMac, chgDstMac ) = func( self, srcMacAddr,
                                                                 dstMacAddr, vlanId,
                                                                 data, srcPortName,
                                                                 intf, finalIntfs )
               if action == 'deny':
                  skipThisIntf = True
                  t4( 'postBridgingHandler denied bridging the frame '
                      'on egress interface ', intf )
                  break # from inner for
               if chgData or chgDstMac or chgSrcMac :
                  dataToSend = chgData
                  sendToMacAddr = chgDstMac
                  sendSrcMacAddr = chgSrcMac
                  break

         if skipThisIntf:
            continue
         # If we're tracing the packet, we don't want to actually send the frame
         if tracePkt:
            egressIntfs.append( intf )
         else:
            self.port[ intf ].sendFrame( dataToSend, sendSrcMacAddr, sendToMacAddr,
                                         srcPortName, priority, vlanId, vlanAction )
      return BridgeFrameInfo( vlanInfo, data, egressIntfs )

   def lookupStpState( self, vlanId, portName ):
      '''Lookup the (vlanId, portName) pair in the Topology structures to find
      the forwarding state to use for packets on that port in that vlan.'''

      if not ( 0 < vlanId < 4095 ):
         t8( 'invalid vlanId' )
         return 'discarding'

      topo = self.topoConfig.vidToTopoMap.get( vlanId, None )
      if not topo:
         t8( 'no topo for vlan', vlanId )
         return 'discarding'

      topoPort = topo.topoPortConfig.get( portName, None )
      if not topoPort:
         t8( 'no topoPort for port', portName, 'in', topo.name )
         return 'discarding'

      return topoPort.state

   def learnAddr( self, vlanId, macAddr, portName, entryType='learnedDynamicMac',
                  moves=None, pktCtx=None ):
      t9( 'Maybe learn vlan %d, mac %s, port %s entryType %s moves %s' %
          ( vlanId, macAddr, portName, entryType, moves ) )

      # Do not learn MAC if learning is disabled in this VLAN or on this port
      vlanConfig = self.brConfig_.vlanConfig.get( vlanId )
      try:
         switchIntfConfig = self.brConfig_.switchIntfConfig.get( portName )
      except IndexError:
         switchIntfConfig = None
      if ( vlanConfig and not vlanConfig.macLearning ) or ( switchIntfConfig and
           not switchIntfConfig.macLearningEnabled ):
         return

      learnedMacs = [ 'learnedDynamicMac' , 'learnedRemoteMac' ]
      configuredRemoteMacs = [ 'configuredRemoteMac' , 'evpnConfiguredRemoteMac' ]
      r = invokeLearningHandlers( vlanId, macAddr, portName, entryType,
                                  pktCtx=pktCtx )
      if( 'dontLearn' in r ):
         return

      # Create these on demand rather than via reactors.
      fid = self.brConfig_.vidToFidMap.get( vlanId )
      if not fid:
         fid = vlanId

      if fid in self.brConfig_.fdbConfig:
         fdbConfig = self.brConfig_.fdbConfig[ fid ]
         staticTable = fdbConfig.configuredHost
         staticHost = macAddr in staticTable
         if staticHost:
            if entryType == 'configuredStaticMac':
               t9( "Adding static entry" )
               staticHostEntry = staticTable[ macAddr ]
               if moves is None:
                  moves = 1
               host = SmashFdbStatus( HostKey( fid, macAddr ) )
               host.intf = staticHostEntry.intf
               host.moves = moves
               host.lastMoveTime = Tac.now()
               host.entryType = "configuredStaticMac"
               self.smashBrStatus_.smashFdbStatus.addMember( host )
               t9( "Entry added in %s" % fid )
            else:
               t9( "Static entry exists, ignoring learn event" )
            return

      if not Ethernet.isUnicast( macAddr ):
         # We don't learn multicast addresses
         t4( 'not learning multicast host vlan %d mac %s' % (vlanId, macAddr) )
         return
      if macAddr == '00:00:00:00:00:00':
         # We don't learn the zero address either
         t4( 'not learning zero host vlan %d mac %s' % (vlanId, macAddr) )
         return

      if( entryType is None ):
         entryType = "learnedDynamicMac"
      key = HostKey( fid, macAddr )
      if key in self.smashBrStatus_.smashFdbStatus:
         host = self.smashBrStatus_.smashFdbStatus[ key ]
         if host.intf == portName and host.entryType.startswith( "peer" ):
            t4( 'Keeping existing learnedHost with type', host.entryType )
            return
         if host.entryType == 'configuredRemoteMac' or \
            host.entryType == 'configuredStaticMac':
            t4( 'Keeping existing learnedHost with type', host.entryType )
            return
         if entryType in learnedMacs and \
            ( host.entryType == 'evpnIntfStaticMac' or \
              host.entryType == 'evpnConfiguredRemoteMac' ):
            t4( 'Keeping existing learnedHost with type', host.entryType )
            return
         # configured MACs have high priority
         if ( ( entryType not in configuredRemoteMacs ) and ( host.intf == portName )
              and ( host.entryType == 'receivedRemoteMac' or
                host.entryType == 'evpnDynamicRemoteMac' ) ):
            t4( 'Keeping existing learnedHost with type', host.entryType )
            return
         if host.intf == portName and entryType in learnedMacs and \
            host.entryType in learnedMacs:
            self.agingTable[ (fid, macAddr) ] = ( Tac.now(), portName )
            return
         t4( 'host entryType %s/%s move to vlanId %d, mac %s, port %s' %
             ( host.entryType, entryType, vlanId, macAddr, portName ) )
         newHost = SmashFdbStatus( key )
         newHost.intf = portName
         if moves is None:
            newHost.moves = ( host.moves + 1 ) & 0xffffffff
         else:
            newHost.moves = moves
         newHost.lastMoveTime = Tac.now()
         newHost.entryType = entryType
         self.smashBrStatus_.smashFdbStatus.addMember( newHost )
      else:
         t4( 'host learn vlanId %d, mac %s, port %s' %
             ( vlanId, macAddr, portName ) )
         if moves is None:
            moves = 1
         host = SmashFdbStatus( HostKey( fid, macAddr ) )
         host.intf = portName
         host.moves = moves
         host.lastMoveTime = Tac.now()
         host.entryType = entryType
         self.smashBrStatus_.smashFdbStatus.addMember( host )

      # Only age learned Mac entries
      if entryType in learnedMacs:
         self.agingTable[ (fid, macAddr) ] = ( Tac.now(), portName )

   def destLookup( self, vlanId, macAddr ):
      evid = Tac.Value( 'Bridging::ExtendedVlanId', vlanId )
      if not evid.inDot1qRange():
         return ( True, [] )

      if macAddr == '00:00:00:00:00:00':
         # Don't forward packets to address zero.
         return (True, [])
      vlanConfig = self.brConfig_.vlanConfig.get( vlanId, None )
      if not vlanConfig:
         return (False, [])

      fid = self.brConfig_.vidToFidMap.get( vlanId )
      if not fid:
         fid = vlanId

      key = HostKey( fid, macAddr )
      if key in self.smashBrStatus_.smashFdbStatus:
         host = self.smashBrStatus_.smashFdbStatus[ key ]
         t4( 'destLookup found dynamic host vlan %d mac %s' % (vlanId, macAddr) )
         intfs = [ host.intf ]
         return (True, intfs)

      return (False, [])

   def hostAging( self ):
      now = Tac.now()
      self.hostAgingActivity_.timeMin = now + self.agingInterval_

      if self.brConfig_.hostAgingTime == 0:
         # Aging time set to 0 means to disable aging.
         return

      c = 0
      for k in list( self.agingTable ):
         ( vlanId, macAddr ) = k
         ( lastSeenTime, _ ) = self.agingTable[ k ]
         if( (now - self.brConfig_.hostAgingTime) > lastSeenTime ):
            t4( 'now', now, 'hostAgingTime', self.brConfig_.hostAgingTime,
                'lastSeenTime', lastSeenTime )
            c += 1
            self.deleteMacAddressEntry( vlanId, macAddr )

      if c > 0:
         t4( 'Aged out ', c, '  MAC address table entries' )

   def macTableFlush( self ):
      shim = getPythonIntoCppShimContext()
      shim.arfaRoot_.arfaPlugins.plugin[
         'CoreBridge' ].flushRequestInternal.hostTableFlushRequest[ "flush" ] = (
            Tac.Type( 'Bridging::HostTableFlushRequest' )( vlanId=0, intf="" ) )

   def macTableFlushPort( self, intfName ):
      if intfName is None:
         # mimic python behavior: calling macTableFlushPort( None ) is
         # a no-op.
         return
      shim = getPythonIntoCppShimContext()
      shim.arfaRoot_.arfaPlugins.plugin[
         'CoreBridge' ].flushRequestInternal.hostTableFlushRequest[
            intfName ] = Tac.Type( 'Bridging::HostTableFlushRequest' )(
               vlanId=0, intf=intfName )

   def deleteMacAddressEntry( self, vlanId, macAddr, entryType=None ):
      t9( 'Deleting MAC entry at vlan ', vlanId, ' mac ', macAddr )
      fid = self.brConfig_.vidToFidMap.get( vlanId )
      if not fid:
         fid = vlanId

      key = HostKey( fid, macAddr )
      if key in self.smashBrStatus_.smashFdbStatus:
         # It is possible that the address was already deleted by a flushing
         # operation.
         hostEntry = self.smashBrStatus_.smashFdbStatus[ key ]
         if hostEntry.entryType == 'configuredStaticMac' and \
            entryType != 'configuredStaticMac':
            t4( "Do not Flush static entry" )
         else:
            del self.smashBrStatus_.smashFdbStatus[ key ]
      if ( fid, macAddr ) in self.agingTable:
         del self.agingTable[ (fid, macAddr) ]
      invokePostAgingHandlers( self, vlanId, macAddr )

