# Copyright (c) 2006-2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

""" This module provides helper functions for dealing with entity mib
objects ( rooted at Tac.root[ sysname ][ 'Sysdb' ][ 'hardware' ][ 'entmib' ] )"""

import Tac
import Tracing
import re

traceHandle = Tracing.Handle( 'Fru' )
t0 = traceHandle.trace0
t1 = traceHandle.trace1

ChipType = Tac.Type( 'EntityMib::Chip::ChipType' )

class IndexAllocator:

   @staticmethod
   def getCardSlotNumber( mibEntity ):
      helper = Tac.newInstance( "EntityMib::Helper" )
      return helper.getCardSlotNumber( mibEntity )
   
   @staticmethod
   def getPath( mibEntity ):
      helper = Tac.newInstance( "EntityMib::Helper" )
      return helper.getPath( mibEntity )

   @staticmethod
   def getChipId( chipType, chipIndex ):
      helper = Tac.newInstance( "EntityMib::Helper" )
      return helper.getChipId( chipType, chipIndex )

   @staticmethod
   # pylint: disable-next=redefined-builtin
   def physicalIndex( container, type, index=None, tag=None ):
      type = type.lower()
      if type in [ 'chassis', 'fixedsystem' ]:
         return 1
      helper = Tac.newInstance( "EntityMib::Helper" )
      if index is None:
         index = 0
      if tag is None:
         tag = ''
      return helper.physicalIndex( container, type, int( index ), tag )
   
   @staticmethod
   def collectionItemPhysicalIndex( collection, index, tag=None ):
      mibEntity = collection.parent
      type = collection.tacAttr.name # pylint: disable=redefined-builtin
      return IndexAllocator.physicalIndex( mibEntity, type, index, tag )

#This cannot be ported to tacc as this is being called from different agents
# with different inventory Component type. e.g. it is being called from ThermoMgr
# as Inventory::ComponentBase whereas from Sb820 as Inventory::Sb820::Sb820TempSensor
def populateMib( physicalEntityMib, invComponent ):
   """ Set the mib attributes for the standard attributes that exist
   in all entities of type Inventory::ComponentBase """

   componentAttrs = [ 'description', 'modelName', 'vendorType', 'hardwareRev',
                      'firmwareRev', 'serialNum', 'mfgName', 'mfgDate' ]
   for attr in componentAttrs:
      value = getattr( invComponent, attr )
      if value != "":
         setattr( physicalEntityMib, attr, value )

   if isinstance( invComponent, Tac.Type( "Inventory::FruBase" ) ):
      # Copy the appropriate Fru attributes to the EntityMib object as well
      for d in invComponent.deviation:
         physicalEntityMib.deviation[ d ] = True

def findParentContainer( parent ):
   helper = Tac.newInstance( "EntityMib::Helper" )
   return helper.findParentContainer( parent )

def _componentNameFromContainer( container, componentTag, componentLabel ):
   helper = Tac.newInstance( "EntityMib::Helper" )
   return helper.componentNameFromContainer( 
         container, componentTag, componentLabel )

def componentName( mib ):
   helper = Tac.newInstance( "EntityMib::Helper" )
   return helper.componentName( mib )

def componentNameFromParent( parentMib, componentTag, componentLabel ):
   helper = Tac.newInstance( "EntityMib::Helper" )
   return helper.componentNameFromParent( parentMib, componentTag, 
         str( componentLabel ) )

def formatEntityMibName( name, detail=False ):
   match = re.match( r"(\D+)(.*)", name )
   if match:
      if detail:
         return match.group( 2 )
      else:
         if match.group( 1 ).startswith( "PowerSupply" ):
            return match.group( 2 ).split( "/" )[ 0 ]
         else:
            return match.group( 2 )
   else:
      return name

class MaxChipIdError( Exception ):
   """Will be thrown if the Entity Mib chipId space is exhausted for a chip.
      See Power/FruPlugin/PowerController.py for an example."""
   pass # pylint: disable-msg=W0107
