#!/usr/bin/env python3
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import os, re, shutil, sys, time, traceback
from subprocess import Popen, PIPE
import SecretCli
import Tac
from Fru.FruBaseVeosDriver import parseVeosConfig
from MultiRangeRule import setFromCanonicalString
import VeosHypervisor
from ArPyUtils import arch
import SimpleConfigFile
from distutils.util import strtobool
import Toggles.EosInitToggleLib
import Toggles.FruToggleLib
from Toggles.EosInitToggleLib import toggleSfeVrfScaleEnabled
from Toggles.EosInitToggleLib import toggleSfeVrfScalev2Enabled

metadataIp = '169.254.169.254'

fabricDevName = 'fabric'
fabricDevMac = '30:00:00:00:12:34'
# Note: no-ETH_P_ARISTA alias is set on the dummy fabric device created below
# so that the findEthProtocol() at /src/Arnet/EthDevPamUtils.h does not treat
# the fabric interface as real and result in overriding the sll_protocol of the
# socket created and bound by EthDevPamBundle::doGetSd() to ETH_P_ARISTA
fabricDevAlias = 'no-ETH_P_ARISTA'
# MTU on the fabric device should be greater than the max allowed MTU on any
# VLAN. Since dynamic VLANs are created with MTU of ~9K, setting fabric device
# MTU to 10000. This is the value used in other platforms too.
fabricDevMtu = '10000'

# EOS paths
userData = "/mnt/flash/.userdata"
kickstartConfig = "/mnt/flash/kickstart-config"
defaultStartupConfig = "/mnt/flash/.default_startup-config"
startupConfig = "/mnt/flash/startup-config"
veosConfigFileName = "veos-config"
veosConfigFile = "/mnt/flash/%s" % veosConfigFileName
veosInternalConfigFileName = ".veos-config-internal"
veosInternalConfigFile = "/mnt/flash/%s" % veosInternalConfigFileName
runtimeVeosConfigFileName = ".veos-config"
runtimeVeosConfigFile = "/var/run/%s" % runtimeVeosConfigFileName
kernelParamsFileName ="kernel-params"
kernelParamsFile= "/mnt/flash/%s" % kernelParamsFileName
# This will help quick debug/test kernel-params changes manually
# otherwise there is not way to boot kernel using custom params
# related to the hugepages/cpu/isolcpu etc which we dynamically
# modify during early boot.
skipSfeKernelSetup = "/mnt/flash/skipSfeKernelSetup"
sysCpuPath = "/sys/devices/system/cpu/"
onlinePath = "online"
threadSiblingsPath = "/topology/thread_siblings_list"
isolcpusKernelParam = "isolcpus="
prefdlFile = "/etc/prefdl"
dhclientPidFile = '/var/lib/dhclient/dhclient-eos-cloud-init.pid'

eosCloudKernelParamHeader = "# CloudEos kernel parameters"
userKernelParamHeader = "# Put user-specified kernel parameters to " \
   "add/overwrite the default kernel parameters below"

# This dumps onto mnt/flash/.CloudInit for persistence
# use carefully to avoid too much info overfiling /mnt/flash
def debug( line ):
   print( time.asctime(), "\tdebug:", line )  
   sys.stdout.flush() 

# Toggle for existing Independence to behave in firewall/monitor mode
def isRuby2ToggleSet():
   return Toggles.FruToggleLib.toggleRubyPlatformEnabled()

def isRuby2Sid():
   indRuby2SidPattern = 'SID: IndependenceRuby2'
   try:
      with open( prefdlFile, 'r' ) as pfile:
         productInfo = pfile.read()
   except ( FileNotFoundError, OSError ) as e:
      debug( "Pre-FDL Read : Unable to read the pre-FDL file to determine SKU" )
      debug( e )
      return False
   return re.search( indRuby2SidPattern, productInfo ) is not None

# SfeVrfScalev2 toggle is introduced as part of further enhancing vrf scale.
# The memory requirements have not changed, the same amount of memory is required
# as when SfeVrfScale toggle is on.
# Need to make sure right memory is allocated  with either toggle on.
def isVrfScaleToggleSet():
   return toggleSfeVrfScaleEnabled() or toggleSfeVrfScalev2Enabled()
#
# If you want cloudinit to halt the boot process with some error code
# make sure to derive from this class with a proper retValue which 
# needs to be synced up with EosCloudInit service which runs at the early boot
# time. This is under /src/Eos/rc/EosCloudInit for now.
# Don't change the existing retValues.
#
class CloudInitException( Exception ):
   # Fatal error codes for dumping on the system screen when the system is
   # totally messed up.
   # Keep in sync with the EosCloudInit service as mentioned above.
   unknownBootError = 255
   badCoreCountError = 1
   tooManyInterfacesError = 2
   unknownCloudPlatformError = 3

   def __init__( self, args ):
      super().__init__( self, args )
      self.errCode = self.unknownBootError

class BadCoreCountException( CloudInitException ):
   def __init__( self ):
      super().__init__(
            "Total cores needs to be at least 2" )
      self.errCode = self.badCoreCountError
      
class TooManyInterfacesException( CloudInitException ):
   def __init__( self ):
      super().__init__(
            "No enough memory to support >8 interfaces" )
      self.errCode = self.tooManyInterfacesError

class UnknownCloudPlatformException( CloudInitException ):
   def __init__( self ):
      super().__init__(
            "Unsupported CloudEOS platfom: %s, contact Arista support" %
            VeosHypervisor.getSysVendorInfo() )
      self.errCode = self.unknownCloudPlatformError

class BessConfigurator:
   # This is used for < 4GB systems.
   # this is actually allocated to bess.
   defaultBessMemInMbForLowMemSystem = 1536 
   # On 32-bit system with DPDK 19.11, we need extra
   # margin as bess needs a bit more memory at least transiently
   # Allocate extra 64 2MB hugepage
   defaultBessMemInMbForLowMemSystemMargin = 64 * 2 
   # This is used for systems with mem > 4GB and <= 8GB
   defaultBessMemInMbForMediumMemSystem = 2560
   # This is used for systems with mem > 8GB and <= 16GB
   defaultBessMemInMbForHighMemSystem = 4608
   # This is used for systems with mem > 16GB
   defaultBessMemInMbForLargeMemSystem = 6656
   defaultBessMemInMbForVrrMode = 1536
   # These are the minimal packets configured in the bess packet pool.
   defaultBessBuffers = 65536
   # For upto 16 interfaces or > 6GB of memory to accommodate tunnel scale
   bessBuffersHighInterfaces = 65536 * 2 
   # tested upto 64.
   maxSupportedVrfScale = 64
   # Caravan platform has total 32GB memory and can support 
   # 8 NAC ports + 2 Fail-to-Wire + upto 2 NIMS with total 8 ports
   defaultBessMemInMbForCaravanSystem = 4096 * 2
   # We need around 16G to support 64 VRFs. The default value of Independence
   # can already support 64 VRFs.
   bessMemInMbForCaravanSystemVrfScale = 8192 * 2
   defaultBessMemInMbForCaravanIndependenceSystem = 20 * 1024 + 512 * 2
   if Toggles.EosInitToggleLib.toggleSfeRssCaravanEnabled():
      defaultBessMemInMbForCaravanIndependenceSystem = 23 * 1024 + 512 * 2
   # For Ruby FCT would be 32M so set BESS memory accordingly
   defaultBessMemInMbForRubyIndependenceSystem = 47 * 1024 + 512 * 2
   if Toggles.EosInitToggleLib.toggleSfeRssCaravanEnabled():
      defaultBessMemInMbForRubyIndependenceSystem = 50 * 1024 + 512 * 2
   defaultBessMemInMbForRuby2SkuSystem = 95 * 1024 + 512 * 2
   bessBuffersCaravanInterfaces = 65536 * 4
   bessBuffersCaravanInterfacesIndependence = 65536 * 8
   # This is used for systems with mem >= 16GB and Wan Delay Emulation
   defaultBessMemInMbForWanEmulation = 8192
   bessBuffersForWanEmulation = 524288
   defaultBessMemForWillametteCaravan = 7*1024

   def __init__( self, kernelPFile=None, veosConfFile=None, 
            userConfigFile=None, 
            runtimeVeosConfFile=runtimeVeosConfigFile ):

      self.flashDir = os.path.join( os.environ.get( "FILESYSTEM_ROOT",
                              "/mnt" ), "flash" ) 
      # Default allocation here applies to 32-bit arch or <= 4GB of mem
      # or vrrMode( ie Virtual Route Reflector Mode )
      self.bessMemoryInMb = self.defaultBessMemInMbForLowMemSystem
      self.bessBuffers = self.defaultBessBuffers
      self.platformRuby = False
      self.kernelParamsFile = kernelPFile if kernelPFile \
               else os.path.join( self.flashDir, kernelParamsFileName ) 
      self.veosConfigFile = veosConfFile if veosConfFile \
               else os.path.join( self.flashDir, veosConfigFileName ) 
      self.userConfigFile = userConfigFile if userConfigFile \
               else os.path.join( self.flashDir, veosInternalConfigFileName ) 
      # Contains final merged configs from CLI, veosConfig etc
      self.runtimeVeosConfigFile = runtimeVeosConfFile
      self.mergedConfig = None
      self.interfaces = {}
      self.kernelStr = None
      self.vrrModeEnabled = False
      self.hyperThreadingEnabled, self.hyperThreadingConfigured = \
            self.getHyperThreadingConf()

      # Handle default core allocation.
      self.totalCores = getSysconf( "SC_NPROCESSORS_ONLN" )
      if self.totalCores < 2:
         raise BadCoreCountException()

      self.totalMem = getSystemMemoryGB()
      self.memComputed = False
      config = self.getMergedUserConfig()
      maxDpCores = int ( config.get( 'maxDataPathCores', 0 ) ) 
      if maxDpCores and maxDpCores <= 2 :
         # If the user has asked us to give <= 2 cores for the data plane
         # We implicitly assume that they want to use Virtual Route Reflector
         # mode.  We can use this mode potentially to change how hugepages are
         # distributed between 1G and 2MB pages.  FOr now this has no effect on
         # memory allocation.
         self.vrrModeEnabled = True

   def getHyperThreadingConf( self ):
      '''Helper function to read the HYPER_THREADING veos-config and return whether
      hyperThreading is enabled/disabled.'''
      config = self.getMergedUserConfig()
      if 'HYPER_THREADING' in config:
         hyperThreadingStr = config.get( 'HYPER_THREADING' )
         return bool( strtobool( hyperThreadingStr.strip() ) ), True
      else:
         # hyperthreading is enabled by default, and wasn't explicitly configured
         return True, False

   def getCpuThreadSiblings( self, cpu ):
      '''Helper function to get the threadSiblings of the "cpu" passed as
      and argument'''
      siblingCpus = []
      threadSiblings = ""
      try:
         siblingStr = os.environ.get( 'TEST_CPU_THREAD_SIBLINGS', None )
         if siblingStr:
            siblings = siblingStr.split()
            threadSiblings = siblings[ cpu ]
         else:
            with open( sysCpuPath + "cpu" + str( cpu ) + threadSiblingsPath ) as f:
               threadSiblings = f.readline().strip()
         siblingCpus = setFromCanonicalString( threadSiblings )
      except Exception as error: # pylint: disable=broad-except
         print( "Unable to get the thread siblings list for cpu:", cpu,
            " error: ", error )
      return sorted( siblingCpus )

   def getOnlineCpus( self ):
      '''Helper function to get a list of cpus which are online'''
      onlineCpuList = []
      onlineCpus = ""
      try:
         onlineCpus = os.environ.get( 'TEST_ONLINE_CORES', None )
         if not onlineCpus:
            with open( sysCpuPath + onlinePath ) as f:
               onlineCpus = f.readline().strip()
         onlineCpuList = setFromCanonicalString( onlineCpus )
      except Exception as error: # pylint: disable=broad-except
         print( "Unable to get the online cpus list, error: ", error )
      return sorted( onlineCpuList )

   def getIsolCpusStr( self ):
      '''Helper function to get the isolcpus kernel command line parameter which
      represents the data plane cpu threads to be isolated.'''
      config = self.getMergedUserConfig()
      maxDpCores = int ( config.get( 'maxDataPathCores', 0 ) )

      #  **** Handle data path Core allocation 
      # We try to reserve upto max DP cores for the data plane.
      # if the total cores are > max_dp_cores, then we limit DP cores
      # to max_dp_cores.
      # Else if the total cores are lower/same as the max, we will try
      # to satisfy as many cores as possible leaving at least 1 for 
      # control plane. 
      # Keep in mind that this is little different behavior than
      # the default allocation in absence of user input.
      # The default behavior is to:
      #     -leave 2 for the control plane
      #      and rest for the data plane provided the total cores are more
      #      than 4,
      #     -else assign 1 for control plane and rest to DP. 
      #
      # Avoid splitting cpus across control plane and data plane, so that control
      # plane and data plane do not share the same physical cpu.
      # Thus, 4vCPUs should have 2 control plane vCPUs and 2vCPUs isn't
      # recommended and will split between 1 control plane and 1 dataplane sharing
      # a physical cpu.
      numCpThreadsFound = 0
      cpThreads = set()
      onlineCores = self.getOnlineCpus()
      actualTotalCores = len( onlineCores )
      numCpThreadsReqd = 1
      if actualTotalCores > 8:
         numCpThreadsReqd = 4
      elif actualTotalCores > 4:
         numCpThreadsReqd = 2
      for i in onlineCores:
         if i in cpThreads: # skip control plane thread already encountered.
            continue
         siblingCpus = self.getCpuThreadSiblings( i )
         cpThreads.update( siblingCpus )
         numCpThreadsFound = len( cpThreads )
         if numCpThreadsFound >= numCpThreadsReqd:
            break
         # Note: numCpThreadsFound can be less than numCpThreadsReqd
         # and even 0, which is an abnormal case and we might end up
         # considering every core/thread as a dataplane(dp) thread in
         # that case.
      dpThreads = set( onlineCores ) - cpThreads
      # Note: Ideally we might want to interpret maxDpCores as whole cpu
      # core and not consider it to be maxDpThreads. But, for now treat it
      # as maxDpThreads unless we need it to be interpreted the other way.
      if maxDpCores:
         dpThreads = list( dpThreads )[ -maxDpCores : ]
      dpThreads = sorted( dpThreads )

      isolCpusStr = ""
      rangeStart = -1
      rangeEnd = -1
      i = 1
      while i != len( dpThreads ) + 1:
         if i != len( dpThreads ) and dpThreads[ i ] == dpThreads[ i - 1 ] + 1:
            if rangeStart == -1:
               rangeStart = dpThreads[ i - 1 ]
         else:
            if rangeStart != -1 and rangeEnd == -1:
               rangeEnd = dpThreads[ i - 1 ]
               isolCpusStr += "%u-%u" % ( rangeStart, rangeEnd )
               rangeStart = rangeEnd = -1
            else:
               isolCpusStr += "%u" % dpThreads[ i - 1 ]
            if i != len( dpThreads ):
               isolCpusStr += ","
         i = i + 1

      debug( "getIsolCpusStr: isolCpusStr = " + isolCpusStr )
      return isolCpusStr

   def dumpConfig( self ):
      userFileContent = None 
      veosConfigContent = None
      try:
         userFileContent = readFile( self.userConfigFile )
         veosConfigContent = readFile( self.veosConfigFile )
      # pylint: disable=bare-except
      except:
         pass
      debug(
"""
**** Dumping BessConfigurator info *** 
total Cores=%d,
BessMemory=%d, BessBuffers=%d
mergedFinalConfig=%s
kernelStr=%s
parsedInterfaces=%s
veosConfig=%s
userVeosConfig=%s
flashDir=%s
veosConfigcontent=%s
userConfigContent=%s
""" % (self.totalCores, self.bessMemoryInMb,
   self.bessBuffers, self.mergedConfig, self.kernelStr,
   self.interfaces, self.veosConfigFile, self.userConfigFile,
   self.flashDir, veosConfigContent, userFileContent
   ) )

   def writeRunTimeVeosConfig( self ):
      cfg = SimpleConfigFile.SimpleConfigFileDict( \
               filename=self.runtimeVeosConfigFile, 
               createIfMissing=True, autoSync=True ) 
      for k, v in self.mergedConfig.items():
         cfg[ k ] = v

   def getMergedUserConfig( self ) :
      if self.mergedConfig:
         return self.mergedConfig
      # Pick from veos-config and .veos-config-internal as configured by the user.
      veosConfig = SimpleConfigFile.SimpleConfigFileDict( \
                              self.veosConfigFile )
      # need this to decouple from the underlying file
      self.mergedConfig = {}
      for k, v in veosConfig.items():
         self.mergedConfig[ k ] = v
      userConfig = SimpleConfigFile.SimpleConfigFileDict( 
            self.userConfigFile )
      for k, v in userConfig.items():
         self.mergedConfig[ k ] = v
      # When the SfeVrfScale/SfeVrfScalev2 toggle is enabled, vrfScaleEnabled flag
      # is not reqd.
      # Ignore it, if present.
      if isVrfScaleToggleSet():
         if 'vrfScaleEnabled' in self.mergedConfig:
            del self.mergedConfig[ 'vrfScaleEnabled' ]
      return self.mergedConfig 

   def getInterfaces( self ):
      if os.environ.get( 'TEST_INTERFACES', None ):
         return os.environ[ 'TEST_INTERFACES' ].split()

      if self.interfaces:
         return self.interfaces 
      # Start picking up  /sys/class/net should have all intfs , we just need
      # to get rid of ma1 and lo as others will always be physical intf
      # so early in the boot time. TBD.
      self.interfaces = Tac.run( ['/bin/ls', '/sys/class/net' ], \
            stdout=Tac.CAPTURE ).split()
      for i in [ 'lo' , 'ma1' ] :
         for k in self.interfaces:
            if i == k:
               del self.interfaces[ self.interfaces.index( k ) ]
      return self.interfaces

   def bessMemForCaravanSystem( self, vrfScaleEnabled ):
      # Check whether its Ruby platform and allocate BESS accordingly
      # Use defaultBessMemInMbForRubyIndependenceSystem

      # We need around 16G to support 64 VRFs. The default value on Independence
      # can support 64 VRFs (DPDK LPM).
      if VeosHypervisor.platformIndependence():
         if isRuby2Sid():
            return self.defaultBessMemInMbForRuby2SkuSystem
         elif isRuby2ToggleSet():
            return self.defaultBessMemInMbForRubyIndependenceSystem
         else:
            return self.defaultBessMemInMbForCaravanIndependenceSystem
      elif VeosHypervisor.platformWillamette():
         return self.defaultBessMemForWillametteCaravan
      if vrfScaleEnabled or ( isVrfScaleToggleSet() and self.totalMem >= 19 ):
         return self.bessMemInMbForCaravanSystemVrfScale
      else:
         return self.defaultBessMemInMbForCaravanSystem

   def computeBessBuffersForCaravan( self ):
      if Toggles.EosInitToggleLib.toggleSfeRssCaravanEnabled():
         if VeosHypervisor.platformIndependence():
            return self.bessBuffersCaravanInterfacesIndependence

      return self.bessBuffersCaravanInterfaces

   def computeBessMemory( self ):
      if self.memComputed:
         return
      config = self.getMergedUserConfig()
      # We overwrite this later if mem etc allow it
      vrfScaleStr = config.get( 'vrfScaleEnabled', 'False' )
      vrfScaleEnabled = strtobool( vrfScaleStr.strip() )
      if not isVrfScaleToggleSet():
         # we will enable later if memory etc are sufficient
         config[ 'vrfScaleEnabled' ] = False
      intfCnt = len( self.getInterfaces() )

      
      # Handle memory/buffers
      # default cases are already handled in init.
      if arch() == 32:
         if intfCnt > 8 and self.totalMem <= 4 :
            # We are not planing to support this case as we need extra
            # memory ~1/2 Gb for intf buffers itself, last time I checked.
            # The default max hugepage memory which can mmaped for Sfe/bess
            # seems limited to ~1.5GB for 32-bit image. Lets bail out for low
            # memory systems with higher intf count. This will cause system to
            # halt during the boot.
            raise TooManyInterfacesException()

      if arch() == 64:
         # Increase packet buffers if there is at least 6GB of memory
         if self.totalMem >= 6:
            self.bessBuffers = self.bessBuffersHighInterfaces

         if self.totalMem > 4 and self.totalMem <= 8:
            self.bessMemoryInMb = self.defaultBessMemInMbForMediumMemSystem
            if self.totalMem >= 6:
               # Add another 1GB as ~500K+ is taken by the extra packet buffers
               self.bessMemoryInMb += 1024
         elif self.totalMem > 8 and self.totalMem <= 16:
            self.bessMemoryInMb = self.defaultBessMemInMbForHighMemSystem
         elif self.totalMem > 16:
            self.bessMemoryInMb = self.defaultBessMemInMbForLargeMemSystem

      # If we have vrfScale enabled then we just grab more. 
      # Not sure why would anyone enable vrrMode too, we just
      # care about vrfScale only and let user worry about 
      # the vrrMode/high mem consumption as the worst case we 
      # will waste more memory. 
      #
      vrfScale = isVrfScaleToggleSet() or vrfScaleEnabled
      if arch() == 64 and vrfScale and self.totalMem >= 19:
         # This is tested to work with upto 64 VRFs.
         # NOTE: The above number is for DPDK LPM and only with IPv4.
         # If IPv6 is also enabled, the total number of VRFs that can be configured
         # is reduced - by half in case of DPDK LPM.
         # With the newer LPM implementation, we have better VRF scale (for IPv4)
         # and that is # set in Sfe (SfeL3AgentSm).
         # We don't use the vrfScaleEnabled or # defaultMaxVrfs attributes in the
         # newer LPM implementation (i.e. with the SfeVrfScale/SfeVrfScalev2 toggle
         # enabled ).
         self.bessMemoryInMb = 14 * 1024
         self.bessBuffers = self.bessBuffersHighInterfaces
         if not isVrfScaleToggleSet():
            # This is not requried when the SfeVrfScale toggle is enabled
            self.mergedConfig[ 'vrfScaleEnabled' ] = True
            self.mergedConfig[ 'defaultfMaxVrfs' ] = self.maxSupportedVrfScale
      if VeosHypervisor.platformCaravan(): 
         self.bessMemoryInMb = self.bessMemForCaravanSystem(
                                    self.mergedConfig.get( 'vrfScaleEnabled' ) )
         self.bessBuffers = self.computeBessBuffersForCaravan()
         self.platformRuby = isRuby2Sid() or isRuby2ToggleSet() 

      if arch() == 64 and isWanEmulationEnabled( modeFile=self.veosConfigFile ):
         # At 5 million pps, 50 msec delay constitutes 250K buffers
         if self.totalMem >= 8 and self.totalMem < 12:
            self.bessBuffers = 128 * 1024 # Upto 25 msec delay
         elif self.totalMem >= 12 and self.totalMem < 16:
            self.bessMemoryInMb = 6 * 1024
            self.bessBuffers = 256 * 1024 # Upto 50 msec delay
         elif self.totalMem >= 16:
            self.bessMemoryInMb = self.defaultBessMemInMbForWanEmulation
            self.bessBuffers = self.bessBuffersForWanEmulation # upto 100 msec delay

      self.mergedConfig[ 'bessMemoryInMb' ] = self.bessMemoryInMb
      self.mergedConfig[ 'bessBuffers' ] = self.bessBuffers
      self.mergedConfig[ 'vrrModeEnabled' ] = self.vrrModeEnabled
      self.mergedConfig[ 'platformRuby' ] = self.platformRuby
      self.memComputed = True

   def getKernelCmdLine( self ):
      if self.kernelStr:
         return self.kernelStr

      if self.memComputed:
         self.computeBessMemory()

      if self.bessMemoryInMb == self.defaultBessMemInMbForLowMemSystem or \
      VeosHypervisor.getPlatform() == 'Caravan':
         if VeosHypervisor.getPlatform() == 'Caravan':
            mb = self.bessMemoryInMb
         else:
            # Seems like we need extra allocation margin for 32-bit systems
            # for DPDK 19.11. Bump up the kernel allocation else it fails.
            # Although the extra memory is freed afterwards
            mb = self.bessMemoryInMb + self.defaultBessMemInMbForLowMemSystemMargin
         if VeosHypervisor.getProductName() == 'Independence':
            kernelStr = "default_hugepagesz=1G hugepagesz=1G hugepages=%d " \
                        "hugepagesz=2M hugepages=%d" % \
                        ( ( mb - 2 * 512 ) // 1024, ( mb - (mb - 2 * 512) ) // 2 )
            # On Independence, the default is to disable hyperthreading unless
            # explicitly configured.
            if not self.hyperThreadingConfigured:
               self.hyperThreadingEnabled = False
         elif VeosHypervisor.getProductName() == 'Willamette':
            kernelStr = "default_hugepagesz=1G hugepagesz=1G hugepages=%d " \
                        % ( mb // 1024 )
            kernelStr += " ixgbe.allow_unsupported_sfp=1 intel_iommu=on iommu=pt"
         else:
            kernelStr = "default_hugepagesz=2M hugepagesz=2M hugepages=%d" % \
                  ( mb // 2 )
         if VeosHypervisor.getPlatform() == 'Caravan' and \
               VeosHypervisor.getProductName() != 'Willamette':
            kernelStr += " modprobe.blacklist=i40evf"
      else:
         gbPages = self.bessMemoryInMb // 1024
         # BUG651720
         # with few features that use flowcache being added, memory usage in
         # Sfe/bessd has gone up. On a 64 bit system, we allocate 512k entries of
         # flowcache. Features using flowcache pre-allocate ( rte_mempool ) that many
         # objects. Each object is 192 bytes. And features such as DPS/Avt allocate
         # twice that because they use one object per direction.
         # 
         # Allocate additional 512M hugepage shared between kernel + Sfe.
         # these are 2M pages
         mbPages = ( self.bessMemoryInMb % 1024 ) // 2 + 256
         if gbPages:
            kernelStr = "default_hugepagesz=1G hugepagesz=1G hugepages=%d" % gbPages
            if mbPages:
               kernelStr += " hugepagesz=2M hugepages=%d" % mbPages 
         else:
            kernelStr = " default_hugepagesz=2M hugepagesz=2M hugepages=%d" % mbPages
      kernelCpuStr = maybeDisableHT( self.hyperThreadingEnabled )
      # Handle core allocation
      # kernel expects isolcpus to be specifed as matching the actual cpus.
      # Earlier 4.9 kernel silently rejects invalid values but 4.9 onwards
      # kernel emits an error when exceeding the actual present cores.

      isolCpuStr = self.getIsolCpusStr()
      kernelStr += " " + isolcpusKernelParam + isolCpuStr
      kernelStr += " nohz=on"
      if isolCpuStr != "":
         kernelStr += " nohz_full=" + isolCpuStr + " rcu_nocbs=" + isolCpuStr
      kernelStr += kernelCpuStr
      self.kernelStr = kernelStr
      return self.kernelStr

   def writeKernelParams( self ):
      if os.path.exists( skipSfeKernelSetup ):
         debug( "Skipping EoSCloudInit kernel parameter setup " \
               "as user requested, removing file" )
         # remove for next iteration in case user forgets to cleanup.
         os.remove( skipSfeKernelSetup )
         return False

      # Get kernel parameters that need to be applied
      ourKstr = self.getKernelCmdLine()   
      
      # split generated kernel paramters
      splitKernelCmdLine = ourKstr.split(' ')
      foundParams = True
      with open( '/proc/cmdline', 'r' ) as fp:
         cmdLine = fp.read()
         cmdLine = cmdLine.split( ' ' )
         for word in splitKernelCmdLine:
            if word not in cmdLine:
               debug( "kernel-param not found:%s" % word )
               foundParams = False
               break
         if 'nosmt=force' in cmdLine and 'nosmt=force' not in splitKernelCmdLine:
            # EosCloudInitLib.py checks that the dynamically generated list of
            # kernel parameters matches what is available in /proc/cmdline.
            # It checks this by iterating over the dynamically generated list
            # against what is in /proc/cmdline. When smt is enabled the
            # "nosmt=force" parameter is omitted. This if-statement ensures checking
            # whether or not smt is enabled by looking for "nosmt=force" in 
            # /proc/cmdline.
            debug( '"nosmt=force" not found in generated parameters but'\
                                    ' found in /proc/cmdline. Rebooting.' )
            foundParams = False

      # if all params have been found, return early and we don't write
      # the generated kernel string. If default kernel parameters differ
      # from generated parameters then we skip the early return and write
      # the kernel-params file
      if foundParams:
         return False
      
      # First boot case where custom kernel parameters have not been applied
      if not os.path.exists( self.kernelParamsFile ):
         with open( self.kernelParamsFile, "w") as f:
            f.write( "\n".join( [ eosCloudKernelParamHeader, ourKstr, \
                  userKernelParamHeader ] ) + '\n' )
            return True

      # get current kernel-params file since it exists, and grab all lines
      with open( self.kernelParamsFile ) as f:
         currContents = f.read().splitlines()
      
      # Normal case during regular boot ( including user-supplied params )
      # e.g. /mnt/flash/kernel-params looks like this:
      # # EosCloud Kernel Parameters
      # < Kernel Parameters from getKernelCmdLine() above >
      # # User-specified Kernel Parameters to add/overwrite below
      # < user param line 1 >
      # < user param line 2 >
      # ...
      if len( currContents ) >= 3 and ( currContents[ 0 : 3 ] == \
            [ eosCloudKernelParamHeader, ourKstr, userKernelParamHeader ] ):
         debug( "kernel-params file exists, " \
               "base EosCloud Kernel-Params already present" )
         return False

      # take any user-specified parameters if they exist
      userContents = []
      if len( currContents ) > 3:
         userContents += currContents[ 3 : ] 

      finalStr = "\n".join( [ eosCloudKernelParamHeader, ourKstr, \
            userKernelParamHeader ] + userContents ) + '\n'

      with open( self.kernelParamsFile, "w") as f:
         f.write( finalStr )
         return True

## Process downloaded user data by find start markers and write contents into
## corresponding config files
def startMark( name ):
   return '%%%s-START%%' % name

def endMark( name ):
   return '%%%s-END%%' % name

class configFile:
   def __init__( self, name, filename, append=False ):
      self.name = name
      self.endMarker = endMark( name )
      self.filename = filename
      self.append = append

forceUserDataString = "%FORCE-USER-DATA%"
eosStartupConfig = 'EOS-STARTUP-CONFIG'
startMarkStartupConfig = startMark( eosStartupConfig )
endMarkStartupConfig = endMark( eosStartupConfig )

emptyUserData = f"{startMarkStartupConfig}\n{endMarkStartupConfig}\n"

## user data needs to have start and end markers to denote the specific config
## eg for EOS-STARTUP-CONFIG the markers need to be
## %EOS-STARTUP-CONFIG-START%
## %EOS-STARTUP-CONFIG-END%
defaultConfigs = {
   startMark( 'EOS-STARTUP-CONFIG' ) : configFile( "EOS-STARTUP-CONFIG",
                                                   startupConfig,
                                                   append=True ),
   startMark( 'CLOUDHA-CONFIG' ) : configFile( "CLOUDHA-CONFIG",
                                               "/mnt/flash/cloud_ha_config.json" ),
   startMark( 'LICENSE-IPSEC' ) : configFile(
      "LICENSE-IPSEC", "/persist/secure/license/store/ipsec_license.json" ),
   startMark( 'LICENSE-BANDWIDTH' ) : configFile(
      "LICENSE-BANDWIDTH", "/persist/secure/license/store/bandwidth_license.json" ),
   startMark( 'VEOS-CONFIG' ) : configFile( "VEOS-CONFIG",
                                            veosConfigFile ),
   # This should not be used unless user wants to process some derived
   # internal config which is sensitive to boot and normally applies after
   # the reboot like 'platform sfe data-plane cpu alloc' etc.
   # This config is either translated from CLI or some non-CLI based.
   # This will also avoid an extra reboot. 
   startMark( 'VEOS-CONFIG-INTERNAL' ) : configFile( "VEOS-CONFIG-INTERNAL",
                                            veosInternalConfigFile ),
}

def runCmd( cmd, dump=True, background=False ):
   p = Popen( cmd, stdout=PIPE, stderr=PIPE ) # pylint: disable=consider-using-with
   if background:
      return None, None
   ( output, error ) = p.communicate()
   rc = p.returncode
   output = output.decode( "utf-8" )
   if dump:
      print( "Execute " + " ".join( cmd ) )
      print( "Output: %s" % output )
      print( "Erorr : %s" % error )
   return output, rc

# Force-kill the dhclient
def killDhclient():
   pid = readFile( dhclientPidFile )
   if pid:
      # pid is a list like [ '485\n' ]
      runCmd( [ 'kill', '-9', pid[ 0 ].rstrip() ] )

def cleanupExit( errStr, rc=0 ):
   print( errStr )
   # cleanup dhclients if any
   killDhclient()
   sys.exit( rc )

def readFile( fileName ):
   f = open( fileName ) # pylint: disable=consider-using-with
   contents = f.readlines()
   f.close()
   return contents

def getNetDevName():
   ## get network device name
   devName = [ 'eth0', 'ma1' ]
   devDir = '/sys/class/net'
   netDevs = [ dev for dev in os.listdir( devDir ) if dev in devName ]
   if netDevs:
      return netDevs[ 0 ]
   return None

def getIP( devname ):
   # pylint: disable-next=consider-using-with
   p = Popen( [ 'ip', 'addr', 'show', 'dev', devname ],
              stdout=PIPE, stderr=PIPE )
   out, _ = p.communicate()
   pat = re.compile( r"inet\s(\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}/\d{1,3})" )
   m = pat.search( out )
   return m.groups()[ 0 ] if m else None

def getIpAndBroadcast( dev ):
   ipRe = r'\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}'
   # pylint: disable-next=consider-using-with
   p = Popen( [ 'ip', 'addr', 'show', 'dev', dev ],
              stdout=PIPE, stderr=PIPE )
   out, _ = p.communicate()
   # Popen returns bytes, so need to decode
   pat = re.compile( r'inet\s(' + ipRe + r'/\d{1,3})\sbrd\s(' + ipRe + r')' )
   m = pat.search( out.decode( 'utf-8' ) )
   return ( m.group( 1 ), m.group( 2 ) ) if m else ( None, None )

def writeFile( fileName, contents ):
   f = open( fileName, "w" ) # pylint: disable=consider-using-with
   contents = "".join( contents )
   f.write( contents )
   f.close()

def getDefaultGw( devname='' ):
   output, _ = runCmd( [ "ip", "route" ], dump=False )
   pat = re.compile( "default via (.*) dev %s" % devname )
   m = pat.search( output )
   if m is not None:
      print( "GW is %s" % m.group( 1 ) )
      return m.group( 1 )
   else:
      return None

def getDhcpLeaseContents():
   leasePaths = [
      # in Azure we noticed sometimes that the default lease
      # file was missing but dhclient.leases was present
      # so including both for redundancy
      '/var/lib/dhclient/dhclient.leases',
      '/var/lib/dhclient/dhclient-default.leases'
   ]
   def leaseFile():
      for leasePath in leasePaths:
         if os.path.isfile( leasePath ):
            return leasePath
      return None

   Tac.waitFor( leaseFile, timeout=10.0 )

   leaseFilePath = leaseFile()
   if leaseFilePath:
      leaseContents = ''.join( readFile( leaseFilePath ) )
   else:
      cleanupExit( "Could not acquire dhcp lease", rc=1 )

   return leaseContents

def getDnsIp():
   leaseContents = getDhcpLeaseContents()
   m = re.search( 'domain-name-servers ([0-9]+.[0-9]+.[0-9]+.[0-9]+)',
                  leaseContents )
   if m:
      return m.groups()[ 0 ]
   else:
      cleanupExit( "DNS server IP not found in dhcp lease", rc=1 )
   return None

def getAllIntfs():
   devDir = '/sys/class/net'
   pattern = r'(eth[0-9]+)'
   netDevs = [ dev for dev in os.listdir( devDir )
                if re.match( pattern, dev ) is not None ]
   return netDevs

def setupConnection( ping=True ):
   ## Bring up the interface
   ## Aboot renames the network interface to ma1 if corresponding network
   ## drivers are enabled in Aboot kernel.
   netDev = getNetDevName()
   if netDev is None:
      cleanupExit( "Unable find eth0/ma1..network Device", rc=1 )

   runCmd( ["ip", "link"] )
   _, rc = runCmd( ["ip", "link", "set", "dev", netDev, "up"] )
   if rc != 0:
      cleanupExit( "Unable to bringup" + netDev + \
                   " no network access. RC %d" % rc, rc )
   runCmd( ["ip", "link"] )

   ## Get IP address using DHCP
   retryCount = 0
   while True:
      print( "dhclient retryCount = %d" % retryCount )
      _, rc = runCmd( [ "dhclient", netDev, '-pf', dhclientPidFile ] )
      if rc == 0:
         break
      retryCount = retryCount + 1
      if retryCount >= 10:
         break
      killDhclient()
      time.sleep(1)

   if rc != 0:
      cleanupExit( "dhclient failed return code %d" % rc, rc )

   runCmd( ["ip", "addr"] )

   ip, brd = getIpAndBroadcast( netDev )

   if ip is None:
      cleanupExit( 'Failed to fetch IP', 1 )

   if brd is None:
      cleanupExit( 'Failed to fetch broadcast address', 1 )

   runCmd( [ 'ip', '-4', 'addr', 'change', ip, 'broadcast', brd, 'dev', netDev,
             'valid_lft', 'forever', 'preferred_lft', 'forever' ] )

   ## get default GW
   gw = getDefaultGw()
   if not gw:
      cleanupExit( "Unable to find default GW..exiting", rc=1 )

   ## allow established connections so we can retrieve provisioning information
   runCmd( [ "iptables-save", "-c" ] )
   _, rc = runCmd( [ "iptables", "-I", "INPUT", "-m", "conntrack", "--ctstate",
                     "established,related", "-j", "ACCEPT" ] )
   if rc != 0:
      print("Could not add iptables rule to accept outgoing connections; continuing")
   else:
      runCmd( [ "iptables-save", "-c" ] )

   ## wait until we can ping the default GW
   if ping:
      for _ in range( 0, 5 ):
         _, rc = runCmd( ["ping", gw, "-w", "5", "-c", "1"] )
         if rc == 0:
            break
      if rc != 0:
         cleanupExit( "Unable to ping GW. RC %d" % rc, rc )

   return gw

def findMarkIndex( mark, contents ):
   markIndices = [ i for i, s in enumerate( contents ) if mark in s ]
   if markIndices:
      return markIndices[ 0 ]
   else:
      return -1

def generateUserConfig( userName, userPassword, keyName='key.pub' ):
   userConfig = []
   if userPassword:
      hashAlg = 'sha512'
      hashPassword = SecretCli.encrypt( userPassword, hashAlgorithm=hashAlg )
      # add username with hashed password
      userConfig.append(
         f'username {userName} secret {hashAlg} {hashPassword}\n' )
   else:
      # add username with no password
      userConfig.append( 'username %s secret *\n' % userName )
   # add public key file location for username
   userConfig.append(
      f'username {userName} ssh-key file flash:{keyName}\n' )
   return userConfig

def firstTimeBoot():
   return not os.path.exists( startupConfig )

def isForcedUserData():
   if not os.path.exists( userData ):
      return False
   with open( userData ) as f:
      if f.readline().strip() == forceUserDataString:
         return True
   return False

def copyDefaultStartupConfig( extraConfig=None ):
   shutil.copyfile( defaultStartupConfig, startupConfig )
   if extraConfig:
      contents = readFile( startupConfig )
      contents.extend( extraConfig )
      writeFile( startupConfig, contents )

def processUserData( configs, extraConfig=None, copyDefaultConfig=True ):
   configs.update( defaultConfigs )

   config = None
   fo = None

   if not os.path.exists( userData ):
      print( "User Data file does not exist" )
      return

   with open( userData ) as f:
      ## %FORCE-USER-DATA% in the first line of the user data will make
      ## EOS treat the boot as a first time boot and honor the user data over
      ## existing configuration
      cline = f.readline().strip()
      if isForcedUserData():
         print( "Forcing first time boot" )
         if os.path.exists( startupConfig ):
            shutil.copyfile( startupConfig, "/mnt/flash/startup-config-backup" )
      elif firstTimeBoot():
         f.seek( 0, 0 )
      else:
         return
      if copyDefaultConfig:
         copyDefaultStartupConfig( extraConfig=extraConfig )
      ## loop through user data splitting into configuration files
      for line in f:
         cline = line.strip()
         ## Process the various configs
         if cline in configs:
            ## found start marker
            if fo != None: # pylint: disable=singleton-comparison
               ## Previous config file is still open
               print( "End marker not found for %s" % config.name )
               fo.close()
            config = configs[ line.strip() ]
            print( "found start marker for %s" % config.name )
            try:
               if config.append is True:
                  fo = open( config.filename, "a" )
               else:
                  # pylint: disable-next=consider-using-with
                  fo = open( config.filename, "w" )
            # pylint: disable=W0703
            except Exception as error:
               print( "Unable to open file to write for {} error {!r}".format(
                     config.name,
                                                                        error ) )
         # pylint: disable-next=singleton-comparison
         elif fo != None and cline == config.endMarker :
            print( "found end marker for %s" % config.name )
            fo.close()
            fo = None
         else:
            if fo is None:
               ## ignore and continue in case we haven't found start marker
               continue
            fo.write( line )
      if fo != None: # pylint: disable=singleton-comparison
         print( "End marker not found for %s" % config.name )
         fo.close()

def isModeSfe( modeFile = veosConfigFile ):
   return parseVeosConfig( modeFile )[ 'MODE' ] == 'sfe' or \
          VeosHypervisor.platformCaravan()


def isModeSfa( modeFile = veosConfigFile ):
   return parseVeosConfig( modeFile )[ 'MODE' ] == 'linux'

def isWanEmulationEnabled( modeFile = veosConfigFile ):
   if 'WANEMULATION' in parseVeosConfig( modeFile ):
      return parseVeosConfig( modeFile )[ 'WANEMULATION' ].lower() == \
         'true'
   else:
      return False

def getSystemMemoryGB():
   testMem = os.environ.get( 'TEST_SYSTEM_MEM_GB', None )
   if testMem:
      return int( testMem ) 
   memGB = 0
   with open('/proc/meminfo') as f:
      meminfo = f.readlines()[ 0 ]
      matched = re.search(r'^MemTotal:\s+(\d+)', meminfo)
      if matched: 
         memKB = int(matched.groups()[0])
         memGB = (memKB // ( 1024 * 1024 ))

   return memGB


def getSysconf( conf ):
   testVar = "TEST_" + conf  
   return  (int)( os.environ.get( testVar, os.sysconf( conf )) )

# Simultaneous Multithreading (SMT) a.k.a Hyper-Threading (HT) allows
# multiple execution threads to be executed on a single physical CPU core.
# HyperThreading is disabled with "nosmt=force" appended to the
# /mnt/flash/kernel-params.
# Returns kernel parameter string to disable HT
def maybeDisableHT( hyperThreadingEnabled ):
   htDisableStr = ""
   if not hyperThreadingEnabled:
      htDisableStr = " nosmt=force"
   return htDisableStr

# Look at CPU and Memory requirements to create a kernel command line
# Also generates a final config for Fru consumption later during boot.
def doSetupSfeKernelParams():
   try:
      bessConf = BessConfigurator()
      bessConf.computeBessMemory()
      bessConf.writeRunTimeVeosConfig()
      ret = bessConf.writeKernelParams()
      bessConf.dumpConfig()
      return ret
   
   except CloudInitException as e:
      debug( "Caught exception: %s " % e )
      # this will force system to halt with proper error string from 
      # EosCloud Service
      sys.exit( e.errCode )
      
   except Exception as e: # pylint: disable=broad-except
      debug( "Caught Exception: %s" % ( e ) ) 
      traceback.print_exc()
      # Should we let it to boot in infinite loop. Probably not
      # a good idea
      return False
   return False 

# Create or append to a kernel params file on /mnt/flash/kernel-params
# It should look something like 
# default_hugepagesz=1G hugepagesz=1G hugepages=2 hugepagesz=2M hugepages=256
# isolcpus=2-7
# If the file did not have relevant config already, we need to trigger a reboot
# after the write. The values of the memory allocation depends on how much actual
# memory is present in the system.
# Returns true if reboot is required
def setupSfeKernelParams():
   return doSetupSfeKernelParams()

def doPlatformCheck():
   if ( isModeSfa() or isModeSfe() ) and not VeosHypervisor.getPlatform():
      debug( "Caught Exception: %s" % ( VeosHypervisor.getPlatform()  ) ) 
      raise UnknownCloudPlatformException()
   return True

# Halt the CloudEOS VM only if the mode is either Sfa/Sfe and the platform is
# unknown. Explicit Sfa/Sfe mode check is required to allow veos-lab VM to boot.
def isSupportedCloudEOSPlatform():
   try:
      return doPlatformCheck()
   except CloudInitException as e:
      debug( "Caught exception: %s " % e )
      #force system to halt
      sys.exit( e.errCode )
   return False

def matchEt100Config( line ):
   tokens = line.split()
   if len( tokens ) != 2 and len( tokens ) != 3:
      # Config can be like "interface Ethernet100" or "interface Ethernet 100"
      return False

   tokens = [ token.strip() for token in tokens ]

   # Match "interface Ethernet100" and "int et100" type of config
   if ( len( tokens ) == 2 and "interface".startswith( tokens[ 0 ] ) and
        "Ethernet".startswith( tokens[ 1 ][ : -3 ] ) and
        tokens[ 1 ].endswith( "100" ) ):
      return True

   # Match "interface Ethernet 100" and "int et 100" type of config
   if ( len( tokens ) == 3 and "interface".startswith( tokens[ 0 ] ) and
        "Ethernet".startswith( tokens[ 1 ] ) and tokens[ 2 ] == "100" ):
      return True

   return False

def deleteFabricDevice():
   cmd = "ip link delete %s" % fabricDevName
   Tac.run( cmd.split( ' ' ), asRoot=True )

def fabricDevicePresent():
   devSysPath = '/sys/class/net/' + fabricDevName
   return os.path.exists( devSysPath )

def createFabricDevice():
   # Create a dummy fabric kernel interface.
   # Sfe based platforms dom't use fabric interface
   # but Ebra needs it to create routed Vlan interfaces ( SVIs )
   # Sfe based platforms don't support SVIs yet. But to
   # make the lo routable using "hardware fordwarding id" Cli,
   # Ebra needs to create SVI for the allocated dynamic Vlan
   configCmds = \
         [ f"ip link add {fabricDevName} type dummy",
           f"ip link set {fabricDevName} address {fabricDevMac} mtu {fabricDevMtu}",
           f"ip link set {fabricDevName} alias {fabricDevAlias} up" ]

   for cmd in configCmds:
      cmdList = cmd.split( " " )
      Tac.run( cmdList, asRoot=True )

def ignoreConfigForDpsIntf( line ):
   # Since Ethernet100 is an Ethernet interface and Dps1 is a virtual interface. Not
   # all config that can be present under Ethernet100 is valid for Dps1. To avoid any
   # errors while parsing this invalid config by Dps1, we transfer only the supported
   # config to Dps1 from Ethernet100
   if line.strip().startswith( "ip add" ):
      return False
   if line.strip().startswith( "mtu" ):
      return False
   if line.strip().startswith( "tcp" ):
      return False
   if line.strip().startswith( "shut" ):
      return False
   if line.strip().startswith( "no shut" ):
      return False
   if line.strip().startswith( "flow tracker" ):
      return False
   return True

def ethernet100ToDpsIntfUpgrade():
   configReplaced = False
   tmpStartupConfig = startupConfig + ".tmp"
   unchangedStartupConfig = startupConfig + ".unchanged"
   insideEthernet100Section = False
   try:
      with open( tmpStartupConfig, "w" ) as nc, open( startupConfig ) as oc:
         for line in oc.readlines():
            if matchEt100Config( line ):
               configReplaced = True
               nc.write( "interface Dps1\n" )
               insideEthernet100Section = True
            else:
               if insideEthernet100Section and not line.startswith( " " ):
                  # Once we enter Ethernet100 section, we stay inside it until we
                  # encounter a config line which doesn't start with a
                  # whitespace. This assumption comes from the fact that the
                  # Ethernet100 config will be indented under "interface Ethernet100"
                  insideEthernet100Section = False
               if insideEthernet100Section and ignoreConfigForDpsIntf( line ):
                  continue
               nc.write( line )
   except Exception as e:       # pylint: disable=broad-except
      debug( "DpsIntf: Error during upgrade." )
      debug( e )
      return

   if not configReplaced:
      try:
         os.remove( tmpStartupConfig )
         return
      # pylint: disable-next=unused-variable
      except Exception as e:    # pylint: disable=broad-except
         debug( "DpsIntf: Error while removing .tmp file" )
         return

   debug( "DpsIntf: Ethernet100 config replaced with Dps1" )
   # TODO: Move the file only after doing some sanity checks that we haven't
   # completely messed up the config. Else delete the .tmp config
   # Not able to figure out a simple check, because we do not want to complicate this
   # script
   try:
      shutil.copyfile( startupConfig, unchangedStartupConfig )
      shutil.move( tmpStartupConfig, startupConfig )
   except Exception as e:       # # pylint: disable=broad-except
      debug( "DpsIntf: Error while copying new startup-config" )
      debug( e )
