#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
import socket
import json
import Tracing
import time
import calendar
from ipaddress import IPv6Address
from IpLibConsts import DEFAULT_VRF
import QuickTrace
import errno
import os
import Toggles.DhcpServerToggleLib
import Tac

__defaultTraceHandle__ = Tracing.Handle( "DhcpServer" )
t8 = Tracing.t8

qt8 = QuickTrace.trace8

aristaOuis = [ "0xfcbd67", "0xc0d682", "0x985d82", "0x001c73", "0x0050c2",
               "0x28993a", "0x7483ef", "0x444ca8", "0x30862d", "0x948ed3",
               "0xd4aff7", "0xc06911", "0xe8aec5" ]

overrideLockfileDir = "/var/lib/run/kea"
tmpKeaConfFile = "/tmp/kea-dhcp{version}-{vrf}.conf.tmp"
tmpToCheckKeaConfFile = "/tmp/kea-dhcp{version}-{vrf}-ToCheck.conf.tmp"
keaLeaseDir = "/var/lib/kea"
keaLeasePath = "/var/lib/kea/dhcp{version}-{vrf}.leases"
keaConfigPath = "/etc/kea/kea-dhcp{version}-{vrf}.conf"
keaControlConfigPath = "/etc/kea/kea-ctrl{version}-{vrf}.conf"
keaPidPath = "/var/lib/run/kea/kea-dhcp{version}-{vrf}.kea-dhcp{version}.pid"
keaServiceName = "keactrl{version}-{vrf}"
keaControlSockAndPidDir = "/var/lib/run/kea"
keaControlSock = "/var/lib/run/kea/kea-dhcp{version}-{vrf}-ctrl.sock"
keaLogPath = "/tmp/keaLogs"

# These are the default lease times for kea
leaseTimeDurationKeaDefault = ( 0, 2, 0 )
leaseTimeSecondsKeaDefault = 7200

clientClassConfigTypes = {
      Tac.Type( 'DhcpServer::GlobalClientClassConfig' ): 'global',
      Tac.Type( 'DhcpServer::GlobalClientClass6Config' ): 'global',
      Tac.Type( 'DhcpServer::ClientClassConfig' ): 'subnet',
      Tac.Type( 'DhcpServer::ClientClass6Config' ): 'subnet',
      Tac.Type( 'DhcpServer::RangeClientClassConfig' ): 'range',
      Tac.Type( 'DhcpServer::RangeClientClass6Config' ): 'range' }
vendorSubOptionType = Tac.Type( 'DhcpServer::VendorSubOptionType' )
OptionType = Tac.Type( 'DhcpServer::OptionType' )
defaultVendorId = 'default'

maxCheckConfigInternalRecursionLevel = 5

# from linux/if_addr.h
IFA_F_TENTATIVE = 0b01000000

# Messages for interface status
class DhcpIntfStatusMessages:
   NO_INTF_STATUS_LOCAL_MSG = "Could not determine VRF"
   NO_INTF_STATUS_ALL_MSG = "No kernel interface name"
   NOT_IN_VRF_MSG = "Not in VRF {}"
   NO_KNI_MSG = "Kernel interface not created"
   NOT_UP_MSG = "Not up"
   NO_IP_ADDRESS_MSG = "No IP address"
   NO_LINK_LOCAL_ADDRESS_MSG = "No Link Local address"
   DHCP_RELAY_CFG_MSG = "DHCP relay is configured for this interface"
   DHCP_RELAY_ALWAYS_MSG = "DHCP relay is always on"

# Messages for subnet direct status
class Dhcp6DirectRelayMessages:
   MULTIPLE_INTERFACES_MATCH_SUBNET = "Multiple interfaces match this subnet: {}"
   MULTIPLE_SUBNETS_MATCH_INTERFACE = "This and other subnets match interface {}"
   NO_IP6_ADDRESS_MATCH = "No IPv6 addresses on interfaces match this subnet"

unknownSubnets = { 4: Arnet.Prefix( "0.0.0.0/0" ),
                   6: Arnet.Ip6Prefix( '::0/0' ) }

def dhcpServerActive( vrfStatus ):
   return vrfStatus.ipv4ServerRunning or vrfStatus.ipv6ServerRunning

def ipAddrToInt( ip, v4=True ):
   if v4:
      return Arnet.IpAddr( ip ).value
   else:
      value = 0
      ip6Addr = Arnet.Ip6Addr( ip )
      for i in [ ip6Addr.word0, ip6Addr.word1, ip6Addr.word2, ip6Addr.word3 ]:
         value <<= 32
         value += i
      return value

def convertLeaseSeconds( seconds ):
   seconds = int( seconds )
   val = seconds // 60
   minutes = val % 60
   val = val // 60
   days = val % 24
   hours = val // 24

   return hours, days, minutes

def convertDaysHoursMinutes( days, hours, minutes ):
   return ( ( days * 24 * 60 * 60 ) +
            ( hours * 60 * 60 ) +
            ( minutes * 60 ) )

def hasLinkLocalAddr( ipAddrs ):
   for addr in ipAddrs:
      if addr.addr.isV6LinkLocal:
         return True
   return False

def rangeV4Contains( subnetRange, ipAddr ):
   ipAddrVal = ipAddr.value
   startVal = Arnet.IpAddr( subnetRange.start ).value
   endVal = Arnet.IpAddr( subnetRange.end ).value
   return startVal <= ipAddrVal <= endVal

def rangeV6Contains( subnetRange, ipAddr ):
   ipAddrVal = int( ipAddr.stringValueExpanded.replace( ':', '' ), 16 )
   startVal = int( subnetRange.start.stringValueExpanded.replace( ':', '' ), 16 )
   endVal = int( subnetRange.end.stringValueExpanded.replace( ':', '' ), 16 )
   return startVal <= ipAddrVal <= endVal

def rangeContains( subnetRange, ipAddr ):
   if isinstance( ipAddr, Tac.Type( 'Arnet::IpAddr' ) ):
      return rangeV4Contains( subnetRange, ipAddr )
   else:
      return rangeV6Contains( subnetRange, ipAddr )

def incrementArnetIpAddr( ip, v4=True ):
   if v4:
      return Tac.Value( 'Arnet::IpAddr', ip.value + 1 )
   else:
      ipv6Addr = IPv6Address( str( ip.stringValue ) )
      return Arnet.Ip6Addr( ipv6Addr + 1 )

def decrementArnetIpAddr( ip, v4=True ):
   if v4:
      return Tac.Value( 'Arnet::IpAddr', ip.value - 1 )
   else:
      ipv6Addr = IPv6Address( str( ip.stringValue ) )
      return Arnet.Ip6Addr( ipv6Addr - 1 )

def featureFlexibleMatchingFuture():
   return Toggles.DhcpServerToggleLib.toggleFlexibleMatchingFutureEnabled()

def featureArbitraryOptionMatch():
   return Toggles.DhcpServerToggleLib.toggleDhcpServerArbitraryOptionMatchEnabled()

def featureEchoClientId():
   return Toggles.DhcpServerToggleLib.toggleDhcpServerEchoClientIdEnabled()

def featureArbitraryOptionV6():
   return Toggles.DhcpServerToggleLib.toggleDhcpServerArbitraryOptionV6Enabled()

def featureArbitraryOptionV6GlobalClientClass():
   return Toggles.DhcpServerToggleLib.\
          toggleDhcpServerArbitraryOptionV6GlobalClientClassEnabled()

def featureArbitraryOptionV6SubnetClientClass():
   return Toggles.DhcpServerToggleLib.\
          toggleDhcpServerArbitraryOptionV6SubnetClientClassEnabled()

def featureArbitraryOptionV6RangeClientClass():
   return Toggles.DhcpServerToggleLib.\
          toggleDhcpServerArbitraryOptionV6RangeClientClassEnabled()

def vrfStr( vrf ):
   if vrf == 'default':
      return ''
   return f'-vrf-{vrf}'

def tacRange( start, end ):
   return Tac.Value( 'DhcpServer::Range', start, end )

def tacRange6( start, end ):
   start = Arnet.Ip6Addr( start )
   end = Arnet.Ip6Addr( end )

   return Tac.Value( 'DhcpServer::Range6', start, end )

def tacLayer2Info( intf, vlan, mac ):
   return Tac.Value( 'DhcpServer::Layer2Info', intf, vlan, mac )

def tacHexOrStr( hexVal, strVal ):
   return Tac.Value( 'DhcpServer::HexOrStr', hexVal, strVal )

def tacCircuitIdRemoteIdHexOrStr( circuitIdHex, circuitIdStr, remoteIdHex,
                                  remoteIdStr ):
   circuitIdHexOrStr = tacHexOrStr( circuitIdHex, circuitIdStr )
   remoteIdHexOrStr = tacHexOrStr( remoteIdHex, remoteIdStr )
   return Tac.Value( 'DhcpServer::CircuitIdRemoteIdHexOrStr', circuitIdHexOrStr,
                     remoteIdHexOrStr )

def tacVendorClassOption( enterpriseId, anyEnterpriseId, dataList ):
   vendorClass = Tac.Value( 'DhcpServer::VendorClassOption', enterpriseId,
                            anyEnterpriseId )
   for i, data in enumerate( dataList ):
      # Note that default values, in this case 0, cannot be used as keys in
      # collections inside Value types in Sysdb collections.
      vendorClass.data[ i + 1 ] = data
   return vendorClass

def formatKeaRemoteIdHexTest( hexVal ):
   remIdHexTestFmt = ( "((substring(relay6[-1].option[37].hex,4,all) == 0x{0})"
                       " or (substring(option[37].hex,4,all) == 0x{0}))" )
   return remIdHexTestFmt.format( hexVal )

def formatKeaRemoteIdStrTest( strVal ):
   remIdStrTestFmt = ( "((substring(relay6[-1].option[37].hex,4,all) == '{0}')"
                       " or (substring(option[37].hex,4,all) == '{0}'))" )
   return remIdStrTestFmt.format( strVal )

def formatL2InfoCmd( l2Info, af ):
   mac = Arnet.EthAddr( l2Info.mac ).displayString
   return reservationsAristaSwitchCmd( l2Info.intf, l2Info.vlanId, mac, af )

def formatRemoteIdHexOrStrCmd( remoteIdHexOrStr ):
   if remoteIdHexOrStr.hex:
      return f'remote-id hex {remoteIdHexOrStr.hex}'
   elif remoteIdHexOrStr.str:
      return f'remote-id string "{remoteIdHexOrStr.str}"'
   else:
      return ''

def formatInfoOptionHexOrStrCmd( circIdRemIdHexOrStr ):
   infoOptCmd = 'information option{}{}{}'
   if circIdRemIdHexOrStr.circuitIdHex:
      circuitIdCmd = f' circuit-id hex {circIdRemIdHexOrStr.circuitIdHex}'
   elif circIdRemIdHexOrStr.circuitIdStr:
      circuitIdCmd = ' circuit-id string "{}"'.format(
                        circIdRemIdHexOrStr.circuitIdStr )
   else:
      circuitIdCmd = ''
   remoteIdCmd = formatRemoteIdHexOrStrCmd( circIdRemIdHexOrStr.remoteIdHexOrStr )
   space = ' ' if remoteIdCmd else ''
   return infoOptCmd.format( circuitIdCmd, space, remoteIdCmd )

def formatVendorClassCmd( vendorClass ):
   vendorClassCmd = 'vendor-class {}{}'
   entIdStr = ( '' if vendorClass.anyEnterpriseId else
                f'enterprise-id {vendorClass.enterpriseId} ' )
   quotedData = [ f'string "{data}"'
                  for data in vendorClass.data.values() ]
   dataStr = ' '.join( quotedData )
   return vendorClassCmd.format( entIdStr, dataStr )

def genOptionCmd( optionString, optionCode, optionType, optionData=None,
                  alwaysSend=False, disable=False, af='ipv4', addQuotes=True ):
   cmd = optionString
   if af:
      cmd += f' {af} {optionCode}'
   else:
      cmd += f' {optionCode}'
   # always send the option to the client
   if alwaysSend:
      cmd += ' always-send'
   # set proper type
   if optionType == OptionType.optionString:
      eosCliType = 'string'
   elif optionType == OptionType.optionFqdn:
      eosCliType = 'fqdn'
   elif optionType == OptionType.optionHex:
      eosCliType = 'hex'
   elif optionType == OptionType.optionIpAddress:
      eosCliType = 'ipv4-address'
   elif optionType == OptionType.optionIp6Address:
      eosCliType = 'ipv6-address'
   else:
      t8( 'Unknown', optionString, 'type:', optionType )
      return ''
   cmd += f' type {eosCliType}'
   if disable:
      return f'no {cmd}'
   # set proper data
   if not optionData:
      t8( 'No option data provided' )
      return ''
   if optionType == OptionType.optionString and addQuotes:
      optionData = '"' + optionData + '"'
   if optionType in ( OptionType.optionIpAddress, OptionType.optionIp6Address ):
      optionData = [ str( ipAddr ) for ipAddr in optionData ]
      optionData = ' '.join( optionData )
   # optionData: can be a double quoted string or a list
   cmd += f' data {optionData}'
   return cmd

def optionNameToCode( optionName, ipVersion='4' ):
   standardNameToCodeV4 = {
      'routers': 3,
      'domain-name-servers': 6,
      'domain-name': 15,
      'vendor-encapsulated-options': 43,
      'tftp-server-name': 66,
      'boot-file-name': 67,
      'tftp-server-address': 150,
   }
   standardNameToCodeV6 = {
      'dns-servers': 23,
      'bootfile-url': 59,
      'domain-search': 24,
   }
   if ipVersion == '4':
      standardNameToCode = standardNameToCodeV4
   else:
      standardNameToCode = standardNameToCodeV6

   if optionName in standardNameToCode:
      return standardNameToCode[ optionName ]
   # option names start with privateOption or arbitraryOption
   elif ( optionName[ : 13 ] == 'privateOption' or
          optionName[ : 15 ] == 'arbitraryOption' ):
      # optionName.partition returns:
      # ( 'private', 'Option', '<code>' ) or ( 'arbitrary', 'Option', '<code>' )
      return int( optionName.partition( 'Option' )[ 2 ] )
   else:
      errMsg = f'Code mapping does not exist for option name: {optionName}'
      assert False, errMsg
   return None

def filterDuplicateOptions( optionList, oldOptionList, opt43DefaultIdConfigured,
                            ipVersion='4' ):
   # helper function for ArbitraryOption
   # filter out options that are already configured
   oldOptionCodes = set()
   if opt43DefaultIdConfigured:
      assert ipVersion == '4'
      oldOptionCodes.add( 43 )
   for option in oldOptionList:
      code = option.get( 'code', None )
      if code is None:
         code = optionNameToCode( option.get( 'name' ), ipVersion )
      oldOptionCodes.add( code )
   uniqueOptions = [ op for op in optionList if op.key.code not in oldOptionCodes ]
   return uniqueOptions

def optionTypeToKeaType( type_ ):
   if type_ == OptionType.optionString:
      return 'string'
   elif type_ == OptionType.optionIpAddress:
      return 'ipv4-address'
   elif type_ == OptionType.optionIp6Address:
      return 'ipv6-address'
   elif type_ == OptionType.optionFqdn:
      return 'fqdn'
   else:
      assert False, "Unknown private-option type"
      return None

def subOptionTypeToKeaType( type_ ):
   if type_ == vendorSubOptionType.string:
      return 'string'
   elif type_ == vendorSubOptionType.ipAddress:
      return 'ipv4-address'
   else:
      assert False, "Unknown sub-option type"
      return None

def fqdnCodeToKeaName( code_, af='ipv4' ):
   codeToNameV4 = { '15': 'domain-name', '88': 'bcms-controller-names',
                    '119': 'domain-search', '141': 'sip-ua-cs-domains',
                    '137': 'v4-lost', '213': 'v4-access-domain' }
   codeToNameV6 = { '21': 'sip-server-dns', '24': 'domain-search',
                    '29': 'nis-domain-name', '30': 'nisp-domain-name',
                    '33': 'bcmcs-server-dns', '51': 'v6-lost',
                    '57': 'v6-access-domain', '58': 'sip-ua-cs-list',
                    '64': 'aftr-name', '65': 'erp-local-domain-name' }
   if af == 'ipv4':
      return codeToNameV4.get( code_, "arbitraryOption" + code_ )
   else:
      return codeToNameV6.get( code_, "arbitraryOption" + code_ )

def optionToHex( optionCode=None, optionType=None, optionData=None,
                 optionDataOnly=False ):
   # Input: 1, vendorSubOptionType.string, "aaa"
   # Output: 1:3:61:61:61
   # Input: 2, vendorSubOptionType.ipAddress, [ '1.0.0.0', '11.0.0.0' ]
   # Output: 2:8:1:0:0:0:b:0:0:0
   # Input: OptionType.optionString, "aaa", optionDataOnly=True
   # Output: 61 61 61
   delim = '' if optionDataOnly else ':'
   if optionCode:
      codeHex = f'{optionCode:x}'

   if optionType in ( vendorSubOptionType.string, OptionType.optionString ):
      lengthHex = f'{len( optionData ):x}'
      # convert string to hex
      dataHex = delim.join( f'{ord( c ):x}' for c in optionData )

   if optionType in ( vendorSubOptionType.ipAddress, OptionType.optionIpAddress ):
      # A single ipv4-address has a length of 4, therefore the length
      # of the sub-option is dependant on how many ipv4-addresses it holds
      lengthHex = f'{len( optionData ) * 4:x}'
      dataHex = []
      # convert ipv4-address(es) to hex
      if not isinstance( optionData, list ):
         optionData = list( optionData.values() )

      for ipAddr in optionData:
         for c in ipAddr.split( '.' ):
            hexVal = f'{int( c ):x}'
            if optionType == OptionType.optionIpAddress:
               hexVal = '0' + hexVal if len( hexVal ) == 1 else hexVal
            dataHex.append( hexVal )
      dataHex = delim.join( dataHex )

   if optionType in ( OptionType.optionIp6Address, 'ipv6-address' ):
      # A single ipv6-address has a length of 8, therefore the length
      # of the sub-option is dependant on how many ipv6-addresses it holds
      lengthHex = f'{len( optionData ) * 8:x}'
      dataHex = []
      # convert ipv6-address(es) to hex
      if isinstance( optionData, str ):
         optionData = list( optionData.split( " " ) )
      if not isinstance( optionData, list ):
         optionData = list( optionData.values() )

      for ip6Addr in optionData:
         if isinstance( ip6Addr, str ):
            ip6Addr = Arnet.Ip6Addr( ip6Addr )
         dataHex.append( ip6Addr.stringValueExpanded.replace( ':', delim ) )
      dataHex = delim.join( dataHex )
   if optionType == OptionType.optionHex:
      lengthHex = f'{len( optionData ) // 2:x}'
      dataHex = optionData
   if optionType == OptionType.optionFqdn:
      lengthHex = f'{len( optionData ):x}'
      dataHex = optionData
   if optionDataOnly:
      return dataHex
   else:
      return "{code}:{length}:{data}".format( code=codeHex, length=lengthHex,
                                              data=dataHex )

def stringToHex( string ):
   return "".join( f"{ord( char ):02x}" for char in string )

def getOptionData( option, raw=False ):
   # raw=False means get optionData "as" is written in the kea config
   # raw=True means get optionData "as" entered by the user.
   # string: double quoted string
   # hex: hex string
   # ipAddress: array of ip addresses
   if option.type in ( OptionType.optionString, OptionType.optionFqdn,
                       OptionType.optionHex ):
      dataRaw = option.data
      # allow "," in the string
      if raw:
         data = dataRaw
      else:
         data = option.data.replace( ",", r"\," )
      return data
   else:
      dataList = list( option.data.values() )
      dataRaw = [ str( data ) for data in dataList ]
      data = dataRaw if raw else ','.join( dataRaw )
      return data

def subOptionCmd( optionCode, optionType=None, optionData=None, disable=False,
                  addQuotes=True, addArrayToken=False ):
   # optionData: can be a double quoted string or a list
   if disable:
      cmd = f'no sub-option {optionCode}'
      return cmd

   cmd = f'sub-option {optionCode} type'

   # set proper type and data
   if optionType == vendorSubOptionType.string:
      eosCliType = 'string'
      if addQuotes:
         optionData = '"' + optionData + '"'

   elif optionType == vendorSubOptionType.ipAddress:
      eosCliType = 'ipv4-address'
      if addArrayToken:
         cmd += ' array'
      optionData = ' '.join( optionData )
   else:
      assert False, "Unknown sub-option type"
      return None

   cmd += f' {eosCliType} data {optionData}'
   return cmd

def getSubOptionData( subOption, raw=False ):
   # raw=False means get optionData "as" is written in the kea config
   # raw=True means get optionData "as" entered by the user.
   # string: double quoted string
   # ipAddress: array of ip addresses
   if subOption.type == vendorSubOptionType.string:
      dataRaw = subOption.dataString
      # allow "," in the string
      data = dataRaw if raw else subOption.dataString.replace( ",", r"\," )
      return data

   else:
      dataRaw = list( subOption.dataIpAddress.values() )
      data = dataRaw if raw else ','.join( dataRaw )
      return data

def reservationsAristaSwitchCmd( intf='', vlan=0, macAddr='0000.0000.0000',
                                af="ipv4" ):
   v4 = af == "ipv4"
   header = "information option" if v4 else "remote-id"
   cmd = f"{header} arista-switch "
   if intf:
      cmd += f"{intf} vlan {vlan}"
      if macAddr != '0000.0000.0000':
         cmd += f" switch-mac {macAddr}"
   else:
      cmd += f"switch-mac {macAddr}"
   return cmd

def tftpServerOptions( option66, option150 ):
   # E.g. one option is configured
   # TFTP Server: 1.1.1.1 (option 66)
   # E.g. both options are configured (print each option on a new line)
   # TFTP Server:
   # 1.1.1.1 (option 66)
   # 2.2.2.2 3.3.3.3 (option 150)
   serverOptions = []
   serverOptionsStr = ''
   if option66:
      serverOptions.append( f"{option66} (Option 66)" )
   if option150:
      serverOptions.append( "{} (Option 150)".format(
         ' '.join( option150 ) ) )
   if serverOptions:
      fmt = '\n' if len( serverOptions ) == 2 else ' '
      serverOptionsStr = 'TFTP server:{}{}'.format(
         fmt, f'{fmt.join( serverOptions )}' )
   return serverOptionsStr

# Returns two strings containing lists of active and inactive client classes
# (with the reason for being inactive)
# activeClasses = 'Foo, Bar, Jar'
# inactiveClasses = 'Mat (no match criteria), Jet (no assignments)'
def filterClientClasses( clientClasses ):
   activeClasses = []
   inactiveClasses = []
   for clientClassName, clientClassDict in iter( clientClasses.items() ):
      inactiveReason = clientClassDict[ 'inactiveReason' ]
      if inactiveReason:
         inactiveClasses.append( clientClassName + ' (' + inactiveReason + ')' )
      else:
         activeClasses.append( clientClassName )
   activeClassesStr = ', '.join( activeClasses )
   inactiveClassesStr = ', '.join( inactiveClasses )
   return activeClassesStr, inactiveClassesStr

# Returns whether a client class has match criteria
def hasMatchCriteria( clientClassConfig, af ):
   if not clientClassConfig.matchCriteriaModeConfig:
      return False
   matchCriteria = clientClassConfig.matchCriteriaModeConfig

   # Check for af agnostic match criteria
   if matchCriteria.l2Info:
      return True

   # Check for af specific match criteria
   if ( af == 'ipv4' and
        ( matchCriteria.vendorId or matchCriteria.hostMacAddress or
          matchCriteria.circuitIdRemoteIdHexOrStr ) ):
      return True

   if ( af == 'ipv6' and
        ( matchCriteria.vendorClass or matchCriteria.duid or
          matchCriteria.remoteIdHexOrStr or matchCriteria.hostMacAddress ) ):
      return True
   return False

# Returns whether a client class has assignment options
def hasAssignmentOptions( optionsConfig, af ):
   # Check for private options
   privateOptionConfig = optionsConfig.privateOptionConfig
   if privateOptionConfig:
      if ( privateOptionConfig.stringPrivateOption or
           privateOptionConfig.ipAddrPrivateOption ):
         return True

   # Check for arbitrary options
   arbitraryOptionConfig = optionsConfig.arbitraryOptionConfig
   if arbitraryOptionConfig:
      if ( arbitraryOptionConfig.stringArbitraryOption or
           arbitraryOptionConfig.ipAddrArbitraryOption or
           arbitraryOptionConfig.hexArbitraryOption ):
         return True

   # Check for af agnostic standard options
   if ( optionsConfig.dnsServers or optionsConfig.domainName or
        optionsConfig.tftpBootFileName ):
      return True

   # Check for af specific standard options
   if ( af == 'ipv4' and
        ( optionsConfig.tftpServerOption66 or optionsConfig.tftpServerOption150 or
          optionsConfig.defaultGateway != '0.0.0.0' ) ):
      return True
   return False

# Returns the inactive reason for the client class
def getClientClassInactiveReason( clientClassConfig, af ):
   noMatchCriteria = not hasMatchCriteria( clientClassConfig, af )
   noAssignments = not hasAssignmentOptions( clientClassConfig, af )
   clientClassConfigType = clientClassConfigTypes.get( type( clientClassConfig ) )
   if clientClassConfigType == 'range':
      noAssignedIp = clientClassConfig.assignedIp == clientClassConfig.ipAddrDefault
      noAssignments = noAssignments and noAssignedIp
   if noMatchCriteria and noAssignments:
      return "no match criteria and no assignments"
   if noMatchCriteria:
      return "no match criteria"
   if noAssignments:
      return "no assignments"
   return ""

class Lease:
   def __init__( self, leaseData, endTime, cltt ):
      self.leaseData = leaseData
      self.ipAddress = None
      self.end = endTime
      self.lastTransaction = cltt

   def cltt( self ):
      return self.lastTransaction

   def endTime( self ):
      return self.end

   def ip( self ):
      return self.ipAddress

   def mac( self ):
      # hw-address is optional for IPv6 - Client may just have a (non-MAC-baseD) DUID
      return self.leaseData.get( 'hw-address', None )

   def __str__( self ):
      return "<Lease {}: E:{} L:{} M:{}>".format(
         self.ip(),
         self.endTime(),
         self.cltt(),
         self.mac() )

   def __repr__( self ):
      return str( self )

class Ipv4Lease( Lease ):
   def __init__( self, leaseData, endTime, cltt, circuitIds, remoteIds,
                opt82UnknownSubOpts ):
      Lease.__init__( self, leaseData, endTime, cltt )
      self.ipAddress = Arnet.IpAddr( leaseData[ 'ip-address' ] )
      self.circuitIds = circuitIds
      self.remoteIds = remoteIds
      self.opt82UnknownSubOpts = opt82UnknownSubOpts

class Ipv6Lease( Lease ):
   def __init__( self, leaseData, endTime, cltt ):
      Lease.__init__( self, leaseData, endTime, cltt )
      self.ipAddress = Arnet.Ip6Addr( leaseData[ 'ip-address' ] )

class Subnet:
   def __init__( self, subnet, ipVersion ):
      self.subnet = subnet
      self.leases = {}
      self.ipVersion = ipVersion

   def addLease( self, leaseData, end, cltt, circuitIds=None, remoteIds=None,
                 opt82UnknownSubOpts=None ):
      if self.ipVersion == 4:
         lease = Ipv4Lease( leaseData, end, cltt, circuitIds, remoteIds,
                            opt82UnknownSubOpts )
      else:
         lease = Ipv6Lease( leaseData, end, cltt )
      # Overrwrite if it exist, since the later one in the file is newer
      self.leases[ lease.ip() ] = lease

   def numActiveLeases( self ):
      return len( self.leases )

   def lease( self, leaseIp ):
      return self.leases[ leaseIp ]

class KeaDhcpLeaseData:
   def __init__( self, subnets, ipVersion=4, vrf=DEFAULT_VRF, keaLimit=1024,
                       filterFunction=None ):
      self.subnetData = {}
      self.subnets = subnets
      self.ipVersion = ipVersion
      self.vrf = vrf
      self.keaLimit = keaLimit
      self.error = False
      try:
         self._parseData( filterFunction=filterFunction )
      except OSError as e:
         # ENOENT when control socket doesn't exist
         # EPIPE when server closes on us
         if e.errno not in [ errno.ENOENT, errno.EPIPE ]:
            qt8( "DhcpServer: Socket had unexpected error: %s, %s" %
                 ( e.errno, os.strerror( e.errno ) ) )

   def _parseOpt82( self, opt82Hex ):
      # opt82Hex will be in the form of:
      #   0xiiLLxxxxxxiiLLxxxx
      # in which ii is the sub option type, LL is the length of the suboption and xx
      # is the contents of the suboption.
      def _parseOpt82SubOption( offset ):
         '''
         Parses a suboption of option 82
         '''
         typeCode = int( opt82Hex[ offset : offset + 2 ], 16 )
         offset += 2
         subOptLen = int( opt82Hex[ offset : offset + 2 ], 16 )
         offset += 2
         if subOptLen == 0:
            return offset, typeCode, ''
         if offset + subOptLen * 2 > len( opt82Hex ):
            raise IndexError
         value = '0x' + opt82Hex[ offset : offset + subOptLen * 2 ]
         offset += subOptLen * 2
         return offset, typeCode, value

      if len( opt82Hex ) % 2:
         # Odd length option 82. This should not be possible because the length
         # cannot be fractional and there cannot be half a byte.
         return None, None, None

      circuitIds = None
      remoteIds = None
      unknownSubOpts = None
      # Start offset as 2 because option 82 data is stored as 0x... in kea
      offset = 2
      while offset < len( opt82Hex ):
         try:
            offset, typeCode, value = _parseOpt82SubOption( offset )
         except ( ValueError, IndexError ) as _:
            # Kea should've discarded any invalid formatted option 82 suboptions, but
            # it's possible that an external program or user may have modified the
            # leases file. In that case, just ignore it.
            return None, None, None
         if typeCode == 1:
            if not circuitIds:
               circuitIds = []
            circuitIds.append( value )
         elif typeCode == 2:
            if not remoteIds:
               remoteIds = []
            remoteIds.append( value )
         else:
            if not unknownSubOpts:
               unknownSubOpts = []
            unknownSubOpts.append( ( typeCode, value ) )
      return circuitIds, remoteIds, unknownSubOpts

   def _parseData( self, filterFunction=None ):
      # Example lease output
      # { u'text': u'1 IPv4 lease(s) found.',
      #   u'arguments': { u'count': 1,
      #                   u'leases': [ { u'fqdn-rev': False,
      #                                  u'state': 0,
      #                                  u'user-context': {
      #                                     u'ISC':
      #                                     { u'relay-agent-info':
      #                                       u'0x0101aa0201bb'} },
      #                                  u'ip-address':
      #                                  u'192.0\.2.1',
      #                                  u'cltt': 1554224535,
      #                                  u'valid-lft': 3600,
      #                                  u'hw-address':
      #                                  u'ba:25:91:1e:3f:1e',
      #                                  u'hostname': u'',
      #                                  u'fqdn-fwd': False,
      #                                  u'subnet-i\d': 1 ]
      #                 },
      #   u'result': 0 }
      now = calendar.timegm( time.gmtime() )
      t8( "GM now", now )
      for leases in leaseDataIter( self.ipVersion, self.vrf, self.keaLimit,
                                   filterFunction=filterFunction ):
         if leases == { "result": -1 }:
            # keactrl error
            self.error = True
            return
         for lease in leases[ "arguments" ][ "leases" ]:
            # Only care about active leases
            # const uint32_t Lease::STATE_DEFAULT = 0x0;
            # const uint32_t Lease::STATE_DECLINED = 0x1;
            # const uint32_t Lease::STATE_EXPIRED_RECLAIMED = 0x2;
            if lease[ "state" ] != 0:
               continue
            cltt = float( lease[ "cltt" ] )
            end = cltt + float( lease[ "valid-lft" ] )
            # Skip if expired
            if end < now:
               continue
            subnet = self.subnets.get( lease[ "subnet-id" ],
                                       unknownSubnets[ self.ipVersion ] )
            try:
               subnetObj = self.subnetData[ subnet ]
            except KeyError:
               subnetObj = Subnet( subnet, self.ipVersion )
               self.subnetData[ subnet ] = subnetObj
            userContext = lease.get( "user-context" )
            circuitIds = None
            remoteIds = None
            unknownSubOpts = None
            if userContext:
               isc = userContext.get( "ISC" )
               if isc:
                  opt82Hex = isc.get( "relay-agent-info" )
                  if opt82Hex:
                     circuitIds, remoteIds, unknownSubOpts = self._parseOpt82(
                           opt82Hex )
            subnetObj.addLease( lease, end, cltt, circuitIds=circuitIds,
                                remoteIds=remoteIds,
                                opt82UnknownSubOpts=unknownSubOpts )

   def subnet( self, subnet ):
      return self.subnetData.get( subnet )

   def totalActiveLeases( self ):
      return sum( data.numActiveLeases() for data in self.subnetData.values() )

_serverPathMapping = {
   "dhcp4": keaControlSock.format( version="4", vrf=DEFAULT_VRF ),
}

class KeaDhcpSocket:
   def __init__( self, ipVersion, vrf, bufSize=4096 ):
      self.bufSize = bufSize
      path = keaControlSock.format( version=ipVersion, vrf=vrf )
      self.sock = socket.socket( socket.AF_UNIX, socket.SOCK_STREAM )
      self.sock.connect( path )

   def runCmd( self, cmdData ):
      try:
         return self._runCmd( cmdData )
      finally:
         # The dhcp server closes the connection after every command, so we should
         # too
         self.sock.close()

   def _runCmd( self, cmdData ):
      self.sock.sendall( json.dumps( cmdData ).encode() )
      resultData = []

      # The server will close the connection, so then recv won't block anymore
      data = self.sock.recv( self.bufSize )
      while data:
         resultData.append( data )
         data = self.sock.recv( self.bufSize )

      if resultData:
         return json.loads( b"".join( resultData ) )
      else:
         return { "result": -1 }

class KeaDhcpCommandError( Exception ):
   pass

def _runKeaDhcpCmdImpl( ipVersion, vrf, cmdData ):
   sock = KeaDhcpSocket( ipVersion, vrf )
   result = sock.runCmd( cmdData )
   # Four result codes possible:
   # 0 No error, results found
   # 1 General error
   # 2 Command not supported
   # 3 No error, no results found
   # -1 Nothing returned from socket
   if result[ "result" ] in [ 1, 2 ]:
      t8( 'KeaDhcpCommandError result:', result[ 'result' ] )
      raise KeaDhcpCommandError( "Command failed with {}".format(
         result[ "result" ] ) )
   return result

def runKeaDhcpCmd( ipVersion, vrf, cmdData ):
   # Retry 5 (arbitrary number of retries) times in case Kea died right after we
   # established the socket.
   for _ in range( 5 ):
      returnCode = _runKeaDhcpCmdImpl( ipVersion, vrf, cmdData )
      if returnCode != { "result": -1 }:
         return returnCode
      else:
         t8( "Kea command failed to run" )
   return { "result": -1 }

def configReloadCmdData():
   return {
      "command": "config-reload"
   }

def _leaseDataIterData( startFrom, limit, ipVersion='4' ):
   return {
      "command": f"lease{ipVersion}-get-page",
      "arguments": {
         "from": startFrom,
         "limit": limit
      }
   }

def clearLeaseCmdData( leaseIpAddress, ipVersion ):
   return {
      "command": f"lease{ipVersion}-del",
      "arguments": {
         "ip-address": leaseIpAddress
      }
   }

def leaseDataIter( ipVersion, vrf, limit=1024, filterFunction=None ):
   # Because we don't currently have a way to mock keactrl responses (BUG698342), our
   # CLI show tests will set the environment variables, DHCPSERVERCLISHOWERRORTEST4
   # and DHCPSERVERCLISHOWERRORTEST6, to signal this function to 'return { "result":
   # -1 }', the return code runKeaDhcpCmd() will return if keactrl did not send back
   # a response. These environment variables should not be set outside of testing.
   if ( ( os.environ.get( 'DHCPSERVERCLISHOWERRORTEST4' ) == vrf and
          ipVersion == 4 ) or
        ( os.environ.get( 'DHCPSERVERCLISHOWERRORTEST6' ) == vrf and
          ipVersion == 6 ) ):
      yield { "result": -1 }
      return
   cmdData = _leaseDataIterData( "start", limit, ipVersion )
   result = runKeaDhcpCmd( ipVersion, vrf, cmdData )
   if result == { "result": -1 }:
      yield result
      return
   while True:
      leaseCount = result[ "arguments" ][ "count" ]
      leases = result[ "arguments" ][ "leases" ]

      # filter leases and update the result
      if filterFunction:
         fLeases = [ l for l in leases if filterFunction( l ) ]
         result[ "arguments" ][ "count" ] = len( fLeases )
         result[ "arguments" ][ "leases" ] = fLeases

      t8( result )
      yield result
      if leaseCount < limit:
         # No more leases
         break
      # Start off from where we left off
      lastIp = result[ "arguments" ][ "leases" ][ -1 ][ "ip-address" ]
      cmdData = _leaseDataIterData( lastIp, limit, ipVersion )
      result = runKeaDhcpCmd( ipVersion, vrf, cmdData )
      if result == { "result": -1 }:
         yield result
         return
