# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import sys
import os
import re
import shutil
import time

import Logging
import Tac # pylint: disable-msg=unused-import
import Swag
import SwagBoot
import Cell

SYSDB_STARTUP_CONFIG_PARSE_ERROR = Logging.LogHandle(
              "SYSDB_STARTUP_CONFIG_PARSE_ERROR",
              severity=Logging.logError,
              fmt="Errors encountered in parsing the startup-config",
              explanation="One or more errors were encountered while parsing the"
                          " startup-config due to invalid, incomplete and/or "
                          "unavailable commands.",
              recommendedAction="Please run 'show startup-config errors' to look "
                                "at the exact error(s) and fix them in the "
                                "startup-config." )

SYSDB_STARTUP_CONFIG_FROM_BOOT_IMAGE = Logging.LogHandle(
              "SYSDB_STARTUP_CONFIG_FROM_BOOT_IMAGE",
              severity=Logging.logInfo,
              fmt="Local startup-config was overwritten by boot image %s",
              explanation="The startup-config file packaged inside the boot"
                          " image was written to /mnt/flash/startup-config."
                          " In order to ensure the integrity of that signed"
                          " config, any untrusted files in /mnt/flash"
                          " were deleted.",
              recommendedAction=Logging.NO_ACTION_REQUIRED )

startupConfigOutputPath = '/var/log/startup-config-output'
fragmentConfigOutputPath = '/var/log/fragment-configs-output'

def getPrefdl():
   fsRoot = os.environ.get( 'FILESYSTEM_ROOT', '/' )
   try:
      with open( os.path.join( fsRoot, 'etc', 'prefdl' ), "rb" ) as f:
         return f.read()
   except OSError:
      return b""

# Get the name of the path of the boot image from boot-config
# boot-config contains SWI=flash:/EOS.swi, what this returns is /mnt/flash/EOS.swi
def getBootImagePathname( ):

   # Find the path for the SWI filename from boot-config filename,
   # we need to copy .boot-image.swi to it
   bootConfig = '/mnt/flash/boot-config'
   if os.path.isfile( bootConfig ):
      with open( bootConfig ) as f:
         config = f.read()
         swiNameRe = re.compile( '^SWI=(.*)$', flags=re.IGNORECASE|re.MULTILINE )
         reMatch = re.search( swiNameRe, config )
         if reMatch:
            swiName = reMatch.group( 1 )

            # Build the bash pathname, '/mnt/flash/EOS.swi'
            (dev, fname) = swiName.split( ':' )
            swiPath = "/mnt/" + dev + "/" + fname
            return swiPath
   return None

# If this is a dut with small flash device, we need to copy
# the /mnt/flash/.boot-image.swi to /mnt/flash/EOS.swi
def restoreASUBootImage( ):

   # Check if this EOS.swi image needs to be copied back, if not, just return
   cmdLineFilename = '/proc/cmdline'
   cmdFile = open( cmdLineFilename ) # pylint: disable=consider-using-with
   m = re.search( r"arista\.doDelayedSwiCopyBack", cmdFile.read() )
   cmdFile.close()

   if not m:
      return

   # Get the bash version of the pathname
   swiPath = getBootImagePathname()
   if not swiPath:
      return

   # Initiate a copy at low prio in the background. Do it in a separate process so
   # that it doesn't affect the ASU timelines.
   forkResult = os.fork()
   if not forkResult:
      # Start the background copy after 5 minutes. All the ASU stages should have
      # completed by then. Record the pid so that we can stop this if a fatal error
      # occurs later on.
      f = open( '/var/imageCopyPidFile', 'w' ) # pylint: disable=consider-using-with
      f.write( str( os.getpid() ) )
      f.close()
      time.sleep( 300.0 )
      shutil.copyfile( '/mnt/flash/.boot-image.swi', '/mnt/flash/.boot-copy.swi' )
      shutil.move( '/mnt/flash/.boot-copy.swi', swiPath )
      os.remove( '/var/imageCopyPidFile' )
      os._exit( 0 ) # pylint: disable-msg=W0212

def moveLauncherProcFile( src ):
   if not os.access( src, os.F_OK | os.R_OK ):
      sys.exit( f"{src} does not exist, or is not accessible" )

   shutil.copy( src, "/etc/ProcMgr.d/inst/Launcher" )

def setUpProcLauncher():
   # for now this is only used in Swag mode -- eventually, ProcLauncher will be
   # started this way across EOS

   if Swag.memberId() is not None:
      instanceId = Swag.memberId()
   else:
      instanceId = Cell.cellId()

   ProcLauncherBaseConfigPath = "/etc/ProcMgr.d/inst/ProcLauncher~"
   ProcLauncherActualConfigPath = "/etc/ProcMgr.d/inst/ProcLauncher"
   if not os.access( ProcLauncherBaseConfigPath, os.F_OK | os.R_OK ):
      sys.exit(
         f"{ProcLauncherBaseConfigPath} does not exist, or is not accessible"
      )

   with open( ProcLauncherBaseConfigPath, 'r' ) as file:
      filedata = file.read()

   filedata = filedata.replace( '%smid', str( instanceId ) )

   # Write the actual ProcLauncher path
   with open( ProcLauncherActualConfigPath, 'w' ) as file:
      file.write( filedata )

def startStage2Agents():
   with open( "/etc/celltype" ) as f:
      cellType = f.read()
   swagRole = Swag.role()

   agentsToStart = []
   if Swag.swagMode():
      # Prepare to start ProcLauncher and Launcher in a SWAG
      setUpProcLauncher()
      agentsToStart.append( "ProcLauncher" )
      moveLauncherProcFile( "/etc/ProcMgr.d/inst/LauncherSwag" )
   else:
      # Use the standalone Launcher with no ProcLauncher outside of a SWAG
      moveLauncherProcFile( "/etc/ProcMgr.d/inst/LauncherNonSwag" )

   if swagRole == "worker":
      agentsToStart += SwagBoot.stage2WorkerAgents()
   elif cellType in ( "supervisor", "fixed", "generic" ):
      agentsToStart += [ "Sysdb", "Launcher", "Fru", "StageMgr", "ConfigAgent" ]
      if Cell.electionMgrSupported():
         agentsToStart.append( "ElectionMgr" )
   else:
      print( f"Unknown cell/swag type.  No agents started by {__file__}" )
      return

   print( f"Starting agents with {swagRole=} {cellType=}")
   print( f"{agentsToStart=}")
   for agentName in agentsToStart:
      Tac.run( [ sys.executable, "/usr/bin/chkagent", "--add", agentName ],
               stdout=Tac.CAPTURE )

   # Restart ProcMgr, so it adopts new agent config files
   Tac.run( "/bin/systemctl reload ProcMgr.service".split(), stdout=Tac.CAPTURE )
