# Copyright (c) 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-with

# This library encapsulates some things about finding out the version
# of hardware and software.

import os
import errno
import six

import ImageVars
import Tac

SwiVersion = Tac.Type( "EosUtils::SwiVersion" )

# For breadth testing, control this by setting os.environ[ "VERSION_ETC_DIR" ].
def etcDir():
   return SwiVersion.etcDir()

# Constants for image format strings

# Latest single-squash format version (i.e. legacy EOS.swi)
swiImageFormatMajorNo = 1
# pylint: disable-next=consider-using-f-string
swiLatestImageFormatVersion = "%d.0" % swiImageFormatMajorNo

# Initial multi-squash format version (SWI modular)
swimLatestImageFormatMajorNo = 2
# pylint: disable-next=consider-using-f-string
swimLatestImageFormatVersion = "%d.0" % swimLatestImageFormatMajorNo

# Optimized multi-squash format version (SWI modular)
swimOptimizedImageFormatMajorNo = 3
# pylint: disable-next=consider-using-f-string
swimOptimizedImageFormatVersion = "%d.0" % swimOptimizedImageFormatMajorNo

# Constants for flavor strings
swiFlavorDefault = ImageVars.SwimFlavor.DEFAULT.value
swiFlavorDPE = ImageVars.SwimFlavor.DPE.value
swiFlavorCloud = "cloud"
swiFlavorPdp = "PDP"
swiFlavorDPEContainer = "DPE-CTNR"

# Header file stored as one of the first files in the SWI
swimHdrFile = 'swimSqshMap'
# New version of optimization/sqsh header file
# stored as one of the first files in the SWI
swimHdrFileV2 = 'swimSqshMapV2'
# Extension header file mapping extension name to extension files
extensionHdrFile = 'extension-data'

# Map of SID to optimization name
# Contains SIDs that use non-Default optimizations
sidToOptimizationFile = 'swimSid2Optimization'

# Constants of swim variant/flavor strings
swimUSDefault = swiFlavorDefault
swimUSDPE = swiFlavorDPE

# Full SWIM Version files installed in the meta image
fullSwimVersions = [
   swimUSDefault,
   swimUSDPE,
]

# Names of rootfs sqshes in SWIM images that have full platform support.
fullSwimFlavors = [
   swimUSDefault,
   swimUSDPE,
   ImageVars.SwimOptimizations.Default,
   ImageVars.SwimOptimizations.Dpe,
]

def swiVersion( filename ):
   '''Returns a VersionInfo for the .swi file given in argument.  The file is
   assumed to be a well-formed ZIP archive.'''
   import zipfile # pylint: disable=import-outside-toplevel
   with zipfile.ZipFile( filename ) as archive:
      # Previously this code was just doing archive.read()
      # but this is unsafe to do because of Python bug 13133
      # which causes a file descriptor leak if the file does
      # not exist.
      try:
         if archive.getinfo( 'version' ):  # If the file exists.
            version = archive.read( 'version' )
      except KeyError:  # File doesn't exist.
         version = ''   # Just pretend it's empty.
   return VersionInfo( None, versionFile=version, arch="" )

class VersionInfo:
   '''This class fetches and aggregates version information for EOS,
   both hardware and software.  Construct this object and pass a
   sysdbRoot (you can pass None, but I may not be able to get all
   fields if you do).  I will then have the following fields:

      modelName():        Model name, such as 'DCS-7148S', or None if not known.

      hardwareRev():      Hardware revision, such as '02.02', or None if not known.

      deviations():       List of deviations as strings, such as ['D000782'],
                          or [] if no deviations.

      serialNum():        Manufacturing serial number, such as 'JFL08290034', or
                          None if not known.

      version():          EOS software version, such as "4.0.0" if blessed, or
                          "4.0.0-121957.2009.intfRange (engineering build)" if not.
                          None if not known.

      internalVersion():  EOS full version, such as "4.0.0-121957.EOS-4.0.0", or
                          None if not known.

      architecture():     CPU architecture, such as 'i386', or None if not known.

      internalBuildId():  Fingerprint like '633eed1e-8f7e-4cec-8af2-16bf267f64aa'.
                          None if not known.

      imageFormatVersion(): EOS image format version, such as '02.01'.
                          None if not known.

      fanDirection():     Returns either 'Forward', 'Reverse' or 'Unknown'
                          (Unknown could signify a mix of forward and reverse fans,
                          no/unrecognized fans, or a modular chassis)

      modelNameExtended():   Returns the model name with the
                             fan direction, if applicable

   Other optional arguments:

      versionFile:        String: If given, will be used as the contents of the
                          version file instead of reading /etc/swi-version.

      arch:               String: If given, will be used as the name of the
                          architecture instead of reading /etc/arch.
   '''

   def __init__( self, sysdbRoot, versionFile=None, arch=None ):
      mfgName = None
      modelName = None
      hardwareRev = None
      serialNum = None
      deviations = []
      fanDirection = None
      modelNameExtended = None
      try:
         entityMibRoot = sysdbRoot and sysdbRoot[ 'hardware' ][ 'entmib' ].root
      except KeyError:
         pass
      else:
         if getattr( entityMibRoot, 'initStatus', None ) == 'ok':
            mfgName = entityMibRoot.mfgName
            modelName = entityMibRoot.modelName
            hardwareRev = entityMibRoot.hardwareRev
            deviations = list( entityMibRoot.deviation )
            serialNum = entityMibRoot.serialNum
            fanDirection = entityMibRoot.fanDirection
            modelNameExtended = entityMibRoot.modelNameExtended

      self.mfgName_ = mfgName
      self.modelName_ = modelName
      self.hardwareRev_ = hardwareRev
      self.deviations_ = deviations
      self.serialNum_ = serialNum
      self.fanDirection_ = fanDirection
      self.modelNameExtended_ = modelNameExtended

      version = six.ensure_str( versionFile ) if versionFile else ""
      self.swiVersion_ = SwiVersion.parse( version )
      try:
         # Note that /etc/arch and /etc/swi-version are generally not present
         # in workspaces, because they are set up by the EosImage makefile
         # (and the "swi installrootfs" tool in EosUtils).
         arch = arch or self.swiVersion_.swiArch or \
                open( os.path.join( etcDir(), 'arch' ) ).read().strip()
      except OSError as e:
         if e.errno != errno.ENOENT:
            raise

      self.architecture_ = arch
      self.swiMaxHwEpoch_ = int( self.swiVersion_.swiMaxHwEpoch )
      self.isCEos_ = False
      cEOSFile = os.path.join( etcDir(), "cEOS-release" )
      if os.path.exists( cEOSFile ):
         self.isCEos_ = True

   def mfgName( self ):
      return self.mfgName_
   def modelName( self ):
      return self.modelName_
   def hardwareRev( self ):
      return self.hardwareRev_
   def deviations( self ):
      return self.deviations_
   def serialNum( self ):
      return self.serialNum_
   def version( self ):
      return self.swiVersion_.version or None
   def internalVersion( self ):
      return self.swiVersion_.internalVersion or None
   def architecture( self ):
      return self.architecture_
   def internalBuildId( self ):
      return self.swiVersion_.internalBuildId or None
   def isCEos( self ):
      return self.isCEos_
   def fanDirection( self ):
      return self.fanDirection_
   def modelNameExtended( self ):
      return self.modelNameExtended_
   def swiArch( self ):
      return self.swiVersion_.swiArch or None
   def swiVariant( self ):
      return self.swiVersion_.swiVariant or None
   def swiFlavor( self ):
      return self.swiVersion_.swiFlavor or None
   def swiOptimization( self ):
      return self.swiVersion_.swiOptimization or None
   def imageFormatVersion( self ):
      return self.swiVersion_.imageFormatVersion or None
   def swiMaxHwEpoch( self ):
      return self.swiMaxHwEpoch_
   def isIntlVersion( self ):
      return self.swiVersion_.isIntlVersion
   def isCloudVersion( self ):
      return self.swiVersion_.isCloudVersion
   def isDPEVersion( self ):
      return self.swiVersion_.isDPEVersion
   def isPdpVersion( self ):
      return self.swiVersion_.isPdpVersion
   def isContainerVersion( self ):
      return self.swiVersion_.isContainerVersion
   def isBlessed( self ):
      return self.swiVersion_.isBlessed
